/*
 * Copyright (c) 2009, 2010, 2011, 2012, 2013, 2014, 2016 Nicira, Inc.
 * Copyright (c) 2019 Samsung Electronics Co.,Ltd.
 * Copyright (c) 2023 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef NETDEV_FLOW_API_PROVIDER_H
#define NETDEV_FLOW_API_PROVIDER_H 1

#include "conntrack-offload.h"
#include "flow.h"
#include "netdev-offload.h"
#include "openvswitch/netdev.h"
#include "openvswitch/types.h"
#include "packets.h"

#ifdef  __cplusplus
extern "C" {
#endif

struct netdev_flow_api {
    char *type;
    /* Flush all offloaded flows from a netdev.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*flow_flush)(struct netdev *);

    /* Flow dumping interface.
     *
     * This is the back-end for the flow dumping interface described in
     * dpif.h.  Please read the comments there first, because this code
     * closely follows it.
     *
     * On success returns 0 and allocates data, on failure returns
     * positive errno. */
    int (*flow_dump_create)(struct netdev *, struct netdev_flow_dump **dump,
                            bool terse);
    int (*flow_dump_destroy)(struct netdev_flow_dump *);

    /* Returns true if there are more flows to dump.
     * 'rbuffer' is used as a temporary buffer and needs to be pre allocated
     * by the caller.  While there are more flows the same 'rbuffer'
     * should be provided. 'wbuffer' is used to store dumped actions and needs
     * to be pre allocated by the caller. */
    bool (*flow_dump_next)(struct netdev_flow_dump *, struct match *,
                           struct nlattr **actions,
                           struct dpif_flow_stats *stats,
                           struct dpif_flow_attrs *attrs, ovs_u128 *ufid,
                           struct ofpbuf *rbuffer, struct ofpbuf *wbuffer);

    /* Offload the given flow on netdev.
     * To modify a flow, use the same ufid.
     * 'actions' are in netlink format, as with struct dpif_flow_put.
     * 'info' is extra info needed to offload the flow.
     * 'stats' is populated according to the rules set out in the description
     * above 'struct dpif_flow_put'.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*flow_put)(struct netdev *, struct match *, struct nlattr *actions,
                    size_t actions_len, const ovs_u128 *ufid,
                    struct offload_info *info, struct dpif_flow_stats *);

    /* Queries a flow specified by ufid on netdev.
     * Fills output buffer as 'wbuffer' in flow_dump_next, which
     * needs to be be pre allocated.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*flow_get)(struct netdev *, struct match *, struct nlattr **actions,
                    const ovs_u128 *ufid, struct dpif_flow_stats *,
                    struct dpif_flow_attrs *, struct ofpbuf *wbuffer,
                    long long now);

    /* Delete a flow specified by ufid from netdev.
     * 'stats' is populated according to the rules set out in the description
     * above 'struct dpif_flow_del'.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*flow_del)(struct netdev *, const ovs_u128 *ufid,
                    struct dpif_flow_stats *);

    /* Get the number of offloads created on netdev.
     * 'n_offloads' is an array of counters, one per offload thread.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*flow_get_n_offloads)(struct netdev *, uint64_t *n_offloads);

    /* Get the number of flows offloaded to netdev.
     * 'n_flows' is an array of counters, one per offload thread.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*flow_get_n_flows)(struct netdev *, uint64_t *n_flows);

    /* Recover the packet state (contents and data) for continued processing
     * in software.
     * Return 0 if successful, otherwise returns a positive errno value and
     * takes ownership of a packet if errno != EOPNOTSUPP. */
    int (*hw_miss_packet_recover)(struct netdev *, struct dp_packet *,
                                  uint8_t *, struct dpif_sflow_attr *);

    /* Initializies the netdev flow api.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*init_flow_api)(struct netdev *);

    /* Uninitializes the netdev flow api. */
    void (*uninit_flow_api)(struct netdev *);

    /* Read the provider stats.
     * 'stats' is a pointer of at least 'MAX_OFFLOAD_THREAD_NB' elements.
     * Returns 0 on success, positive errno otherwise. */
    int (*get_stats)(struct netdev *netdev,
                     struct netdev_offload_stats stats[MAX_OFFLOAD_THREAD_NB]);

    /* Queries a CT counter object. */
    int (*ct_counter_query)(struct netdev *, uintptr_t, long long, long long,
                            struct dpif_flow_stats *);

    /* Offload a connection on a netdev.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*conn_add)(struct netdev *, struct ct_flow_offload_item *ct_offload);

    /* Delete a connection offload from a netdev.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*conn_del)(struct netdev *, struct ct_flow_offload_item *ct_offload);

    /* Queries a connection offloaded on a netdev.
     * Return 0 if successful, otherwise returns a positive errno value. */
    int (*conn_stats)(struct netdev *, struct ct_flow_offload_item *ct_offload,
                      struct dpif_flow_stats *, struct dpif_flow_attrs *,
                      long long int);

    /* This function must be called periodically to maintain internal
     * netdev offload structures.
     * If 'quiescing' is true, the caller signals to the provider that
     * it might be the last upkeep before an arbitrary waiting period.
     */
    void (*upkeep)(struct netdev *netdev, bool quiescing);

    /* Get the hash value as if calculated by the HW for the provided packet. */
    int (*packet_hw_hash)(struct netdev *, struct dp_packet *, uint32_t *);
};

int netdev_register_flow_api_provider(const struct netdev_flow_api *);
int netdev_unregister_flow_api_provider(const char *type);
bool netdev_flow_api_equals(const struct netdev *, const struct netdev *);

#ifdef __linux__
extern const struct netdev_flow_api netdev_offload_tc;
#endif

#ifdef DPDK_NETDEV
extern const struct netdev_flow_api netdev_offload_dpdk;
#endif

#ifdef  __cplusplus
}
#endif

#endif /* NETDEV_FLOW_API_PROVIDER_H */
