/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2022-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#include <config.h>

#include <infiniband/verbs.h>
#include <unistd.h>

#include <rte_dev.h>
#include <rte_ethdev.h>
#include <rte_malloc.h>
#include <rte_mtr.h>
#include <rte_pci.h>

#include <doca_bitfield.h>
#include <doca_dev.h>
#include <doca_dpdk.h>
#include <doca_flow.h>
#include <doca_flow_ct.h>
#include <doca_flow_definitions.h>
#include <doca_log.h>
#include <doca_rdma_bridge.h>
#include <doca_version.h>

#include "coverage.h"
#include "conntrack-offload.h"
#include "dp-packet.h"
#include "dpdk.h"
#include "dpdk-offload-doca.h"
#include "dpif-doca.h"
#include "netdev-doca.h"
#include "netdev-offload.h"
#include "netdev-offload-doca.h"
#include "netdev-offload-provider.h"
#include "dpdk-offload-doca.h"
#include "openvswitch/dynamic-string.h"
#include "openvswitch/vlog.h"
#include "ovs-doca.h"
#include "refmap.h"
#include "unixctl.h"
#include "util.h"

VLOG_DEFINE_THIS_MODULE(ovs_doca);

COVERAGE_DEFINE_ERR(ovs_doca_no_mark);
COVERAGE_DEFINE_ERR(ovs_doca_invalid_classify_port);

#define OVS_DOCA_CONGESTION_THRESHOLD_DEFAULT 80
#define OVS_DOCA_CONGESTION_THRESHOLD_MIN (DOCA_RESIZED_PIPE_CONGESTION_LEVEL + 1)
#define OVS_DOCA_CONGESTION_THRESHOLD_MAX 90
#define MAX_PORT_STR_LEN 128
#define MAX_DOCA_CT_ACTIONS 16000000
/* Value defined privately in DOCA as CT_MAX_NUM_RULES_PER_MATCHER. */
#define DOCA_CT_MAX_NUM_RULES_PER_MATCHER 0x400000
#define MAX_DOCA_CT_CONNECTIONS DOCA_CT_MAX_NUM_RULES_PER_MATCHER

/* Take RSS distribution imbalance into account. */
#define OVS_DOCA_CT_ENTRIES_RSS_FACTOR 1.05

OVS_ASSERT_PACKED(struct ovs_doca_esw_key,
    struct rte_pci_addr rte_pci;
);

struct ovs_doca_esw_ctx_arg {
    struct ovs_doca_esw_key *esw_key;
    struct ovs_doca_netdev_data *dev_data;
};

/* Indicates successful initialization of DOCA. */
static atomic_bool doca_enabled = false;
static atomic_bool doca_initialized = false;
static atomic_bool doca_offload_trace_enabled = false;
static FILE *log_stream = NULL;       /* Stream for DOCA log redirection */
static struct doca_log_backend *ovs_doca_log = NULL;
static struct refmap *ovs_doca_esw_rfm;
struct ovs_mutex mgmt_queue_lock = OVS_MUTEX_INITIALIZER;
static enum doca_flow_port_operation_state init_op_state = DOCA_FLOW_PORT_OPERATION_STATE_ACTIVE;

unsigned int ovs_doca_max_eswitch_num = OVS_DOCA_DEFAULT_MAX_ESW;
unsigned int doca_congestion_threshold = OVS_DOCA_CONGESTION_THRESHOLD_DEFAULT;
/* Size of control pipes. If zero, DOCA uses its default value. */
uint32_t ctl_pipe_size = 1024;
/* Size 0 means to set as ctl_pipe_size. */
uint32_t ctl_pipe_infra_size = 0;
static uint32_t per_port_meter_rate = 0;
static uint32_t per_core_meter_rate = DEFAULT_PER_CORE_METER_RATE;
static unsigned int ovs_doca_max_megaflows_counters;
static enum ovs_doca_gpr_mode gpr_mode = OVS_DOCA_GPR_MODE_DISABLED;
atomic_bool ovs_doca_eswitch_active_ids[OVS_DOCA_MAX_SUPPORTED_ESW];
static unsigned int n_offload_queues;

#define OVS_DOCA_MAX_SW_METER_COUNTERS RTE_MAX_ETHPORTS
#define OVS_DOCA_SLOWPATH_COUNTERS ((OVS_DOCA_RSS_NUM_ENTRIES + 1) * RTE_MAX_ETHPORTS)
#define OVS_DOCA_MAX_SAMPLE_COUNTERS (4 * ovs_doca_max_eswitch_num_get())

static const char *
ovs_doca_stats_name(enum ovs_doca_rss_type type);

bool
ovs_doca_enabled(void)
{
    bool enabled;

    atomic_read_relaxed(&doca_enabled, &enabled);
    return enabled;
}

bool
ovs_doca_initialized(void)
{
    bool initialized;

    atomic_read_relaxed(&doca_initialized, &initialized);
    return initialized;
}

unsigned int
ovs_doca_max_eswitch_num_get(void)
{
    return ovs_doca_max_eswitch_num;
}

unsigned int
ovs_doca_n_offload_queues(void)
{
    return n_offload_queues;
}

unsigned int
ovs_doca_max_counters(void)
{
    return ovs_doca_max_megaflows_counters + OVS_DOCA_MAX_SW_METER_COUNTERS +
           OVS_DOCA_SLOWPATH_COUNTERS + OVS_DOCA_MAX_SAMPLE_COUNTERS;
}

unsigned int
ovs_doca_get_max_megaflows_counters(void)
{
    return ovs_doca_max_megaflows_counters;
}

/* Returns 'true' if 'qid' is an offload queue that
 * is expected to interact with resizable pipes. */
bool
ovs_doca_queue_use_resize(unsigned int qid)
{
    /* Invalid input or DOCA-CT queues. */
    if (qid >= ovs_doca_n_offload_queues()) {
        return false;
    }
    /* Main thread. */
    if (qid == AUX_QUEUE) {
        return false;
    }
    /* Polling threads. */
    if (qid > netdev_offload_thread_nb()) {
        return false;
    }
    /* Only remaining are hw-offload threads. */
    return true;
}

uint32_t
ovs_doca_per_port_meter_rate(void)
{
    return per_port_meter_rate;
}

uint32_t
ovs_doca_per_core_meter_rate(void)
{
    return per_core_meter_rate;
}

uint32_t
ovs_doca_gpr_mode(void)
{
    return gpr_mode;
}

bool
ovs_doca_port_rate_configured(void)
{
    return per_port_meter_rate != 0;
}

static ssize_t
ovs_doca_log_write(void *c OVS_UNUSED, const char *buf, size_t size)
{
    static struct vlog_rate_limit rl = VLOG_RATE_LIMIT_INIT(600, 600);
    static struct vlog_rate_limit dbg_rl = VLOG_RATE_LIMIT_INIT(600, 600);

    switch (doca_log_level_get_global_sdk_limit()) {
        case DOCA_LOG_LEVEL_TRACE:
            VLOG_DBG_RL(&dbg_rl, "%.*s", (int) size, buf);
            break;
        case DOCA_LOG_LEVEL_DEBUG:
            VLOG_DBG_RL(&dbg_rl, "%.*s", (int) size, buf);
            break;
        case DOCA_LOG_LEVEL_INFO:
            VLOG_INFO_RL(&rl, "%.*s", (int) size, buf);
            break;
        case DOCA_LOG_LEVEL_WARNING:
            VLOG_WARN_RL(&rl, "%.*s", (int) size, buf);
            break;
        case DOCA_LOG_LEVEL_ERROR:
            VLOG_ERR_RL(&rl, "%.*s", (int) size, buf);
            break;
        case DOCA_LOG_LEVEL_CRIT:
            VLOG_EMER("%.*s", (int) size, buf);
            break;
        default:
            OVS_NOT_REACHED();
    }

    return size;
}

static cookie_io_functions_t ovs_doca_log_func = {
    .write = ovs_doca_log_write,
};

static void
ovs_doca_unixctl_mem_stream(struct unixctl_conn *conn, int argc OVS_UNUSED,
                            const char *argv[] OVS_UNUSED, void *aux)
{
    void (*callback)(FILE *) = aux;
    char *response = NULL;
    FILE *stream;
    size_t size;

    stream = open_memstream(&response, &size);
    if (!stream) {
        response = xasprintf("Unable to open memstream: %s.",
                             ovs_strerror(errno));
        unixctl_command_reply_error(conn, response);
        goto out;
    }

    callback(stream);
    fclose(stream);
    unixctl_command_reply(conn, response);
out:
    free(response);
}

static const char * const levels[] = {
    [DOCA_LOG_LEVEL_CRIT]    = "critical",
    [DOCA_LOG_LEVEL_ERROR]   = "error",
    [DOCA_LOG_LEVEL_WARNING] = "warning",
    [DOCA_LOG_LEVEL_INFO]    = "info",
    [DOCA_LOG_LEVEL_DEBUG]   = "debug",
    [DOCA_LOG_LEVEL_TRACE]   = "trace",
};

static int
ovs_doca_parse_log_level(const char *s)
{
    int i;

    for (i = 0; i < ARRAY_SIZE(levels); ++i) {
        if (levels[i] && !strcmp(s, levels[i])) {
            return i;
        }
    }
    return -1;
}

static const char *
ovs_doca_log_level_to_str(uint32_t log_level)
{
    int i;

    for (i = 0; i < ARRAY_SIZE(levels); ++i) {
        if (i == log_level && levels[i]) {
            return levels[i];
        }
    }
    return NULL;
}

static void
ovs_doca_unixctl_log_set(struct unixctl_conn *conn, int argc,
                         const char *argv[], void *aux OVS_UNUSED)
{
    int level = DOCA_LOG_LEVEL_DEBUG;

    /* With no argument, level is set to 'debug'. */

    if (argc == 2) {
        const char *level_string;

        level_string = argv[1];
        level = ovs_doca_parse_log_level(level_string);
        if (level == -1) {
            char *err_msg;

            err_msg = xasprintf("invalid log level: '%s'", level_string);
            unixctl_command_reply_error(conn, err_msg);
            free(err_msg);
            return;
        }
    }

    doca_log_level_set_global_sdk_limit(level);
    unixctl_command_reply(conn, NULL);
}

static void
ovs_doca_log_dump(FILE *stream)
{
    uint32_t log_level;

    log_level = doca_log_level_get_global_sdk_limit();
    fprintf(stream, "DOCA log level is %s", ovs_doca_log_level_to_str(log_level));
}

static void
ovs_doca_parse_gpr_mode(const char *gpr_mode_str, enum ovs_doca_gpr_mode *req_gpr_mode)
{
    static bool should_warn = true;
    const char *gpr_mode_names[] = {
        [OVS_DOCA_GPR_MODE_DISABLED] = "disabled",
        [OVS_DOCA_GPR_MODE_REP_ONLY] = "rep-only",
        [OVS_DOCA_GPR_MODE_ALL_PORTS] = "all-ports",
    };

    *req_gpr_mode = OVS_DOCA_GPR_MODE_DISABLED;
    if (!gpr_mode_str) {
        return;
    }

    for (int i = 0; i < ARRAY_SIZE(gpr_mode_names); i++) {
        if (!strncmp(gpr_mode_str, gpr_mode_names[i], strlen(gpr_mode_names[i]))) {
            *req_gpr_mode = (enum ovs_doca_gpr_mode) i;
            should_warn = true;
            return;
        }
    }
    if (should_warn) {
        VLOG_WARN("gpr mode %s is not supported, using gpr mode disabled", gpr_mode_str);
        should_warn = false;
    }
}

static void
ovs_doca_dynamic_config(const struct smap *config)
{
    uint32_t req_port_meter_rate, req_core_meter_rate;
    enum ovs_doca_gpr_mode req_gpr_mode;
    uint32_t req_ctl_pipe_size;
    const char *gpr_mode_str;

    if (!smap_get_bool(config, "doca-init", false)) {
        return;
    }

    /* Once the user request a DOCA run, we must modify all future logic
     * (DPDK port probing) to take it into account, even if it results
     * in a failure. Once set, this value won't change. */
    if (!ovs_doca_enabled()) {
        VLOG_INFO("Using DOCA %s", ovs_doca_get_version());
        atomic_store(&doca_enabled, true);
    }

    req_ctl_pipe_size = smap_get_uint(config, "ctl-pipe-size", ctl_pipe_size);
    if (req_ctl_pipe_size != ctl_pipe_size) {
        VLOG_INFO("Changing DOCA ctl-pipe-size from %"PRIu32" to %"PRIu32,
                  ctl_pipe_size, req_ctl_pipe_size);

        ctl_pipe_size = req_ctl_pipe_size;
    } else {
        VLOG_INFO_ONCE("DOCA ctl-pipe-size is %"PRIu32, ctl_pipe_size);
    }

    if (!smap_get_bool(config, "hw-offload", false)) {
        return;
    }

    if (smap_get_bool(config, "doca-offload-trace", false) ||
        smap_get_bool(config, "dpdk-offload-trace", false)) {
        if (!ovs_doca_is_offload_trace_enabled()) {
            atomic_store_relaxed(&doca_offload_trace_enabled, true);
            VLOG_INFO("DOCA offload trace enabled.");
        }
    } else {
        atomic_store_relaxed(&doca_offload_trace_enabled, false);
    }

    gpr_mode_str = smap_get(config, "gpr-mode");
    ovs_doca_parse_gpr_mode(gpr_mode_str, &req_gpr_mode);
    if (req_gpr_mode != OVS_DOCA_GPR_MODE_DISABLED) {
        bool should_modify = false;

        req_core_meter_rate = smap_get_uint(config, "per-core-meter-rate",
                                            DEFAULT_PER_CORE_METER_RATE);
        if (req_core_meter_rate != per_core_meter_rate) {
            VLOG_INFO("Changing gpr core meter rate from %"PRIu32" to %"PRIu32, per_core_meter_rate,
                      req_core_meter_rate);
            per_core_meter_rate = req_core_meter_rate;
            should_modify = true;
        }

        req_port_meter_rate = smap_get_uint(config, "per-port-meter-rate", 0);
        if (req_port_meter_rate != per_port_meter_rate) {
            VLOG_INFO("Changing gpr port meter rate from %"PRIu32" to %"PRIu32, per_port_meter_rate,
                      req_port_meter_rate);
            per_port_meter_rate = req_port_meter_rate;
            should_modify = true;
        }

        if (should_modify) {
            doca_offload_gpr_meter_rate_modify(per_core_meter_rate,
                                               per_port_meter_rate);
        }
    }
    if (req_gpr_mode != gpr_mode && ovs_doca_initialized()) {
        doca_offload_gpr_mode_set(req_gpr_mode);
        gpr_mode = req_gpr_mode;
        VLOG_INFO("gpr mode %s", gpr_mode == OVS_DOCA_GPR_MODE_DISABLED ? "disabled" :
                  gpr_mode_str);
    }
}

static doca_error_t
ovs_doca_entries_process_sync(struct ovs_doca_netdev_data *dev_data)
    OVS_EXCLUDED(mgmt_queue_lock)
{
    struct doca_flow_port *port = doca_flow_port_switch_get(dev_data->port);
    long long int timeout_ms = time_msec() + 10 * 1000;
    struct ovs_doca_esw_ctx *ctx = dev_data->esw_ctx;
    doca_error_t ret = DOCA_SUCCESS;

    if (!ctx) {
        return ret;
    }

    ovs_doca_mgmt_queue_lock();
    while (ctx->queue.n_waiting_entries) {
        if (time_msec() > timeout_ms) {
            ovs_abort(0, "Timeout reached trying to complete the main queue: %u remaining entries",
                      ctx->queue.n_waiting_entries);
        }
        ret = doca_flow_entries_process(port, AUX_QUEUE, ENTRY_PROCESS_TIMEOUT_US, 0);
        if (DOCA_IS_ERROR(ret)) {
            break;
        }
    }
    ovs_doca_mgmt_queue_unlock();
    return ret;
}

static void
ovs_doca_pipe_uninit(struct doca_flow_pipe **ppipe)
{
    if (!*ppipe) {
        return;
    }

    doca_flow_pipe_destroy(*ppipe);
    *ppipe = NULL;
}

static void
ovs_doca_entry_uninit(struct ovs_doca_netdev_data *dev_data,
                      struct doca_flow_pipe_entry **pentry)
{
    ovs_assert(dev_data && dev_data->esw_ctx);

    if (!*pentry) {
        return;
    }

    ovs_doca_mgmt_queue_lock();
    doca_flow_pipe_remove_entry(AUX_QUEUE, DOCA_FLOW_NO_WAIT, *pentry);
    ovs_doca_mgmt_queue_unlock();
    *pentry = NULL;
    dev_data->esw_ctx->queue.n_waiting_entries++;
}

static struct doca_flow_pipe *
ovs_doca_egress_pipe_init(struct ovs_doca_netdev_data *dev_data)
{
    const char *pci = dev_data->esw_ctx->pci_addr;
    struct doca_flow_pipe *pipe = NULL;
    struct doca_flow_monitor monitor;
    struct ovs_doca_flow_match match;
    struct doca_flow_pipe_cfg *cfg;
    struct doca_flow_fwd fwd;
    char pipe_name[50];
    int ret;

    memset(&match, 0, sizeof match);
    memset(&fwd, 0, sizeof fwd);
    memset(&monitor, 0, sizeof monitor);

    /* Meta to match on is defined per entry */
    match.d.meta.pkt_meta = (OVS_FORCE doca_be32_t) UINT32_MAX;
    /* Port ID to forward to is defined per entry */
    fwd.type = DOCA_FLOW_FWD_PORT;
    fwd.port_id = UINT16_MAX;
    snprintf(pipe_name, sizeof pipe_name, "OVS_EGRESS_PIPE_%s", pci);

    monitor.counter_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;

    ret = doca_flow_pipe_cfg_create(&cfg, dev_data->port);
    if (ret) {
        VLOG_ERR("%s: Could not create doca_flow_pipe_cfg for egress", pci);
        return NULL;
    }

    if (doca_flow_pipe_cfg_set_name(cfg, pipe_name) ||
        doca_flow_pipe_cfg_set_type(cfg, DOCA_FLOW_PIPE_BASIC) ||
        doca_flow_pipe_cfg_set_domain(cfg, DOCA_FLOW_PIPE_DOMAIN_EGRESS) ||
        doca_flow_pipe_cfg_set_match(cfg, &match.d, &match.d) ||
        doca_flow_pipe_cfg_set_is_root(cfg, true) ||
        doca_pipe_cfg_allow_one_queue(cfg, AUX_QUEUE) ||
        doca_flow_pipe_cfg_set_monitor(cfg, &monitor) ||
        doca_flow_pipe_cfg_set_nr_entries(cfg, RTE_MAX_ETHPORTS)) {
        VLOG_ERR("%s: Could not set doca_flow_pipe_cfg for egress", pci);
        goto out;
    }

    ret = doca_flow_pipe_create(cfg, &fwd, NULL, &pipe);
    if (ret) {
        VLOG_ERR("%s: Failed to create egress pipe. Error %d (%s)", pci, ret,
                 doca_error_get_descr(ret));
        goto out;
    }

out:
    doca_flow_pipe_cfg_destroy(cfg);
    return pipe;
}

static void
ovs_doca_rss_uninit(struct ovs_doca_esw_ctx *esw_ctx)
{
    ovs_doca_pipe_uninit(&esw_ctx->rss_pipe);
}

static uint32_t
ovs_doca_rss_flags(enum ovs_doca_rss_type type)
{
    switch (type) {
    case OVS_DOCA_RSS_IPV4_TCP:
        return DOCA_FLOW_RSS_IPV4 | DOCA_FLOW_RSS_TCP;
    case OVS_DOCA_RSS_IPV4_UDP:
        return DOCA_FLOW_RSS_IPV4 | DOCA_FLOW_RSS_UDP;
    case OVS_DOCA_RSS_IPV4_ICMP:
        return DOCA_FLOW_RSS_IPV4;
    case OVS_DOCA_RSS_IPV4_ESP:
        return DOCA_FLOW_RSS_IPV4;
    case OVS_DOCA_RSS_IPV4_OTHER:
        return DOCA_FLOW_RSS_IPV4;
    case OVS_DOCA_RSS_IPV6_TCP:
        return DOCA_FLOW_RSS_IPV6 | DOCA_FLOW_RSS_TCP;
    case OVS_DOCA_RSS_IPV6_UDP:
        return DOCA_FLOW_RSS_IPV6 | DOCA_FLOW_RSS_UDP;
    case OVS_DOCA_RSS_IPV6_ICMP:
        return DOCA_FLOW_RSS_IPV6;
    case OVS_DOCA_RSS_IPV6_ESP:
        return DOCA_FLOW_RSS_IPV6;
    case OVS_DOCA_RSS_IPV6_OTHER:
        return DOCA_FLOW_RSS_IPV6;
    case OVS_DOCA_RSS_OTHER:
        return 0;
    }
    OVS_NOT_REACHED();
    return 0;
}

struct rss_match_type {
    enum doca_flow_l3_meta l3_type;
    enum doca_flow_l4_meta l4_type;
};

static struct rss_match_type
ovs_doca_rss_match_type(enum ovs_doca_rss_type type)
{
    switch (type) {
    case OVS_DOCA_RSS_IPV4_TCP:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV4,
            .l4_type = DOCA_FLOW_L4_META_TCP,
        };
    case OVS_DOCA_RSS_IPV4_UDP:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV4,
            .l4_type = DOCA_FLOW_L4_META_UDP,
        };
    case OVS_DOCA_RSS_IPV4_ICMP:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV4,
            .l4_type = DOCA_FLOW_L4_META_ICMP,
        };
    case OVS_DOCA_RSS_IPV4_ESP:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV4,
            .l4_type = DOCA_FLOW_L4_META_ESP,
        };
    case OVS_DOCA_RSS_IPV4_OTHER:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV4,
            .l4_type = DOCA_FLOW_L4_META_NONE,
        };
    case OVS_DOCA_RSS_IPV6_TCP:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV6,
            .l4_type = DOCA_FLOW_L4_META_TCP,
        };
    case OVS_DOCA_RSS_IPV6_UDP:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV6,
            .l4_type = DOCA_FLOW_L4_META_UDP,
        };
    case OVS_DOCA_RSS_IPV6_ICMP:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV6,
            .l4_type = DOCA_FLOW_L4_META_ICMP,
        };
    case OVS_DOCA_RSS_IPV6_ESP:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV6,
            .l4_type = DOCA_FLOW_L4_META_ESP,
        };
    case OVS_DOCA_RSS_IPV6_OTHER:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_IPV6,
            .l4_type = DOCA_FLOW_L4_META_NONE,
        };
    case OVS_DOCA_RSS_OTHER:
        return (struct rss_match_type) {
            .l3_type = DOCA_FLOW_L3_META_NONE,
            .l4_type = DOCA_FLOW_L4_META_NONE,
        };
    }
    OVS_NOT_REACHED();
    return (struct rss_match_type) {};
}

static int
ovs_doca_rss_entries_init(struct ovs_doca_netdev_data *dev_data)
{
    struct ovs_doca_esw_ctx *ctx = dev_data->esw_ctx;
    struct ovs_doca_flow_actions actions;
    uint16_t port_id = dev_data->port_id;
    struct doca_flow_pipe_entry *entry;
    struct ovs_doca_flow_match match;
    unsigned int num_of_queues;
    struct doca_flow_fwd fwd;
    int entry_ret, sync_ret;
    uint16_t *rss_queues;
    int i;

    num_of_queues = ctx->n_rxq;

    rss_queues = xcalloc(num_of_queues, sizeof *rss_queues);
    for (i = 0; i < num_of_queues; i++) {
        rss_queues[i] = i;
    }

    memset(&match, 0, sizeof match);
    memset(&actions, 0, sizeof actions);
    memset(&fwd, 0, sizeof fwd);

    fwd.type = DOCA_FLOW_FWD_RSS;
    fwd.rss.queues_array = rss_queues;
    fwd.rss.nr_queues = num_of_queues;
    fwd.rss_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;

    match.d.parser_meta.port_id = ovs_doca_port_id_pkt_meta(port_id);
    actions.mark = (OVS_FORCE doca_be32_t) DOCA_HTOBE32(ovs_doca_port_id_pkt_meta(port_id));

    for (i = 0; i < OVS_DOCA_RSS_NUM_ENTRIES; i++) {
        struct rss_match_type match_type = ovs_doca_rss_match_type(i);

        /* The configuration is of the inner since the outer is the meta-push encapsulation
         * header.
         */
        match.d.parser_meta.inner_l3_type = match_type.l3_type;
        match.d.parser_meta.inner_l4_type = match_type.l4_type;
        fwd.rss.inner_flags = ovs_doca_rss_flags(i);

        ovs_doca_mgmt_queue_lock();
        entry_ret = doca_flow_pipe_add_entry(AUX_QUEUE, ctx->rss_pipe, &match.d, 0, &actions.d,
                                             NULL, &fwd, DOCA_FLOW_NO_WAIT, &ctx->queue, &entry);
        ovs_doca_mgmt_queue_unlock();
        if (entry_ret) {
            VLOG_ERR("%s: Failed to create '%s' rss entry: Error %d (%s)", dev_data->devargs,
                     ovs_doca_stats_name(i), entry_ret, doca_error_get_descr(entry_ret));
            break;
        }
        ctx->queue.n_waiting_entries++;
        dev_data->rss_entries[i] = entry;
    }

    sync_ret = ovs_doca_entries_process_sync(dev_data);
    if (sync_ret) {
        VLOG_ERR("%s: Failed to process rss pipe entries. Error: %d (%s)", dev_data->devargs,
                 sync_ret, doca_error_get_descr(sync_ret));
    }

    free(rss_queues);

    return entry_ret || sync_ret;
}

static int
ovs_doca_rss_pipe_init(struct ovs_doca_netdev_data *dev_data)
{
    const char *pci = dev_data->esw_ctx->pci_addr;
    struct doca_flow_actions *actions_masks_list;
    struct ovs_doca_flow_actions actions_masks;
    struct doca_flow_actions *actions_list;
    struct ovs_doca_flow_actions actions;
    struct doca_flow_pipe *pipe = NULL;
    struct doca_flow_monitor monitor;
    struct ovs_doca_flow_match match;
    struct doca_flow_pipe_cfg *cfg;
    struct doca_flow_fwd fwd;
    char pipe_name[50];
    int ret;

    memset(&match, 0, sizeof match);
    memset(&fwd, 0, sizeof fwd);
    memset(&actions, 0, sizeof actions);
    memset(&actions_masks, 0, sizeof actions_masks);
    memset(&monitor, 0, sizeof monitor);

    memset(&match.d.parser_meta.port_id, 0xFF, sizeof match.d.parser_meta.port_id);
    /* The configuration is of the inner since the outer is the meta-push encapsulation header. */
    memset(&match.d.parser_meta.inner_l3_type, 0xFF, sizeof match.d.parser_meta.inner_l3_type);
    memset(&match.d.parser_meta.inner_l4_type, 0xFF, sizeof match.d.parser_meta.inner_l4_type);

    actions_list = &actions.d;
    actions_masks_list = &actions_masks.d;
    actions_masks.mark = (OVS_FORCE doca_be32_t) UINT32_MAX;
    actions.mark = (OVS_FORCE doca_be32_t) UINT32_MAX;

    monitor.counter_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;

    fwd.type = DOCA_FLOW_FWD_RSS;
    fwd.rss_type = DOCA_FLOW_RESOURCE_TYPE_NON_SHARED;
    memset(&fwd.rss.nr_queues, 0xFF, sizeof fwd.rss.nr_queues);

    snprintf(pipe_name, sizeof pipe_name, "OVS_RSS_PIPE_%s", pci);

    ret = doca_flow_pipe_cfg_create(&cfg, dev_data->port);
    if (ret) {
        VLOG_ERR("%s: Could not create doca_flow_pipe_cfg for RSS", pci);
        goto out;
    }

    if (doca_flow_pipe_cfg_set_name(cfg, pipe_name) ||
        doca_flow_pipe_cfg_set_type(cfg, DOCA_FLOW_PIPE_BASIC) ||
        doca_flow_pipe_cfg_set_match(cfg, &match.d, &match.d) ||
        doca_flow_pipe_cfg_set_actions(cfg, &actions_list, &actions_masks_list, NULL, 1) ||
        doca_flow_pipe_cfg_set_monitor(cfg, &monitor) ||
        doca_pipe_cfg_allow_one_queue(cfg, AUX_QUEUE) ||
        doca_flow_pipe_cfg_set_nr_entries(cfg, OVS_DOCA_RSS_NUM_ENTRIES * RTE_MAX_ETHPORTS)) {
        VLOG_ERR("%s: Could not set doca_flow_pipe_cfg for RSS", pci);
        goto out;
    }

    ret = doca_flow_pipe_create(cfg, &fwd, NULL, &pipe);
    if (ret) {
        VLOG_ERR("%s: Failed to create rss pipe. Error %d (%s)", pci, ret,
                 doca_error_get_descr(ret));
        goto out;
    }

    dev_data->esw_ctx->rss_pipe = pipe;

out:
    doca_flow_pipe_cfg_destroy(cfg);
    if (ret) {
        ovs_doca_pipe_uninit(&pipe);
    }

    return ret;
}

static struct ds *
dump_ovs_doca_esw(struct ds *s, void *key_,
                  void *ctx_ OVS_UNUSED, void *arg_ OVS_UNUSED)
{
    struct ovs_doca_esw_key *key = key_;
    char pci_addr[PCI_PRI_STR_SIZE];

    rte_pci_device_name(&key->rte_pci, pci_addr, sizeof pci_addr);
    ds_put_format(s, "pci=%s, ", pci_addr);

    return s;
}

static int
ovs_doca_dev_open_pci(struct rte_pci_addr *rte_pci, struct doca_dev **pdev)
{
    struct doca_devinfo **dev_list;
    char pci[PCI_PRI_STR_SIZE];
    uint8_t is_esw_manager = 0;
    uint8_t is_addr_equal = 0;
    uint32_t nb_devs;
    size_t i;
    int res;

    /* Set default return value */
    *pdev = NULL;

    res = doca_devinfo_create_list(&dev_list, &nb_devs);
    if (res != DOCA_SUCCESS) {
        VLOG_ERR("Failed to load doca devices list. Doca_error value: %d",
                 res);
        return res;
    }

    rte_pci_device_name(rte_pci, pci, sizeof pci);
    /* Search */
    for (i = 0; i < nb_devs; i++) {
        res = doca_devinfo_is_equal_pci_addr(dev_list[i], pci, &is_addr_equal);
        if (res != DOCA_SUCCESS || !is_addr_equal) {
            continue;
        }
        res = doca_dpdk_cap_is_rep_port_supported(dev_list[i], &is_esw_manager);
        if (res != DOCA_SUCCESS || !is_esw_manager) {
            continue;
        }
        VLOG_INFO("Opening '%s'", pci);
        res = doca_dev_open(dev_list[i], pdev);
        if (res != DOCA_SUCCESS) {
            VLOG_ERR("Failed to open DOCA device: %s",
                     doca_error_get_descr(res));
        }
        goto out;
    }

    VLOG_WARN("No matching doca device found");
    res = DOCA_ERROR_NOT_FOUND;

out:
    doca_devinfo_destroy_list(dev_list);
    return res;
}

static void
ovs_doca_esw_ctx_uninit(void *ctx_)
{
    struct ovs_doca_esw_ctx *ctx = ctx_;

    memset(ctx->pci_addr, 0, sizeof ctx->pci_addr);
}

static int
ovs_doca_esw_ctx_init(void *ctx_, void *arg_)
{
    struct ovs_doca_esw_ctx_arg *arg = arg_;
    struct ovs_doca_esw_ctx *ctx = ctx_;

    if (ovs_doca_dev_open_pci(&arg->esw_key->rte_pci, &ctx->dev)) {
        return ENODEV;
    }
    rte_pci_device_name(&arg->esw_key->rte_pci, ctx->pci_addr,
                        sizeof ctx->pci_addr);
    ctx->cmd_fd = -1;
    ctx->queue.n_waiting_entries = 0;

    return 0;
}

static int
ovs_doca_flow_ct_init(void)
{
    struct doca_flow_meta zone_match_mask;
    unsigned int n_ct_sessions_ipv6 = 0;
    struct doca_flow_ct_cfg *ct_cfg;
    unsigned int n_ct_sessions;
    unsigned int n_ct_actions;
    unsigned int n_ct_queues;
    doca_error_t err;

    memset(&zone_match_mask, 0, sizeof zone_match_mask);
    err = doca_flow_ct_cfg_create(&ct_cfg);
    if (err) {
        VLOG_ERR("Could not create DOCA-CT configuration: (%s)", doca_error_get_descr(err));
        return err;
    }

    /* One per polling thread.
     * Queries are done using the insertion thread ID.
     */
    n_ct_queues = dpif_doca_get_n_pmd_threads();
    /* DOCA-CT enforces that this value is a multiple of 64.
     * It incorrectly assumes that this is aligned on the cacheline size.
     * Each DOCA-CT queue uses 2 caches of 1024 user actions each.
     * Those caches can deplete the available actions for other queues.
     * To ensure that all requested actions can be used, account for the caches as well.
     */
    n_ct_actions = ROUND_UP(ovs_doca_max_ct_rules() + 1024 * n_ct_queues * 2, CACHE_LINE_SIZE);
    if (n_ct_actions > MAX_DOCA_CT_ACTIONS) {
        n_ct_actions = MAX_DOCA_CT_ACTIONS;
    }

    n_ct_sessions = ovs_doca_max_ct_conns();
    if (n_ct_queues > 1) {
        n_ct_sessions *= OVS_DOCA_CT_ENTRIES_RSS_FACTOR;
    }
    if (n_ct_sessions > MAX_DOCA_CT_CONNECTIONS) {
        if (ovs_doca_max_ct_conns() > MAX_DOCA_CT_CONNECTIONS) {
            VLOG_WARN_ONCE("Requested number of connections %u is not supported, reducing to %u",
                           ovs_doca_max_ct_conns(), MAX_DOCA_CT_CONNECTIONS);
        } else {
            VLOG_WARN_ONCE("Effective number of connections %u is not supported, reducing to %u",
                           n_ct_sessions, MAX_DOCA_CT_CONNECTIONS);
        }
        n_ct_sessions = MAX_DOCA_CT_CONNECTIONS;
    }
    /* IPv6 connections uses the legacy basic-pipe CT implementation
     * unless explicitly enabled.
     */
    if (conntrack_offload_doca_ct_ipv6_enabled) {
        n_ct_sessions_ipv6 = n_ct_sessions;
    }

    doca_offload_set_reg_mask(&zone_match_mask, REG_FIELD_CT_ZONE);
    /* DOCA-CT is configured by default in managed mode. */
    if (doca_flow_ct_cfg_set_flags(ct_cfg, DOCA_FLOW_CT_FLAG_NO_AGING) ||
        doca_flow_ct_cfg_set_queues(ct_cfg, n_ct_queues) ||
        doca_flow_ct_cfg_set_ctrl_queues(ct_cfg, n_ct_queues) ||
        doca_flow_ct_cfg_set_user_actions(ct_cfg, n_ct_actions) ||
        doca_flow_ct_cfg_set_max_connections_per_zone(ct_cfg, n_ct_sessions) ||
        /* The meta_modify_mask field covers modification to DOCA-CT entry metadata,
         * not CT metadata. The DOCA-CT entry metadata are: zone and 'DOCA-CT mark id',
         * which is *NOT* ct-mark. As such, we do not modify either of those
         * fields within the CT pipe and this mask should remain NULL. */
        doca_flow_ct_cfg_set_direction(ct_cfg, false, false, &zone_match_mask, NULL) ||
        doca_flow_ct_cfg_set_direction(ct_cfg, true, false, &zone_match_mask, NULL) ||
        doca_flow_ct_cfg_set_queue_depth(ct_cfg, OVS_DOCA_CT_QUEUE_DEPTH) ||
        doca_flow_ct_cfg_set_connections(ct_cfg, n_ct_sessions, n_ct_sessions_ipv6,
                                         n_ct_sessions + n_ct_sessions_ipv6)) {
        VLOG_ERR("Could not set DOCA-CT configuration");
        err = -1;
        goto out;
    }

    err = doca_flow_ct_init(ct_cfg);
    if (DOCA_IS_ERROR(err)) {
        VLOG_ERR("Failed to initialize DOCA-CT: %s", doca_error_get_descr(err));
    } else {
        VLOG_INFO("DOCA-CT Enabled - initialized");
    }

out:
    doca_flow_ct_cfg_destroy(ct_cfg);
    return err;
}

void
ovs_doca_register_classes(const struct smap *ovs_other_config)
{
    netdev_doca_register(ovs_other_config);
    netdev_register_flow_api_provider(&netdev_offload_doca);
}

bool
ovs_doca_is_offload_trace_enabled(void)
{
    bool val;

    atomic_read_relaxed(&doca_offload_trace_enabled, &val);
    return val;
}

static void
ovs_doca_destroy_defs(struct doca_flow_definitions *defs,
                      struct doca_flow_definitions_cfg *defs_cfg)
{
    if (defs) {
        doca_flow_definitions_destroy(defs);
    }
    if (defs_cfg) {
        doca_flow_definitions_cfg_destroy(defs_cfg);
    }
}

static doca_error_t
ovs_doca_init_defs(struct doca_flow_cfg *cfg,
                   struct doca_flow_definitions **defs,
                   struct doca_flow_definitions_cfg **defs_cfg)
{
    doca_error_t result;
    const char *def_str;

    result = doca_flow_definitions_cfg_create(defs_cfg);
    if (result != DOCA_SUCCESS) {
        VLOG_ERR("Failed to create defs cfg: %d(%s)", result, doca_error_get_name(result));
        return result;
    }
    result = doca_flow_definitions_create(*defs_cfg, defs);
    if (result != DOCA_SUCCESS) {
        VLOG_ERR("definitions creation error: %d(%s)", result, doca_error_get_name(result));
        goto out;
    }

    def_str = "actions.packet.meta.mark";
    result = doca_flow_definitions_add_field(*defs, def_str,
                                             offsetof(struct ovs_doca_flow_actions, mark),
                                             MEMBER_SIZEOF(struct ovs_doca_flow_actions, mark));
    if (result != DOCA_SUCCESS) {
        VLOG_ERR("definitions add field '%s' error: result=%d(%s)", def_str,
                 result, doca_error_get_name(result));
        goto out;
    }
    def_str = "actions.packet.meta.path_selector";
    result = doca_flow_definitions_add_field(*defs, def_str,
                                             offsetof(struct ovs_doca_flow_actions, path_selector),
                                             MEMBER_SIZEOF(struct ovs_doca_flow_actions,
                                                           path_selector));
    if (result != DOCA_SUCCESS) {
        VLOG_ERR("definitions add field '%s' error: result=%d(%s)", def_str,
                  result, doca_error_get_name(result));
         goto out;
    }
    def_str = "match.packet.meta.path_selector";
    result = doca_flow_definitions_add_field(*defs, def_str,
                                             offsetof(struct ovs_doca_flow_match, path_selector),
                                             MEMBER_SIZEOF(struct ovs_doca_flow_match,
                                                           path_selector));
    if (result != DOCA_SUCCESS) {
        VLOG_ERR("definitions add field '%s' error: result=%d(%s)", def_str,
                 result, doca_error_get_name(result));
        goto out;
    }

    result = doca_flow_cfg_set_definitions(cfg, *defs);
    if (result != DOCA_SUCCESS) {
        VLOG_ERR("Failed to set doca_flow_cfg defs: %d(%s)", result, doca_error_get_name(result));
        goto out;
    }

out:
    if (result) {
        ovs_doca_destroy_defs(*defs, *defs_cfg);
    }
    return result;
}

int
ovs_doca_init(const struct smap *ovs_other_config)
{
    static struct ovsthread_once once_enable = OVSTHREAD_ONCE_INITIALIZER;
    const char *cmask = smap_get(ovs_other_config, "pmd-cpu-mask");
    unsigned int nb_threads = DEFAULT_OFFLOAD_THREAD_NB;
    struct doca_flow_definitions_cfg *defs_cfg = NULL;
    unsigned int req_doca_congestion_threshold;
    struct doca_flow_definitions *defs = NULL;
    unsigned int req_doca_max_eswitch_num;
    uint32_t req_ctl_pipe_infra_size;
    static bool enabled = false;
    struct doca_flow_cfg *cfg;
    doca_error_t err;
    int ret = 0;

    /* Dynamic configuration:
     * This section can be modified without restarting the process. */

    ovs_doca_dynamic_config(ovs_other_config);

    /* DPDK usage in dpif-doca configuration:
     * When configuration is dpdk-init=true and dpdk-doca=false. */
    if (dpdk_available()) {
        static struct ovsthread_once once = OVSTHREAD_ONCE_INITIALIZER;

        if (ovsthread_once_start(&once)) {
            rte_flow_dynf_metadata_register();
            ovsthread_once_done(&once);
        }
    }

    /* Static configuration:
     * This section is set once, restart is required after a change. */

    if (!ovsthread_once_start(&once_enable)) {
        return 0;
    }

    offload_provider_api_init();

    if (enabled) {
        goto out;
    }

    if (!smap_get_bool(ovs_other_config, "doca-init", false)) {
        VLOG_INFO("DOCA Disabled - Use other_config:doca-init to enable");
        goto out;
    }

    /* Since it is the first time the configuration is parsed, set it in dpif-netdev layer, to be
     * available later to use the getter.
     */
    dpif_doca_set_n_pmd_threads(cmask);

    req_ctl_pipe_infra_size = smap_get_uint(ovs_other_config,
                                            "ctl-pipe-infra-size", ctl_pipe_infra_size);
    if (req_ctl_pipe_infra_size != ctl_pipe_infra_size) {
        VLOG_INFO("Changing DOCA ctl-pipe-infra-size from %"PRIu32" to %"PRIu32,
                  ctl_pipe_infra_size, req_ctl_pipe_infra_size);

        ctl_pipe_infra_size = req_ctl_pipe_infra_size;
    } else {
        ctl_pipe_infra_size = ctl_pipe_size;
        VLOG_INFO("DOCA ctl-pipe-infra-size is %"PRIu32, ctl_pipe_infra_size);
    }

    log_stream = fopencookie(NULL, "w+", ovs_doca_log_func);
    if (log_stream == NULL) {
        VLOG_ERR("Can't redirect DOCA log: %s.", ovs_strerror(errno));
    } else {
        /* Create a logger backend that prints to the redirected log */
        err = doca_log_backend_create_with_file_sdk(log_stream, &ovs_doca_log);
        if (err != DOCA_SUCCESS) {
            ovsthread_once_done(&once_enable);
            return EXIT_FAILURE;
        }
        doca_log_level_set_global_sdk_limit(DOCA_LOG_LEVEL_WARNING);
    }
    unixctl_command_register("doca/log-set", "{level}. level=critical/error/"
                             "warning/info/debug", 0, 1,
                             ovs_doca_unixctl_log_set, NULL);
    unixctl_command_register("doca/log-get", "", 0, 0,
                             ovs_doca_unixctl_mem_stream, ovs_doca_log_dump);
    unixctl_command_register("doca/dump-offloads", "", 0, 0,
                             netdev_offload_doca_dump_traces,
                             NULL);

    unixctl_command_register("dpdk/dump-offloads", "", 0, 0,
                             netdev_offload_doca_dump_traces,
                             NULL);

    ovs_doca_esw_rfm = refmap_create("ovs-doca-esw",
                                     sizeof(struct ovs_doca_esw_key),
                                     sizeof(struct ovs_doca_esw_ctx),
                                     ovs_doca_esw_ctx_init,
                                     ovs_doca_esw_ctx_uninit,
                                     dump_ovs_doca_esw, false);

    /* Set dpdk-init to be true if not already set */
    smap_replace(CONST_CAST(struct smap *, ovs_other_config),
                 "dpdk-init", "true");
    dpdk_init(ovs_other_config);
    rte_flow_dynf_metadata_register();

    /* OVS-DOCA configuration happens earlier than dpif-netdev's.
     * To avoid reorganizing them, read the relevant item directly. */
    conntrack_offload_config(ovs_other_config);

    ovs_doca_max_megaflows_counters = smap_get_uint(ovs_other_config, "flow-limit",
                                                    OVS_DOCA_MAX_MEGAFLOWS_COUNTERS);

    /* Due to limitation in doca, only one offload thread is currently
     * supported.
     * */
    if (smap_get_uint(ovs_other_config,"n-offload-threads", 1) !=
        nb_threads) {
        smap_replace(CONST_CAST(struct smap *, ovs_other_config),
                     "n-offload-threads", "1");
        VLOG_WARN_ONCE("Only %u offload thread is currently supported with"
                       "doca, ignoring n-offload-threads configuration",
                       nb_threads);
    }

    req_doca_congestion_threshold =
        smap_get_uint(ovs_other_config, "doca-congestion-threshold",
                      OVS_DOCA_CONGESTION_THRESHOLD_DEFAULT);
    if (req_doca_congestion_threshold != doca_congestion_threshold) {
        if (req_doca_congestion_threshold < OVS_DOCA_CONGESTION_THRESHOLD_MIN ||
            req_doca_congestion_threshold > OVS_DOCA_CONGESTION_THRESHOLD_MAX ) {
            VLOG_WARN("doca-congestion-threshold (%u) is not within expected"
                      " range (%u, %u)", req_doca_congestion_threshold,
                      OVS_DOCA_CONGESTION_THRESHOLD_MIN,
                      OVS_DOCA_CONGESTION_THRESHOLD_MAX);
        }
        VLOG_INFO("Changing DOCA doca-congestion-threshold from %u to %u",
                  doca_congestion_threshold, req_doca_congestion_threshold);

        doca_congestion_threshold = req_doca_congestion_threshold;
    } else {
        VLOG_INFO_ONCE("DOCA doca-congestion-threhold is %u",
                       doca_congestion_threshold);
    }

    req_doca_max_eswitch_num = smap_get_uint(ovs_other_config, "doca-eswitch-max",
                                             OVS_DOCA_DEFAULT_MAX_ESW);
    if (req_doca_max_eswitch_num != ovs_doca_max_eswitch_num) {
        if (req_doca_max_eswitch_num == 0 ||
            req_doca_max_eswitch_num > OVS_DOCA_MAX_SUPPORTED_ESW) {
            VLOG_WARN("other_config doca-eswitch-max=%u not supported, setting as default=%u",
                      req_doca_max_eswitch_num, OVS_DOCA_DEFAULT_MAX_ESW);
        } else {
            ovs_doca_max_eswitch_num = req_doca_max_eswitch_num;
        }
    }
    VLOG_INFO("Maximum supported E-Switches: %u", ovs_doca_max_eswitch_num);

    ovs_assert(!doca_flow_cfg_create(&cfg));
    /* Offload threads, PMD threads, one management thread. */
    n_offload_queues =
        smap_get_uint(ovs_other_config, "n-offload-threads", 1) +
        dpif_doca_get_n_pmd_threads() +
        1;
    ovs_assert(!doca_flow_cfg_set_pipe_queues(cfg, n_offload_queues));

    ovs_assert(!doca_flow_cfg_set_resource_mode(cfg, DOCA_FLOW_RESOURCE_MODE_PORT));
    ovs_assert(!doca_flow_cfg_set_mode_args(cfg, "switch,hws,cpds,isolated,expert"));
    ovs_assert(!doca_flow_cfg_set_queue_depth(cfg, OVS_DOCA_QUEUE_DEPTH));
    ovs_assert(!doca_flow_cfg_set_cb_entry_process(cfg,
                                                   doca_offload_entry_process));
    ovs_assert(!doca_flow_cfg_set_cb_pipe_process(cfg,
                                                  ovs_doca_pipe_process_cb));
    ovs_assert(!ovs_doca_init_defs(cfg, &defs, &defs_cfg));

    VLOG_INFO("DOCA Enabled - initializing...");
    err = doca_flow_init(cfg);
    ovs_doca_destroy_defs(defs, defs_cfg);
    ovs_assert(!doca_flow_cfg_destroy(cfg));
    if (err) {
        VLOG_ERR("Error initializing doca flow offload. Error %d (%s)\n",
                err, doca_error_get_descr(err));

        ovsthread_once_done(&once_enable);
        ovs_abort(err, "Cannot init DOCA");
        return err;
    }

    enabled = true;
    VLOG_INFO("DOCA Enabled - initialized");

    /* Must be called after DOCA-flow initialization and
     * before any port start. */
    if (conntrack_offload_doca_ct_enabled) {
        if (ovs_doca_flow_ct_init()) {
            ret = -1;
            goto out;
        }
    } else {
        VLOG_INFO("DOCA-CT - disabled");
    }

    if (conntrack_offload_doca_ct_ipv6_enabled) {
        VLOG_INFO("DOCA-CT-IPv6 - enabled");
    } else if (conntrack_offload_ipv6_is_enabled()) {
        VLOG_INFO("DOCA-CT-IPv6 - disabled, using legacy CT implementation");
    }

    atomic_store_relaxed(&doca_initialized, enabled);

out:
    ovsthread_once_done(&once_enable);

    return ret;
}

/* Extract the PCI part to the provided esw_key argument, and rte_devargs.
 * Return -EINVAL for error or 0 for success.
 */
static int
ovs_doca_esw_key_parse(const char *devargs,
                       struct ovs_doca_esw_key *esw_key)
{
    memset(esw_key, 0, sizeof *esw_key);

    if (!netdev_doca_parse_dpdk_devargs_pci(devargs, &esw_key->rte_pci)) {
        return -1;
    }

    return 0;
}

static struct doca_flow_pipe_entry *
ovs_doca_egress_entry_init(struct ovs_doca_netdev_data *dev_data,
                           struct doca_flow_pipe *pipe,
                           int port_id)
{
    static struct doca_flow_pipe_entry *entry;
    struct ovs_doca_flow_match match;
    struct doca_flow_fwd fwd;
    doca_error_t err;

    memset(&match, 0, sizeof match);
    memset(&fwd, 0, sizeof fwd);

    match.d.meta.pkt_meta =
        (OVS_FORCE doca_be32_t) DOCA_HTOBE32(ovs_doca_port_id_pkt_meta(port_id));

    fwd.type = DOCA_FLOW_FWD_PORT;
    fwd.port_id = port_id;

    ovs_doca_mgmt_queue_lock();
    err = doca_flow_pipe_add_entry(AUX_QUEUE, pipe, &match.d, 0, NULL, NULL, &fwd,
                                   DOCA_FLOW_NO_WAIT, &dev_data->esw_ctx->queue,
                                   &entry);
    ovs_doca_mgmt_queue_unlock();

    if (err) {
        VLOG_ERR("Failed to create egress pipe entry. Error %d (%s)", err,
                 doca_error_get_descr(err));
        return NULL;
    }
    dev_data->esw_ctx->queue.n_waiting_entries++;

    err = ovs_doca_entries_process_sync(dev_data);
    if (err) {
        VLOG_ERR("Failed to process egress pipe entry. Error: %d (%s)", err,
                 doca_error_get_descr(err));
        return NULL;
    }

    return entry;
}

static void
ovs_doca_esw_port_uninit(struct ovs_doca_esw_ctx *ctx)
{
    uint16_t pid;

    if (!ctx) {
        return;
    }

    for (pid = 0; pid < RTE_MAX_ETHPORTS; pid++) {
        if (ctx->port_queues[pid]) {
            rte_free(ctx->port_queues[pid]);
            ctx->port_queues[pid] = NULL;
        }
    }
    ovs_doca_rss_uninit(ctx);
    ovs_doca_pipe_uninit(&ctx->egress_pipe);
}

static int
ovs_doca_esw_port_init(struct ovs_doca_netdev_data *dev_data, int socket_id)
{
    struct ovs_doca_esw_ctx *ctx = dev_data->esw_ctx;
    uint16_t pid;

    ctx->n_rxq = *dev_data->n_rxq;

    ctx->egress_pipe = ovs_doca_egress_pipe_init(dev_data);
    if (ctx->egress_pipe == NULL) {
        goto err;
    }
    if (ovs_doca_rss_pipe_init(dev_data)) {
        goto err;
    }

    ctx->port_id = dev_data->port_id;
    for (pid = 0; pid < RTE_MAX_ETHPORTS; pid++) {
        uint16_t qid;

        ctx->port_queues[pid] = rte_calloc_socket("port_queues", ctx->n_rxq,
                                                  sizeof(struct ovs_doca_port_queue),
                                                  RTE_CACHE_LINE_SIZE,
                                                  socket_id);
        if (!ctx->port_queues[pid]) {
            goto err;
        }
        for (qid = 0; qid < ctx->n_rxq; qid++) {
            dp_packet_batch_init(&ctx->port_queues[pid][qid].batch);
            atomic_init(&ctx->port_queues[pid][qid].n_packets, 0);
            atomic_init(&ctx->port_queues[pid][qid].n_bytes, 0);
        }
    }

    ctx->op_state = init_op_state;

    return 0;
err:
    ovs_doca_esw_port_uninit(ctx);
    return -1;
}

int
ovs_doca_sw_meter_init(struct netdev *netdev)
{
    return dpdk_offload_doca_sw_meter_init(netdev);
}

void
ovs_doca_sw_meter_uninit(struct netdev *netdev)
{
    dpdk_offload_doca_sw_meter_uninit(netdev);
}

int
ovs_doca_port_start(struct netdev *netdev, struct ovs_doca_netdev_data *dev_data, int socket_id)
{
    const char *devargs = dev_data->devargs;
    uint16_t port_id = dev_data->port_id;
    struct doca_flow_port_cfg *port_cfg;
    struct ovs_doca_esw_ctx *esw_ctx;
    int err;

    if (!ovs_doca_initialized()) {
        VLOG_ERR("DOCA is not initialized: Failed to initialize port %u",
                 port_id);
        return -1;
    }

    if (!rte_eth_dev_is_valid_port(*dev_data->esw_mgr_port_id)) {
        VLOG_ERR("Cannot start port %d '%s', invalid proxy port", port_id, devargs);
        return -1;
    }

    err = doca_flow_port_cfg_create(&port_cfg);
    if (err) {
        VLOG_ERR("Failed to create doca flow port_cfg. Error: %d (%s)",
                 err, doca_error_get_descr(err));
        return -1;
    }

    if (ovs_doca_dev_probe(dev_data, devargs)) {
        err = -1;
        goto out;
    }
    esw_ctx = dev_data->esw_ctx;
    if (!esw_ctx) {
        err = -1;
        goto out;
    }

    err = doca_flow_port_cfg_set_port_id(port_cfg, port_id);
    if (err) {
        VLOG_ERR("%s: Failed to set doca flow port_cfg port_id %d. Error: %d (%s)",
                 netdev_get_name(netdev), port_id, err, doca_error_get_descr(err));
        goto out;
    }

    if (!netdev_doca_is_esw_mgr(netdev)) {
        VLOG_INFO("%s: Opening doca dev_rep for port_id %d", netdev_get_name(netdev), port_id);
        err = doca_dpdk_open_dev_rep_by_port_id(port_id, esw_ctx->dev, &dev_data->dev_rep);
        if (err) {
            VLOG_ERR("%s: Failed to open doca dev_rep for port_id %d. Error: %d (%s)",
                     netdev_get_name(netdev), port_id, err, doca_error_get_descr(err));
            goto out;
        }

        err = doca_flow_port_cfg_set_dev_rep(port_cfg, dev_data->dev_rep);
        if (err) {
            VLOG_ERR("%s: Failed to set doca flow port_cfg dev_rep. Error: %d (%s)",
                     netdev_get_name(netdev), err, doca_error_get_descr(err));
            goto out;
        }
    }

    err = doca_flow_port_cfg_set_dev(port_cfg, esw_ctx->dev);
    if (err) {
        VLOG_ERR("%s: Failed to set doca flow port_cfg dev. Error: %d (%s)",
                 netdev_get_name(netdev), err, doca_error_get_descr(err));
        goto out;
    }

    VLOG_INFO("%s: Starting '%s', port_id=%d", netdev_get_name(netdev), devargs, port_id);
    if (dev_data->port_id == *dev_data->esw_mgr_port_id) {
        err = doca_flow_port_cfg_set_operation_state(port_cfg, init_op_state);
        ovs_assert(!err);
        err = doca_flow_port_cfg_set_actions_mem_size(port_cfg, OVS_DOCA_DEFAULT_ACTIONS_MEM_SIZE);
        ovs_assert(!err);
        err = rte_eth_dev_start(dev_data->port_id);
        if (err) {
            VLOG_ERR("Failed to start dpdk port_id %"PRIu16". Error: %d (%s)",
                     dev_data->port_id, err, doca_error_get_descr(err));
            goto out;
        }

        err = doca_flow_port_cfg_set_nr_resources(port_cfg, DOCA_FLOW_RESOURCE_COUNTER,
                                                  ovs_doca_max_counters() +
                                                  ovs_doca_max_shared_counters_per_esw());
        ovs_assert(!err);
        err = doca_flow_port_cfg_set_nr_resources(port_cfg, DOCA_FLOW_RESOURCE_ENCAP,
                                                  OVS_DOCA_MAX_SHARED_ENCAPS);
        ovs_assert(!err);
        err = doca_flow_port_cfg_set_nr_resources(port_cfg, DOCA_FLOW_RESOURCE_METER,
                                                  OVS_DOCA_MAX_METERS_PER_ESW);
        ovs_assert(!err);
    }
    err = doca_flow_port_start(port_cfg, &dev_data->port);
    if (err) {
        VLOG_ERR("Failed to start doca flow port_id %"PRIu16". Error: %d (%s)",
                 port_id, err, doca_error_get_descr(err));
        goto out;
    }

    if (dev_data->port_id == *dev_data->esw_mgr_port_id &&
        ovs_doca_esw_port_init(dev_data, socket_id)) {
        err = -1;
        goto out;
    }
    dev_data->egress_entry = ovs_doca_egress_entry_init(dev_data,
                                                        esw_ctx->egress_pipe,
                                                        port_id);
    if (!dev_data->egress_entry) {
        err = -1;
        goto out;
    }
    if (ovs_doca_rss_entries_init(dev_data)) {
        err = -1;
        goto out;
    }

out:
    doca_flow_port_cfg_destroy(port_cfg);
    if (err) {
        ovs_doca_port_stop(netdev, dev_data);
    }
    return err;
}

int
ovs_doca_port_stop(struct netdev *netdev, struct ovs_doca_netdev_data *dev_data)
{
    int err = 0;
    int i;

    ovs_assert(ovs_doca_initialized());

    VLOG_INFO("%s: Stopping '%s', port_id=%d", netdev_get_name(netdev), dev_data->devargs,
              dev_data->port_id);

    for (i = 0; i < OVS_DOCA_RSS_NUM_ENTRIES; i++) {
        ovs_doca_entry_uninit(dev_data, &dev_data->rss_entries[i]);
    }
    ovs_doca_entry_uninit(dev_data, &dev_data->egress_entry);

    /* Synchronize all entries removal before clearing the pipes. */
    ovs_doca_entries_process_sync(dev_data);

    if (dev_data->port_id == *dev_data->esw_mgr_port_id) {
        ovs_doca_esw_port_uninit(dev_data->esw_ctx);
    }

    if (dev_data->port) {
        err = doca_flow_port_stop(dev_data->port);
        dev_data->port = NULL;
    }

    rte_eth_dev_stop(dev_data->port_id);

    return err;
}

void
ovs_doca_set_properties(struct ovs_doca_netdev_data *dev_data,
                        uint16_t port_id,
                        const char *devargs,
                        uint16_t *esw_mgr_port_id,
                        bool *attached,
                        int *n_rxq)
{
    dev_data->port_id = port_id;
    dev_data->esw_mgr_port_id = esw_mgr_port_id;
    dev_data->devargs = devargs;
    dev_data->attached = attached;
    dev_data->n_rxq = n_rxq;
}

void
ovs_doca_status(const struct ovsrec_open_vswitch *cfg)
{
    if (!cfg) {
        return;
    }
    ovsrec_open_vswitch_set_doca_initialized(cfg, ovs_doca_initialized());
    ovsrec_open_vswitch_set_doca_version(cfg, doca_version_runtime());
}

void
print_ovs_doca_release(void)
{
    printf("RELEASE %s\n", ovs_git_sha_str);
}

void
print_doca_version(void)
{
    printf("DOCA %s\n", doca_version_runtime());
}

const char *
ovs_doca_get_version(void)
{
    return doca_version_runtime();
}

static void
fill_meter_profile(struct doca_flow_resource_meter_cfg *meter_cfg,
                   struct ofputil_meter_config *config)
{
    if (config->flags & OFPMF13_PKTPS) {
        meter_cfg->limit_type = DOCA_FLOW_METER_LIMIT_TYPE_PACKETS;
        meter_cfg->cir = config->bands[0].rate;
        meter_cfg->cbs = config->bands[0].burst_size;
    } else {
        meter_cfg->limit_type = DOCA_FLOW_METER_LIMIT_TYPE_BYTES;
        /* Convert from kilobits per second to bytes per second */
        meter_cfg->cir = ((uint64_t) config->bands[0].rate) * 125;
        meter_cfg->cbs = ((uint64_t) config->bands[0].burst_size) * 125;
    }
}

int
ovs_doca_create_meter(uint32_t of_meter_id OVS_UNUSED,
                      struct ofputil_meter_config *config,
                      struct doca_flow_resource_meter_cfg *meter_cfg,
                      struct rte_mtr_error *error OVS_UNUSED)
{
    if (config->n_bands != 1) {
        return -1;
    }

    memset(meter_cfg, 0, sizeof *meter_cfg);
    fill_meter_profile(meter_cfg, config);
    doca_offload_shared_meter_set(of_meter_id, meter_cfg);

    return 0;
}

int
ovs_doca_delete_meter(uint16_t port_id OVS_UNUSED, uint32_t of_meter_id,
                      struct rte_mtr_error *error OVS_UNUSED)
{
    doca_offload_shared_meter_unset(of_meter_id);
    return 0;
}

int
ovs_doca_mtr_stats_read(uint16_t port_id OVS_UNUSED, uint32_t dp_meter_id,
                        struct rte_mtr_stats *stats,
                        struct rte_mtr_error *error OVS_UNUSED)
{
    struct doca_flow_resource_query red_stats, green_stats;

    doca_offload_shared_meter_stats_get(dp_meter_id, &red_stats, &green_stats);

    stats->n_pkts[RTE_COLOR_GREEN] = green_stats.counter.total_pkts;
    stats->n_pkts[RTE_COLOR_YELLOW] = 0;
    stats->n_pkts[RTE_COLOR_RED] = red_stats.counter.total_pkts;

    stats->n_bytes[RTE_COLOR_GREEN] = green_stats.counter.total_bytes;
    stats->n_bytes[RTE_COLOR_YELLOW] = 0;
    stats->n_bytes[RTE_COLOR_RED] = red_stats.counter.total_bytes;

    stats->n_pkts_dropped = red_stats.counter.total_pkts;
    stats->n_bytes_dropped = red_stats.counter.total_bytes;

    return 0;
}

unsigned int
ovs_doca_max_ct_conns(void)
{
    return conntrack_offload_size();
}

int
ovs_doca_dev_probe(struct ovs_doca_netdev_data *dev_data,
                   const char *devargs)
{
    struct ds rte_devargs = DS_EMPTY_INITIALIZER;
    struct ovs_doca_esw_ctx_arg ctx_arg;
    struct ovs_doca_esw_key esw_key;
    struct ibv_pd *pd;
    int rv = 0;

    if (dev_data->esw_ctx) {
        /* Already probed. do nothing. */
        return 0;
    }

    if (ovs_doca_esw_key_parse(devargs, &esw_key)) {
        return EINVAL;
    }

    ctx_arg = (struct ovs_doca_esw_ctx_arg) {
        .esw_key = &esw_key,
        .dev_data = dev_data,
    };

    dev_data->esw_ctx = refmap_ref(ovs_doca_esw_rfm, &esw_key, &ctx_arg);
    if (!dev_data->esw_ctx) {
        VLOG_ERR("Could not get esw context for %s", devargs);
        return EINVAL;
    } else if (*dev_data->attached) {
        /* When a representor is probed before the eswitch, dpdk implicitly
         * probes the latter, thus probe is not called from
         * netdev_dpdk_process_devargs(). In this case we call probe at
         * ovs_doca_port_start(), to take the reference and initialize esw_ctx
         * handle in dev_data.
         */
        goto out;
    }

    if (doca_rdma_bridge_get_dev_pd(dev_data->esw_ctx->dev, &pd)) {
        VLOG_ERR("Could not get pd for %s", devargs);
        rv = EINVAL;
        goto out;
    }
    if (dev_data->esw_ctx->cmd_fd == -1) {
        dev_data->esw_ctx->cmd_fd = dup(pd->context->cmd_fd);
        if (dev_data->esw_ctx->cmd_fd == -1) {
            VLOG_ERR("Could not dup fd for %s. Error %s", devargs,
                     ovs_strerror(errno));
            rv = EBADF;
            goto out;
        }
    }

    ds_put_format(&rte_devargs, "%s,cmd_fd=%d,pd_handle=%u", devargs,
                  dev_data->esw_ctx->cmd_fd, pd->handle);

    VLOG_INFO("Probing '%s'", ds_cstr(&rte_devargs));
    if (rte_dev_probe(ds_cstr(&rte_devargs))) {
        rv = ENODEV;
        goto out;
    }

out:
    ds_destroy(&rte_devargs);
    if (rv) {
        ovs_doca_dev_close(dev_data);
    }
    return rv;
}

void
ovs_doca_dev_close(struct ovs_doca_netdev_data *dev_data)
{
    struct ovs_doca_esw_ctx *ctx = dev_data->esw_ctx;
    struct rte_eth_dev_info dev_info;
    char *pci_addr;
    bool last;
    int err;

    memset(&dev_info, 0, sizeof dev_info);

    if (rte_eth_dev_is_valid_port(dev_data->port_id)) {
        err = rte_eth_dev_info_get(dev_data->port_id, &dev_info);
        if (err) {
            VLOG_ERR("Failed to get info of port %d: %s", dev_data->port_id,
                     rte_strerror(-err));
        }
        err = rte_eth_dev_close(dev_data->port_id);
        if (err) {
            VLOG_ERR("Failed to close port %d: %s", dev_data->port_id,
                     rte_strerror(-err));
        }
    }

    if (!ctx) {
        return;
    }

    pci_addr = xstrdup(ctx->pci_addr);
    if (dev_data->port_id != *dev_data->esw_mgr_port_id && dev_data->dev_rep) {
        err = doca_dev_rep_close(dev_data->dev_rep);
        if (err) {
            VLOG_ERR("Failed to close doca dev_rep with port id %d. Error: %d (%s)",
                     dev_data->port_id, err, doca_error_get_descr(err));
        }
        dev_data->dev_rep = NULL;
    }

    last = refmap_unref(ovs_doca_esw_rfm, ctx);
    if (last && ctx->dev) {
        if (rte_eth_dev_is_valid_port(*dev_data->esw_mgr_port_id)) {
            err = rte_eth_dev_close(*dev_data->esw_mgr_port_id);
            if (err) {
                VLOG_ERR("Failed to close esw_mgr port %d: %s",
                         *dev_data->esw_mgr_port_id, rte_strerror(-err));
            }
        }
        /* ctx->cmd_fd is closed inside. */
        if (dev_info.device) {
            err = rte_dev_remove(dev_info.device);
            if (err) {
                VLOG_ERR("Failed to remove device %s: %s", dev_data->devargs,
                         rte_strerror(-err));
            }
        }

        VLOG_INFO("Closing '%s'", pci_addr);
        err = doca_dev_close(ctx->dev);
        if (err) {
            VLOG_ERR("Failed to close doca dev %s. Error: %d (%s)", pci_addr,
                     err, doca_error_get_descr(err));
        }
        ctx->dev = NULL;
        ctx->cmd_fd = -1;
    }

    dev_data->esw_ctx = NULL;
    free(pci_addr);
}

static void
classify_in_port(struct dp_packet_batch *rx_batch,
                 struct ovs_doca_port_queue *pq[RTE_MAX_ETHPORTS],
                 uint16_t queue_id)
{
    struct dp_packet *pkt;
    uint64_t old_count;
    uint32_t pkt_size;
    uint32_t port_id;

    DP_PACKET_BATCH_FOR_EACH (i, pkt, rx_batch) {
        if (!dp_packet_has_flow_mark(pkt, &port_id)) {
            COVERAGE_INC(ovs_doca_no_mark);
            dp_packet_delete(pkt);
            continue;
        }
        dp_packet_reset_flow_mark(pkt);
        if (!rte_eth_dev_is_valid_port(port_id)) {
            COVERAGE_INC(ovs_doca_invalid_classify_port);
            dp_packet_delete(pkt);
            continue;
        }
        pkt_size = dp_packet_size(pkt);
        dp_packet_batch_add(&pq[port_id][queue_id].batch, pkt);
        atomic_add_relaxed(&pq[port_id][queue_id].n_bytes, pkt_size, &old_count);
    }
}

int
ovs_doca_rx_burst(struct ovs_doca_netdev_data *dev_data,
                  uint16_t queue_id,
                  struct dp_packet_batch *batch)
{
    struct dp_packet_batch rx_batch;
    struct ovs_doca_port_queue *pq;
    uint16_t esw_mgr_port_id;
    uint64_t old_count;
    uint16_t port_id;

    esw_mgr_port_id = dev_data->esw_ctx->port_id;
    port_id = dev_data->port_id;

    if (port_id == esw_mgr_port_id) {
        rx_batch.count = rte_eth_rx_burst(esw_mgr_port_id, queue_id,
                                          (struct rte_mbuf **) rx_batch.packets,
                                          NETDEV_MAX_BURST);
        if (rx_batch.count == 0) {
            return 0;
        }
        classify_in_port(&rx_batch, dev_data->esw_ctx->port_queues, queue_id);
    }

    pq = &dev_data->esw_ctx->port_queues[port_id][queue_id];
    memcpy(batch, &pq->batch, sizeof *batch);
    atomic_add_relaxed(&pq->n_packets, batch->count, &old_count);
    dp_packet_batch_init(&pq->batch);

    return batch->count;
}

void
ovs_doca_mgmt_queue_lock(void)
{
    ovs_mutex_lock(&mgmt_queue_lock);
}

void
ovs_doca_mgmt_queue_unlock(void)
{
    ovs_mutex_unlock(&mgmt_queue_lock);
}

int
ovs_doca_eth_stats_get(struct ovs_doca_netdev_data *dev_data,
                       struct rte_eth_stats *rte_stats)
{
    struct doca_flow_resource_query stats;
    int err;
    int i;

    memset(rte_stats, 0, sizeof *rte_stats);

    for (i = 0; i < OVS_DOCA_RSS_NUM_ENTRIES; i++) {
        err = doca_flow_resource_query_entry(dev_data->rss_entries[i], &stats);
        if (err) {
            VLOG_ERR("%s: Failed to query RSS entry. Error %d (%s)", dev_data->devargs, err,
                     doca_error_get_descr(err));
            return -1;
        }
        rte_stats->ipackets += stats.counter.total_pkts;
        rte_stats->ibytes += stats.counter.total_bytes;
    }

    err = doca_flow_resource_query_entry(dev_data->egress_entry, &stats);
    if (err) {
        VLOG_ERR("%s: Failed to query EGRESS entry. Error %d (%s)", dev_data->devargs, err,
                 doca_error_get_descr(err));
        return -1;
    }
    rte_stats->opackets = stats.counter.total_pkts;
    rte_stats->obytes = stats.counter.total_bytes;

    return 0;
}

static const char *
ovs_doca_stats_name(enum ovs_doca_rss_type type)
{
    switch (type) {
    case OVS_DOCA_RSS_IPV4_TCP:
        return "rx_ipv4_tcp";
    case OVS_DOCA_RSS_IPV4_UDP:
        return "rx_ipv4_udp";
    case OVS_DOCA_RSS_IPV4_ICMP:
        return "rx_ipv4_icmp";
    case OVS_DOCA_RSS_IPV4_ESP:
        return "rx_ipv4_esp";
    case OVS_DOCA_RSS_IPV4_OTHER:
        return "rx_ipv4_other";
    case OVS_DOCA_RSS_IPV6_TCP:
        return "rx_ipv6_tcp";
    case OVS_DOCA_RSS_IPV6_UDP:
        return "rx_ipv6_udp";
    case OVS_DOCA_RSS_IPV6_ICMP:
        return "rx_ipv6_icmp";
    case OVS_DOCA_RSS_IPV6_ESP:
        return "rx_ipv6_esp";
    case OVS_DOCA_RSS_IPV6_OTHER:
        return "rx_ipv6_other";
    case OVS_DOCA_RSS_OTHER:
        return "rx_other";
    }
    OVS_NOT_REACHED();
    return "ERR";
}

int
ovs_doca_get_custom_stats(const struct netdev *netdev,
                          const struct ovs_doca_netdev_data *dev_data,
                          const struct ovs_doca_tx_stats *tx_stats,
                          struct netdev_custom_stats *custom_stats)
{
    struct ovs_doca_esw_ctx *ctx = dev_data->esw_ctx;
    struct doca_flow_resource_query stats;
    struct netdev_custom_counter *counter;
    unsigned int n_rxq = *dev_data->n_rxq;
    uint16_t port_id = dev_data->port_id;
    uint64_t n_sw_packets, n_sw_bytes;
    uint64_t mgr_entries_num = 0;
    uint64_t n_packets, n_bytes;
    int n_txq = netdev->n_txq;
    int sw_stats_size;
    int err;
    enum {
        PACKETS,
        BYTES,
    };
    int i;

    if (dev_data->port_id == *dev_data->esw_mgr_port_id) {
        mgr_entries_num = 1;
    }
    sw_stats_size = custom_stats->size;
    custom_stats->size += 2 * (OVS_DOCA_RSS_NUM_ENTRIES + n_rxq + n_txq + 1 +
                               mgr_entries_num);

    custom_stats->counters = xrealloc(custom_stats->counters,
                                      custom_stats->size *
                                      sizeof *custom_stats->counters);
    counter = &custom_stats->counters[sw_stats_size];

    for (i = 0; i < OVS_DOCA_RSS_NUM_ENTRIES; i++, counter += 2) {
        const char *stats_name = ovs_doca_stats_name(i);

        err = doca_flow_resource_query_entry(dev_data->rss_entries[i], &stats);
        if (err) {
            VLOG_ERR("%s: Failed to query '%s' RSS entry. Error %d (%s)", dev_data->devargs,
                     stats_name, err, doca_error_get_descr(err));
            return -1;
        }

        counter[PACKETS].value = stats.counter.total_pkts;
        snprintf(counter[PACKETS].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "%s_packets", stats_name);
        counter[BYTES].value = stats.counter.total_bytes;
        snprintf(counter[BYTES].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "%s_bytes", stats_name);
    }

    n_sw_packets = 0;
    n_sw_bytes = 0;

    for (i = 0; i < n_rxq; i++, counter += 2) {
        atomic_read_relaxed(&ctx->port_queues[port_id][i].n_packets, &n_packets);
        atomic_read_relaxed(&ctx->port_queues[port_id][i].n_bytes, &n_bytes);

        n_sw_packets += n_packets;
        n_sw_bytes += n_bytes;

        counter[PACKETS].value = n_packets;
        snprintf(counter[PACKETS].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "rx_q%d_packets", i);
        counter[BYTES].value = n_bytes;
        snprintf(counter[BYTES].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "rx_q%d_bytes", i);
    }

    counter[PACKETS].value = n_sw_packets;
    snprintf(counter[PACKETS].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "sw_rx_packets");
    counter[BYTES].value = n_sw_bytes;
    snprintf(counter[BYTES].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "sw_rx_bytes");
    counter += 2;

    for (i = 0; i < n_txq; i++, counter += 2) {
        atomic_read_relaxed(&tx_stats[i].n_packets, &n_packets);
        atomic_read_relaxed(&tx_stats[i].n_bytes, &n_bytes);

        counter[PACKETS].value = n_packets;
        snprintf(counter[PACKETS].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "tx_q%d_packets", i);
        counter[BYTES].value = n_bytes;
        snprintf(counter[BYTES].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "tx_q%d_bytes", i);
        counter->value = n_packets;
    }

    if (mgr_entries_num) {
        struct doca_flow_pipe_entry *sample_entry = dpdk_offload_sample_entry_get(netdev);
        const char *stats_name = "hw_sampled";

        counter[PACKETS].value = 0;
        snprintf(counter[PACKETS].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "%s_packets", stats_name);
        counter[BYTES].value = 0;
        snprintf(counter[BYTES].name, NETDEV_CUSTOM_STATS_NAME_SIZE, "%s_bytes", stats_name);

        if (sample_entry) {
            err = doca_flow_resource_query_entry(sample_entry, &stats);
            if (err) {
                VLOG_ERR("%s: Failed to query eswitch manager custom stat entry. Error %d (%s)",
                        dev_data->devargs, err, doca_error_get_descr(err));
                return -1;
            }
            counter[PACKETS].value = stats.counter.total_pkts;
            counter[BYTES].value = stats.counter.total_bytes;
        }
    }

    return 0;
}

static bool
set_op_state_cb(struct netdev *netdev, odp_port_t port_no OVS_UNUSED, void *aux_)
{
    struct ovs_doca_esw_ctx *ctx;
    struct doca_flow_port *port;
    uint32_t *op_state = aux_;

    if (!netdev_doca_is_ethdev(netdev) || !netdev_doca_is_esw_mgr(netdev)) {
        return false;
    }

    port = netdev_doca_port_get(netdev);
    if (!port) {
        return false;
    }

    ctx = netdev_doca_ovs_doca_esw_ctx(netdev);
    if (!ctx) {
        return false;
    }

    if (doca_flow_port_operation_state_modify(port, *op_state)) {
        return false;
    }

    ctx->op_state = *op_state;
    return false;
}

void
ovs_doca_set_op_state(uint32_t op_state)
{
    init_op_state = op_state;
    netdev_ports_traverse("doca", set_op_state_cb, &op_state);
}

#define DMP_CACHE_SIZE 32

void
ovs_doca_mempool_destroy(struct ovs_doca_mempool *odmp)
{
    rte_mempool_free((struct rte_mempool *) odmp);
}

struct ovs_doca_mempool *
ovs_doca_mempool_create(unsigned n, unsigned elt_size)
{
    char mp_name[RTE_MEMPOOL_NAMESIZE];
    static int mp_count = 0;
    struct rte_mempool *mp;

    snprintf(mp_name, sizeof mp_name, "ovs_doca_mempool%d", mp_count++);
    /* The worst case is that all the caches (per l-core) are full except the
     * current one. Increase the number of elements by this number to handle
     * this case.
     */
    n += (RTE_MAX_LCORE - 1) * DMP_CACHE_SIZE;
    mp = rte_mempool_create(mp_name, n, elt_size, DMP_CACHE_SIZE, 0, NULL,
                            NULL, NULL, NULL, SOCKET_ID_ANY, 0);

    return (struct ovs_doca_mempool *) mp;
}

void
ovs_doca_mempool_free(struct ovs_doca_mempool *odmp, void *obj)
{
    rte_mempool_put((struct rte_mempool *) odmp, obj);
}

int
ovs_doca_mempool_alloc(struct ovs_doca_mempool *odmp, void **obj_p)
{
    return rte_mempool_get((struct rte_mempool *) odmp, obj_p);
}

struct ds *
ovs_doca_packet_ds_put_hex(struct ds *s, struct dp_packet *b,
                           uint32_t max_bytes)
{
    uint32_t mark, meta;
    uint32_t size, i;
    uint8_t *p;

    if (dp_packet_has_flow_mark(b, &mark)) {
        ds_put_format(s, "mark=%"PRIu32", ", mark);
    }
    if (dp_packet_get_meta(b, &meta)) {
        ds_put_format(s, "meta=0x%08"PRIx32", ", meta);
    }
    size = dp_packet_size(b);
    ds_put_format(s, "size=%"PRIu32". ", size);

    ds_put_cstr(s, "Raw(hex_bytes('");
    for (i = 0, p = dp_packet_data(b); i < MIN(size, max_bytes) && p; i++, p++) {
        ds_put_format(s, "%02"PRIx8, *p);
    }
    ds_put_cstr(s, "'))");

    if (max_bytes < size) {
        ds_put_format(s, "/Raw('\\x00' * %"PRIu32")", size - max_bytes);
    }

    return s;
}

int
ovs_doca_get_esw_n_rxq(struct ovs_doca_netdev_data *dev_data)
{
    if (!dev_data || !dev_data->esw_ctx) {
        return -1;
    }

    return dev_data->esw_ctx->n_rxq;
}
