/*
 * Copyright (c) 2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef ID_REFMAP_H
#define ID_REFMAP_H

#include <config.h>

#include <stddef.h>
#include <stdint.h>

#include "openvswitch/dynamic-string.h"

/*
 * ID Reference map
 * ================
 *
 * Reference-counted key-value store, where the key is an arbitrary
 * type and the value is a uint32_t.
 *
 * The key to ID relations are reference-counted: duplicates create
 * a reference, and the last dereference frees the associated ID.
 *
 * Thread safety
 * =============
 *
 * MT-unsafe:
 *   * id_refmap_create
 *   * id_refmap_destroy
 *
 * MT-safe:
 *   * id_refmap_find
 *   * id_refmap_ref
 *   * id_refmap_unref
 *
 */

struct id_refmap;

/* Format a key in 's'. */
typedef struct ds *(*id_refmap_format)(struct ds *s, void *key);

/* Allocate and return a map handle.
 *
 * The user must ensure that the 'key' type (of which 'key_size' is the size)
 * does not contain padding. The macros 'OVS_PACKED' or 'OVS_ASSERT_PACKED'
 * (if one does not want a packed struct) can be used to enforce this property.
 */
struct id_refmap *
id_refmap_create(unsigned int nb_thread, const char *name,
                 size_t key_size, id_refmap_format format,
                 uint32_t base_id, uint32_t n_ids,
                 unsigned int id_release_delay_ms,
                 bool disable_map_shrink);

/* Frees the map memory. */
void id_refmap_destroy(struct id_refmap *irfm);

void
id_refmap_upkeep(struct id_refmap *irfm, unsigned int tid,
                 long long int now);

uint32_t id_refmap_find(struct id_refmap *irfm, void *key);
void *id_refmap_find_key(struct id_refmap *irfm, uint32_t id);

/* Returns 'true' if no error happened and 'id' has been set to a valid ID.
 * Returns 'false' otherwise, in which case 'id' is not written. */
bool id_refmap_ref(struct id_refmap *irfm, unsigned int tid, void *key, uint32_t *id);

/* Returns 'true' if this was the last 'id' reference released.
 * If a delay has been configured in the map, the ID will become available only
 * after this delay. If a reference on the same key is attempted before, the
 * same ID will be re-used. */
bool id_refmap_unref(struct id_refmap *irfm, unsigned int tid, uint32_t id);

#endif /* ID_REFMAP_H */
