/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2023-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#ifndef DPIF_DOCA_EXT_H
#define DPIF_DOCA_EXT_H 1

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include "dpif.h"
#include "dp-packet.h"
#include "dpif-doca-private-thread.h"
#include "netdev-offload-doca.h"
#include "netdev-offload.h"
#include "openvswitch/match.h"
#include "openvswitch/types.h"
#include "packets.h"

#ifdef  __cplusplus
extern "C" {
#endif

struct unixctl_conn;
struct dp_offload_thread;
struct dp_offload_thread_item;
struct dp_packet_flow_map;

extern unsigned int max_recirc_depth;

struct ovs_numa_dump *dp_doca_pmd_cmask2cores(const char *pmd_cmask);
unsigned int dpif_doca_get_n_pmd_threads(void);
void dpif_doca_set_n_pmd_threads(const char *pmd_cmask);

void
dp_doca_dump_packets_toggle(struct unixctl_conn *conn, int argc,
                            const char *argv[], void *aux OVS_UNUSED);

void
dp_doca_read_dump_packets_enabled(bool *flag);

int
dpif_doca_offload_stats_get(struct dpif *dpif,
                            struct netdev_custom_stats *stats);

int
dpif_doca_offload_stats_clear(struct dpif *dpif OVS_UNUSED);

void
dpif_doca_set_static_config_ct_add_queue_size(const struct smap *other_config);

void
dp_doca_offload_thread_uninit(struct dp_offload_thread *thread);

void
dp_doca_offload_thread_init(struct dp_offload_thread *thread);

struct dp_offload_thread *
dp_doca_offload_thread_next(struct dp_offload_thread *start,
                            unsigned int *tid, bool include_main);

void
packet_enqueue_to_flow_map(struct dp_packet *packet,
                           struct dp_doca_flow *flow,
                           uint16_t tcp_flags,
                           struct dp_packet_flow_map *flow_map,
                           size_t index);

unsigned int
dp_doca_offload_thread_nb(void);

void
pmd_thread_offload_upkeep(struct dp_doca_pmd_thread *pmd);

unsigned int
pmd_thread_offload_process(struct dp_doca_pmd_thread *pmd, unsigned int limit);

void
pmd_thread_offload_init(struct dp_doca_pmd_thread *pmd);

void
pmd_thread_offload_uninit(struct dp_doca_pmd_thread *pmd);

void
pmd_thread_offload_disable(struct dp_doca_pmd_thread *pmd, struct netdev *netdev);

void
dp_doca_pmd_idle_begin(struct dp_doca_pmd_thread *pmd);

void
dp_doca_pmd_idle_end(struct dp_doca_pmd_thread *pmd);

void
dp_doca_flow_format(const char *prefix,
                    struct ds *s,
                    const struct dp_doca_flow *dp_flow);

long long int
dp_offload_measure_latency(struct dp_offload_thread *thread,
                           long long int enqueue_time_us,
                           long long int finish_time_us);

void
dp_offload_process(struct dp_offload_thread *thread,
                   struct dp_offload_thread_item *msg);

void
dp_doca_offload_poll_queues(struct dp_offload_thread *ofl_thread,
                            struct dp_offload_thread_item **offload_item);

void
dp_doca_ct_offload_init(struct dp_doca *dp);

void
dp_doca_ct_offload_uninit(struct dp_doca *dp);

/* Iterator over all active offload threads.
 *
 * THREAD (struct dp_offload_thread *):
 *   Pointer to a thread object. Will be set to each valid
 *   thread in succession and then NULL after completion.
 */
#define DP_NETDEV_OFFLOAD_FOREACH_THREAD_SHORT(INCLUDE_MAIN, THREAD) \
    for ((THREAD) = dp_doca_offload_thread_next(NULL, NULL, INCLUDE_MAIN); (THREAD); \
         (THREAD) = dp_doca_offload_thread_next((THREAD), NULL, INCLUDE_MAIN))

/* Iterator over all active offload threads.
 *
 * THREAD (struct dp_offload_thread *):
 *   Pointer to a thread object. Will be set to each valid
 *   thread in succession and then NULL after completion.
 *
 * ID (unsigned int):
 *   Variable that will contain the thread-id during each
 *   step of the iteration.
 */
#define DP_NETDEV_OFFLOAD_FOREACH_THREAD_LONG(INCLUDE_MAIN, THREAD, ID) \
    for ((THREAD) = dp_doca_offload_thread_next(NULL, &(ID), INCLUDE_MAIN); (THREAD); \
         (THREAD) = dp_doca_offload_thread_next((THREAD), &(ID), INCLUDE_MAIN))

/* Iterator over all active offload threads.
 *
 * This is a variadic macro that can take either SHORT or LONG
 * form: passing only (THREAD), or (THREAD, ID)) according to each
 * variant signature above.
 *
 */
#define DP_NETDEV_OFFLOAD_FOREACH_THREAD(...) \
    OVERLOAD_SAFE_MACRO(DP_NETDEV_OFFLOAD_FOREACH_THREAD_LONG, \
                        DP_NETDEV_OFFLOAD_FOREACH_THREAD_SHORT, \
                        3, true, __VA_ARGS__)

/* Iterator over all active offload threads except 'main'.
 *
 * This is a variadic macro that can take either SHORT or LONG
 * form: passing only (THREAD), or (THREAD, ID)) according to each
 * variant signature above.
 *
 * The reason to exclude the main sometimes is that this special thread
 * does not handle offload messages. If the iteration is done to emit
 * messages, the main thread must be excluded.
 *
 */
#define DP_NETDEV_OFFLOAD_FOREACH_THREAD_NO_MAIN(...) \
    OVERLOAD_SAFE_MACRO(DP_NETDEV_OFFLOAD_FOREACH_THREAD_LONG, \
                        DP_NETDEV_OFFLOAD_FOREACH_THREAD_SHORT, \
                        3, false, __VA_ARGS__)

void
dp_doca_offload_thread_enqueue(struct dp_offload_thread *thread,
                               struct dp_offload_thread_item *offload);

void
dpif_doca_set_ext_config(struct dp_doca *dp, const struct smap *other_config,
                         bool first_set_config);

int
dpif_doca_ct_get_stats(struct dpif *dpif,
                       struct ct_dpif_stats *stats);

int
dp_doca_offload_netdev_meter_set(uint32_t meter_id_,
                                 struct ofputil_meter_config *config);

int
dp_doca_offload_netdev_meter_get(uint32_t meter_id_,
                                 struct ofputil_meter_stats *stats,
                                 uint16_t n_bands);
int
dp_doca_offload_netdev_meter_del(uint32_t meter_id_,
                                 struct ofputil_meter_stats *stats,
                                 uint16_t n_bands);

#ifdef  __cplusplus
}
#endif

#endif /* dpif-doca-ext.h */
