/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2022-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#ifndef DPDK_OFFLOAD_PROVIDER_H
#define DPDK_OFFLOAD_PROVIDER_H

#ifdef DPDK_NETDEV

#include <stdbool.h>
#include <stdint.h>

#include <rte_flow.h>

#ifdef DOCA_OFFLOAD
#include "doca-pipe-group.h"
#else
struct doca_pipe_group_mask_entry {
};
#endif

#include "dp-packet.h"
#include "netdev-provider.h"
#include "openvswitch/match.h"

#define SAMPLE_TABLE_ID                 0xf1000000
#define SAMPLE_POSTMIRROR_TABLE_ID      0xf2000000
#define MISS_TABLE_ID                   0xf3000000
#define POST_ACT_SPLIT_TABLE_ID         0xf4000000
#define MIRROR_TARGET_GROUP_TABLE_ID    0xf5000000
#define SPLIT_POSTPREFIX_TABLE_ID       0xfa000000
#define POSTHASH_TABLE_ID               0xfb000000
#define CT_TABLE_ID                     0xfc000000
#define CTNAT_TABLE_ID                  0xfc100000
#define POSTCT_TABLE_ID                 0xfd000000
#define POSTMETER_TABLE_ID              0xff000000
#define KNOWN_TABLE_ID_MASK             0xfff00000

#define MAX_SPLIT_DEPTH 10
#define IS_SPLIT_TABLE_ID(group_id) (group_id == SPLIT_POSTPREFIX_TABLE_ID)
#define POST_ACT_SPLIT_TABLE_ID_MAX (POST_ACT_SPLIT_TABLE_ID + MATCH_RECIRC_ID_MASK)

#define MIN_TABLE_ID     1
#define MAX_TABLE_ID     0xf0000000
#define NUM_TABLE_ID     (MAX_TABLE_ID - MIN_TABLE_ID + 1)
#define MIN_ZONE_ID     1
#define MAX_ZONE_ID     0x000000ff
#define NUM_ZONE_ID     (MAX_ZONE_ID - MIN_ZONE_ID + 1)
#define CT_ZONE_IDX(zone_id) ((zone_id) - MIN_ZONE_ID)

static inline const char *
dpdk_offload_table_id_str(uint32_t table_id)
{
    struct {
        uint32_t table_id;
        char *desc;
    } known_tables[] = {
        { .table_id = SAMPLE_TABLE_ID, .desc = "sample", },
        { .table_id = SAMPLE_POSTMIRROR_TABLE_ID, .desc = "sample-post-mirror", },
        { .table_id = MISS_TABLE_ID, .desc = "miss", },
        { .table_id = POST_ACT_SPLIT_TABLE_ID, .desc = "post-act-split", },
        { .table_id = MIRROR_TARGET_GROUP_TABLE_ID, .desc = "mirror-target", },
        { .table_id = SPLIT_POSTPREFIX_TABLE_ID, .desc = "split-post-prefix", },
        { .table_id = POSTHASH_TABLE_ID, .desc = "post-hash", },
        { .table_id = CT_TABLE_ID, .desc = "ct", },
        { .table_id = CTNAT_TABLE_ID, .desc = "ct-nat", },
        { .table_id = POSTCT_TABLE_ID, .desc = "post-ct", },
        { .table_id = POSTMETER_TABLE_ID, .desc = "post-meter", },
    };
    static char ret[50];
    char *str;
    int i;

    str = xasprintf("0x%08x", table_id);

    for (i = 0; i < ARRAY_SIZE(known_tables); i++) {
        uint32_t base_table_id = known_tables[i].table_id;
        char *desc = known_tables[i].desc;
        char *tmp;

        if ((table_id & KNOWN_TABLE_ID_MASK) != base_table_id) {
            continue;
        }

        if (table_id == base_table_id) {
            tmp = str;
            str = xasprintf("%s (%s)", str, desc);
            free(tmp);
            break;
        }

        tmp = str;
        str = xasprintf("%s (%s-%d)", str, desc, table_id - base_table_id);
        free(tmp);
        break;
    }

    ovs_strzcpy(ret, str, sizeof ret);
    free(str);
    return ret;
}

struct ufid_to_rte_flow_data;

struct doca_action_split;
struct doca_flow_pipe_entry;
struct doca_pipe_group_ctx;
struct doca_pipe_group_mask_ctx;

enum {
    DPDK_OFFLOAD_PRIORITY_DOCA_SAMPLE = 0,
    DPDK_OFFLOAD_PRIORITY_HIGH,
    DPDK_OFFLOAD_PRIORITY_MED,
    DPDK_OFFLOAD_PRIORITY_SPLIT_HIGH,
    DPDK_OFFLOAD_PRIORITY_SPLIT_MED,
    DPDK_OFFLOAD_PRIORITY_LOW,
    DPDK_OFFLOAD_PRIORITY_MISS,
};

enum ovs_shared_type {
    OVS_SHARED_UNDEFINED,
    OVS_SHARED_CT_COUNT,
    OVS_SHARED_CT_ACTIONS,
};

struct ovs_shared_ctx {
    enum ovs_shared_type res_type;
    union {
        void *esw_offload_ctx;
        struct rte_flow_action_handle *act_hdl;
    };
    int port_id;
    uint32_t act_type;
    uint32_t res_id;
};

struct ovs_shared_arg {
    struct netdev *netdev;
    struct rte_flow_action *action;
    struct shared_ct_actions_key *ct_actions_key;
    enum ovs_shared_type type;
    unsigned int tid;
};

OVS_PACKED(struct shared_ct_actions_key {
    uint32_t esw_mgr_port_id;
    uint32_t ct_label;
    uint32_t ct_mark;
    uint8_t ct_state;
    unsigned int tid;
});

struct raw_encap_data {
    struct rte_flow_action_raw_encap conf;
    uint8_t headroom[8];
    uint8_t data[TNL_PUSH_HEADER_SIZE - 8];
    uint32_t tnl_type;
};
BUILD_ASSERT_DECL(offsetof(struct raw_encap_data, conf) == 0);

struct meter_data {
    struct rte_flow_action_meter conf;
    uint32_t flow_id;
};
BUILD_ASSERT_DECL(offsetof(struct meter_data, conf) == 0);

struct action_set_data {
    uint8_t value[16];
    uint8_t mask[16];
    size_t size;
};
BUILD_ASSERT_DECL(offsetof(struct action_set_data, value) == 0);

struct hash_data {
    uint32_t flow_id;
    uint32_t seed;
};

struct vxlan_data {
    struct rte_flow_item_vxlan conf;
    uint16_t gbp_id;
    uint8_t gbp_flags;
};
BUILD_ASSERT_DECL(offsetof(struct vxlan_data, conf) == 0);

struct doca_flow_handle_resources {
    struct doca_pipe_group_mask_ctx *mask_ctx;
    struct doca_pipe_group_ctx *self_group_ctx;
    struct doca_pipe_group_ctx *next_group_ctx;
    struct doca_split_prefix_ctx *split_prefix_ctx;
    struct ovs_list *meters_ctx;
    struct doca_pipe_group_mask_entry post_hash_entry_mctx;
    struct doca_action_split *action_split;
    bool sampled;
    struct doca_mirror_ctx *mirror_ctx;
};

struct doca_flow_handle {
    struct doca_flow_pipe_entry *flow;
    struct doca_flow_handle_resources flow_res;
};

struct offload_trace;
struct dpdk_offload_handle {
    union {
        struct rte_flow *rte_flow;
        struct doca_flow_handle dfh;
    };
    struct offload_trace *trace;
    bool valid;
};
BUILD_ASSERT_DECL(offsetof(struct dpdk_offload_handle, rte_flow) ==
                  offsetof(struct dpdk_offload_handle, dfh.flow));
BUILD_ASSERT_DECL(MEMBER_SIZEOF(struct dpdk_offload_handle, rte_flow) ==
                  MEMBER_SIZEOF(struct dpdk_offload_handle, dfh.flow));

#define NUM_HANDLE_PER_ITEM 2
struct flow_item {
    struct dpdk_offload_handle doh[NUM_HANDLE_PER_ITEM];
    bool flow_offload;
};

struct conn_item {
    union {
        void *flow;
        struct rte_flow *rte_flow;
        struct doca_flow_pipe_entry *doca_flow;
    };
    union {
        struct rte_flow *ct_ct_nat;
        struct doca_split_prefix_ctx *curr_split_ctx;
    };
    struct offload_trace *trace;
    bool use_doca_ct;
    bool valid;
};
BUILD_ASSERT_DECL(offsetof(struct conn_item, rte_flow) ==
                  offsetof(struct conn_item, doca_flow));
BUILD_ASSERT_DECL(MEMBER_SIZEOF(struct conn_item, rte_flow) ==
                  MEMBER_SIZEOF(struct conn_item, doca_flow));

struct netdev_offload_dpdk_data {
    struct cmap ufid_to_rte_flow;
    uint64_t offload_counters[MAX_OFFLOAD_THREAD_NB];
    uint64_t flow_counters[MAX_OFFLOAD_THREAD_NB];
    uint64_t conn_counters[MAX_OFFLOAD_THREAD_NB];
    uint64_t unidir_conn_counters[MAX_OFFLOAD_THREAD_NB];
    struct ovs_mutex map_lock;
    struct dpdk_offload_handle ct_nat_miss;
    struct dpdk_offload_handle zone_flows[2][2][MAX_ZONE_ID + 1];
    struct dpdk_offload_handle hairpin;
    struct dpdk_offload_handle sw_meter_apply;
    struct dpdk_offload_handle sw_meter_match;
    void *eswitch_ctx;
};

struct vxlan_meta_header {
    union meta_eth_dst {
        struct eth_addr eth_dst;
        struct {
            ovs_be16 reg_field_ct_zone;
            uint8_t reg_field_ct_state;
            uint8_t pad[3];
        };
    } eth_dst;
    union meta_eth_src {
        struct eth_addr eth_src;
        struct {
            ovs_16aligned_be32 reg_field_multi_plane;
            uint8_t pad[2];
        };
    } eth_src;
    ovs_be16 eth_type;
    uint8_t ip_ihl_ver;
    uint8_t ip_tos;
    ovs_be16 ip_tot_len;
    ovs_be16 ip_id;
    ovs_be16 ip_frag_off;
    uint8_t ip_ttl;
    uint8_t ip_proto;
    ovs_be16 ip_csum;
    union {
        ovs_16aligned_be32 ip_src;
        ovs_16aligned_be32 reg_field_ct_label_id;
    };
    union {
        ovs_16aligned_be32 ip_dst;
        ovs_16aligned_be32 reg_field_ct_mark;
    };
    struct udp_header udp;
    struct vxlanhdr vxlan;
};

/* Proprietary rte-flow action enums. */
enum {
    OVS_RTE_FLOW_ACTION_TYPE_FLOW_INFO = INT_MIN,
    OVS_RTE_FLOW_ACTION_TYPE_CT_INFO,
    OVS_RTE_FLOW_ACTION_TYPE_PRE_CT_END,
    OVS_RTE_FLOW_ACTION_TYPE_HASH,
    OVS_RTE_FLOW_ACTION_TYPE_SET_UDP_SRC,
    OVS_RTE_FLOW_ACTION_TYPE_SET_UDP_DST,
    OVS_RTE_FLOW_ACTION_TYPE_SET_TCP_SRC,
    OVS_RTE_FLOW_ACTION_TYPE_SET_TCP_DST,
    OVS_RTE_FLOW_ACTION_TYPE_SFLOW,
    OVS_RTE_FLOW_ACTION_TYPE_USERSPACE,
    OVS_RTE_FLOW_ACTION_TYPE_NESTED,
};

#define OVS_RTE_FLOW_ACTION_TYPE(TYPE) \
    ((enum rte_flow_action_type) OVS_RTE_FLOW_ACTION_TYPE_##TYPE)

/* Proprietary rte-flow item enums. */
enum {
    OVS_RTE_FLOW_ITEM_TYPE_FLOW_INFO = INT_MIN,
    OVS_RTE_FLOW_ITEM_TYPE_HASH,
    OVS_RTE_FLOW_ITEM_TYPE_NV_MP_PID,
    OVS_RTE_FLOW_ITEM_TYPE_NV_MP_PREFERRED,
    OVS_RTE_FLOW_ITEM_TYPE_NV_MP_STRICT,
};

#define OVS_RTE_FLOW_ITEM_TYPE(TYPE) \
    ((enum rte_flow_item_type) OVS_RTE_FLOW_ITEM_TYPE_##TYPE)

union nv_mp {
    ovs_be32 data;
#if RTE_BYTE_ORDER == RTE_LITTLE_ENDIAN
    struct {
        uint32_t reserved:3;
        uint32_t strict:1;
        uint32_t preferred:1;
        uint32_t pid:3;
#else
        uint32_t reserved:27;
        uint32_t strict:1;
        uint32_t preferred:1;
        uint32_t pid:3;
#endif
    };
};

struct reg_tags {
    uint32_t ct_state;
    uint32_t ct_zone;
    uint32_t ct_mark;
    uint32_t ct_label;
    union nv_mp nv_mp;
};
BUILD_ASSERT_DECL(sizeof(union nv_mp) == sizeof(uint32_t));

struct dpdk_offload_recovery_info {
    struct reg_tags reg_tags;
    uint32_t flow_miss_id;
    uint32_t ct_miss_id;
    uint32_t sflow_id;
    uint32_t dp_hash;
};

enum dpdk_reg_id {
    REG_FIELD_CT_STATE,
    REG_FIELD_CT_ZONE,
    REG_FIELD_CT_MARK,
    REG_FIELD_CT_LABEL_ID,
    REG_FIELD_TUN_INFO,
    REG_FIELD_PORT_METER_COLOR = REG_FIELD_TUN_INFO,
    REG_FIELD_CT_CTX,
    REG_FIELD_SFLOW_CTX,
    REG_FIELD_FLOW_INFO,
    REG_FIELD_DP_HASH,
    REG_FIELD_SCRATCH,
    REG_FIELD_RECIRC,
    REG_FIELD_TAG0,
    REG_FIELD_NUM,
};

enum reg_type {
    REG_TYPE_TAG,
    REG_TYPE_META,
    REG_TYPE_MARK,
};

struct reg_field {
    enum reg_type type;
    uint8_t index;
    uint32_t offset;
    uint32_t mask;
};

#define REG_TAG_INDEX_NUM 5

uint32_t
reg_field_spec(enum dpdk_reg_id reg_id, uint32_t val);

uint32_t
reg_field_mask(enum dpdk_reg_id reg_id, uint32_t mask);

struct dpdk_offload_api {
    void (*upkeep)(struct netdev *netdev, bool quiescing);

    /* Offload insertion / deletion */
    int (*create)(struct netdev *netdev,
                  const struct rte_flow_attr *attr,
                  struct rte_flow_item *items,
                  struct rte_flow_action *actions,
                  struct dpdk_offload_handle *doh,
                  struct rte_flow_error *error);
    int (*destroy)(struct netdev *netdev,
                   struct dpdk_offload_handle *doh,
                   struct rte_flow_error *error,
                   bool esw_port_id);
    int (*query_count)(struct netdev *netdev,
                       struct dpdk_offload_handle *doh,
                       struct rte_flow_query_count *query,
                       struct rte_flow_error *error);
    int (*shared_create)(struct ovs_shared_ctx *ctx,
                         struct ovs_shared_arg *arg,
                         struct rte_flow_error *error);
    int (*shared_destroy)(struct ovs_shared_ctx *ctx,
                          struct rte_flow_error *error);
    int (*shared_query)(struct ovs_shared_ctx *ctx,
                        void *data,
                        struct rte_flow_error *error);

    bool (*get_packet_recover_info)(struct dp_packet *p,
                                    struct dpdk_offload_recovery_info *info);

    int (*insert_conns)(struct netdev *netdevs[CT_DIR_NUM], struct batch *conns);
    int (*destroy_conn)(struct netdev *netdevs[CT_DIR_NUM], struct conn *conn,
                        struct rte_flow_error *error);
    int (*query_conn)(struct netdev *netdevs[CT_DIR_NUM], struct conn *conn,
                      struct dpif_flow_stats *stats,
                      struct dpif_flow_attrs *attrs,
                      struct rte_flow_error *error,
                      long long int now);

    struct reg_field *(*reg_fields)(void);

    void (*update_stats)(struct dpif_flow_stats *stats,
                         struct dpif_flow_stats *baseline_stats,
                         struct dpif_flow_attrs *attrs,
                         struct rte_flow_query_count *query);
    int (*aux_tables_init)(struct netdev *netdev);
    void (*aux_tables_uninit)(struct netdev *netdev);
    int (*packet_hw_hash)(struct netdev *, struct dp_packet *, uint32_t, uint32_t *);
    int (*packet_hw_entropy)(struct netdev *, struct dp_packet *, uint16_t *);
};

extern struct dpdk_offload_api dpdk_offload_api_rte;
extern struct dpdk_offload_api dpdk_offload_api_doca;

bool
dpdk_offload_get_reg_field(struct dp_packet *packet,
                           enum dpdk_reg_id reg_id,
                           uint32_t *val);

void *
find_raw_encap_spec(const struct raw_encap_data *raw_encap_data,
                    enum rte_flow_item_type type);

static inline void
dpdk_offload_counter_inc(struct netdev *netdev)
{
    unsigned int tid = netdev_offload_thread_id();
    struct netdev_offload_dpdk_data *data;

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);
    data->offload_counters[tid]++;
}

static inline void
dpdk_offload_counter_dec(struct netdev *netdev)
{
    unsigned int tid = netdev_offload_thread_id();
    struct netdev_offload_dpdk_data *data;

    data = (struct netdev_offload_dpdk_data *)
        ovsrcu_get(void *, &netdev->hw_info.offload_data);
    /* Decrement can be done during delayed unref of flow resources,
     * which can be executed after the port has been uninit already.
     * In that case, the offload data is not available and there is
     * nothing to count. */
    if (data) {
        data->offload_counters[tid]--;
    }
}

enum ct_zone_cls_flow_type {
    CT_ZONE_FLOW_REVISIT,
    CT_ZONE_FLOW_UPHOLD_IP4_UDP,
    CT_ZONE_FLOW_UPHOLD_IP4_TCP,
    CT_ZONE_FLOW_UPHOLD_IP6_UDP,
    CT_ZONE_FLOW_UPHOLD_IP6_TCP,
    CT_ZONE_FLOW_MISS,
    CT_ZONE_FLOWS_NUM,
};

#define NUM_CT_ZONES (1 << 16)

#else /* DPDK_NETDEV */

struct flow_item {
};

struct conn_item {
};

#endif /* DPDK_NETDEV */

#if DOCA_OFFLOAD

#define DPDK_OFFLOAD_MAX_METERS_PER_FLOW 4

#else /* DOCA_OFFLOAD */

#define DPDK_OFFLOAD_MAX_METERS_PER_FLOW 1

#endif /* DOCA_OFFLOAD */

#endif /* DPDK_OFFLOAD_PROVIDER_H */
