/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2014-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#include <config.h>
#include "netdev-doca.h"

#include <errno.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <linux/if.h>

#include <rte_bus.h>
#include <rte_config.h>
#include <rte_cycles.h>
#include <rte_dev.h>
#include <rte_errno.h>
#include <rte_ethdev.h>
#include <rte_flow.h>
#include <rte_malloc.h>
#include <rte_mbuf.h>
#include <rte_pci.h>
#include <rte_version.h>
#include <rte_vhost.h>

#include "cmap.h"
#include "coverage.h"
#include "dirs.h"
#include "dp-packet.h"
#include "dpdk.h"
#include "dpdk-offload-provider.h"
#include "dpif-doca.h"
#include "fatal-signal.h"
#include "if-notifier.h"
#include "mpsc-queue.h"
#include "netdev-offload-doca.h"
#include "netdev-provider.h"
#include "netdev-doca-vdpa.h"
#include "netdev-vport.h"
#include "odp-util.h"
#include "openflow/openflow.h"
#include "openvswitch/dynamic-string.h"
#include "openvswitch/list.h"
#include "openvswitch/match.h"
#include "openvswitch/ofp-parse.h"
#include "openvswitch/ofp-print.h"
#include "openvswitch/poll-loop.h"
#include "openvswitch/shash.h"
#include "openvswitch/vlog.h"
#include "ovs-doca.h"
#include "ovs-numa.h"
#include "ovs-rcu.h"
#include "ovs-thread.h"
#include "packets.h"
#include "rtnetlink.h"
#include "smap.h"
#include "sset.h"
#include "timeval.h"
#include "unaligned.h"
#include "unixctl.h"
#include "userspace-tso.h"
#include "util.h"
#include "uuid.h"

VLOG_DEFINE_THIS_MODULE(netdev_doca);
static struct vlog_rate_limit rl = VLOG_RATE_LIMIT_INIT(5, 20);
static bool netdev_doca_initialized = false;

COVERAGE_DEFINE_WARN(netdev_doca_drop_oversized);

static bool per_port_memory = false; /* Status of per port memory support */

static atomic_bool netdev_doca_pending_reset[RTE_MAX_ETHPORTS];

#define DPDK_PORT_WATCHDOG_INTERVAL 5

#define OVS_CACHE_LINE_SIZE CACHE_LINE_SIZE

/*
 * need to reserve tons of extra space in the mbufs so we can align the
 * DMA addresses to 4KB.
 * The minimum mbuf size is limited to avoid scatter behaviour and drop in
 * performance for standard Ethernet MTU.
 */
#define ETHER_HDR_MAX_LEN           (RTE_ETHER_HDR_LEN + RTE_ETHER_CRC_LEN \
                                     + (2 * VLAN_HEADER_LEN))
#define MTU_TO_FRAME_LEN(mtu)       ((mtu) + RTE_ETHER_HDR_LEN + \
                                     RTE_ETHER_CRC_LEN)
#define MTU_TO_MAX_FRAME_LEN(mtu)   ((mtu) + ETHER_HDR_MAX_LEN)
#define FRAME_LEN_TO_MTU(frame_len) ((frame_len)                    \
                                     - RTE_ETHER_HDR_LEN - RTE_ETHER_CRC_LEN)
#define NETDEV_DPDK_MBUF_ALIGN      1024
#define NETDEV_DPDK_MAX_PKT_LEN     9728

/* Max and min number of packets in the mempool. OVS tries to allocate a
 * mempool with MAX_NB_MBUF: if this fails (because the system doesn't have
 * enough hugepages) we keep halving the number until the allocation succeeds
 * or we reach MIN_NB_MBUF */

#define MAX_NB_MBUF          (4096 * 64)
#define MIN_NB_MBUF          (4096 * 4)
#define MP_CACHE_SZ          RTE_MEMPOOL_CACHE_MAX_SIZE

/* MAX_NB_MBUF can be divided by 2 many times, until MIN_NB_MBUF */
BUILD_ASSERT_DECL(MAX_NB_MBUF % ROUND_DOWN_POW2(MAX_NB_MBUF / MIN_NB_MBUF)
                  == 0);

/* The smallest possible NB_MBUF that we're going to try should be a multiple
 * of MP_CACHE_SZ. This is advised by DPDK documentation. */
BUILD_ASSERT_DECL((MAX_NB_MBUF / ROUND_DOWN_POW2(MAX_NB_MBUF / MIN_NB_MBUF))
                  % MP_CACHE_SZ == 0);

#define SOCKET0              0

/* Default size of Physical NIC RXQ */
#define NIC_PORT_DEFAULT_RXQ_SIZE 2048
/* Default size of Physical NIC TXQ */
#define NIC_PORT_DEFAULT_TXQ_SIZE 2048
/* Maximum size of Physical NIC Queues */
#define NIC_PORT_MAX_Q_SIZE 4096

#define OVS_VHOST_MAX_QUEUE_NUM 1024  /* Maximum number of vHost TX queues. */
#define OVS_VHOST_QUEUE_MAP_UNKNOWN (-1) /* Mapping not initialized. */
#define OVS_VHOST_QUEUE_DISABLED    (-2) /* Queue was disabled by guest and not
                                          * yet mapped to another queue. */

#define DPDK_ETH_PORT_ID_INVALID    RTE_MAX_ETHPORTS

/* DPDK library uses uint16_t for port_id. */
typedef uint16_t dpdk_port_t;
#define DPDK_PORT_ID_FMT "%"PRIu16

/* Minimum amount of vhost tx retries, effectively a disable. */
#define VHOST_ENQ_RETRY_MIN 0
/* Maximum amount of vhost tx retries. */
#define VHOST_ENQ_RETRY_MAX 32
/* Legacy default value for vhost tx retries. */
#define VHOST_ENQ_RETRY_DEF 8

#define IF_NAME_SZ (PATH_MAX > IFNAMSIZ ? PATH_MAX : IFNAMSIZ)

/* List of required flags advertised by the hardware that will be used
 * if TSO is enabled. Ideally this should include
 * RTE_ETH_TX_OFFLOAD_SCTP_CKSUM. However, very few drivers support that
 * at the moment and SCTP is not a widely used protocol like TCP and UDP,
 * so it's optional. */
#define DPDK_TX_TSO_OFFLOAD_FLAGS (RTE_ETH_TX_OFFLOAD_TCP_TSO        \
                                   | RTE_ETH_TX_OFFLOAD_TCP_CKSUM    \
                                   | RTE_ETH_TX_OFFLOAD_UDP_CKSUM    \
                                   | RTE_ETH_TX_OFFLOAD_IPV4_CKSUM)


static const struct rte_eth_conf port_conf = {
    .rxmode = {
        .offloads = 0,
    },
    .rx_adv_conf = {
        .rss_conf = {
            .rss_key = NULL,
            .rss_hf = RTE_ETH_RSS_IP | RTE_ETH_RSS_UDP | RTE_ETH_RSS_TCP,
        },
    },
    .txmode = {
        .mq_mode = RTE_ETH_MQ_TX_NONE,
    },
};

/* Custom software stats for dpdk ports */
struct netdev_doca_sw_stats {
    /* No. of retries when unable to transmit. */
    uint64_t tx_retries;
    /* Packet drops when unable to transmit; Probably Tx queue is full. */
    uint64_t tx_failure_drops;
    /* Packet length greater than device MTU. */
    uint64_t tx_mtu_exceeded_drops;
    /* Packet drops in egress policer processing. */
    uint64_t tx_qos_drops;
    /* Packet drops in ingress policer processing. */
    uint64_t rx_qos_drops;
    /* Packet drops in HWOL processing. */
    uint64_t tx_invalid_hwol_drops;
};

enum dpdk_dev_type {
    DPDK_DEV_ETH = 0,
    DPDK_DEV_VHOST = 1,
};

/* Quality of Service */

/* An instance of a QoS configuration.  Always associated with a particular
 * network device.
 *
 * Each QoS implementation subclasses this with whatever additional data it
 * needs.
 */
struct qos_conf {
    const struct dpdk_qos_ops *ops;
    rte_spinlock_t lock;
};

/* QoS queue information used by the netdev queue dump functions. */
struct netdev_doca_queue_state {
    uint32_t *queues;
    size_t cur_queue;
    size_t n_queues;
};

/* A particular implementation of dpdk QoS operations.
 *
 * The functions below return 0 if successful or a positive errno value on
 * failure, except where otherwise noted. All of them must be provided, except
 * where otherwise noted.
 */
struct dpdk_qos_ops {

    /* Name of the QoS type */
    const char *qos_name;

    /* Called to construct a qos_conf object. The implementation should make
     * the appropriate calls to configure QoS according to 'details'.
     *
     * The contents of 'details' should be documented as valid for 'ovs_name'
     * in the "other_config" column in the "QoS" table in vswitchd/vswitch.xml
     * (which is built as ovs-vswitchd.conf.db(8)).
     *
     * This function must return 0 if and only if it sets '*conf' to an
     * initialized 'struct qos_conf'.
     *
     * For all QoS implementations it should always be non-null.
     */
    int (*qos_construct)(const struct smap *details, struct qos_conf **conf);

    /* Destroys the data structures allocated by the implementation as part of
     * 'qos_conf'.
     *
     * For all QoS implementations it should always be non-null.
     */
    void (*qos_destruct)(struct qos_conf *conf);

    /* Retrieves details of 'conf' configuration into 'details'.
     *
     * The contents of 'details' should be documented as valid for 'ovs_name'
     * in the "other_config" column in the "QoS" table in vswitchd/vswitch.xml
     * (which is built as ovs-vswitchd.conf.db(8)).
     */
    int (*qos_get)(const struct qos_conf *conf, struct smap *details);

    /* Returns true if 'conf' is already configured according to 'details'.
     *
     * The contents of 'details' should be documented as valid for 'ovs_name'
     * in the "other_config" column in the "QoS" table in vswitchd/vswitch.xml
     * (which is built as ovs-vswitchd.conf.db(8)).
     *
     * For all QoS implementations it should always be non-null.
     */
    bool (*qos_is_equal)(const struct qos_conf *conf,
                         const struct smap *details);

    /* Modify an array of rte_mbufs. The modification is specific to
     * each qos implementation.
     *
     * The function should take and array of mbufs and an int representing
     * the current number of mbufs present in the array.
     *
     * After the function has performed a qos modification to the array of
     * mbufs it returns an int representing the number of mbufs now present in
     * the array. This value is can then be passed to the port send function
     * along with the modified array for transmission.
     *
     * For all QoS implementations it should always be non-null.
     */
    int (*qos_run)(struct qos_conf *qos_conf, struct rte_mbuf **pkts,
                   int pkt_cnt, bool should_steal);

    /* Called to construct a QoS Queue. The implementation should make
     * the appropriate calls to configure QoS Queue according to 'details'.
     *
     * The contents of 'details' should be documented as valid for 'ovs_name'
     * in the "other_config" column in the "QoS" table in vswitchd/vswitch.xml
     * (which is built as ovs-vswitchd.conf.db(8)).
     *
     * This function must return 0 if and only if it constructs
     * QoS queue successfully.
     */
    int (*qos_queue_construct)(const struct smap *details,
                               uint32_t queue_id, struct qos_conf *conf);

    /* Destroys the QoS Queue. */
    void (*qos_queue_destruct)(struct qos_conf *conf, uint32_t queue_id);

    /* Retrieves details of QoS Queue configuration into 'details'.
     *
     * The contents of 'details' should be documented as valid for 'ovs_name'
     * in the "other_config" column in the "QoS" table in vswitchd/vswitch.xml
     * (which is built as ovs-vswitchd.conf.db(8)).
     */
    int (*qos_queue_get)(struct smap *details, uint32_t queue_id,
                         const struct qos_conf *conf);

    /* Retrieves statistics of QoS Queue configuration into 'stats'. */
    int (*qos_queue_get_stats)(const struct qos_conf *conf, uint32_t queue_id,
                               struct netdev_queue_stats *stats);

    /* Setup the 'netdev_doca_queue_state' structure used by the dpdk queue
     * dump functions.
     */
    int (*qos_queue_dump_state_init)(const struct qos_conf *conf,
                                     struct netdev_doca_queue_state *state);
};

static struct ovs_mutex dpdk_mutex = OVS_MUTEX_INITIALIZER;

/* Contains all 'struct dpdk_dev's. */
static struct ovs_list dpdk_list OVS_GUARDED_BY(dpdk_mutex)
    = OVS_LIST_INITIALIZER(&dpdk_list);

static struct ovs_mutex dpdk_mp_mutex OVS_ACQ_AFTER(dpdk_mutex)
    = OVS_MUTEX_INITIALIZER;

/* Contains all 'struct dpdk_mp's. */
static struct ovs_list dpdk_mp_list OVS_GUARDED_BY(dpdk_mp_mutex)
    = OVS_LIST_INITIALIZER(&dpdk_mp_list);

struct dpdk_mp {
     struct rte_mempool *mp;
     int mtu;
     int socket_id;
     dpdk_port_t esw_mgr_port_id;
     int refcount;
     struct ovs_list list_node OVS_GUARDED_BY(dpdk_mp_mutex);
};

struct user_mempool_config {
    int adj_mtu;
    int socket_id;
};

static struct user_mempool_config *user_mempools = NULL;
static int n_user_mempools;

/* There should be one 'struct dpdk_tx_queue' created for
 * each netdev tx queue. */
struct dpdk_tx_queue {
    /* Padding to make dpdk_tx_queue exactly one cache line long. */
    PADDED_MEMBERS(CACHE_LINE_SIZE,
        /* Protects the members and the NIC queue from concurrent access.
         * It is used only if the queue is shared among different pmd threads
         * (see 'concurrent_txq'). */
        rte_spinlock_t tx_lock;
        /* Mapping of configured vhost-user queue to enabled by guest. */
        int map;
    );
};

struct ingress_policer {
    struct rte_meter_srtcm_params app_srtcm_params;
    struct rte_meter_srtcm in_policer;
    struct rte_meter_srtcm_profile in_prof;
    rte_spinlock_t policer_lock;
};

enum dpdk_hw_ol_features {
    NETDEV_RX_CHECKSUM_OFFLOAD = 1 << 0,
    NETDEV_RX_HW_CRC_STRIP = 1 << 1,
    NETDEV_RX_HW_SCATTER = 1 << 2,
    NETDEV_TX_IPV4_CKSUM_OFFLOAD = 1 << 3,
    NETDEV_TX_TCP_CKSUM_OFFLOAD = 1 << 4,
    NETDEV_TX_UDP_CKSUM_OFFLOAD = 1 << 5,
    NETDEV_TX_SCTP_CKSUM_OFFLOAD = 1 << 6,
    NETDEV_TX_TSO_OFFLOAD = 1 << 7,
    NETDEV_TX_VXLAN_TNL_TSO_OFFLOAD = 1 << 8,
    NETDEV_TX_GENEVE_TNL_TSO_OFFLOAD = 1 << 9,
    NETDEV_TX_OUTER_IP_CKSUM_OFFLOAD = 1 << 10,
    NETDEV_TX_OUTER_UDP_CKSUM_OFFLOAD = 1 << 11,
};

/*
 * In order to avoid confusion in variables names, following naming convention
 * should be used, if possible:
 *
 *     'struct netdev'          : 'netdev'
 *     'struct netdev_doca'     : 'dev'
 *     'struct netdev_rxq'      : 'rxq'
 *     'struct netdev_rxq_dpdk' : 'rx'
 *
 * Example:
 *     struct netdev *netdev = netdev_from_name(name);
 *     struct netdev_doca *dev = netdev_doca_cast(netdev);
 *
 *  Also, 'netdev' should be used instead of 'dev->up', where 'netdev' was
 *  already defined.
 */

struct netdev_doca {
    PADDED_MEMBERS_CACHELINE_MARKER(CACHE_LINE_SIZE, cacheline0,
        struct eth_addr hwaddr;
        dpdk_port_t port_id;
        int mtu;
        int socket_id;
        int max_packet_len;
        enum dpdk_dev_type type;
        enum netdev_flags flags;
        int link_reset_cnt;
        union {
            /* Device arguments for dpdk ports. */
            char *devargs;
            /* Identifier used to distinguish vhost devices from each other. */
            char *vhost_id;
        };
        struct dpdk_tx_queue *tx_q;
        struct rte_eth_link link;
        /* 1 pad bytes here. */
        dpdk_port_t esw_mgr_port_id;
        /* If true, device was attached by rte_eth_dev_attach(). */
        bool attached;
        /* If true, rte_eth_dev_start() was successfully called */
        atomic_bool started;
        /* If true, this is a port representor. */
        bool is_representor;
    );

    PADDED_MEMBERS_CACHELINE_MARKER(CACHE_LINE_SIZE, cacheline1,
        struct ovs_mutex mutex OVS_ACQ_AFTER(dpdk_mutex);
        struct dpdk_mp *dpdk_mp;
    );

    PADDED_MEMBERS(CACHE_LINE_SIZE,
        struct netdev up;
        /* In dpdk_list. */
        struct ovs_list list_node OVS_GUARDED_BY(dpdk_mutex);

        /* Array of vhost rxq states, see vring_state_changed. */
        bool *vhost_rxq_enabled;

        /* Ensures that Rx metadata delivery is configured only once. */
        bool rx_metadata_delivery_configured;
    );

    PADDED_MEMBERS(CACHE_LINE_SIZE,
        struct netdev_stats stats;
        struct netdev_doca_sw_stats *sw_stats;
        struct ovs_doca_tx_stats *sw_tx_stats;
        /* Protects stats */
        rte_spinlock_t stats_lock;
        /* 8 pad bytes here. */
    );

    PADDED_MEMBERS(CACHE_LINE_SIZE,
        /* The following properties cannot be changed when a device is running,
         * so we remember the request and update them next time
         * netdev_doca*_reconfigure() is called */
        int requested_mtu;
        int requested_n_txq;
        /* User input for n_rxq (see dpdk_set_rxq_config). */
        int user_n_rxq;
        /* user_n_rxq + an optional rx steering queue (see
         * netdev_doca_reconfigure). This field is different from the other
         * requested_* fields as it may contain a different value than the user
         * input. */
        int requested_n_rxq;
        int requested_rxq_size;
        int requested_txq_size;

        /* Number of rx/tx descriptors for physical devices */
        int rxq_size;
        int txq_size;

        /* Socket ID detected when vHost device is brought up */
        int requested_socket_id;

        /* Denotes whether vHost port is client/server mode */
        uint64_t vhost_driver_flags;

        /* DPDK-ETH Flow control */
        struct rte_eth_fc_conf fc_conf;

        /* DPDK-ETH hardware offload features,
         * from the enum set 'dpdk_hw_ol_features' */
        uint32_t hw_ol_features;

        /* Properties for link state change detection mode.
         * If lsc_interrupt_mode is set to false, poll mode is used,
         * otherwise interrupt mode is used. */
        bool requested_lsc_interrupt_mode;
        bool lsc_interrupt_mode;

        /* VF configuration. */
        struct eth_addr requested_hwaddr;

        /* Software-meter configuration. */
        struct netdev_doca_sw_meter requested_sw_meter;
        struct netdev_doca_sw_meter sw_meter;
    );

    PADDED_MEMBERS(CACHE_LINE_SIZE,
        /* Names of all XSTATS counters */
        struct rte_eth_xstat_name *rte_xstats_names;
        int rte_xstats_names_size;
        int rte_xstats_ids_size;
        uint64_t *rte_xstats_ids;
    );

    struct netdev_doca_vdpa_relay *relay;
    char *peer_name;
    bool netdev_rep;
    struct ovs_doca_netdev_data doca_netdev_data;
    enum netdev_doca_port_dir port_dir;
};

struct netdev_rxq_dpdk {
    struct netdev_rxq up;
    dpdk_port_t port_id;
};

static void netdev_doca_destruct(struct netdev *netdev);
static void netdev_doca_vdpa_destruct(struct netdev *netdev);

static int netdev_doca_get_sw_custom_stats(const struct netdev *,
                                           struct netdev_custom_stats *);
static void netdev_doca_configure_xstats(struct netdev_doca *dev);
static void netdev_doca_clear_xstats(struct netdev_doca *dev);
static struct netdev_doca *
netdev_doca_lookup_by_port_id(dpdk_port_t port_id);
static bool
netdev_doca_foreach_representor(struct netdev_doca *esw_dev,
                                bool (*cb)(struct netdev_doca *, dpdk_port_t));

struct ingress_policer *
netdev_doca_get_ingress_policer(const struct netdev_doca *dev);

static void netdev_doca_mbuf_dump(const char *prefix, const char *message,
                                  const struct rte_mbuf *);

bool
is_doca_class(const struct netdev_class *class)
{
    return class->destruct == netdev_doca_destruct
           || class->destruct == netdev_doca_vdpa_destruct;
}

/* DPDK NIC drivers allocate RX buffers at a particular granularity, typically
 * aligned at 1k or less. If a declared mbuf size is not a multiple of this
 * value, insufficient buffers are allocated to accomodate the packet in its
 * entirety. Furthermore, certain drivers need to ensure that there is also
 * sufficient space in the Rx buffer to accommodate two VLAN tags (for QinQ
 * frames). If the RX buffer is too small, then the driver enables scatter RX
 * behaviour, which reduces performance. To prevent this, use a buffer size
 * that is closest to 'mtu', but which satisfies the aforementioned criteria.
 */
static uint32_t
dpdk_buf_size(int mtu)
{
    return ROUND_UP(MTU_TO_MAX_FRAME_LEN(mtu), NETDEV_DPDK_MBUF_ALIGN)
            + RTE_PKTMBUF_HEADROOM;
}

static int
dpdk_get_user_adjusted_mtu(int port_adj_mtu, int port_mtu, int port_socket_id)
{
    int best_adj_user_mtu = INT_MAX;

    for (unsigned i = 0; i < n_user_mempools; i++) {
        int user_adj_mtu, user_socket_id;

        user_adj_mtu = user_mempools[i].adj_mtu;
        user_socket_id = user_mempools[i].socket_id;
        if (port_adj_mtu > user_adj_mtu
            || (user_socket_id != INT_MAX
                && user_socket_id != port_socket_id)) {
            continue;
        }
        if (user_adj_mtu < best_adj_user_mtu) {
            /* This is the is the lowest valid user MTU. */
            best_adj_user_mtu = user_adj_mtu;
            if (best_adj_user_mtu == port_adj_mtu) {
                /* Found an exact fit, no need to keep searching. */
                break;
            }
        }
    }
    if (best_adj_user_mtu == INT_MAX) {
        VLOG_DBG("No user configured shared mempool mbuf sizes found "
                 "suitable for port with MTU %d, NUMA %d.", port_mtu,
                 port_socket_id);
        best_adj_user_mtu = port_adj_mtu;
    } else {
        VLOG_DBG("Found user configured shared mempool with mbufs "
                 "of size %d, suitable for port with MTU %d, NUMA %d.",
                 MTU_TO_FRAME_LEN(best_adj_user_mtu), port_mtu,
                 port_socket_id);
    }
    return best_adj_user_mtu;
}

/* Allocates an area of 'sz' bytes from DPDK.  The memory is zero'ed.
 *
 * Unlike xmalloc(), this function can return NULL on failure. */
static void *
doca_rte_mzalloc(const char *type, size_t sz)
{
    return rte_zmalloc(type, sz, OVS_CACHE_LINE_SIZE);
}

/* This should match the implementation in netdev-dpdk.c
 * as dp-packet.c calls free_dpdk_buf(). */
static void
ovs_rte_pktmbuf_init(struct rte_mempool *mp OVS_UNUSED,
                     void *opaque_arg OVS_UNUSED,
                     void *_p,
                     unsigned i OVS_UNUSED)
{
    struct rte_mbuf *pkt = _p;

    dp_packet_init_dpdk((struct dp_packet *) pkt);
}

static int
dpdk_mp_full(const struct rte_mempool *mp) OVS_REQUIRES(dpdk_mp_mutex)
{
    /* At this point we want to know if all the mbufs are back
     * in the mempool. rte_mempool_full() is not atomic but it's
     * the best available and as we are no longer requesting mbufs
     * from the mempool, it means mbufs will not move from
     * 'mempool ring' --> 'mempool cache'. In rte_mempool_full()
     * the ring is counted before caches, so we won't get false
     * positives in this use case and we handle false negatives.
     *
     * If future implementations of rte_mempool_full() were to change
     * it could be possible for a false positive. Even that would
     * likely be ok, as there are additional checks during mempool
     * freeing but it would make things racey.
     */
    return rte_mempool_full(mp);
}

/* Free unused mempools. */
static void
dpdk_mp_sweep(void) OVS_REQUIRES(dpdk_mp_mutex)
{
    struct dpdk_mp *dmp;

    LIST_FOR_EACH_SAFE (dmp, list_node, &dpdk_mp_list) {
        if (!dmp->refcount && dpdk_mp_full(dmp->mp)) {
            VLOG_DBG("Freeing mempool \"%s\"", dmp->mp->name);
            ovs_list_remove(&dmp->list_node);
            rte_mempool_free(dmp->mp);
            rte_free(dmp);
        }
    }
}

#ifdef DOCA_OFFLOAD

static uint32_t
doca_calculate_mbufs(struct netdev_doca *dev)
{
     /* In DOCA mode, Shared RQ is used which mean RX queue is
     * only allocated on the eswitch manager. This mean we can determine
     * the max number of RX queue is the system.
     * In this case we will use the calculation of of number of mbufs
     * in per_port_memory mode. Rough estimation of number of mbufs
     * required for the eswitch manager and all its ports:
     * (<packets required to fill the device rxqs>
     * + <packets that could be stuck on other ports txqs>
     * + <packets in the pmd threads>
     * + <additional memory for corner cases>)
     */
    uint32_t n_pmd = dpif_doca_get_n_pmd_threads();
    uint32_t n_mbufs;

    n_mbufs = n_pmd * dev->requested_rxq_size
              + (n_pmd + 1) * dev->requested_txq_size
              + MIN(RTE_MAX_LCORE, n_pmd) * NETDEV_MAX_BURST
              + MIN_NB_MBUF;
    return n_mbufs;
}

#endif

static uint32_t
dpdk_calculate_mbufs__(struct netdev_doca *dev, int mtu)
{
    uint32_t n_mbufs;

    if (!per_port_memory) {
        /* Shared memory are being used.
         * XXX: this is a really rough method of provisioning memory.
         * It's impossible to determine what the exact memory requirements are
         * when the number of ports and rxqs that utilize a particular mempool
         * can change dynamically at runtime. For now, use this rough
         * heurisitic.
         */
        if (mtu >= RTE_ETHER_MTU) {
            n_mbufs = MAX_NB_MBUF;
        } else {
            n_mbufs = MIN_NB_MBUF;
        }
    } else {
        /* Per port memory is being used.
         * XXX: rough estimation of number of mbufs required for this port:
         * <packets required to fill the device rxqs>
         * + <packets that could be stuck on other ports txqs>
         * + <packets in the pmd threads>
         * + <additional memory for corner cases>
         */
        n_mbufs = dev->requested_n_rxq * dev->requested_rxq_size
                  + dev->requested_n_txq * dev->requested_txq_size
                  + MIN(RTE_MAX_LCORE, dev->requested_n_rxq) * NETDEV_MAX_BURST
                  + MIN_NB_MBUF;
    }

    return n_mbufs;
}

/* Calculating the required number of mbufs differs depending on the
 * mempool model being used. Check if per port memory is in use before
 * calculating.
 */
static uint32_t
dpdk_calculate_mbufs(struct netdev_doca *dev, int mtu)
{
#ifdef DOCA_OFFLOAD
    if (ovs_doca_initialized()) {
        return doca_calculate_mbufs(dev);
    }
#endif
    return dpdk_calculate_mbufs__(dev, mtu);
}

static struct dpdk_mp *
dpdk_mp_create(struct netdev_doca *dev, int mtu)
{
    char mp_name[RTE_MEMPOOL_NAMESIZE];
    const char *netdev_name = netdev_get_name(&dev->up);
    int socket_id = dev->requested_socket_id;
    uint32_t n_mbufs = 0;
    uint32_t mbuf_size = 0;
    uint32_t aligned_mbuf_size = 0;
    uint32_t mbuf_priv_data_len = 0;
    uint32_t pkt_size = 0;
    uint32_t hash = hash_string(netdev_name, 0);
    struct dpdk_mp *dmp = NULL;
    int ret;

    dmp = doca_rte_mzalloc("ovs_doca_mp", sizeof *dmp);
    if (!dmp) {
        return NULL;
    }
    dmp->socket_id = socket_id;
    dmp->esw_mgr_port_id = dev->esw_mgr_port_id;
    dmp->mtu = mtu;
    dmp->refcount = 1;

    /* Get the size of each mbuf, based on the MTU */
    mbuf_size = MTU_TO_FRAME_LEN(mtu);

    n_mbufs = dpdk_calculate_mbufs(dev, mtu);

    do {
        /* Full DPDK memory pool name must be unique and cannot be
         * longer than RTE_MEMPOOL_NAMESIZE. Note that for the shared
         * mempool case this can result in one device using a mempool
         * which references a different device in it's name. However as
         * mempool names are hashed, the device name will not be readable
         * so this is not an issue for tasks such as debugging.
         */
        ret = snprintf(mp_name, RTE_MEMPOOL_NAMESIZE,
                       "ovs%08x%02d%05d%07u",
                        hash, socket_id, mtu, n_mbufs);
        if (ret < 0 || ret >= RTE_MEMPOOL_NAMESIZE) {
            VLOG_DBG("snprintf returned %d. "
                     "Failed to generate a mempool name for \"%s\". "
                     "Hash:0x%x, socket_id: %d, mtu:%d, mbufs:%u.",
                     ret, netdev_name, hash, socket_id, mtu, n_mbufs);
            break;
        }

        VLOG_DBG("Port %s: Requesting a mempool of %u mbufs of size %u "
                  "on socket %d for eswitch manager port %u with %d Rx and %d Tx queues, "
                  "cache line size of %u",
                  netdev_name, n_mbufs, mbuf_size, socket_id, dev->esw_mgr_port_id,
                  dev->requested_n_rxq, dev->requested_n_txq,
                  RTE_CACHE_LINE_SIZE);

        /* The size of the mbuf's private area (i.e. area that holds OvS'
         * dp_packet data)*/
        mbuf_priv_data_len = sizeof(struct dp_packet) -
                                 sizeof(struct rte_mbuf);
        /* The size of the entire dp_packet. */
        pkt_size = sizeof(struct dp_packet) + mbuf_size;
        /* mbuf size, rounded up to cacheline size. */
        aligned_mbuf_size = ROUND_UP(pkt_size, RTE_CACHE_LINE_SIZE);
        /* If there is a size discrepancy, add padding to mbuf_priv_data_len.
         * This maintains mbuf size cache alignment, while also honoring RX
         * buffer alignment in the data portion of the mbuf. If this adjustment
         * is not made, there is a possiblity later on that for an element of
         * the mempool, buf, buf->data_len < (buf->buf_len - buf->data_off).
         * This is problematic in the case of multi-segment mbufs, particularly
         * when an mbuf segment needs to be resized (when [push|popp]ing a VLAN
         * header, for example.
         */
        mbuf_priv_data_len += (aligned_mbuf_size - pkt_size);

        dmp->mp = rte_pktmbuf_pool_create(mp_name, n_mbufs, MP_CACHE_SZ,
                                          mbuf_priv_data_len,
                                          mbuf_size,
                                          socket_id);

        if (dmp->mp) {
            VLOG_DBG("Allocated \"%s\" mempool with %u mbufs",
                     mp_name, n_mbufs);
            /* rte_pktmbuf_pool_create has done some initialization of the
             * rte_mbuf part of each dp_packet, while ovs_rte_pktmbuf_init
             * initializes some OVS specific fields of dp_packet.
             */
            rte_mempool_obj_iter(dmp->mp, ovs_rte_pktmbuf_init, NULL);
            return dmp;
        } else if (rte_errno == EEXIST) {
            /* A mempool with the same name already exists.  We just
             * retrieve its pointer to be returned to the caller. */
            dmp->mp = rte_mempool_lookup(mp_name);
            /* As the mempool create returned EEXIST we can expect the
             * lookup has returned a valid pointer.  If for some reason
             * that's not the case we keep track of it. */
            VLOG_DBG("A mempool with name \"%s\" already exists at %p.",
                     mp_name, dmp->mp);
            return dmp;
        } else {
            VLOG_DBG("Failed to create mempool \"%s\" with a request of "
                     "%u mbufs, retrying with %u mbufs",
                     mp_name, n_mbufs, n_mbufs / 2);
        }
    } while (!dmp->mp && rte_errno == ENOMEM && (n_mbufs /= 2) >= MIN_NB_MBUF);

    VLOG_ERR("Failed to create mempool \"%s\" with a request of %u mbufs",
             mp_name, n_mbufs);

    rte_free(dmp);
    return NULL;
}

static struct dpdk_mp *
dpdk_mp_get(struct netdev_doca *dev, int mtu)
{
    struct dpdk_mp *dmp = NULL, *next;
    bool reuse = false;

    ovs_mutex_lock(&dpdk_mp_mutex);
    /* Check if shared memory is being used, if so check existing mempools
     * to see if reuse is possible. */
    if (!per_port_memory) {
        /* If user has provided defined mempools, check if one is suitable
         * and get new buffer size.*/
        mtu = dpdk_get_user_adjusted_mtu(mtu, dev->requested_mtu,
                                         dev->requested_socket_id);
        LIST_FOR_EACH (dmp, list_node, &dpdk_mp_list) {
            if (dmp->socket_id == dev->requested_socket_id
                && dmp->esw_mgr_port_id == dev->esw_mgr_port_id
                && dmp->mtu == mtu) {
                VLOG_DBG("Reusing mempool \"%s\"", dmp->mp->name);
                dmp->refcount++;
                reuse = true;
                break;
            }
        }
    }
    /* Sweep mempools after reuse or before create. */
    dpdk_mp_sweep();

    if (!reuse) {
        dmp = dpdk_mp_create(dev, mtu);
        if (dmp) {
            /* Shared memory will hit the reuse case above so will not
             * request a mempool that already exists but we need to check
             * for the EEXIST case for per port memory case. Compare the
             * mempool returned by dmp to each entry in dpdk_mp_list. If a
             * match is found, free dmp as a new entry is not required, set
             * dmp to point to the existing entry and increment the refcount
             * to avoid being freed at a later stage.
             */
            if (per_port_memory && rte_errno == EEXIST) {
                LIST_FOR_EACH (next, list_node, &dpdk_mp_list) {
                    if (dmp->mp == next->mp) {
                        rte_free(dmp);
                        dmp = next;
                        dmp->refcount++;
                    }
                }
            } else {
                ovs_list_push_back(&dpdk_mp_list, &dmp->list_node);
            }
        }
    }

    ovs_mutex_unlock(&dpdk_mp_mutex);

    return dmp;
}

/* Decrement reference to a mempool. */
static void
dpdk_mp_put(struct dpdk_mp *dmp)
{
    if (!dmp) {
        return;
    }

    ovs_mutex_lock(&dpdk_mp_mutex);
    ovs_assert(dmp->refcount);
    dmp->refcount--;
    ovs_mutex_unlock(&dpdk_mp_mutex);
}

/* Depending on the memory model being used this function tries to
 * identify and reuse an existing mempool or tries to allocate a new
 * mempool on requested_socket_id with mbuf size corresponding to the
 * requested_mtu. On success, a new configuration will be applied.
 * On error, device will be left unchanged. */
static int
netdev_doca_mempool_configure(struct netdev_doca *dev)
    OVS_REQUIRES(dev->mutex)
{
    uint32_t buf_size = dpdk_buf_size(dev->requested_mtu);
    struct dpdk_mp *dmp;
    int ret = 0;

    /* With shared memory we do not need to configure a mempool if the MTU
     * and socket ID have not changed, the previous configuration is still
     * valid so return 0 */
    if (!per_port_memory && dev->mtu == dev->requested_mtu
        && dev->socket_id == dev->requested_socket_id) {
        return ret;
    }

    dmp = dpdk_mp_get(dev, FRAME_LEN_TO_MTU(buf_size));
    if (!dmp) {
        VLOG_ERR("Failed to create memory pool for netdev "
                 "%s, with MTU %d on socket %d: %s\n",
                 dev->up.name, dev->requested_mtu, dev->requested_socket_id,
                 rte_strerror(rte_errno));
        ret = rte_errno;
    } else {
        /* Check for any pre-existing dpdk_mp for the device before accessing
         * the associated mempool.
         */
        if (dev->dpdk_mp != NULL) {
            /* A new MTU was requested, decrement the reference count for the
             * devices current dpdk_mp. This is required even if a pointer to
             * same dpdk_mp is returned by dpdk_mp_get. The refcount for dmp
             * has already been incremented by dpdk_mp_get at this stage so it
             * must be decremented to keep an accurate refcount for the
             * dpdk_mp.
             */
            dpdk_mp_put(dev->dpdk_mp);
        }
        dev->dpdk_mp = dmp;
        dev->mtu = dev->requested_mtu;
        dev->socket_id = dev->requested_socket_id;
        dev->max_packet_len = MTU_TO_FRAME_LEN(dev->mtu);
    }

    return ret;
}

static void
check_link_status(struct netdev_doca *dev)
{
    struct rte_eth_link link;

    if (!rte_eth_dev_is_valid_port(dev->port_id)) {
        memset(&link, 0, sizeof link);
    } else {
        if (rte_eth_link_get_nowait(dev->port_id, &link) < 0) {
            VLOG_DBG_RL(&rl,
                        "Failed to retrieve link status for port "DPDK_PORT_ID_FMT,
                        dev->port_id);
            return;
        }
    }

    if (dev->link.link_status != link.link_status) {
        netdev_change_seq_changed(&dev->up);

        dev->link_reset_cnt++;
        dev->link = link;
        if (dev->link.link_status) {
            VLOG_DBG_RL(&rl,
                        "Port "DPDK_PORT_ID_FMT" Link Up - speed %u Mbps - %s",
                        dev->port_id, (unsigned) dev->link.link_speed,
                        (dev->link.link_duplex == RTE_ETH_LINK_FULL_DUPLEX)
                        ? "full-duplex" : "half-duplex");
        } else {
            VLOG_DBG_RL(&rl, "Port "DPDK_PORT_ID_FMT" Link Down",
                        dev->port_id);
        }
    }
}

static void *
dpdk_watchdog(void *dummy OVS_UNUSED)
{
    struct netdev_doca *dev;

    pthread_detach(pthread_self());

    for (;;) {
        ovs_mutex_lock(&dpdk_mutex);
        LIST_FOR_EACH (dev, list_node, &dpdk_list) {
            ovs_mutex_lock(&dev->mutex);
            if (!dev->netdev_rep) {
                ovs_mutex_unlock(&dev->mutex);
                continue;
            }

            if (dev->type == DPDK_DEV_ETH) {
                check_link_status(dev);
            }
            ovs_mutex_unlock(&dev->mutex);
        }
        ovs_mutex_unlock(&dpdk_mutex);
        xsleep(DPDK_PORT_WATCHDOG_INTERVAL);
    }

    return NULL;
}

static void
netdev_doca_update_netdev_flag(struct netdev_doca *dev,
                               enum dpdk_hw_ol_features hw_ol_features,
                               enum netdev_ol_flags flag)
    OVS_REQUIRES(dev->mutex)
{
    struct netdev *netdev = &dev->up;

    if (dev->hw_ol_features & hw_ol_features) {
        netdev->ol_flags |= flag;
    } else {
        netdev->ol_flags &= ~flag;
    }
}

static void
netdev_doca_update_netdev_flags(struct netdev_doca *dev)
    OVS_REQUIRES(dev->mutex)
{
    netdev_doca_update_netdev_flag(dev, NETDEV_TX_IPV4_CKSUM_OFFLOAD,
                                   NETDEV_TX_OFFLOAD_IPV4_CKSUM);
    netdev_doca_update_netdev_flag(dev, NETDEV_TX_TCP_CKSUM_OFFLOAD,
                                   NETDEV_TX_OFFLOAD_TCP_CKSUM);
    netdev_doca_update_netdev_flag(dev, NETDEV_TX_UDP_CKSUM_OFFLOAD,
                                   NETDEV_TX_OFFLOAD_UDP_CKSUM);
    netdev_doca_update_netdev_flag(dev, NETDEV_TX_SCTP_CKSUM_OFFLOAD,
                                   NETDEV_TX_OFFLOAD_SCTP_CKSUM);
    netdev_doca_update_netdev_flag(dev, NETDEV_TX_TSO_OFFLOAD,
                                   NETDEV_TX_OFFLOAD_TCP_TSO);
    netdev_doca_update_netdev_flag(dev, NETDEV_TX_VXLAN_TNL_TSO_OFFLOAD,
                                   NETDEV_TX_VXLAN_TNL_TSO);
    netdev_doca_update_netdev_flag(dev, NETDEV_TX_GENEVE_TNL_TSO_OFFLOAD,
                                   NETDEV_TX_GENEVE_TNL_TSO);
    netdev_doca_update_netdev_flag(dev, NETDEV_TX_OUTER_IP_CKSUM_OFFLOAD,
                                   NETDEV_TX_OFFLOAD_OUTER_IP_CKSUM);
    netdev_doca_update_netdev_flag(dev, NETDEV_TX_OUTER_UDP_CKSUM_OFFLOAD,
                                   NETDEV_TX_OFFLOAD_OUTER_UDP_CKSUM);
}

static int
dpdk_eth_dev_port_config_complete(struct netdev_doca *dev, int n_rxq, int n_txq)
{
    uint32_t extra = 0;
    uint16_t conf_mtu;
    int diag = 0;

    free(dev->sw_tx_stats);
    dev->sw_tx_stats = xcalloc(n_txq, sizeof(struct ovs_doca_tx_stats));
    for (int i = 0; i < n_txq; i++) {
        atomic_init(&dev->sw_tx_stats[i].n_packets, 0);
        atomic_init(&dev->sw_tx_stats[i].n_bytes, 0);
    }

    dev->up.n_rxq = n_rxq;
    dev->up.n_txq = n_txq;

    /* Save additional MTU size for miss path ENCAP */
    extra = sizeof(struct vxlan_meta_header);
    diag = rte_eth_dev_set_mtu(dev->port_id, dev->mtu + extra);

    if (diag) {
        /* A device may not support rte_eth_dev_set_mtu, in this case
         * flag a warning to the user and include the devices configured
         * MTU value that will be used instead. */
        if (-ENOTSUP == diag) {
            rte_eth_dev_get_mtu(dev->port_id, &conf_mtu);
            VLOG_WARN("Interface %s does not support MTU configuration, "
                      "max packet size supported is %"PRIu16".",
                      dev->up.name, conf_mtu);
        } else {
            VLOG_ERR("Interface %s MTU (%d) setup error: %s",
                     dev->up.name, dev->mtu, rte_strerror(-diag));
        }
    }

    return diag;
}

static int
dpdk_eth_dev_port_config(struct netdev_doca *dev,
                         const struct rte_eth_dev_info *info,
                         int n_rxq, int n_txq)
{
    struct rte_eth_conf conf = port_conf;
    struct rte_eth_hairpin_conf hairpin_conf = {
        .peer_count = 1,
    };
    int diag = 0;
    int i;

    /* As of DPDK 17.11.1 a few PMDs require to explicitly enable
     * scatter to support jumbo RX.
     * Setting scatter for the device is done after checking for
     * scatter support in the device capabilites. */
    if (dev->mtu > RTE_ETHER_MTU) {
        if (dev->hw_ol_features & NETDEV_RX_HW_SCATTER) {
            conf.rxmode.offloads |= RTE_ETH_RX_OFFLOAD_SCATTER;
        }
    }

    conf.intr_conf.lsc = dev->lsc_interrupt_mode;

    if (dev->hw_ol_features & NETDEV_RX_CHECKSUM_OFFLOAD) {
        conf.rxmode.offloads |= RTE_ETH_RX_OFFLOAD_CHECKSUM;
    }

    if (!(dev->hw_ol_features & NETDEV_RX_HW_CRC_STRIP)
        && info->rx_offload_capa & RTE_ETH_RX_OFFLOAD_KEEP_CRC) {
        conf.rxmode.offloads |= RTE_ETH_RX_OFFLOAD_KEEP_CRC;
    }

    if (dev->hw_ol_features & NETDEV_TX_IPV4_CKSUM_OFFLOAD) {
        conf.txmode.offloads |= RTE_ETH_TX_OFFLOAD_IPV4_CKSUM;
    }

    if (dev->hw_ol_features & NETDEV_TX_TCP_CKSUM_OFFLOAD) {
        conf.txmode.offloads |= RTE_ETH_TX_OFFLOAD_TCP_CKSUM;
    }

    if (dev->hw_ol_features & NETDEV_TX_UDP_CKSUM_OFFLOAD) {
        conf.txmode.offloads |= RTE_ETH_TX_OFFLOAD_UDP_CKSUM;
    }

    if (dev->hw_ol_features & NETDEV_TX_SCTP_CKSUM_OFFLOAD) {
        conf.txmode.offloads |= RTE_ETH_TX_OFFLOAD_SCTP_CKSUM;
    }

    if (dev->hw_ol_features & NETDEV_TX_TSO_OFFLOAD) {
        conf.txmode.offloads |= RTE_ETH_TX_OFFLOAD_TCP_TSO;
    }

    if (dev->hw_ol_features & NETDEV_TX_VXLAN_TNL_TSO_OFFLOAD) {
        conf.txmode.offloads |= RTE_ETH_TX_OFFLOAD_VXLAN_TNL_TSO;
    }

    if (dev->hw_ol_features & NETDEV_TX_GENEVE_TNL_TSO_OFFLOAD) {
        conf.txmode.offloads |= RTE_ETH_TX_OFFLOAD_GENEVE_TNL_TSO;
    }

    if (dev->hw_ol_features & NETDEV_TX_OUTER_IP_CKSUM_OFFLOAD) {
        conf.txmode.offloads |= RTE_ETH_TX_OFFLOAD_OUTER_IPV4_CKSUM;
    }

    if (dev->hw_ol_features & NETDEV_TX_OUTER_UDP_CKSUM_OFFLOAD) {
        conf.txmode.offloads |= RTE_ETH_TX_OFFLOAD_OUTER_UDP_CKSUM;
    }

    /* Limit configured rss hash functions to only those supported
     * by the eth device. */
    conf.rx_adv_conf.rss_conf.rss_hf &= info->flow_type_rss_offloads;
    if (conf.rx_adv_conf.rss_conf.rss_hf == 0) {
        conf.rxmode.mq_mode = RTE_ETH_MQ_RX_NONE;
    } else {
        conf.rxmode.mq_mode = RTE_ETH_MQ_RX_RSS;
    }

    if (!netdev_doca_is_esw_mgr(&dev->up)) {
        rte_eth_dev_configure(dev->port_id, 0, 0, &conf);
        return dpdk_eth_dev_port_config_complete(dev, n_rxq, n_txq);
    }

    /* A device may report more queues than it makes available (this has
     * been observed for Intel xl710, which reserves some of them for
     * SRIOV):  rte_eth_*_queue_setup will fail if a queue is not
     * available.  When this happens we can retry the configuration
     * and request less queues */
    while (n_rxq && n_txq) {
        if (diag) {
            VLOG_INFO("Retrying setup with (rxq:%d txq:%d hairpinq:%d)", n_rxq,
                      n_txq, NR_HAIRPINQ);
        }

        diag = rte_eth_dev_configure(dev->port_id, n_rxq + NR_HAIRPINQ,
                                     n_txq + NR_HAIRPINQ, &conf);
        if (diag) {
            VLOG_WARN("Interface %s eth_dev setup error %s\n",
                      dev->up.name, rte_strerror(-diag));
            break;
        }

        for (i = 0; i < n_txq; i++) {
            diag = rte_eth_tx_queue_setup(dev->port_id, i, dev->txq_size,
                                          dev->socket_id, NULL);
            if (diag) {
                VLOG_INFO("Interface %s unable to setup txq(%d): %s",
                          dev->up.name, i, rte_strerror(-diag));
                break;
            }
        }

        if (i != n_txq) {
            /* Retry with less tx queues */
            n_txq = i;
            continue;
        }

        for (i = 0; i < n_rxq; i++) {
            diag = rte_eth_rx_queue_setup(dev->port_id, i, dev->rxq_size,
                                          dev->socket_id, NULL,
                                          dev->dpdk_mp->mp);
            if (diag) {
                VLOG_INFO("Interface %s unable to setup rxq(%d): %s",
                          dev->up.name, i, rte_strerror(-diag));
                break;
            }
        }

        if (i != n_rxq) {
            /* Retry with less rx queues */
            n_rxq = i;
            continue;
        }

        diag = netdev_doca_setup_hairpin_queues(&dev->up, dev->port_id,
                                                dev->rxq_size, dev->txq_size,
                                                n_rxq, n_txq, &hairpin_conf);
        if (diag) {
            return diag;
        }

        return dpdk_eth_dev_port_config_complete(dev, n_rxq, n_txq);
    }

    return diag;
}

static void
dpdk_eth_flow_ctrl_setup(struct netdev_doca *dev) OVS_REQUIRES(dev->mutex)
{
    if (rte_eth_dev_flow_ctrl_set(dev->port_id, &dev->fc_conf)) {
        VLOG_WARN("Failed to enable flow control on device "DPDK_PORT_ID_FMT,
                  dev->port_id);
    }
}

static void
dpdk_eth_dev_init_rx_metadata(struct netdev_doca *dev)
{
    uint64_t rx_metadata = 0;
    int ret;

    if (dev->rx_metadata_delivery_configured) {
        return;
    }

    /* For the fallback offload (non-"transfer" rules). */
    rx_metadata |= RTE_ETH_RX_METADATA_USER_MARK;

#ifdef ALLOW_EXPERIMENTAL_API
    /* For the tunnel offload.  */
    rx_metadata |= RTE_ETH_RX_METADATA_TUNNEL_ID;
#endif /* ALLOW_EXPERIMENTAL_API */

    ret = rte_eth_rx_metadata_negotiate(dev->port_id, &rx_metadata);
    if (ret == 0) {
        if (!(rx_metadata & RTE_ETH_RX_METADATA_USER_MARK)) {
            VLOG_DBG("%s: The NIC will not provide per-packet USER_MARK",
                     netdev_get_name(&dev->up));
        }
#ifdef ALLOW_EXPERIMENTAL_API
        if (!(rx_metadata & RTE_ETH_RX_METADATA_TUNNEL_ID)) {
            VLOG_DBG("%s: The NIC will not provide per-packet TUNNEL_ID",
                     netdev_get_name(&dev->up));
        }
#endif /* ALLOW_EXPERIMENTAL_API */
    } else {
        VLOG(ret == -ENOTSUP ? VLL_DBG : VLL_WARN,
             "%s: Cannot negotiate Rx metadata: %s",
             netdev_get_name(&dev->up), rte_strerror(-ret));
    }

    dev->rx_metadata_delivery_configured = true;
}

static bool
netdev_doca_rep_set_esw_mgr_port_id(struct netdev_doca *dev,
                                    dpdk_port_t esw_mgr_port_id)
{
    dev->esw_mgr_port_id = esw_mgr_port_id;
    netdev_set_need_restart(&dev->up);

    return true;
}

static void
set_etheraddr(struct netdev_doca *dev)
{
    struct rte_ether_addr eth_addr;
    int err;

    memset(&eth_addr, 0, sizeof eth_addr);

    err = -1;
    if (dev->peer_name) {
        struct eth_addr peer_hwaddr;
        struct netdev *peer;

        err = netdev_open(dev->peer_name, NULL, &peer);
        if (!err) {
            err = netdev_get_etheraddr(peer, &peer_hwaddr);
            memcpy(eth_addr.addr_bytes, peer_hwaddr.ea, ETH_ADDR_LEN);
            netdev_close(peer);
        }
    }
    if (err) {
        rte_eth_macaddr_get(dev->port_id, &eth_addr);
    }
    VLOG_INFO_RL(&rl, "Port "DPDK_PORT_ID_FMT": "ETH_ADDR_FMT,
                 dev->port_id, ETH_ADDR_BYTES_ARGS(eth_addr.addr_bytes));

    memcpy(dev->hwaddr.ea, eth_addr.addr_bytes, ETH_ADDR_LEN);
}

static int
dpdk_eth_dev_init(struct netdev_doca *dev)
    OVS_REQUIRES(dpdk_mutex)
    OVS_REQUIRES(dev->mutex)
{
    struct netdev *netdev = &dev->up;
    struct rte_eth_dev_info info;
    int diag;
    int n_rxq, n_txq;
    uint32_t rx_chksm_offload_capa = RTE_ETH_RX_OFFLOAD_UDP_CKSUM |
                                     RTE_ETH_RX_OFFLOAD_TCP_CKSUM |
                                     RTE_ETH_RX_OFFLOAD_IPV4_CKSUM;
    if (netdev_is_flow_api_enabled()) {
        /*
         * Full tunnel offload requires that tunnel ID metadata be
         * delivered with "miss" packets from the hardware to the
         * PMD. The same goes for megaflow mark metadata which is
         * used in MARK + RSS offload scenario.
         *
         * Request delivery of such metadata.
         */
        dpdk_eth_dev_init_rx_metadata(dev);
    }

    diag = rte_eth_dev_info_get(dev->port_id, &info);
    if (diag < 0) {
        VLOG_ERR("Interface %s rte_eth_dev_info_get error: %s",
                 dev->up.name, rte_strerror(-diag));
        return -diag;
    }

    dev->is_representor = !!(*info.dev_flags & RTE_ETH_DEV_REPRESENTOR);

    if (strstr(info.driver_name, "vf") != NULL) {
        VLOG_INFO("Virtual function detected, HW_CRC_STRIP will be enabled");
        dev->hw_ol_features |= NETDEV_RX_HW_CRC_STRIP;
    } else {
        dev->hw_ol_features &= ~NETDEV_RX_HW_CRC_STRIP;
    }

    if ((info.rx_offload_capa & rx_chksm_offload_capa) !=
            rx_chksm_offload_capa) {
        VLOG_WARN("Rx checksum offload is not supported on port "
                  DPDK_PORT_ID_FMT, dev->port_id);
        dev->hw_ol_features &= ~NETDEV_RX_CHECKSUM_OFFLOAD;
    } else {
        dev->hw_ol_features |= NETDEV_RX_CHECKSUM_OFFLOAD;
    }

    if (info.rx_offload_capa & RTE_ETH_RX_OFFLOAD_SCATTER) {
        dev->hw_ol_features |= NETDEV_RX_HW_SCATTER;
    } else {
        /* Do not warn on lack of scatter support */
        dev->hw_ol_features &= ~NETDEV_RX_HW_SCATTER;
    }

    if (!strcmp(info.driver_name, "net_tap")) {
        /* FIXME: L4 checksum offloading is broken in DPDK net/tap driver.
         * This workaround can be removed once the fix makes it to a DPDK
         * LTS release used by OVS. */
        VLOG_INFO("%s: disabled Tx L4 checksum offloads for a net/tap port.",
                  netdev_get_name(&dev->up));
        info.tx_offload_capa &= ~RTE_ETH_TX_OFFLOAD_UDP_CKSUM;
        info.tx_offload_capa &= ~RTE_ETH_TX_OFFLOAD_TCP_CKSUM;
    }

    if (info.tx_offload_capa & RTE_ETH_TX_OFFLOAD_IPV4_CKSUM) {
        dev->hw_ol_features |= NETDEV_TX_IPV4_CKSUM_OFFLOAD;
    } else {
        dev->hw_ol_features &= ~NETDEV_TX_IPV4_CKSUM_OFFLOAD;
    }

    if (info.tx_offload_capa & RTE_ETH_TX_OFFLOAD_TCP_CKSUM) {
        dev->hw_ol_features |= NETDEV_TX_TCP_CKSUM_OFFLOAD;
    } else {
        dev->hw_ol_features &= ~NETDEV_TX_TCP_CKSUM_OFFLOAD;
    }

    if (info.tx_offload_capa & RTE_ETH_TX_OFFLOAD_UDP_CKSUM) {
        dev->hw_ol_features |= NETDEV_TX_UDP_CKSUM_OFFLOAD;
    } else {
        dev->hw_ol_features &= ~NETDEV_TX_UDP_CKSUM_OFFLOAD;
    }

    if (info.tx_offload_capa & RTE_ETH_TX_OFFLOAD_SCTP_CKSUM) {
        dev->hw_ol_features |= NETDEV_TX_SCTP_CKSUM_OFFLOAD;
    } else {
        dev->hw_ol_features &= ~NETDEV_TX_SCTP_CKSUM_OFFLOAD;
    }

    if (info.tx_offload_capa & RTE_ETH_TX_OFFLOAD_OUTER_IPV4_CKSUM) {
        dev->hw_ol_features |= NETDEV_TX_OUTER_IP_CKSUM_OFFLOAD;
    } else {
        dev->hw_ol_features &= ~NETDEV_TX_OUTER_IP_CKSUM_OFFLOAD;
    }

    if (info.tx_offload_capa & RTE_ETH_TX_OFFLOAD_OUTER_UDP_CKSUM) {
        dev->hw_ol_features |= NETDEV_TX_OUTER_UDP_CKSUM_OFFLOAD;
    } else {
        dev->hw_ol_features &= ~NETDEV_TX_OUTER_UDP_CKSUM_OFFLOAD;
    }

    dev->hw_ol_features &= ~NETDEV_TX_TSO_OFFLOAD;
    if (userspace_tso_enabled()) {
        if (info.tx_offload_capa & RTE_ETH_TX_OFFLOAD_TCP_TSO) {
            dev->hw_ol_features |= NETDEV_TX_TSO_OFFLOAD;
        } else {
            VLOG_WARN("%s: Tx TSO offload is not supported.",
                      netdev_get_name(&dev->up));
        }

        if (info.tx_offload_capa & RTE_ETH_TX_OFFLOAD_VXLAN_TNL_TSO) {
            dev->hw_ol_features |= NETDEV_TX_VXLAN_TNL_TSO_OFFLOAD;
        } else {
            VLOG_WARN("%s: Tx Vxlan tunnel TSO offload is not supported.",
                      netdev_get_name(&dev->up));
        }

        if (info.tx_offload_capa & RTE_ETH_TX_OFFLOAD_GENEVE_TNL_TSO) {
            dev->hw_ol_features |= NETDEV_TX_GENEVE_TNL_TSO_OFFLOAD;
        } else {
            VLOG_WARN("%s: Tx Geneve tunnel TSO offload is not supported.",
                      netdev_get_name(&dev->up));
        }
    }

    n_rxq = MIN(info.max_rx_queues, dev->up.n_rxq);
    n_txq = MIN(info.max_tx_queues, dev->up.n_txq);

    diag = dpdk_eth_dev_port_config(dev, &info, n_rxq, n_txq);
    if (diag) {
        VLOG_ERR("Interface %s(rxq:%d txq:%d lsc interrupt mode:%s) "
                 "configure error: %s",
                 dev->up.name, n_rxq, n_txq,
                 dev->lsc_interrupt_mode ? "true" : "false",
                 rte_strerror(-diag));
        return -diag;
    }

    /* Try again if eswitch manager is not UP. */
    if (!netdev_doca_is_esw_mgr(netdev) &&
        netdev_doca_get_esw_mgr_port_id(netdev) == -1) {
        return EAGAIN;
    }

    netdev_doca_initialized = true;
    /* When a representor is probed before the eswitch, dpdk implicitly
        * probes the latter, thus probe is not called from
        * netdev_doca_process_devargs(). In this case we call probe at
        * ovs_doca_port_start(), and make sure the device is marked as
        * "attached".
        */
    dev->attached = true;
    if (ovs_doca_port_start(netdev, &dev->doca_netdev_data, dev->socket_id)) {
        VLOG_ERR("Failed to init DOCA port %s port_id %"PRIu16,
                    netdev_get_name(netdev), dev->port_id);
        return -1;
    }

    atomic_store(&dev->started, true);
    ovs_mutex_unlock(&dev->mutex);
    if (!netdev_get_dpif_type(netdev)) {
        netdev_set_dpif_type(netdev, "doca");
    }

    if (netdev_init_flow_api(netdev)) {
        /* Initialize AUX tables also when offload is disabled.
            * This is needed in order to support slow path offloads
            * like SW-meter, meta copy to SW and LACP/LLDP/802.1X
            * forward to Kernel.
            */
        diag = netdev_offload_doca_init_flow_api(netdev);
        if (diag) {
            VLOG_ERR("Interface %s DOCA init flow api error: %s", dev->up.name,
                        rte_strerror(-diag));
            goto err_doca_init_flow_api;
        }
    }

    ovs_mutex_lock(&dev->mutex);

    netdev_doca_configure_xstats(dev);

    set_etheraddr(dev);

    if (rte_eth_link_get_nowait(dev->port_id, &dev->link) < 0) {
        memset(&dev->link, 0, sizeof dev->link);
    }

    /* Upon success of esw_mgr port, update the representor's field of it. */
    if (netdev_doca_get_esw_mgr_port_id(netdev) == dev->port_id &&
        netdev_doca_foreach_representor(dev, netdev_doca_rep_set_esw_mgr_port_id)) {
        /* If a representor is reconfigured a result of its eswitch manager change, it might not be
         * synced in the bridge's database. Signal it to reconfigure, to make it right.
         */
        rtnetlink_report_link();
    }
    return 0;

err_doca_init_flow_api:
    ovs_mutex_lock(&dev->mutex);
    return -diag;
}

static struct netdev_doca *
netdev_doca_cast(const struct netdev *netdev)
{
    return CONTAINER_OF(netdev, struct netdev_doca, up);
}

struct doca_flow_port *
netdev_doca_port_get(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    return dev->doca_netdev_data.port;
}

bool
netdev_doca_parse_dpdk_devargs_pci(const char *devargs, struct rte_pci_addr *rte_pci)
{
    struct rte_devargs da;
    bool rv = true;

    if (rte_devargs_parse(&da, devargs)) {
        return false; /* Error has already been printed by the RTE function. */
    }

    if (rte_pci_addr_parse(da.name, rte_pci)) {
        rv = false;
        goto out;
    }

out:
    rte_devargs_reset(&da);
    return rv;
}

struct ovs_doca_esw_ctx *
netdev_doca_ovs_doca_esw_ctx(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    if (!is_doca_class(netdev->netdev_class)) {
        return NULL;
    }

    return dev->doca_netdev_data.esw_ctx;
}

static struct netdev *
netdev_doca_alloc(void)
{
    struct netdev_doca *dev;

    dev = doca_rte_mzalloc("ovs_doca_netdev", sizeof *dev);
    if (!dev) {
        return NULL;
    }

    ovs_doca_set_properties(&dev->doca_netdev_data, DPDK_ETH_PORT_ID_INVALID, NULL,
                            &dev->esw_mgr_port_id, &dev->attached, &dev->up.n_rxq);

    return &dev->up;
}

static struct dpdk_tx_queue *
netdev_doca_alloc_txq(unsigned int n_txqs)
{
    struct dpdk_tx_queue *txqs;
    unsigned i;

    txqs = doca_rte_mzalloc("ovs_doca_txq", n_txqs * sizeof *txqs);
    if (txqs) {
        for (i = 0; i < n_txqs; i++) {
            /* Initialize map for vhost devices. */
            txqs[i].map = OVS_VHOST_QUEUE_MAP_UNKNOWN;
            rte_spinlock_init(&txqs[i].tx_lock);
        }
    }

    return txqs;
}

static int
common_construct(struct netdev *netdev, dpdk_port_t port_no,
                 enum dpdk_dev_type type, int socket_id)
    OVS_REQUIRES(dpdk_mutex)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    ovs_mutex_init(&dev->mutex);

    rte_spinlock_init(&dev->stats_lock);

    /* If the 'sid' is negative, it means that the kernel fails
     * to obtain the pci numa info.  In that situation, always
     * use 'SOCKET0'. */
    dev->socket_id = socket_id < 0 ? SOCKET0 : socket_id;
    dev->requested_socket_id = dev->socket_id;
    dev->port_id = port_no;
    dev->esw_mgr_port_id = port_no;
    dev->type = type;
    dev->flags = 0;
    dev->requested_mtu = RTE_ETHER_MTU;
    dev->max_packet_len = MTU_TO_FRAME_LEN(dev->mtu);
    dev->requested_lsc_interrupt_mode = 0;
    dev->attached = false;
    atomic_store(&dev->started, false);

    netdev->n_rxq = 0;
    netdev->n_txq = 0;
    dev->user_n_rxq = NR_QUEUE;
    dev->requested_n_rxq = NR_QUEUE;
    dev->requested_n_txq = NR_QUEUE;
    dev->requested_rxq_size = NIC_PORT_DEFAULT_RXQ_SIZE;
    dev->requested_txq_size = NIC_PORT_DEFAULT_TXQ_SIZE;

    /* Initialize the flow control to NULL */
    memset(&dev->fc_conf, 0, sizeof dev->fc_conf);

    /* Initilize the hardware offload flags to 0 */
    dev->hw_ol_features = 0;

    dev->rx_metadata_delivery_configured = false;

    dev->flags = NETDEV_UP | NETDEV_PROMISC;

    ovs_list_push_back(&dpdk_list, &dev->list_node);

    netdev_request_reconfigure(netdev);

    dev->rte_xstats_names = NULL;
    dev->rte_xstats_names_size = 0;

    dev->rte_xstats_ids = NULL;
    dev->rte_xstats_ids_size = 0;

    dev->sw_stats = xzalloc(sizeof *dev->sw_stats);
    dev->sw_stats->tx_retries = UINT64_MAX;

    return 0;
}

static int
netdev_doca_construct(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    char peer[IFNAMSIZ];
    int err;

    ovs_mutex_lock(&dpdk_mutex);
    err = common_construct(netdev, DPDK_ETH_PORT_ID_INVALID,
                           DPDK_DEV_ETH, SOCKET0);
    ovs_mutex_unlock(&dpdk_mutex);

    if (get_rep_peer(netdev_get_name(netdev), peer, sizeof peer)) {
        goto out;
    }

    dev->peer_name = xstrdup(peer);
out:
    return err;
}

static int
netdev_doca_vdpa_construct(struct netdev *netdev)
{
    struct netdev_doca *dev;
    int err;

    err = netdev_doca_construct(netdev);
    if (err) {
        VLOG_ERR("netdev_doca_construct failed. Port: %s\n", netdev->name);
        goto out;
    }

    ovs_mutex_lock(&dpdk_mutex);
    dev = netdev_doca_cast(netdev);
    dev->relay = netdev_doca_vdpa_alloc_relay();
    if (!dev->relay) {
        err = ENOMEM;
    }

    ovs_mutex_unlock(&dpdk_mutex);
out:
    return err;
}

static void
common_destruct(struct netdev_doca *dev)
    OVS_REQUIRES(dpdk_mutex)
    OVS_EXCLUDED(dev->mutex)
{
    rte_free(dev->tx_q);
    dpdk_mp_put(dev->dpdk_mp);

    ovs_list_remove(&dev->list_node);
    free(dev->sw_tx_stats);
    free(dev->sw_stats);
    ovs_mutex_destroy(&dev->mutex);
}

static bool
netdev_doca_rep_stop(struct netdev_doca *dev,
                     dpdk_port_t esw_mgr_port_id OVS_UNUSED)
{
    bool started;

    netdev_offload_doca_uninit_flow_api(&dev->up);
    atomic_read(&dev->started, &started);
    if (!started) {
        return false;
    }

    atomic_store(&dev->started, false);
    if (ovs_doca_enabled()) {
        struct ovs_doca_netdev_data *dev_data = &dev->doca_netdev_data;

        ovs_doca_port_stop(&dev->up, dev_data);
        ovs_doca_dev_close(dev_data);
    } else {
        rte_eth_dev_stop(dev->port_id);
        rte_eth_dev_close(dev->port_id);
    }
    atomic_store_relaxed(&netdev_doca_pending_reset[dev->port_id], true);
    dev->port_id = DPDK_ETH_PORT_ID_INVALID;
    dev->esw_mgr_port_id = -1;
    dev->attached = false;

    return true;
}

static bool
netdev_doca_foreach_representor(struct netdev_doca *esw_dev,
                                bool (*cb)(struct netdev_doca *, dpdk_port_t))
    OVS_REQUIRES(dpdk_mutex)
{
    bool need_reconfigure = false;
    struct rte_pci_addr esw_pci;
    struct rte_pci_addr rep_pci;
    struct netdev_doca *dev;

    if (!netdev_doca_parse_dpdk_devargs_pci(esw_dev->devargs, &esw_pci)) {
        return false;
    }

    LIST_FOR_EACH (dev, list_node, &dpdk_list) {
        if (esw_dev == dev) {
            continue;
        }
        if (!dev->devargs || !netdev_doca_parse_dpdk_devargs_pci(dev->devargs, &rep_pci)) {
            continue;
        }
        if (rte_pci_addr_cmp(&rep_pci, &esw_pci)) {
            continue;
        }

        ovs_mutex_lock(&dev->mutex);
        need_reconfigure |= cb(dev, esw_dev->port_id);
        ovs_mutex_unlock(&dev->mutex);
        netdev_request_reconfigure(&dev->up);
    }

    return need_reconfigure;
}

static void
netdev_doca_destruct(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    bool started;

    ovs_mutex_lock(&dpdk_mutex);
    atomic_read(&dev->started, &started);

    if (started) {
        if (netdev_doca_get_esw_mgr_port_id(netdev) == dev->port_id &&
            netdev_doca_foreach_representor(dev, netdev_doca_rep_stop)) {
            /* If a representor is reconfigured a result of its eswitch manager change, it might
             * not be synced in the bridge's database. Signal it to reconfigure, to make it right.
             */
            rtnetlink_report_link();
        }

        netdev_offload_doca_uninit_flow_api(&dev->up);

        atomic_store(&dev->started, false);
        if (ovs_doca_enabled()) {
            ovs_doca_port_stop(netdev, &dev->doca_netdev_data);
        } else {
            rte_eth_dev_stop(dev->port_id);
        }
    }

    if (dev->attached) {
        /* Remove the eth device. */
        ovs_doca_dev_close(&dev->doca_netdev_data);
        dev->port_id = DPDK_ETH_PORT_ID_INVALID;
        VLOG_INFO("Device '%s' has been removed", dev->devargs);
    }

    netdev_doca_clear_xstats(dev);
    free(dev->devargs);
    free(dev->peer_name);
    common_destruct(dev);

    ovs_mutex_unlock(&dpdk_mutex);
}

static void
netdev_doca_dealloc(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    rte_free(dev);
}

static void
netdev_doca_clear_xstats(struct netdev_doca *dev)
    OVS_REQUIRES(dev->mutex)
{
    free(dev->rte_xstats_names);
    dev->rte_xstats_names = NULL;
    dev->rte_xstats_names_size = 0;
    free(dev->rte_xstats_ids);
    dev->rte_xstats_ids = NULL;
    dev->rte_xstats_ids_size = 0;
}

static const char *
netdev_doca_get_xstat_name(struct netdev_doca *dev, uint64_t id)
    OVS_REQUIRES(dev->mutex)
{
    if (id >= dev->rte_xstats_names_size) {
        return "UNKNOWN";
    }
    return dev->rte_xstats_names[id].name;
}

static bool
is_queue_stat(const char *s)
{
    uint16_t tmp;

    return (s[0] == 'r' || s[0] == 't') &&
            (ovs_scan(s + 1, "x_q%"SCNu16"_packets", &tmp) ||
             ovs_scan(s + 1, "x_q%"SCNu16"_bytes", &tmp));
}

static void
netdev_doca_configure_xstats(struct netdev_doca *dev)
    OVS_REQUIRES(dev->mutex)
{
    struct rte_eth_xstat_name *rte_xstats_names = NULL;
    struct rte_eth_xstat *rte_xstats = NULL;
    int rte_xstats_names_size;
    int rte_xstats_len;
    const char *name;
    uint64_t id;

    netdev_doca_clear_xstats(dev);

    rte_xstats_names_size = rte_eth_xstats_get_names(dev->port_id, NULL, 0);
    if (rte_xstats_names_size < 0) {
        VLOG_WARN("Cannot get XSTATS names for port: "DPDK_PORT_ID_FMT,
                  dev->port_id);
        goto out;
    }

    rte_xstats_names = xcalloc(rte_xstats_names_size,
                               sizeof *rte_xstats_names);
    rte_xstats_len = rte_eth_xstats_get_names(dev->port_id,
                                              rte_xstats_names,
                                              rte_xstats_names_size);
    if (rte_xstats_len < 0 || rte_xstats_len != rte_xstats_names_size) {
        VLOG_WARN("Cannot get XSTATS names for port: "DPDK_PORT_ID_FMT,
                  dev->port_id);
        goto out;
    }

    rte_xstats = xcalloc(rte_xstats_names_size, sizeof *rte_xstats);
    rte_xstats_len = rte_eth_xstats_get(dev->port_id, rte_xstats,
                                        rte_xstats_names_size);
    if (rte_xstats_len < 0 || rte_xstats_len != rte_xstats_names_size) {
        VLOG_WARN("Cannot get XSTATS for port: "DPDK_PORT_ID_FMT,
                  dev->port_id);
        goto out;
    }

    dev->rte_xstats_names = rte_xstats_names;
    rte_xstats_names = NULL;
    dev->rte_xstats_names_size = rte_xstats_names_size;

    dev->rte_xstats_ids = xcalloc(rte_xstats_names_size,
                                  sizeof *dev->rte_xstats_ids);
    for (unsigned int i = 0; i < rte_xstats_names_size; i++) {
        id = rte_xstats[i].id;
        name = netdev_doca_get_xstat_name(dev, id);

        /* For custom stats, we filter out everything except per rxq/txq basic
         * stats, and dropped, error and management counters. */
        if (is_queue_stat(name) ||
            string_ends_with(name, "_errors") ||
            strstr(name, "_management_") ||
            string_ends_with(name, "_dropped")) {

            dev->rte_xstats_ids[dev->rte_xstats_ids_size] = id;
            dev->rte_xstats_ids_size++;
        }
    }

out:
    free(rte_xstats);
    free(rte_xstats_names);
}

#define NV_BF3_DEVICE_ID "0xa2dc"

int
netdev_doca_is_bf3(struct netdev *netdev, bool *is_bf3)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    char pci_addr[PCI_PRI_STR_SIZE];
    struct rte_pci_addr rte_pci;
    char device_id[IFNAMSIZ];

    if (!dev->devargs) {
        return -1;
    }
    if (!netdev_doca_parse_dpdk_devargs_pci(dev->devargs, &rte_pci)) {
        return -1;
    }
    rte_pci_device_name(&rte_pci, pci_addr, sizeof pci_addr);

    if (get_sys("bus/pci/devices", pci_addr, "device", device_id, sizeof device_id)) {
        VLOG_ERR("%s: Failed getting device id", netdev_get_name(netdev));
        return -1;
    }
    *is_bf3 = strncmp(device_id, NV_BF3_DEVICE_ID, sizeof device_id) == 0;
    return 0;
}

static int
netdev_doca_get_config(const struct netdev *netdev, struct smap *args)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    ovs_mutex_lock(&dev->mutex);

    if (dev->devargs && dev->devargs[0]) {
        smap_add_format(args, "dpdk-devargs", "%s", dev->devargs);
    }

    smap_add_format(args, "n_rxq", "%d", dev->user_n_rxq);

    netdev_doca_get_config_sw_meter(args, &dev->sw_meter, &dev->requested_sw_meter);

    if (dev->fc_conf.mode == RTE_ETH_FC_TX_PAUSE ||
        dev->fc_conf.mode == RTE_ETH_FC_FULL) {
        smap_add(args, "rx-flow-ctrl", "true");
    }

    if (dev->fc_conf.mode == RTE_ETH_FC_RX_PAUSE ||
        dev->fc_conf.mode == RTE_ETH_FC_FULL) {
        smap_add(args, "tx-flow-ctrl", "true");
    }

    if (dev->fc_conf.autoneg) {
        smap_add(args, "flow-ctrl-autoneg", "true");
    }

    smap_add_format(args, "n_rxq_desc", "%d", dev->rxq_size);
    smap_add_format(args, "n_txq_desc", "%d", dev->txq_size);

    smap_add(args, "dpdk-lsc-interrupt",
             dev->lsc_interrupt_mode ? "true" : "false");

    if (dev->is_representor) {
        smap_add_format(args, "dpdk-vf-mac", ETH_ADDR_FMT,
                        ETH_ADDR_ARGS(dev->requested_hwaddr));
    }

    ovs_mutex_unlock(&dev->mutex);

    return 0;
}

static struct netdev_doca *
netdev_doca_lookup_by_port_id(dpdk_port_t port_id)
    OVS_REQUIRES(dpdk_mutex)
{
    struct netdev_doca *dev;

    LIST_FOR_EACH (dev, list_node, &dpdk_list) {
        if (dev->port_id == port_id) {
            return dev;
        }
    }

    return NULL;
}

static dpdk_port_t
netdev_doca_get_port_by_mac(const char *mac_str)
{
    dpdk_port_t port_id;
    struct eth_addr mac, port_mac;

    if (!eth_addr_from_string(mac_str, &mac)) {
        VLOG_ERR("invalid mac: %s", mac_str);
        return DPDK_ETH_PORT_ID_INVALID;
    }

    RTE_ETH_FOREACH_DEV (port_id) {
        struct rte_ether_addr ea;

        rte_eth_macaddr_get(port_id, &ea);
        memcpy(port_mac.ea, ea.addr_bytes, ETH_ADDR_LEN);
        if (eth_addr_equals(mac, port_mac)) {
            return port_id;
        }
    }

    return DPDK_ETH_PORT_ID_INVALID;
}

/* Return the first DPDK port id matching the devargs pattern. */
static dpdk_port_t netdev_doca_get_port_by_devargs(const char *devargs)
    OVS_REQUIRES(dpdk_mutex)
{
    dpdk_port_t port_id;
    struct rte_dev_iterator iterator;

    RTE_ETH_FOREACH_MATCHING_DEV (port_id, devargs, &iterator) {
        /* If a break is done - must call rte_eth_iterator_cleanup. */
        rte_eth_iterator_cleanup(&iterator);
        break;
    }

    return port_id;
}

/*
 * Normally, a PCI id (optionally followed by a representor identifier)
 * is enough for identifying a specific DPDK port.
 * However, for some NICs having multiple ports sharing the same PCI
 * id, using PCI id won't work then.
 *
 * To fix that, here one more method is introduced: "class=eth,mac=$MAC".
 *
 * Note that the compatibility is fully kept: user can still use the
 * PCI id for adding ports (when it's enough for them).
 */
static dpdk_port_t
netdev_doca_process_devargs(struct netdev_doca *dev,
                            const char *devargs, char **errp)
    OVS_REQUIRES(dpdk_mutex)
{
    dpdk_port_t new_port_id;

    if (strncmp(devargs, "class=eth,mac=", 14) == 0) {
        new_port_id = netdev_doca_get_port_by_mac(&devargs[14]);
    } else {
        new_port_id = netdev_doca_get_port_by_devargs(devargs);
        if (!rte_eth_dev_is_valid_port(new_port_id)) {
            int err;

            /* Device not found in DPDK, attempt to attach it */
            err = ovs_doca_dev_probe(&dev->doca_netdev_data, devargs);
            if (err) {
                new_port_id = DPDK_ETH_PORT_ID_INVALID;
            } else {
                new_port_id = netdev_doca_get_port_by_devargs(devargs);
                if (rte_eth_dev_is_valid_port(new_port_id)) {
                    /* Attach successful */
                    dev->attached = true;
                    VLOG_INFO("Device '%s' attached to DPDK", devargs);
                } else {
                    /* Attach unsuccessful */
                    new_port_id = DPDK_ETH_PORT_ID_INVALID;
                }
            }
        }
    }

    if (new_port_id == DPDK_ETH_PORT_ID_INVALID) {
        VLOG_WARN_BUF(errp, "Error attaching device '%s' to DPDK", devargs);
    }

    if (new_port_id == NETDEV_DPDK_METER_PORT_ID && !ovs_doca_enabled()) {
        netdev_doca_add_meter_policy(NETDEV_DPDK_METER_PORT_ID,
                         NETDEV_DPDK_METER_POLICY_ID);
    }

    return new_port_id;
}

static struct seq *netdev_doca_reset_seq;
static uint64_t netdev_doca_last_reset_seq;

static void
netdev_doca_wait(const struct netdev_class *netdev_class OVS_UNUSED)
{
    uint64_t last_reset_seq = seq_read(netdev_doca_reset_seq);

    if (netdev_doca_last_reset_seq == last_reset_seq) {
        seq_wait(netdev_doca_reset_seq, netdev_doca_last_reset_seq);
    } else {
        poll_immediate_wake();
    }
}

static void
netdev_doca_run(const struct netdev_class *netdev_class OVS_UNUSED)
{
    uint64_t reset_seq = seq_read(netdev_doca_reset_seq);

    if (reset_seq != netdev_doca_last_reset_seq) {
        dpdk_port_t port_id;

        netdev_doca_last_reset_seq = reset_seq;

        for (port_id = 0; port_id < RTE_MAX_ETHPORTS; port_id++) {
            struct netdev_doca *dev;
            bool pending_reset;

            atomic_read_relaxed(&netdev_doca_pending_reset[port_id],
                                &pending_reset);
            if (!pending_reset) {
                continue;
            }

            ovs_mutex_lock(&dpdk_mutex);
            dev = netdev_doca_lookup_by_port_id(port_id);
            if (dev) {
                ovs_mutex_lock(&dev->mutex);
                netdev_request_reconfigure(&dev->up);
                VLOG_DBG_RL(&rl, "%s: Device reset requested.",
                            netdev_get_name(&dev->up));
                ovs_mutex_unlock(&dev->mutex);
            }
            ovs_mutex_unlock(&dpdk_mutex);
        }
    }
}

static int
dpdk_eth_event_callback(dpdk_port_t port_id, enum rte_eth_event_type type,
                        void *param OVS_UNUSED, void *ret_param OVS_UNUSED)
{
    switch ((int) type) {
    case RTE_ETH_EVENT_INTR_RESET:
        atomic_store_relaxed(&netdev_doca_pending_reset[port_id], true);
        seq_change(netdev_doca_reset_seq);
        break;

    default:
        /* Ignore all other types. */
        break;
    }
    return 0;
}

static void
dpdk_set_rxq_config(struct netdev_doca *dev, const struct smap *args)
    OVS_REQUIRES(dev->mutex)
{
    int new_n_rxq;

    new_n_rxq = MAX(smap_get_int(args, "n_rxq", NR_QUEUE), 1);
    if (new_n_rxq != dev->user_n_rxq) {
        dev->user_n_rxq = new_n_rxq;
        netdev_request_reconfigure(&dev->up);
    }
}

static void
dpdk_process_queue_size(struct netdev *netdev, const struct smap *args,
                        struct rte_eth_dev_info *info, bool is_rx)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    struct rte_eth_desc_lim *lim;
    int default_size, queue_size, cur_size, new_requested_size;
    int *cur_requested_size;
    bool reconfig = false;

    if (is_rx) {
        default_size = NIC_PORT_DEFAULT_RXQ_SIZE;
        new_requested_size = smap_get_int(args, "n_rxq_desc", default_size);
        cur_size = dev->rxq_size;
        cur_requested_size = &dev->requested_rxq_size;
        lim = info ? &info->rx_desc_lim : NULL;
    } else {
        default_size = NIC_PORT_DEFAULT_TXQ_SIZE;
        new_requested_size = smap_get_int(args, "n_txq_desc", default_size);
        cur_size = dev->txq_size;
        cur_requested_size = &dev->requested_txq_size;
        lim = info ? &info->tx_desc_lim : NULL;
    }

    queue_size = new_requested_size;

    /* Check for OVS limits. */
    if (queue_size <= 0 || queue_size > NIC_PORT_MAX_Q_SIZE
            || !is_pow2(queue_size)) {
        queue_size = default_size;
    }

    if (lim) {
        /* Check for device limits. */
        if (lim->nb_align) {
            queue_size = ROUND_UP(queue_size, lim->nb_align);
        }
        queue_size = MIN(queue_size, lim->nb_max);
        queue_size = MAX(queue_size, lim->nb_min);
    }

    *cur_requested_size = queue_size;

    if (cur_size != queue_size) {
        netdev_request_reconfigure(netdev);
        reconfig = true;
    }
    if (new_requested_size != queue_size) {
        VLOG(reconfig ? VLL_INFO : VLL_DBG,
             "%s: Unable to set the number of %s descriptors to %d. "
             "Adjusted to %d.", netdev_get_name(netdev),
             is_rx ? "rx": "tx", new_requested_size, queue_size);
    }
}

static dpdk_port_t
netdev_doca_find_esw_mgr_port_id(uint16_t dev_port_id)
    OVS_REQUIRES(dpdk_mutex)
{
    struct rte_eth_dev_info info;
    struct netdev_doca *dev;
    uint16_t domain_id;

    if (!rte_eth_dev_is_valid_port(dev_port_id)) {
        return -1;
    }
    if (rte_eth_dev_info_get(dev_port_id, &info) < 0) {
        VLOG_DBG_RL(&rl, "Failed to retrieve device info for port "DPDK_PORT_ID_FMT, dev_port_id);
        return -1;
    }
    domain_id = info.switch_info.domain_id;
    LIST_FOR_EACH (dev, list_node, &dpdk_list) {
        if (!rte_eth_dev_is_valid_port(dev->port_id)) {
            continue;
        }
        if (rte_eth_dev_info_get(dev->port_id, &info) < 0) {
            VLOG_DBG_RL(&rl, "Failed to retrieve device info for port "DPDK_PORT_ID_FMT,
                        dev->port_id);
            continue;
        }
        if (info.switch_info.domain_id == domain_id &&
            !(*info.dev_flags & RTE_ETH_DEV_REPRESENTOR)) {
            VLOG_INFO("Found ESW manager port "DPDK_PORT_ID_FMT" for device "
                      DPDK_PORT_ID_FMT, dev->port_id, dev_port_id);
            return dev->port_id;
        }
    }

    return -1;
}

static int
netdev_doca_set_config(struct netdev *netdev, const struct smap *args,
                       char **errp)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    bool rx_fc_en, tx_fc_en, autoneg, lsc_interrupt_mode;
    bool flow_control_requested = true;
    enum rte_eth_fc_mode fc_mode;
    static const enum rte_eth_fc_mode fc_mode_set[2][2] = {
        {RTE_ETH_FC_NONE,     RTE_ETH_FC_TX_PAUSE},
        {RTE_ETH_FC_RX_PAUSE, RTE_ETH_FC_FULL    }
    };
    struct rte_eth_dev_info info;
    char generated[PATH_MAX];
    const char *new_devargs;
    char iface[IFNAMSIZ];
    const char *vf_mac;
    bool started;
    int err = 0;
    int ret;

    ovs_mutex_lock(&dpdk_mutex);
    ovs_mutex_lock(&dev->mutex);

    dpdk_set_rxq_config(dev, args);

    new_devargs = smap_get(args, "dpdk-devargs");

    if (new_devargs == NULL) {
        const char *dev_name;

        memset(iface, 0, sizeof iface);
        dev_name = netdev_get_name(netdev);
        new_devargs = netdev_doca_generate_devargs(dev_name, generated, sizeof generated, iface);
        if (new_devargs) {
            smap_add(CONST_CAST(struct smap *, args), "dpdk-devargs", new_devargs);
        } else {
            VLOG_DBG("%s: Configured port name was not found in sysfs, skipping DPDK probe",
                     netdev_get_name(netdev));
            err = ENODEV;
            goto out;
        }
    }

    if (dev->devargs && new_devargs && strcmp(new_devargs, dev->devargs)) {
        /* The user requested a new device.  If we return error, the caller
         * will delete this netdev and try to recreate it. */
        err = EAGAIN;
        goto out;
    }

    /* dpdk-devargs is required for device configuration */
    if (new_devargs && new_devargs[0]) {
        /* Don't process dpdk-devargs if value is unchanged and port id
         * is valid */
        if (!(dev->devargs && !strcmp(dev->devargs, new_devargs)
               && rte_eth_dev_is_valid_port(dev->port_id) && dev->attached)) {
            dpdk_port_t new_port_id = netdev_doca_process_devargs(dev,
                                                                  new_devargs,
                                                                  errp);
            if (!rte_eth_dev_is_valid_port(new_port_id)) {
                err = EINVAL;
            } else if (new_port_id == dev->port_id) {
                /* Already configured, do not reconfigure again */
                err = 0;
            } else {
                struct netdev_doca *dup_dev;

                dup_dev = netdev_doca_lookup_by_port_id(new_port_id);
                if (dup_dev) {
                    VLOG_WARN_BUF(errp, "'%s' is trying to use device '%s' "
                                  "which is already in use by '%s'",
                                  netdev_get_name(netdev), new_devargs,
                                  netdev_get_name(&dup_dev->up));
                    err = EADDRINUSE;
                } else {
                    int sid = rte_eth_dev_socket_id(new_port_id);

                    dev->requested_socket_id = sid < 0 ? SOCKET0 : sid;
                    dev->devargs = xstrdup(new_devargs);
                    dev->port_id = new_port_id;
                    dev->esw_mgr_port_id =
                        netdev_doca_find_esw_mgr_port_id(new_port_id);
                    ovs_doca_set_properties(&dev->doca_netdev_data,
                                            dev->port_id, dev->devargs, &dev->esw_mgr_port_id,
                                            &dev->attached, &netdev->n_rxq);
                    netdev_request_reconfigure(&dev->up);
                    err = 0;
                }
            }
        }
    } else {
        VLOG_WARN_BUF(errp, "'%s' is missing 'options:dpdk-devargs'. "
                            "The old 'dpdk<port_id>' names are not supported",
                      netdev_get_name(netdev));
        err = EINVAL;
    }

    if (err) {
        goto out;
    }

    ret = rte_eth_dev_info_get(dev->port_id, &info);

    dpdk_process_queue_size(netdev, args, !ret ? &info : NULL, true);
    dpdk_process_queue_size(netdev, args, !ret ? &info : NULL, false);

    err = netdev_doca_set_config_sw_meter(netdev, args,
                                          &dev->requested_sw_meter,
                                          dev->started, errp);
    if (err) {
        goto out;
    }

    atomic_read_explicit(&dev->started, &started, memory_order_acquire);
    if (ovs_doca_enabled() && started) {
        const char *pre_miss_rules = smap_get(args, "doca-pre-miss-rules");

        if (!netdev_doca_is_esw_mgr(netdev)) {
            if (pre_miss_rules) {
                VLOG_WARN("%s: doca-pre-miss-rules requested on non-ESW manager "
                          "port. ignoring", netdev_get_name(netdev));
            }
        } else if (ovs_doca_pre_miss_rules_update(netdev, pre_miss_rules)) {
            goto out;
        }
    }

    netdev_doca_set_config_port_dir(dev->devargs, &dev->port_dir);

    vf_mac = smap_get(args, "dpdk-vf-mac");
    if (vf_mac) {
        struct eth_addr mac;

        if (!dev->is_representor) {
            VLOG_WARN("'%s' is trying to set the VF MAC '%s' "
                      "but 'options:dpdk-vf-mac' is only supported for "
                      "VF representors.",
                      netdev_get_name(netdev), vf_mac);
        } else if (!eth_addr_from_string(vf_mac, &mac)) {
            VLOG_WARN("interface '%s': cannot parse VF MAC '%s'.",
                      netdev_get_name(netdev), vf_mac);
        } else if (eth_addr_is_multicast(mac)) {
            VLOG_WARN("interface '%s': cannot set VF MAC to multicast "
                      "address '%s'.", netdev_get_name(netdev), vf_mac);
        } else if (!eth_addr_equals(dev->requested_hwaddr, mac)) {
            dev->requested_hwaddr = mac;
            netdev_request_reconfigure(netdev);
        }
    }

    lsc_interrupt_mode = smap_get_bool(args, "dpdk-lsc-interrupt", false);
    if (dev->requested_lsc_interrupt_mode != lsc_interrupt_mode) {
        dev->requested_lsc_interrupt_mode = lsc_interrupt_mode;
        netdev_request_reconfigure(netdev);
    }

    rx_fc_en = smap_get_bool(args, "rx-flow-ctrl", false);
    tx_fc_en = smap_get_bool(args, "tx-flow-ctrl", false);
    autoneg = smap_get_bool(args, "flow-ctrl-autoneg", false);

    fc_mode = fc_mode_set[tx_fc_en][rx_fc_en];

    if (!smap_get(args, "rx-flow-ctrl") && !smap_get(args, "tx-flow-ctrl")
        && !smap_get(args, "flow-ctrl-autoneg")) {
        /* FIXME: User didn't ask for flow control configuration.
         *        For now we'll not print a warning if flow control is not
         *        supported by the DPDK port. */
        flow_control_requested = false;
    }

    /* Get the Flow control configuration. */
    err = -rte_eth_dev_flow_ctrl_get(dev->port_id, &dev->fc_conf);
    if (err) {
        if (err == ENOTSUP) {
            if (flow_control_requested) {
                VLOG_WARN("%s: Flow control is not supported.",
                          netdev_get_name(netdev));
            }
            err = 0; /* Not fatal. */
        } else {
            VLOG_WARN_BUF(errp, "%s: Cannot get flow control parameters: %s",
                          netdev_get_name(netdev), rte_strerror(err));
        }
        goto out;
    }

    if (dev->fc_conf.mode != fc_mode || autoneg != dev->fc_conf.autoneg) {
        dev->fc_conf.mode = fc_mode;
        dev->fc_conf.autoneg = autoneg;
        dpdk_eth_flow_ctrl_setup(dev);
    }

out:
    ovs_mutex_unlock(&dev->mutex);
    ovs_mutex_unlock(&dpdk_mutex);

    return err;
}

static int
netdev_doca_vdpa_set_config(struct netdev *netdev, const struct smap *args,
                            char **errp)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    const char *vdpa_accelerator_devargs =
                smap_get(args, "vdpa-accelerator-devargs");
    const char *vdpa_socket_path =
                smap_get(args, "vdpa-socket-path");
    const char *rep = smap_get(args, "dpdk-devargs");
    int err = 0;

    if ((vdpa_accelerator_devargs == NULL) || (vdpa_socket_path == NULL)) {
        VLOG_ERR("netdev_doca_vdpa_set_config failed."
                 "Required arguments are missing for VDPA port %s",
                 netdev->name);
        goto free_relay;
    }
    dev->netdev_rep = rep || !netdev_doca_get_phys_port_name(netdev_get_name(netdev), NULL, 0);

    if (dev->netdev_rep) {
        err = netdev_doca_set_config(netdev, args, errp);
        if (err) {
            VLOG_ERR("netdev_doca_set_config failed. Port: %s", netdev->name);
            goto free_relay;
        }
    }

    err = netdev_doca_vdpa_config_impl(dev->relay, vdpa_socket_path,
                                       vdpa_accelerator_devargs);
    if (err) {
        VLOG_ERR("netdev_doca_vdpa_config_impl failed. Port %s",
                 netdev->name);
        goto free_relay;
    }

    goto out;

free_relay:
    rte_free(dev->relay);
out:
    return err;
}

static int
netdev_doca_get_numa_id(const struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    return dev->socket_id;
}

/* Sets the number of tx queues for the dpdk interface. */
static int
netdev_doca_set_tx_multiq(struct netdev *netdev, unsigned int n_txq)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    ovs_mutex_lock(&dev->mutex);

    if (dev->requested_n_txq == n_txq) {
        goto out;
    }

    dev->requested_n_txq = n_txq;
    netdev_request_reconfigure(netdev);

out:
    ovs_mutex_unlock(&dev->mutex);
    return 0;
}

static struct netdev_rxq *
netdev_doca_rxq_alloc(void)
{
    struct netdev_rxq_dpdk *rx = doca_rte_mzalloc("ovs_doca_rxq", sizeof *rx);

    if (rx) {
        return &rx->up;
    }

    return NULL;
}

static struct netdev_rxq_dpdk *
netdev_rxq_dpdk_cast(const struct netdev_rxq *rxq)
{
    return CONTAINER_OF(rxq, struct netdev_rxq_dpdk, up);
}

static int
netdev_doca_rxq_construct(struct netdev_rxq *rxq)
{
    struct netdev_rxq_dpdk *rx = netdev_rxq_dpdk_cast(rxq);
    struct netdev_doca *dev = netdev_doca_cast(rxq->netdev);

    ovs_mutex_lock(&dev->mutex);
    rx->port_id = dev->port_id;
    ovs_mutex_unlock(&dev->mutex);

    return 0;
}

static void
netdev_doca_rxq_destruct(struct netdev_rxq *rxq OVS_UNUSED)
{
}

static void
netdev_doca_rxq_dealloc(struct netdev_rxq *rxq)
{
    struct netdev_rxq_dpdk *rx = netdev_rxq_dpdk_cast(rxq);

    rte_free(rx);
}

/* Prepare the packet for HWOL.
 * Return True if the packet is OK to continue. */
static bool
netdev_doca_prep_hwol_packet(struct netdev_doca *dev, struct rte_mbuf *mbuf)
{
    struct dp_packet *pkt = CONTAINER_OF(mbuf, struct dp_packet, mbuf);
    void *l2;
    void *l3;
    void *l4;

    const uint64_t all_inner_requests = (RTE_MBUF_F_TX_IP_CKSUM |
                                         RTE_MBUF_F_TX_L4_MASK |
                                         RTE_MBUF_F_TX_TCP_SEG);
    const uint64_t all_outer_requests = (RTE_MBUF_F_TX_OUTER_IP_CKSUM |
                                         RTE_MBUF_F_TX_OUTER_UDP_CKSUM);
    const uint64_t all_requests = all_inner_requests | all_outer_requests;
    const uint64_t all_inner_marks = (RTE_MBUF_F_TX_IPV4 |
                                      RTE_MBUF_F_TX_IPV6);
    const uint64_t all_outer_marks = (RTE_MBUF_F_TX_OUTER_IPV4 |
                                      RTE_MBUF_F_TX_OUTER_IPV6 |
                                      RTE_MBUF_F_TX_TUNNEL_MASK);
    const uint64_t all_marks = all_inner_marks | all_outer_marks;

    if (!(mbuf->ol_flags & all_requests)) {
        /* No offloads requested, no marks should be set. */
        mbuf->ol_flags &= ~all_marks;

        uint64_t unexpected = mbuf->ol_flags & RTE_MBUF_F_TX_OFFLOAD_MASK;
        if (OVS_UNLIKELY(unexpected)) {
            VLOG_WARN_RL(&rl, "%s: Unexpected Tx offload flags: %#"PRIx64,
                         netdev_get_name(&dev->up), unexpected);
            netdev_doca_mbuf_dump(netdev_get_name(&dev->up),
                                  "Packet with unexpected ol_flags", mbuf);
            return false;
        }
        return true;
    }

    const uint64_t tunnel_type = mbuf->ol_flags & RTE_MBUF_F_TX_TUNNEL_MASK;
    if (OVS_UNLIKELY(tunnel_type &&
                     tunnel_type != RTE_MBUF_F_TX_TUNNEL_GENEVE &&
                     tunnel_type != RTE_MBUF_F_TX_TUNNEL_VXLAN)) {
        VLOG_WARN_RL(&rl, "%s: Unexpected tunnel type: %#"PRIx64,
                     netdev_get_name(&dev->up), tunnel_type);
        netdev_doca_mbuf_dump(netdev_get_name(&dev->up),
                              "Packet with unexpected tunnel type", mbuf);
        return false;
    }

    if (tunnel_type && (mbuf->ol_flags & all_inner_requests)) {
        if (mbuf->ol_flags & all_outer_requests) {
            mbuf->outer_l2_len = (char *) dp_packet_l3(pkt) -
                                 (char *) dp_packet_eth(pkt);
            mbuf->outer_l3_len = (char *) dp_packet_l4(pkt) -
                                 (char *) dp_packet_l3(pkt);

            /* Inner L2 length must account for the tunnel header length. */
            l2 = dp_packet_l4(pkt);
            l3 = dp_packet_inner_l3(pkt);
            l4 = dp_packet_inner_l4(pkt);
        } else {
            /* If no outer offloading is requested, clear outer marks. */
            mbuf->ol_flags &= ~all_outer_marks;
            mbuf->outer_l2_len = 0;
            mbuf->outer_l3_len = 0;

            /* Skip outer headers. */
            l2 = dp_packet_eth(pkt);
            l3 = dp_packet_inner_l3(pkt);
            l4 = dp_packet_inner_l4(pkt);
        }
    } else {
        if (tunnel_type) {
            /* No inner offload is requested, fallback to non tunnel
             * checksum offloads. */
            mbuf->ol_flags &= ~all_inner_marks;
            if (mbuf->ol_flags & RTE_MBUF_F_TX_OUTER_IP_CKSUM) {
                mbuf->ol_flags |= RTE_MBUF_F_TX_IP_CKSUM;
                mbuf->ol_flags |= RTE_MBUF_F_TX_IPV4;
            }
            if (mbuf->ol_flags & RTE_MBUF_F_TX_OUTER_UDP_CKSUM) {
                mbuf->ol_flags |= RTE_MBUF_F_TX_UDP_CKSUM;
                mbuf->ol_flags |= mbuf->ol_flags & RTE_MBUF_F_TX_OUTER_IPV4
                                  ? RTE_MBUF_F_TX_IPV4 : RTE_MBUF_F_TX_IPV6;
            }
            mbuf->ol_flags &= ~(all_outer_requests | all_outer_marks);
        }
        mbuf->outer_l2_len = 0;
        mbuf->outer_l3_len = 0;

        l2 = dp_packet_eth(pkt);
        l3 = dp_packet_l3(pkt);
        l4 = dp_packet_l4(pkt);
    }

    ovs_assert(l4);

    mbuf->l2_len = (char *) l3 - (char *) l2;
    mbuf->l3_len = (char *) l4 - (char *) l3;

    if (mbuf->ol_flags & RTE_MBUF_F_TX_TCP_SEG) {
        struct tcp_header *th = l4;
        uint16_t link_tso_segsz;
        int hdr_len;

        mbuf->l4_len = TCP_OFFSET(th->tcp_ctl) * 4;
        if (tunnel_type) {
            link_tso_segsz = dev->mtu - mbuf->l2_len - mbuf->l3_len -
                             mbuf->l4_len - mbuf->outer_l3_len;
        } else {
            link_tso_segsz = dev->mtu - mbuf->l3_len - mbuf->l4_len;
        }

        if (mbuf->tso_segsz > link_tso_segsz) {
            mbuf->tso_segsz = link_tso_segsz;
        }

        hdr_len = mbuf->l2_len + mbuf->l3_len + mbuf->l4_len;
        if (OVS_UNLIKELY((hdr_len + mbuf->tso_segsz) > dev->max_packet_len)) {
            VLOG_WARN_RL(&rl, "%s: Oversized TSO packet. hdr: %"PRIu32", "
                         "gso: %"PRIu32", max len: %"PRIu32"",
                         dev->up.name, hdr_len, mbuf->tso_segsz,
                         dev->max_packet_len);
            return false;
        }
    }

    /* If L4 checksum is requested, IPv4 should be requested as well. */
    if (mbuf->ol_flags & RTE_MBUF_F_TX_L4_MASK
        && mbuf->ol_flags & RTE_MBUF_F_TX_IPV4) {
        mbuf->ol_flags |= RTE_MBUF_F_TX_IP_CKSUM;
    }

    return true;
}

/* Prepare a batch for HWOL.
 * Return the number of good packets in the batch. */
static int
netdev_doca_prep_hwol_batch(struct netdev_doca *dev, struct rte_mbuf **pkts,
                            int pkt_cnt)
{
    int i = 0;
    int cnt = 0;
    struct rte_mbuf *pkt;

    /* Prepare and filter bad HWOL packets. */
    for (i = 0; i < pkt_cnt; i++) {
        pkt = pkts[i];
        if (!netdev_doca_prep_hwol_packet(dev, pkt)) {
            rte_pktmbuf_free(pkt);
            continue;
        }

        if (OVS_UNLIKELY(i != cnt)) {
            pkts[cnt] = pkt;
        }
        cnt++;
    }

    return cnt;
}

static void
netdev_doca_mbuf_dump(const char *prefix, const char *message,
                      const struct rte_mbuf *mbuf)
{
    static struct vlog_rate_limit dump_rl = VLOG_RATE_LIMIT_INIT(5, 5);
    char *response = NULL;
    FILE *stream;
    size_t size;

    if (VLOG_DROP_DBG(&dump_rl)) {
        return;
    }

    stream = open_memstream(&response, &size);
    if (!stream) {
        VLOG_ERR("Unable to open memstream for mbuf dump: %s.",
                 ovs_strerror(errno));
        return;
    }

    rte_pktmbuf_dump(stream, mbuf, rte_pktmbuf_pkt_len(mbuf));

    fclose(stream);

    VLOG_DBG(prefix ? "%s: %s:\n%s" : "%s%s:\n%s",
             prefix ? prefix : "", message, response);
    free(response);
}

/* Tries to transmit 'pkts' to txq 'qid' of device 'dev'.  Takes ownership of
 * 'pkts', even in case of failure.
 *
 * Returns the number of packets that weren't transmitted. */
static inline int
netdev_doca_eth_tx_burst(struct netdev_doca *dev, int qid,
                         struct rte_mbuf **pkts, int cnt)
{
    uint16_t nb_tx_prep = cnt;
    uint32_t nb_tx = 0;
    uint16_t port_id;
    bool started;

    atomic_read_explicit(&dev->started, &started, memory_order_acquire);
    if (OVS_UNLIKELY(!started)) {
        goto out;
    }

    port_id = netdev_doca_initialized ? dev->esw_mgr_port_id : dev->port_id;

    nb_tx_prep = rte_eth_tx_prepare(port_id, qid, pkts, cnt);
    if (nb_tx_prep != cnt) {
        VLOG_WARN_RL(&rl, "%s: Output batch contains invalid packets. "
                     "Only %u/%u are valid: %s", netdev_get_name(&dev->up),
                     nb_tx_prep, cnt, rte_strerror(rte_errno));
        netdev_doca_mbuf_dump(netdev_get_name(&dev->up),
                              "First invalid packet", pkts[nb_tx_prep]);
    }

    while (nb_tx != nb_tx_prep) {
        uint32_t ret;

        ret = rte_eth_tx_burst(port_id, qid, pkts + nb_tx, nb_tx_prep - nb_tx);
        if (!ret) {
            break;
        }

        nb_tx += ret;
    }

out:
    if (OVS_UNLIKELY(nb_tx != cnt)) {
        /* Free buffers, which we couldn't transmit. */
        rte_pktmbuf_free_bulk(&pkts[nb_tx], cnt - nb_tx);
    }

    return cnt - nb_tx;
}

static int
netdev_doca_rxq_recv(struct netdev_rxq *rxq, struct dp_packet_batch *batch,
                     int *qfill)
{
    struct netdev_rxq_dpdk *rx = netdev_rxq_dpdk_cast(rxq);
    struct netdev_doca *dev = netdev_doca_cast(rxq->netdev);
    int nb_rx;
    bool started;

    atomic_read_explicit(&dev->started, &started, memory_order_acquire);

    if (OVS_UNLIKELY(!(dev->flags & NETDEV_UP) || !started)) {
        return EAGAIN;
    }

    if (netdev_doca_initialized) {
        nb_rx = ovs_doca_rx_burst(&dev->doca_netdev_data, rxq->queue_id,
                                  batch);
    } else {
        nb_rx = rte_eth_rx_burst(rx->port_id, rxq->queue_id,
                                 (struct rte_mbuf **) batch->packets,
                                 NETDEV_MAX_BURST);
    }
    if (!nb_rx) {
        return EAGAIN;
    }

    batch->count = nb_rx;
    dp_packet_batch_init_packet_fields(batch);

    if (qfill) {
        if (nb_rx == NETDEV_MAX_BURST) {
            *qfill = rte_eth_rx_queue_count(rx->port_id, rxq->queue_id);
        } else {
            *qfill = 0;
        }
    }

    return 0;
}

static int
netdev_doca_vdpa_rxq_recv(struct netdev_rxq *rxq,
                          struct dp_packet_batch *batch,
                          int *qfill)
{
    struct netdev_doca *dev = netdev_doca_cast(rxq->netdev);

    if (dev->netdev_rep) {
        return netdev_doca_rxq_recv(rxq, batch, qfill);
    }

    return 0;
}

static int
netdev_doca_filter_packet_len(struct netdev_doca *dev, struct rte_mbuf **pkts,
                              int pkt_cnt)
{
    int i = 0;
    int cnt = 0;
    struct rte_mbuf *pkt;

    /* Filter oversized packets. The TSO packets are filtered out
     * during the offloading preparation for performance reasons. */
    for (i = 0; i < pkt_cnt; i++) {
        pkt = pkts[i];
        if (OVS_UNLIKELY((pkt->pkt_len > dev->max_packet_len)
            && !(pkt->ol_flags & RTE_MBUF_F_TX_TCP_SEG))) {
            VLOG_WARN_RL(&rl, "%s: Too big size %" PRIu32 " "
                         "max_packet_len %d", dev->up.name, pkt->pkt_len,
                         dev->max_packet_len);
            COVERAGE_INC(netdev_doca_drop_oversized);
            rte_pktmbuf_free(pkt);
            continue;
        }

        if (OVS_UNLIKELY(i != cnt)) {
            pkts[cnt] = pkt;
        }
        cnt++;
    }

    return cnt;
}

static void
netdev_doca_extbuf_free(void *addr OVS_UNUSED, void *opaque)
{
    rte_free(opaque);
}

static struct rte_mbuf *
dpdk_pktmbuf_attach_extbuf(struct rte_mbuf *pkt, uint32_t data_len)
{
    uint32_t total_len = RTE_PKTMBUF_HEADROOM + data_len;
    struct rte_mbuf_ext_shared_info *shinfo = NULL;
    uint16_t buf_len;
    void *buf;

    total_len += sizeof *shinfo + sizeof(uintptr_t);
    total_len = RTE_ALIGN_CEIL(total_len, sizeof(uintptr_t));

    if (OVS_UNLIKELY(total_len > UINT16_MAX)) {
        VLOG_ERR("Can't copy packet: too big %u", total_len);
        return NULL;
    }

    buf_len = total_len;
    buf = rte_malloc("ovs_doca_pktmbuf", buf_len, RTE_CACHE_LINE_SIZE);
    if (OVS_UNLIKELY(buf == NULL)) {
        VLOG_ERR("Failed to allocate memory using rte_malloc: %u", buf_len);
        return NULL;
    }

    /* Initialize shinfo. */
    shinfo = rte_pktmbuf_ext_shinfo_init_helper(buf, &buf_len,
                                                netdev_doca_extbuf_free,
                                                buf);
    if (OVS_UNLIKELY(shinfo == NULL)) {
        rte_free(buf);
        VLOG_ERR("Failed to initialize shared info for mbuf while "
                 "attempting to attach an external buffer.");
        return NULL;
    }

    rte_pktmbuf_attach_extbuf(pkt, buf, rte_malloc_virt2iova(buf), buf_len,
                              shinfo);
    rte_pktmbuf_reset_headroom(pkt);

    return pkt;
}

static struct rte_mbuf *
dpdk_pktmbuf_alloc(struct rte_mempool *mp, uint32_t data_len)
{
    struct rte_mbuf *pkt = rte_pktmbuf_alloc(mp);

    if (OVS_UNLIKELY(!pkt)) {
        return NULL;
    }

    if (rte_pktmbuf_tailroom(pkt) >= data_len) {
        return pkt;
    }

    if (dpdk_pktmbuf_attach_extbuf(pkt, data_len)) {
        return pkt;
    }

    rte_pktmbuf_free(pkt);

    return NULL;
}

static struct dp_packet *
dpdk_copy_dp_packet_to_mbuf(struct rte_mempool *mp, struct dp_packet *pkt_orig)
{
    struct rte_mbuf *mbuf_dest;
    struct dp_packet *pkt_dest;
    uint32_t pkt_len;

    pkt_len = dp_packet_size(pkt_orig);
    mbuf_dest = dpdk_pktmbuf_alloc(mp, pkt_len);
    if (OVS_UNLIKELY(mbuf_dest == NULL)) {
            return NULL;
    }

    pkt_dest = CONTAINER_OF(mbuf_dest, struct dp_packet, mbuf);
    memcpy(dp_packet_data(pkt_dest), dp_packet_data(pkt_orig), pkt_len);
    dp_packet_set_size(pkt_dest, pkt_len);

    mbuf_dest->tx_offload = pkt_orig->mbuf.tx_offload;
    mbuf_dest->packet_type = pkt_orig->mbuf.packet_type;
    mbuf_dest->ol_flags |= (pkt_orig->mbuf.ol_flags &
                            ~(RTE_MBUF_F_EXTERNAL | RTE_MBUF_F_INDIRECT));
    mbuf_dest->tso_segsz = pkt_orig->mbuf.tso_segsz;

    memcpy(&pkt_dest->l2_pad_size, &pkt_orig->l2_pad_size,
           sizeof(struct dp_packet) - offsetof(struct dp_packet, l2_pad_size));

    if (dp_packet_l3(pkt_dest)) {
        if (dp_packet_eth(pkt_dest)) {
            mbuf_dest->l2_len = (char *) dp_packet_l3(pkt_dest)
                                - (char *) dp_packet_eth(pkt_dest);
        } else {
            mbuf_dest->l2_len = 0;
        }
        if (dp_packet_l4(pkt_dest)) {
            mbuf_dest->l3_len = (char *) dp_packet_l4(pkt_dest)
                                - (char *) dp_packet_l3(pkt_dest);
        } else {
            mbuf_dest->l3_len = 0;
        }
    }

    return pkt_dest;
}

/* Replace packets in a 'batch' with their corresponding copies using
 * DPDK memory.
 *
 * Returns the number of good packets in the batch. */
static size_t
dpdk_copy_batch_to_mbuf(struct netdev *netdev, struct dp_packet_batch *batch)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    size_t i, size = dp_packet_batch_size(batch);
    struct dp_packet *packet;

    DP_PACKET_BATCH_REFILL_FOR_EACH (i, size, packet, batch) {
        if (OVS_UNLIKELY(packet->source == DPBUF_DPDK)) {
            dp_packet_batch_refill(batch, packet, i);
        } else {
            struct dp_packet *pktcopy;

            pktcopy = dpdk_copy_dp_packet_to_mbuf(dev->dpdk_mp->mp, packet);
            if (pktcopy) {
                dp_packet_batch_refill(batch, pktcopy, i);
            }

            dp_packet_delete(packet);
        }
    }

    return dp_packet_batch_size(batch);
}

static size_t
netdev_doca_common_send(struct netdev *netdev, struct dp_packet_batch *batch,
                        struct netdev_doca_sw_stats *stats)
{
    struct rte_mbuf **pkts = (struct rte_mbuf **) batch->packets;
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    size_t cnt, pkt_cnt = dp_packet_batch_size(batch);
    struct dp_packet *packet;
    bool need_copy = false;

    memset(stats, 0, sizeof *stats);

    DP_PACKET_BATCH_FOR_EACH (i, packet, batch) {
        if (packet->source != DPBUF_DPDK) {
            need_copy = true;
            break;
        }
    }

    /* Copy dp-packets to mbufs. */
    if (OVS_UNLIKELY(need_copy)) {
        cnt = dpdk_copy_batch_to_mbuf(netdev, batch);
        stats->tx_failure_drops += pkt_cnt - cnt;
        pkt_cnt = cnt;
    }

    /* Drop oversized packets. */
    cnt = netdev_doca_filter_packet_len(dev, pkts, pkt_cnt);
    stats->tx_mtu_exceeded_drops += pkt_cnt - cnt;
    pkt_cnt = cnt;

    /* Prepare each mbuf for hardware offloading. */
    cnt = netdev_doca_prep_hwol_batch(dev, pkts, pkt_cnt);
    stats->tx_invalid_hwol_drops += pkt_cnt - cnt;
    pkt_cnt = cnt;

    return cnt;
}

static int
netdev_doca_eth_send(struct netdev *netdev, int qid,
                     struct dp_packet_batch *batch, bool concurrent_txq)
{
    struct rte_mbuf **pkts = (struct rte_mbuf **) batch->packets;
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int batch_cnt = dp_packet_batch_size(batch);
    struct netdev_doca_sw_stats stats;
    int cnt, dropped;

    if (OVS_UNLIKELY(!(dev->flags & NETDEV_UP))) {
        rte_spinlock_lock(&dev->stats_lock);
        dev->stats.tx_dropped += dp_packet_batch_size(batch);
        rte_spinlock_unlock(&dev->stats_lock);
        dp_packet_delete_batch(batch, true);
        return 0;
    }

    if (OVS_UNLIKELY(concurrent_txq)) {
        qid = qid % dev->up.n_txq;
        rte_spinlock_lock(&dev->tx_q[qid].tx_lock);
    }

    cnt = netdev_doca_common_send(netdev, batch, &stats);

    if (netdev_doca_initialized) {
        uint32_t port_id_meta =
            ovs_doca_port_id_pkt_meta(netdev_doca_get_port_id(netdev));
        struct dp_packet *packet;
        uint64_t n_bytes = 0;
        uint64_t old_count;

        DP_PACKET_BATCH_FOR_EACH (i, packet, batch) {
            /* Set metadata for egress pipe rules to match on. */
            dp_packet_set_meta(packet, port_id_meta);
            n_bytes += dp_packet_size(packet);
        }
        atomic_add_relaxed(&dev->sw_tx_stats[qid].n_packets, batch->count, &old_count);
        atomic_add_relaxed(&dev->sw_tx_stats[qid].n_bytes, n_bytes, &old_count);
    }

    dropped = netdev_doca_eth_tx_burst(dev, qid, pkts, cnt);
    stats.tx_failure_drops += dropped;
    dropped += batch_cnt - cnt;
    if (OVS_UNLIKELY(dropped)) {
        struct netdev_doca_sw_stats *sw_stats = dev->sw_stats;

        rte_spinlock_lock(&dev->stats_lock);
        dev->stats.tx_dropped += dropped;
        sw_stats->tx_failure_drops += stats.tx_failure_drops;
        sw_stats->tx_mtu_exceeded_drops += stats.tx_mtu_exceeded_drops;
        sw_stats->tx_qos_drops += stats.tx_qos_drops;
        sw_stats->tx_invalid_hwol_drops += stats.tx_invalid_hwol_drops;
        rte_spinlock_unlock(&dev->stats_lock);
    }

    if (OVS_UNLIKELY(concurrent_txq)) {
        rte_spinlock_unlock(&dev->tx_q[qid].tx_lock);
    }

    return 0;
}

static int
netdev_doca_get_addr_list(const struct netdev *netdev,
                          struct in6_addr **addr,
                          struct in6_addr **mask,
                          int *n_cnt)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int rv;

    if (!dev->peer_name) {
        return -1;
    }
    rv = netdev_get_addrs(dev->peer_name, addr, mask, n_cnt);
    return rv;
}

struct get_rep_aux {
    const char *dev;
    char *rep;
    size_t maxlen;
    int rv;
};

static bool
get_rep_cb(struct netdev *netdev,
           odp_port_t odp_port OVS_UNUSED,
           void *aux_)
{
    struct netdev_doca *rep_dev = netdev_doca_cast(netdev);
    const char *name = netdev_get_name(netdev);
    struct get_rep_aux *aux = aux_;

    if (netdev_doca_is_ethdev(netdev) && nullable_string_is_equal(rep_dev->peer_name, aux->dev)) {
        ovs_strlcpy(aux->rep, name, aux->maxlen);
        aux->rv = 0;
        return true;
    }

    return false;
}

int
netdev_doca_get_rep_name(const char dev[], char *rep, size_t maxlen)
{
    struct get_rep_aux aux = {
        .dev = dev,
        .rep = rep,
        .maxlen = maxlen,
        .rv = -1,
    };

    netdev_ports_traverse("doca", get_rep_cb, &aux);
    return aux.rv;
}

static int
netdev_doca_vdpa_send(struct netdev *netdev,
                      int qid,
                      struct dp_packet_batch *batch,
                      bool concurrent_txq)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    if (!dev->netdev_rep) {
        return 0;
    } else {
        return netdev_doca_eth_send(netdev, qid, batch, concurrent_txq);
    }
}

static int
netdev_doca_set_etheraddr__(struct netdev_doca *dev, const struct eth_addr mac)
    OVS_REQUIRES(dev->mutex)
{
    int err = 0;

    if (dev->type == DPDK_DEV_ETH) {
        struct rte_ether_addr ea;

        memcpy(ea.addr_bytes, mac.ea, ETH_ADDR_LEN);
        err = -rte_eth_dev_default_mac_addr_set(dev->port_id, &ea);
    }
    if (!err) {
        dev->hwaddr = mac;
    } else {
        VLOG_WARN("%s: Failed to set requested mac("ETH_ADDR_FMT"): %s",
                  netdev_get_name(&dev->up), ETH_ADDR_ARGS(mac),
                  rte_strerror(err));
    }

    return err;
}

static int
netdev_doca_set_etheraddr(struct netdev *netdev, const struct eth_addr mac)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int err = 0;

    ovs_mutex_lock(&dev->mutex);
    if (!eth_addr_equals(dev->hwaddr, mac)) {
        err = netdev_doca_set_etheraddr__(dev, mac);
        if (!err) {
            netdev_change_seq_changed(netdev);
        }
    }
    ovs_mutex_unlock(&dev->mutex);

    return err;
}

static int
netdev_doca_get_etheraddr(const struct netdev *netdev, struct eth_addr *mac)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    ovs_mutex_lock(&dev->mutex);
    *mac = dev->hwaddr;
    ovs_mutex_unlock(&dev->mutex);

    return 0;
}

static int
netdev_doca_get_mtu(const struct netdev *netdev, int *mtup)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    ovs_mutex_lock(&dev->mutex);
    *mtup = dev->mtu;
    ovs_mutex_unlock(&dev->mutex);

    return 0;
}

static int
netdev_doca_set_mtu(struct netdev *netdev, int mtu)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    /* XXX: Ensure that the overall frame length of the requested MTU does not
     * surpass the NETDEV_DPDK_MAX_PKT_LEN. DPDK device drivers differ in how
     * the L2 frame length is calculated for a given MTU when
     * rte_eth_dev_set_mtu(mtu) is called e.g. i40e driver includes 2 x vlan
     * headers, the em driver includes 1 x vlan header, the ixgbe driver does
     * not include vlan headers. As such we should use
     * MTU_TO_MAX_FRAME_LEN(mtu) which includes an additional 2 x vlan headers
     * (8 bytes) for comparison. This avoids a failure later with
     * rte_eth_dev_set_mtu(). This approach should be used until DPDK provides
     * a method to retrieve the upper bound MTU for a given device.
     */
    if (MTU_TO_MAX_FRAME_LEN(mtu) > NETDEV_DPDK_MAX_PKT_LEN
        || mtu < RTE_ETHER_MIN_MTU) {
        VLOG_WARN("%s: unsupported MTU %d\n", dev->up.name, mtu);
        return EINVAL;
    }

    ovs_mutex_lock(&dev->mutex);
    if (dev->requested_mtu != mtu) {
        dev->requested_mtu = mtu;
        netdev_request_reconfigure(netdev);
    }
    ovs_mutex_unlock(&dev->mutex);

    return 0;
}

static int
netdev_doca_get_carrier(const struct netdev *netdev, bool *carrier);

static int
netdev_doca_get_stats(const struct netdev *netdev, struct netdev_stats *stats)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    struct rte_eth_stats rte_stats;
    bool gg, started;
    int rv;

    netdev_doca_get_carrier(netdev, &gg);
    ovs_mutex_lock(&dev->mutex);

    atomic_read(&dev->started, &started);

    if (!started) {
        ovs_mutex_unlock(&dev->mutex);
        return -1;
    }

    rv = ovs_doca_eth_stats_get(&dev->doca_netdev_data, &rte_stats);
    if (rv) {
        VLOG_ERR("Can't get ETH statistics for port: "DPDK_PORT_ID_FMT,
                 dev->port_id);
        ovs_mutex_unlock(&dev->mutex);
        return EPROTO;
    }

    stats->rx_packets = rte_stats.ipackets;
    stats->tx_packets = rte_stats.opackets;
    stats->rx_bytes = rte_stats.ibytes;
    stats->tx_bytes = rte_stats.obytes;
    stats->rx_errors = rte_stats.ierrors;
    stats->tx_errors = rte_stats.oerrors;

    rte_spinlock_lock(&dev->stats_lock);
    stats->tx_dropped = dev->stats.tx_dropped;
    stats->rx_dropped = dev->stats.rx_dropped;
    rte_spinlock_unlock(&dev->stats_lock);

    /* These are the available DPDK counters for packets not received due to
     * local resource constraints in DPDK and NIC respectively. */
    stats->rx_dropped += rte_stats.rx_nombuf + rte_stats.imissed;
    stats->rx_missed_errors = rte_stats.imissed;

    ovs_mutex_unlock(&dev->mutex);

    return 0;
}

static int
netdev_doca_vdpa_get_stats(const struct netdev *netdev,
                           struct netdev_stats *stats)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int ret = 0;

    if (dev->netdev_rep) {
        ret = netdev_doca_get_stats(netdev, stats);
    }

    return ret;
}

static int
netdev_doca_get_custom_stats(const struct netdev *netdev,
                             struct netdev_custom_stats *custom_stats)
{
    uint32_t i;
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int rte_xstats_ret, sw_stats_size;
    bool started;

    atomic_read(&dev->started, &started);

    if (!started) {
        return -1;
    }

    netdev_doca_get_sw_custom_stats(netdev, custom_stats);

    netdev_doca_get_custom_stats_sw_meter(netdev, custom_stats);

    if (netdev_doca_initialized) {
        ovs_doca_get_custom_stats(netdev, &dev->doca_netdev_data, dev->sw_tx_stats,
                                  custom_stats);
        /* For unified representors on DOCA statistics XSTATS are not supported. */
        return 0;
    }

    ovs_mutex_lock(&dev->mutex);

    if (dev->rte_xstats_ids_size > 0) {
        uint64_t *values = xcalloc(dev->rte_xstats_ids_size,
                                   sizeof(uint64_t));

        rte_xstats_ret =
                rte_eth_xstats_get_by_id(dev->port_id, dev->rte_xstats_ids,
                                         values, dev->rte_xstats_ids_size);

        if (rte_xstats_ret > 0 &&
            rte_xstats_ret <= dev->rte_xstats_ids_size) {

            sw_stats_size = custom_stats->size;
            custom_stats->size += rte_xstats_ret;
            custom_stats->counters = xrealloc(custom_stats->counters,
                                              custom_stats->size *
                                              sizeof *custom_stats->counters);

            for (i = 0; i < rte_xstats_ret; i++) {
                ovs_strlcpy(custom_stats->counters[sw_stats_size + i].name,
                            netdev_doca_get_xstat_name(dev,
                                                       dev->rte_xstats_ids[i]),
                            NETDEV_CUSTOM_STATS_NAME_SIZE);
                custom_stats->counters[sw_stats_size + i].value = values[i];
            }
        } else {
            VLOG_WARN("Cannot get XSTATS values for port: "DPDK_PORT_ID_FMT,
                      dev->port_id);
        }

        free(values);
    }

    ovs_mutex_unlock(&dev->mutex);

    return 0;
}

static int
netdev_doca_get_sw_custom_stats(const struct netdev *netdev,
                                struct netdev_custom_stats *custom_stats)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int i, n;

#define SW_CSTATS                    \
    SW_CSTAT(tx_retries)             \
    SW_CSTAT(tx_failure_drops)       \
    SW_CSTAT(tx_mtu_exceeded_drops)  \
    SW_CSTAT(tx_qos_drops)           \
    SW_CSTAT(rx_qos_drops)           \
    SW_CSTAT(tx_invalid_hwol_drops)

#define SW_CSTAT(NAME) + 1
    custom_stats->size = SW_CSTATS;
#undef SW_CSTAT
    custom_stats->counters = xcalloc(custom_stats->size,
                                     sizeof *custom_stats->counters);

    ovs_mutex_lock(&dev->mutex);

    rte_spinlock_lock(&dev->stats_lock);
    i = 0;
#define SW_CSTAT(NAME) \
    custom_stats->counters[i++].value = dev->sw_stats->NAME;
    SW_CSTATS;
#undef SW_CSTAT
    rte_spinlock_unlock(&dev->stats_lock);

    ovs_mutex_unlock(&dev->mutex);

    i = 0;
    n = 0;
#define SW_CSTAT(NAME) \
    if (custom_stats->counters[i].value != UINT64_MAX) {                   \
        ovs_strlcpy(custom_stats->counters[n].name,                        \
                    "ovs_"#NAME, NETDEV_CUSTOM_STATS_NAME_SIZE);           \
        custom_stats->counters[n].value = custom_stats->counters[i].value; \
        n++;                                                               \
    }                                                                      \
    i++;
    SW_CSTATS;
#undef SW_CSTAT

    custom_stats->size = n;
    return 0;
}

static int
netdev_doca_vdpa_get_custom_stats(const struct netdev *netdev,
                                  struct netdev_custom_stats *custom_stats)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int err = 0;

    err = netdev_doca_vdpa_get_custom_stats_impl
        (dev->relay, custom_stats, &dev->mutex, netdev,
         dev->netdev_rep ? netdev_doca_get_custom_stats : NULL);
    if (err) {
        VLOG_ERR("netdev_doca_vdpa_get_custom_stats_impl failed."
                 "Port %s\n", netdev->name);
    }

    return err;
}

static int
netdev_doca_get_features(const struct netdev *netdev,
                         enum netdev_features *current,
                         enum netdev_features *advertised,
                         enum netdev_features *supported,
                         enum netdev_features *peer)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    struct rte_eth_link link;
    uint32_t feature = 0;

    ovs_mutex_lock(&dev->mutex);
    link = dev->link;
    ovs_mutex_unlock(&dev->mutex);

    /* Match against OpenFlow defined link speed values. */
    if (link.link_duplex == RTE_ETH_LINK_FULL_DUPLEX) {
        switch (link.link_speed) {
        case RTE_ETH_SPEED_NUM_10M:
            feature |= NETDEV_F_10MB_FD;
            break;
        case RTE_ETH_SPEED_NUM_100M:
            feature |= NETDEV_F_100MB_FD;
            break;
        case RTE_ETH_SPEED_NUM_1G:
            feature |= NETDEV_F_1GB_FD;
            break;
        case RTE_ETH_SPEED_NUM_10G:
            feature |= NETDEV_F_10GB_FD;
            break;
        case RTE_ETH_SPEED_NUM_40G:
            feature |= NETDEV_F_40GB_FD;
            break;
        case RTE_ETH_SPEED_NUM_100G:
            feature |= NETDEV_F_100GB_FD;
            break;
        default:
            feature |= NETDEV_F_OTHER;
        }
    } else if (link.link_duplex == RTE_ETH_LINK_HALF_DUPLEX) {
        switch (link.link_speed) {
        case RTE_ETH_SPEED_NUM_10M:
            feature |= NETDEV_F_10MB_HD;
            break;
        case RTE_ETH_SPEED_NUM_100M:
            feature |= NETDEV_F_100MB_HD;
            break;
        case RTE_ETH_SPEED_NUM_1G:
            feature |= NETDEV_F_1GB_HD;
            break;
        default:
            feature |= NETDEV_F_OTHER;
        }
    }

    if (link.link_autoneg) {
        feature |= NETDEV_F_AUTONEG;
    }

    *current = feature;
    *advertised = *supported = *peer = 0;

    return 0;
}

static int
netdev_doca_get_speed(const struct netdev *netdev, uint32_t *current,
                      uint32_t *max)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    struct rte_eth_dev_info dev_info;
    struct rte_eth_link link;
    int diag = 0;

    ovs_mutex_lock(&dev->mutex);
    link = dev->link;

    if (!rte_eth_dev_is_valid_port(dev->port_id)) {
        memset(&dev_info, 0, sizeof dev_info);
    } else {
        diag = rte_eth_dev_info_get(dev->port_id, &dev_info);
    }

    ovs_mutex_unlock(&dev->mutex);

    *current = link.link_speed != RTE_ETH_SPEED_NUM_UNKNOWN
               ? link.link_speed : 0;

    if (diag < 0) {
        *max = 0;
        goto out;
    }

    if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_200G) {
        *max = RTE_ETH_SPEED_NUM_200G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_100G) {
        *max = RTE_ETH_SPEED_NUM_100G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_56G) {
        *max = RTE_ETH_SPEED_NUM_56G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_50G) {
        *max = RTE_ETH_SPEED_NUM_50G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_40G) {
        *max = RTE_ETH_SPEED_NUM_40G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_25G) {
        *max = RTE_ETH_SPEED_NUM_25G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_20G) {
        *max = RTE_ETH_SPEED_NUM_20G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_10G) {
        *max = RTE_ETH_SPEED_NUM_10G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_5G) {
        *max = RTE_ETH_SPEED_NUM_5G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_2_5G) {
        *max = RTE_ETH_SPEED_NUM_2_5G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_1G) {
        *max = RTE_ETH_SPEED_NUM_1G;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_100M ||
        dev_info.speed_capa & RTE_ETH_LINK_SPEED_100M_HD) {
        *max = RTE_ETH_SPEED_NUM_100M;
    } else if (dev_info.speed_capa & RTE_ETH_LINK_SPEED_10M ||
        dev_info.speed_capa & RTE_ETH_LINK_SPEED_10M_HD) {
        *max = RTE_ETH_SPEED_NUM_10M;
    } else {
        *max = 0;
    }

out:
    return 0;
}

static int
netdev_doca_get_ifindex(const struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    ovs_mutex_lock(&dev->mutex);
    /* Calculate hash from the netdev name. Ensure that ifindex is a 24-bit
     * postive integer to meet RFC 2863 recommendations.
     */
    int ifindex = hash_string(netdev->name, 0) % 0xfffffe + 1;
    ovs_mutex_unlock(&dev->mutex);

    return ifindex;
}

static int
netdev_doca_get_carrier(const struct netdev *netdev, bool *carrier)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    ovs_mutex_lock(&dev->mutex);
    check_link_status(dev);
    *carrier = dev->link.link_status;

    ovs_mutex_unlock(&dev->mutex);

    return 0;
}

static long long int
netdev_doca_get_carrier_resets(const struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    long long int carrier_resets;

    ovs_mutex_lock(&dev->mutex);
    carrier_resets = dev->link_reset_cnt;
    ovs_mutex_unlock(&dev->mutex);

    return carrier_resets;
}

static int
netdev_doca_set_miimon(struct netdev *netdev OVS_UNUSED,
                       long long int interval OVS_UNUSED)
{
    return EOPNOTSUPP;
}

static int
netdev_doca_update_flags__(struct netdev_doca *dev,
                           enum netdev_flags off, enum netdev_flags on,
                           enum netdev_flags *old_flagsp)
    OVS_REQUIRES(dev->mutex)
{
    if ((off | on) & ~(NETDEV_UP | NETDEV_PROMISC)) {
        return EINVAL;
    }

    *old_flagsp = dev->flags;
    dev->flags |= on;
    dev->flags &= ~off;

    if (dev->flags == *old_flagsp) {
        return 0;
    }

    if (dev->type == DPDK_DEV_ETH) {

        if ((dev->flags ^ *old_flagsp) & NETDEV_UP) {
            int err;

            if (dev->flags & NETDEV_UP) {
                err = rte_eth_dev_set_link_up(dev->port_id);
            } else {
                err = rte_eth_dev_set_link_down(dev->port_id);
            }
            if (err == -ENOTSUP) {
                VLOG_INFO("Interface %s does not support link state "
                          "configuration", netdev_get_name(&dev->up));
            } else if (err < 0) {
                VLOG_ERR("Interface %s link change error: %s",
                         netdev_get_name(&dev->up), rte_strerror(-err));
                dev->flags = *old_flagsp;
                return -err;
            }
        }

        if (dev->flags & NETDEV_PROMISC) {
            rte_eth_promiscuous_enable(dev->port_id);
        }

        netdev_change_seq_changed(&dev->up);
    }

    return 0;
}

static int
netdev_doca_update_flags(struct netdev *netdev,
                         enum netdev_flags off, enum netdev_flags on,
                         enum netdev_flags *old_flagsp)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int error;

    ovs_mutex_lock(&dev->mutex);
    error = netdev_doca_update_flags__(dev, off, on, old_flagsp);
    ovs_mutex_unlock(&dev->mutex);

    return error;
}

/*
 * Convert a given uint32_t link speed defined in DPDK to a string
 * equivalent.
 */
static const char *
netdev_doca_link_speed_to_str__(uint32_t link_speed)
{
    switch (link_speed) {
    case RTE_ETH_SPEED_NUM_10M:    return "10Mbps";
    case RTE_ETH_SPEED_NUM_100M:   return "100Mbps";
    case RTE_ETH_SPEED_NUM_1G:     return "1Gbps";
    case RTE_ETH_SPEED_NUM_2_5G:   return "2.5Gbps";
    case RTE_ETH_SPEED_NUM_5G:     return "5Gbps";
    case RTE_ETH_SPEED_NUM_10G:    return "10Gbps";
    case RTE_ETH_SPEED_NUM_20G:    return "20Gbps";
    case RTE_ETH_SPEED_NUM_25G:    return "25Gbps";
    case RTE_ETH_SPEED_NUM_40G:    return "40Gbps";
    case RTE_ETH_SPEED_NUM_50G:    return "50Gbps";
    case RTE_ETH_SPEED_NUM_56G:    return "56Gbps";
    case RTE_ETH_SPEED_NUM_100G:   return "100Gbps";
    default:                       return "Not Defined";
    }
}

static int
netdev_doca_get_status(const struct netdev *netdev, struct smap *args)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    struct rte_eth_dev_info dev_info;
    uint32_t link_speed;
    int diag;

    if (!rte_eth_dev_is_valid_port(dev->port_id)) {
        return ENODEV;
    }

    ovs_mutex_lock(&dpdk_mutex);
    ovs_mutex_lock(&dev->mutex);
    diag = rte_eth_dev_info_get(dev->port_id, &dev_info);
    link_speed = dev->link.link_speed;
    ovs_mutex_unlock(&dev->mutex);
    ovs_mutex_unlock(&dpdk_mutex);

    smap_add_format(args, "port_no", DPDK_PORT_ID_FMT, dev->port_id);
    smap_add_format(args, "numa_id", "%d",
                           rte_eth_dev_socket_id(dev->port_id));
    if (!diag) {
        smap_add_format(args, "driver_name", "%s", dev_info.driver_name);
        smap_add_format(args, "driver_version", "%s", rte_version());
        smap_add_format(args, "min_rx_bufsize", "%u", dev_info.min_rx_bufsize);
    }
    smap_add_format(args, "max_rx_pktlen", "%u", dev->max_packet_len);
    if (!diag) {
        smap_add_format(args, "max_rx_queues", "%u", dev_info.max_rx_queues);
        smap_add_format(args, "max_tx_queues", "%u", dev_info.max_tx_queues);
        smap_add_format(args, "max_mac_addrs", "%u", dev_info.max_mac_addrs);
        smap_add_format(args, "max_hash_mac_addrs", "%u",
                           dev_info.max_hash_mac_addrs);
        smap_add_format(args, "max_vfs", "%u", dev_info.max_vfs);
        smap_add_format(args, "max_vmdq_pools", "%u", dev_info.max_vmdq_pools);
    }

    smap_add_format(args, "n_rxq", "%d", netdev->n_rxq);
    smap_add_format(args, "n_txq", "%d", netdev->n_txq);

    smap_add(args, "rx_csum_offload",
             dev->hw_ol_features & NETDEV_RX_CHECKSUM_OFFLOAD
             ? "true" : "false");

    /* Querying the DPDK library for iftype may be done in future, pending
     * support; cf. RFC 3635 Section 3.2.4. */
    enum { IF_TYPE_ETHERNETCSMACD = 6 };

    smap_add_format(args, "if_type", "%"PRIu32, IF_TYPE_ETHERNETCSMACD);
    smap_add_format(args, "if_descr", "%s %s", rte_version(),
                    diag < 0 ? "<unknown>" : dev_info.driver_name);
    if (!diag) {
        const char *bus_info = rte_dev_bus_info(dev_info.device);
        smap_add_format(args, "bus_info", "bus_name=%s%s%s",
                        rte_bus_name(rte_dev_bus(dev_info.device)),
                        bus_info != NULL ? ", " : "",
                        bus_info != NULL ? bus_info : "");
    }

    /* Not all link speeds are defined in the OpenFlow specs e.g. 25 Gbps.
     * In that case the speed will not be reported as part of the usual
     * call to get_features(). Get the link speed of the device and add it
     * to the device status in an easy to read string format.
     */
    smap_add(args, "link_speed",
             netdev_doca_link_speed_to_str__(link_speed));

    if (dev->is_representor) {
        smap_add_format(args, "dpdk-vf-mac", ETH_ADDR_FMT,
                        ETH_ADDR_ARGS(dev->hwaddr));
    }

    smap_add(args, "rx-steering", "rss");

    return 0;
}

static void
netdev_doca_set_admin_state__(struct netdev_doca *dev, bool admin_state)
    OVS_REQUIRES(dev->mutex)
{
    enum netdev_flags old_flags;

    if (admin_state) {
        netdev_doca_update_flags__(dev, 0, NETDEV_UP, &old_flags);
    } else {
        netdev_doca_update_flags__(dev, NETDEV_UP, 0, &old_flags);
    }
}

static void
netdev_doca_set_admin_state(struct unixctl_conn *conn, int argc,
                            const char *argv[], void *aux OVS_UNUSED)
{
    bool up;

    if (!strcasecmp(argv[argc - 1], "up")) {
        up = true;
    } else if ( !strcasecmp(argv[argc - 1], "down")) {
        up = false;
    } else {
        unixctl_command_reply_error(conn, "Invalid Admin State");
        return;
    }

    if (argc > 2) {
        struct netdev *netdev = netdev_from_name(argv[1]);

        if (netdev && is_doca_class(netdev->netdev_class)) {
            struct netdev_doca *dev = netdev_doca_cast(netdev);

            ovs_mutex_lock(&dev->mutex);
            netdev_doca_set_admin_state__(dev, up);
            ovs_mutex_unlock(&dev->mutex);

            netdev_close(netdev);
        } else {
            unixctl_command_reply_error(conn, "Not a DPDK Interface");
            netdev_close(netdev);
            return;
        }
    } else {
        struct netdev_doca *dev;

        ovs_mutex_lock(&dpdk_mutex);
        LIST_FOR_EACH (dev, list_node, &dpdk_list) {
            ovs_mutex_lock(&dev->mutex);
            netdev_doca_set_admin_state__(dev, up);
            ovs_mutex_unlock(&dev->mutex);
        }
        ovs_mutex_unlock(&dpdk_mutex);
    }
    unixctl_command_reply(conn, "OK");
}

static void
netdev_doca_detach(struct unixctl_conn *conn, int argc OVS_UNUSED,
                   const char *argv[], void *aux OVS_UNUSED)
{
    struct ds used_interfaces = DS_EMPTY_INITIALIZER;
    struct rte_eth_dev_info dev_info;
    dpdk_port_t sibling_port_id;
    dpdk_port_t port_id;
    bool used = false;
    char *response;
    int diag;

    ovs_mutex_lock(&dpdk_mutex);

    port_id = netdev_doca_get_port_by_devargs(argv[1]);
    if (!rte_eth_dev_is_valid_port(port_id)) {
        response = xasprintf("Device '%s' not found in DPDK", argv[1]);
        goto error;
    }

    ds_put_format(&used_interfaces,
                  "Device '%s' is being used by the following interfaces:",
                  argv[1]);

    RTE_ETH_FOREACH_DEV_SIBLING (sibling_port_id, port_id) {
        struct netdev_doca *dev;

        LIST_FOR_EACH (dev, list_node, &dpdk_list) {
            if (dev->port_id != sibling_port_id) {
                continue;
            }
            used = true;
            ds_put_format(&used_interfaces, " %s",
                          netdev_get_name(&dev->up));
            break;
        }
    }

    if (used) {
        ds_put_cstr(&used_interfaces, ". Remove them before detaching.");
        response = ds_steal_cstr(&used_interfaces);
        ds_destroy(&used_interfaces);
        goto error;
    }
    ds_destroy(&used_interfaces);

    diag = rte_eth_dev_info_get(port_id, &dev_info);
    rte_eth_dev_close(port_id);
    if (diag < 0 || rte_dev_remove(dev_info.device) < 0) {
        response = xasprintf("Device '%s' can not be detached", argv[1]);
        goto error;
    }

    response = xasprintf("All devices shared with device '%s' "
                         "have been detached", argv[1]);

    ovs_mutex_unlock(&dpdk_mutex);
    unixctl_command_reply(conn, response);
    free(response);
    return;

error:
    ovs_mutex_unlock(&dpdk_mutex);
    unixctl_command_reply_error(conn, response);
    free(response);
}

static void
netdev_doca_get_mempool_info(struct unixctl_conn *conn,
                             int argc, const char *argv[],
                             void *aux OVS_UNUSED)
{
    struct netdev *netdev = NULL;
    const char *error = NULL;
    char *response = NULL;
    FILE *stream;
    size_t size;

    if (argc == 2) {
        netdev = netdev_from_name(argv[1]);
        if (!netdev || !is_doca_class(netdev->netdev_class)) {
            unixctl_command_reply_error(conn, "Not a DPDK Interface");
            goto out;
        }
    }

    stream = open_memstream(&response, &size);
    if (!stream) {
        response = xasprintf("Unable to open memstream: %s.",
                             ovs_strerror(errno));
        unixctl_command_reply_error(conn, response);
        goto out;
    }

    if (netdev) {
        struct netdev_doca *dev = netdev_doca_cast(netdev);

        ovs_mutex_lock(&dev->mutex);
        ovs_mutex_lock(&dpdk_mp_mutex);

        if (dev->dpdk_mp) {
            rte_mempool_dump(stream, dev->dpdk_mp->mp);
            fprintf(stream, "    count: avail (%u), in use (%u)\n",
                    rte_mempool_avail_count(dev->dpdk_mp->mp),
                    rte_mempool_in_use_count(dev->dpdk_mp->mp));
        } else {
            error = "Not allocated";
        }

        ovs_mutex_unlock(&dpdk_mp_mutex);
        ovs_mutex_unlock(&dev->mutex);
    } else {
        ovs_mutex_lock(&dpdk_mp_mutex);
        rte_mempool_list_dump(stream);
        ovs_mutex_unlock(&dpdk_mp_mutex);
    }

    fclose(stream);

    if (error) {
        unixctl_command_reply_error(conn, error);
    } else {
        unixctl_command_reply(conn, response);
    }
out:
    free(response);
    netdev_close(netdev);
}

static int
netdev_doca_class_init(void)
{
    static struct ovsthread_once once = OVSTHREAD_ONCE_INITIALIZER;

    /* This function can be called for different classes.  The initialization
     * needs to be done only once */
    if (ovsthread_once_start(&once)) {
        int ret;

        ovs_thread_create("dpdk_watchdog", dpdk_watchdog, NULL);
        unixctl_command_register("netdev-doca/set-admin-state",
                                 "[netdev] up|down", 1, 2,
                                 netdev_doca_set_admin_state, NULL);

        unixctl_command_register("netdev-doca/detach",
                                 "pci address of device", 1, 1,
                                 netdev_doca_detach, NULL);

        unixctl_command_register("netdev-doca/get-mempool-info",
                                 "[netdev]", 0, 1,
                                 netdev_doca_get_mempool_info, NULL);

        netdev_doca_reset_seq = seq_create();
        netdev_doca_last_reset_seq = seq_read(netdev_doca_reset_seq);
        ret = rte_eth_dev_callback_register(RTE_ETH_ALL,
                                            RTE_ETH_EVENT_INTR_RESET,
                                            dpdk_eth_event_callback, NULL);
        if (ret != 0) {
            VLOG_ERR("Ethernet device callback register error: %s",
                     rte_strerror(-ret));
        }

        netdev_doca_meters_init();

        ovsthread_once_done(&once);
    }

    return 0;
}

/* Retrieves the 'dry_run' attribute of the sw-meter */
bool
netdev_doca_sw_meter_get_dry_run(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    return dev->sw_meter.dry_run;
}

bool
netdev_doca_sw_meter_get_protected(struct netdev *netdev,
                                   struct ofputil_meter_config *config,
                                   struct ofputil_meter_band *band)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    bool valid_meter;

    if (!is_doca_class(netdev->netdev_class)) {
        return false;
    }

    valid_meter = !!dev->sw_meter.band.rate;
    if (valid_meter && config && band) {
        memcpy(config, &dev->sw_meter.config, sizeof config[0]);
        memcpy(band, &dev->sw_meter.band, sizeof band[0]);
        /* Do not export the reference to the band within the netdev-doca
         * that was copied with mempcy. */
        config->bands = band;
    }

    return valid_meter;
}

/* Returns true if the sw-meter for that port is enabled,
 * false otherwise. If true is returned and the two parameters
 * 'config' and 'band' are not NULL, they receive the sw-meter
 * configuration currently set on this netdev.
 *
 * Beware that config->bands is set to point to 'band'. */
bool
netdev_doca_sw_meter_get(struct netdev *netdev,
                         struct ofputil_meter_config *config,
                         struct ofputil_meter_band *band)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    bool valid_meter;

    /* Do not allow a potential reconfigure executing
     * in the main thread to modify the sw-meter config
     * while we read it. */
    ovs_mutex_lock(&dev->mutex);
    valid_meter = netdev_doca_sw_meter_get_protected(netdev, config, band);
    ovs_mutex_unlock(&dev->mutex);

    return valid_meter;
}

static bool
netdev_doca_reconfigure_require_restart(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    struct ds change_list = DS_EMPTY_INITIALIZER;
    bool pending_reset, started;
    bool must_restart = false;

    if (netdev->n_txq != dev->requested_n_txq) {
        ds_put_cstr(&change_list, "txq");
    }

    if (netdev->n_rxq != dev->requested_n_rxq) {
        if (change_list.length) {
            ds_put_cstr(&change_list, ", ");
        }
        ds_put_cstr(&change_list, "rxq");
    }

    if (dev->mtu != dev->requested_mtu) {
        if (change_list.length) {
            ds_put_cstr(&change_list, ", ");
        }
        ds_put_cstr(&change_list, "MTU");
    }

    if (dev->lsc_interrupt_mode != dev->requested_lsc_interrupt_mode) {
        if (change_list.length) {
            ds_put_cstr(&change_list, ", ");
        }
        ds_put_cstr(&change_list, "lsc_interrupt_mode");
    }

    if (!eth_addr_equals(dev->hwaddr, dev->requested_hwaddr)) {
        if (change_list.length) {
            ds_put_cstr(&change_list, ", ");
        }
        ds_put_cstr(&change_list, "MAC");
    }

    /* Some configuration elements require the port to be stopped
     * to be applied. Calling rte_eth_dev_stop however will not work
     * while offload rules still exist. While it is not possible to
     * flush those rules from this module, we must instead ignore
     * the new configuration. The only way to apply it properly being
     * to restart the OVS daemon and configure the port in the requested
     * way from the beginning.
     *
     * Warn the user about it.
     */

    atomic_read_relaxed(&netdev_doca_pending_reset[dev->port_id],
                        &pending_reset);
    atomic_read(&dev->started, &started);
    if (change_list.length && started && !pending_reset) {
        VLOG_WARN("%s: the requested '%s' configuration cannot be applied on the fly. Restarting"
                  " the port", netdev_get_name(netdev), ds_cstr(&change_list));
        must_restart = true;
    }

    ds_destroy(&change_list);
    return must_restart;
}

#define RETA_CONF_SIZE (RTE_ETH_RSS_RETA_SIZE_512 / RTE_ETH_RETA_GROUP_SIZE)

static int
netdev_doca_reconfigure(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    bool pending_reset, started;
    int err = 0;

    /* If eswitch manager is not UP then representors rte_eth devices are closed
     * We cant reconfigure the netdev before probing the reps again.
     */
    if (!netdev_doca_is_esw_mgr(netdev) &&
        netdev_doca_get_esw_mgr_port_id(netdev) == -1) {
        return EAGAIN;
    }

    ovs_mutex_lock(&dpdk_mutex);
    ovs_mutex_lock(&dev->mutex);

    dev->requested_n_rxq = dev->user_n_rxq;

    atomic_read_relaxed(&netdev_doca_pending_reset[dev->port_id],
                        &pending_reset);
    atomic_read(&dev->started, &started);

    if (started) {
        netdev_doca_sw_meter_copy(&dev->sw_meter,
                                  &dev->requested_sw_meter);
        ovs_doca_sw_meter_init(netdev);
    }

    if (netdev->n_txq == dev->requested_n_txq
        && netdev->n_rxq == dev->requested_n_rxq
        && dev->mtu == dev->requested_mtu
        && dev->lsc_interrupt_mode == dev->requested_lsc_interrupt_mode
        && dev->rxq_size == dev->requested_rxq_size
        && dev->txq_size == dev->requested_txq_size
        && eth_addr_equals(dev->hwaddr, dev->requested_hwaddr)
        && dev->socket_id == dev->requested_socket_id
        && started && !pending_reset) {
        /* Reconfiguration is unnecessary */

        goto out;
    }

    if (ovs_doca_enabled() &&
        netdev_doca_reconfigure_require_restart(netdev)) {
        /* Restart is necessary. */
        netdev_set_need_restart(netdev);
        rtnetlink_report_link();
        goto out;
    }

    if (pending_reset) {
        /*
         * Set false before reset to avoid missing a new reset interrupt event
         * in a race with event callback.
         */
        atomic_store_relaxed(&netdev_doca_pending_reset[dev->port_id], false);
        rte_eth_dev_reset(dev->port_id);
        if_notifier_manual_report();
    } else {
        rte_eth_dev_stop(dev->port_id);
    }

    /* DOCA might implicitly start probed representor ports that are not meant to be started yet.
     * Make sure the port is stopped. If the port is already stopped, it will just have an INFO
     * level log from DPDK that it is already stopped.
     */
    atomic_store(&dev->started, false);
    rte_eth_dev_stop(dev->port_id);

    err = netdev_doca_mempool_configure(dev);
    if (err && err != EEXIST) {
        goto out;
    }

    dev->lsc_interrupt_mode = dev->requested_lsc_interrupt_mode;

    netdev->n_txq = dev->requested_n_txq;
    netdev->n_rxq = dev->requested_n_rxq;
    if (ovs_doca_enabled()) {
        if (!netdev_doca_is_esw_mgr(netdev)) {
            int esw_n_rxq;

            esw_n_rxq = ovs_doca_get_esw_n_rxq(&dev->doca_netdev_data);
            if (esw_n_rxq < 0) {
                err = -1;
                goto out;
            }
            if (dev->requested_n_rxq != esw_n_rxq) {
                VLOG_WARN("%s: requested_n_rxq=%d is ignored. DOCA binds the number of rx "
                          "queues to the esw's n_rxq=%d", netdev_get_name(netdev),
                          dev->requested_n_rxq, esw_n_rxq);
            }
            netdev->n_rxq = esw_n_rxq;
        }
    }

    dev->rxq_size = dev->requested_rxq_size;
    dev->txq_size = dev->requested_txq_size;

    rte_free(dev->tx_q);
    dev->tx_q = NULL;

    if (!eth_addr_equals(dev->hwaddr, dev->requested_hwaddr)) {
        err = netdev_doca_set_etheraddr__(dev, dev->requested_hwaddr);
        if (err) {
            goto out;
        }
    }

    err = dpdk_eth_dev_init(dev);
    if (err) {
        goto out;
    }
    netdev_doca_update_netdev_flags(dev);

    if (!started) {
        netdev_doca_sw_meter_copy(&dev->sw_meter,
                                  &dev->requested_sw_meter);
        ovs_doca_sw_meter_init(netdev);
    }
    /* If both requested and actual hwaddr were previously
     * unset (initialized to 0), then first device init above
     * will have set actual hwaddr to something new.
     * This would trigger spurious MAC reconfiguration unless
     * the requested MAC is kept in sync.
     *
     * This is harmless in case requested_hwaddr was
     * configured by the user, as netdev_doca_set_etheraddr__()
     * will have succeeded to get to this point.
     */
    dev->requested_hwaddr = dev->hwaddr;

    dev->tx_q = netdev_doca_alloc_txq(netdev->n_txq);
    if (!dev->tx_q) {
        err = ENOMEM;
    }

    netdev_change_seq_changed(netdev);

out:
    ovs_mutex_unlock(&dev->mutex);
    ovs_mutex_unlock(&dpdk_mutex);
    return err;
}

static int
netdev_doca_vdpa_reconfigure(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int err = 0;

    if (dev->netdev_rep) {
        err = netdev_doca_reconfigure(netdev);
        if (err) {
            if (err != EAGAIN) {
                VLOG_ERR("netdev_doca_reconfigure failed. Port %s", netdev->name);
            }
            return err;
        }
    } else {
        netdev->n_txq = dev->requested_n_txq;
        netdev->n_rxq = dev->requested_n_rxq;
        ovs_mutex_lock(&dev->mutex);

        err = netdev_doca_mempool_configure(dev);
        if (err && err != EEXIST) {
            ovs_mutex_unlock(&dev->mutex);
            return err;
        }
        ovs_mutex_unlock(&dev->mutex);
    }

    return err;
}

int
netdev_doca_get_esw_mgr_port_id(const struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int ret = -1;

    if (!is_doca_class(netdev->netdev_class) ||
        !rte_eth_dev_is_valid_port(dev->port_id) ||
        !rte_eth_dev_is_valid_port(dev->esw_mgr_port_id)) {
        goto out;
    }

    ret = dev->esw_mgr_port_id;
out:
    return ret;
}

int
netdev_doca_get_port_id(const struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    int ret = -1;

    if (!is_doca_class(netdev->netdev_class) ||
        !rte_eth_dev_is_valid_port(dev->port_id)) {
        goto out;
    }

    ret = dev->port_id;
out:
    return ret;
}

enum netdev_doca_port_dir
netdev_doca_get_port_dir(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    return dev->port_dir;
}

bool
netdev_doca_flow_api_supported(struct netdev *netdev)
{
    struct netdev_doca *dev;
    bool ret = false;

    if ((!strcmp(netdev_get_type(netdev), "vxlan") ||
         !strcmp(netdev_get_type(netdev), "gre") ||
         !strcmp(netdev_get_type(netdev), "ip6gre") ||
         !strcmp(netdev_get_type(netdev), "geneve") ||
         !strcmp(netdev_get_type(netdev), "tap")) &&
        !strcmp(netdev_get_dpif_type(netdev), "doca")) {
        ret = true;
        goto out;
    }

    if (!is_doca_class(netdev->netdev_class)) {
        goto out;
    }

    if (netdev_doca_get_esw_mgr_port_id(netdev) == -1) {
        goto out;
    }

    dev = netdev_doca_cast(netdev);
    if (dev->type == DPDK_DEV_ETH) {
        uint16_t proxy_port_id;

        /* TODO: Check if we able to offload some minimal flow. */
        ret = ovs_doca_initialized() ||
            !rte_flow_pick_transfer_proxy(dev->port_id, &proxy_port_id, NULL);
    }
out:
    return ret;
}

int
netdev_doca_rte_flow_destroy(struct netdev *netdev,
                             struct rte_flow *rte_flow,
                             struct rte_flow_error *error,
                             bool esw_port_id)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    dpdk_port_t pid = esw_port_id ? dev->esw_mgr_port_id : dev->port_id;
    int ret;

    ret = rte_flow_destroy(pid, rte_flow, error);
    return ret;
}

struct rte_flow *
netdev_doca_rte_flow_create(struct netdev *netdev,
                            const struct rte_flow_attr *attr,
                            const struct rte_flow_item *items,
                            const struct rte_flow_action *actions,
                            struct rte_flow_error *error)
{
    struct rte_flow *flow;
    struct netdev_doca *dev = netdev_doca_cast(netdev);
    dpdk_port_t pid = attr->transfer ? dev->esw_mgr_port_id : dev->port_id;

    flow = rte_flow_create(pid, attr, items, actions, error);
    return flow;
}

int
netdev_doca_rte_flow_query_count(struct netdev *netdev,
                                 struct rte_flow *rte_flow,
                                 struct rte_flow_query_count *query,
                                 struct rte_flow_error *error)
{
    struct rte_flow_action_count count = { .id = 0, };
    const struct rte_flow_action actions[] = {
        {
            .type = RTE_FLOW_ACTION_TYPE_COUNT,
            .conf = &count,
        },
        {
            .type = RTE_FLOW_ACTION_TYPE_END,
        },
    };
    struct netdev_doca *dev;
    int ret;

    if (!is_doca_class(netdev->netdev_class)) {
        return -1;
    }

    dev = netdev_doca_cast(netdev);
    ret = rte_flow_query(dev->esw_mgr_port_id, rte_flow, actions, query,
                         error);
    return ret;
}

static void
parse_mempool_config(const struct smap *ovs_other_config)
{
    per_port_memory = smap_get_bool(ovs_other_config,
                                    "per-port-memory", false);
    VLOG_INFO("Per port memory for DPDK devices %s.",
              per_port_memory ? "enabled" : "disabled");
}

static void
parse_user_mempools_list(const struct smap *ovs_other_config)
{
    const char *mtus = smap_get(ovs_other_config, "shared-mempool-config");
    char *list, *copy, *key, *value;
    int error = 0;

    if (!mtus) {
        return;
    }

    n_user_mempools = 0;
    list = copy = xstrdup(mtus);

    while (ofputil_parse_key_value(&list, &key, &value)) {
        int socket_id, mtu, adj_mtu;

        if (!str_to_int(key, 0, &mtu) || mtu < 0) {
            error = EINVAL;
            VLOG_WARN("Invalid user configured shared mempool MTU.");
            break;
        }

        if (!str_to_int(value, 0, &socket_id)) {
            /* No socket specified. It will apply for all numas. */
            socket_id = INT_MAX;
        } else if (socket_id < 0) {
            error = EINVAL;
            VLOG_WARN("Invalid user configured shared mempool NUMA.");
            break;
        }

        user_mempools = xrealloc(user_mempools, (n_user_mempools + 1) *
                                 sizeof(struct user_mempool_config));
        adj_mtu = FRAME_LEN_TO_MTU(dpdk_buf_size(mtu));
        user_mempools[n_user_mempools].adj_mtu = adj_mtu;
        user_mempools[n_user_mempools].socket_id = socket_id;
        n_user_mempools++;
        VLOG_INFO("User configured shared mempool set for: MTU %d, NUMA %s.",
                  mtu, socket_id == INT_MAX ? "ALL" : value);
    }

    if (error) {
        VLOG_WARN("User configured shared mempools will not be used.");
        n_user_mempools = 0;
        free(user_mempools);
        user_mempools = NULL;
    }
    free(copy);
}

void
netdev_doca_dev_lock(struct netdev *netdev)
    OVS_ACQUIRES(netdev_doca_cast(netdev)->mutex)
{
    ovs_mutex_lock(&netdev_doca_cast(netdev)->mutex);
}

void
netdev_doca_dev_unlock(struct netdev *netdev)
    OVS_RELEASES(netdev_doca_cast(netdev)->mutex)
{
    ovs_mutex_unlock(&netdev_doca_cast(netdev)->mutex);
}

static void
netdev_doca_vdpa_destruct(struct netdev *netdev)
{
    struct netdev_doca *dev = netdev_doca_cast(netdev);

    ovs_mutex_lock(&dpdk_mutex);
    netdev_doca_vdpa_destruct_impl(dev->relay);
    rte_free(dev->relay);
    ovs_mutex_unlock(&dpdk_mutex);

    netdev_doca_destruct(netdev);
}

#define NETDEV_DPDK_CLASS_COMMON                            \
    .is_pmd = true,                                         \
    .alloc = netdev_doca_alloc,                             \
    .dealloc = netdev_doca_dealloc,                         \
    .get_numa_id = netdev_doca_get_numa_id,                 \
    .set_etheraddr = netdev_doca_set_etheraddr,             \
    .get_etheraddr = netdev_doca_get_etheraddr,             \
    .get_mtu = netdev_doca_get_mtu,                         \
    .set_mtu = netdev_doca_set_mtu,                         \
    .get_ifindex = netdev_doca_get_ifindex,                 \
    .get_carrier_resets = netdev_doca_get_carrier_resets,   \
    .set_miimon_interval = netdev_doca_set_miimon,          \
    .update_flags = netdev_doca_update_flags,               \
    .rxq_alloc = netdev_doca_rxq_alloc,                     \
    .rxq_construct = netdev_doca_rxq_construct,             \
    .rxq_destruct = netdev_doca_rxq_destruct,               \
    .rxq_dealloc = netdev_doca_rxq_dealloc

#define NETDEV_DPDK_CLASS_BASE                          \
    NETDEV_DPDK_CLASS_COMMON,                           \
    .init = netdev_doca_class_init,                     \
    .run = netdev_doca_run,                             \
    .wait = netdev_doca_wait,                           \
    .destruct = netdev_doca_destruct,                   \
    .set_tx_multiq = netdev_doca_set_tx_multiq,         \
    .get_carrier = netdev_doca_get_carrier,             \
    .get_stats = netdev_doca_get_stats,                 \
    .get_custom_stats = netdev_doca_get_custom_stats,   \
    .get_features = netdev_doca_get_features,           \
    .get_speed = netdev_doca_get_speed,                 \
    .get_status = netdev_doca_get_status,               \
    .reconfigure = netdev_doca_reconfigure,             \
    .rxq_recv = netdev_doca_rxq_recv

static const struct netdev_class doca_class = {
    .type = "doca",
    NETDEV_DPDK_CLASS_BASE,
    .construct = netdev_doca_construct,
    .get_config = netdev_doca_get_config,
    .set_config = netdev_doca_set_config,
    .send = netdev_doca_eth_send,
    .get_addr_list = netdev_doca_get_addr_list,
    .get_datapath_priority = netdev_doca_get_datapath_priority,
#ifdef ALLOW_EXPERIMENTAL_API
    .hw_miss_packet_recover = netdev_offload_doca_hw_miss_packet_recover,
#endif
};

static const struct netdev_class doca_vdpa_class = {
    .type = "docavdpa",
    NETDEV_DPDK_CLASS_COMMON,
    .construct = netdev_doca_vdpa_construct,
    .destruct = netdev_doca_vdpa_destruct,
    .rxq_recv = netdev_doca_vdpa_rxq_recv,
    .set_config = netdev_doca_vdpa_set_config,
    .reconfigure = netdev_doca_vdpa_reconfigure,
    .get_stats = netdev_doca_vdpa_get_stats,
    .get_custom_stats = netdev_doca_vdpa_get_custom_stats,
    .send = netdev_doca_vdpa_send,
#ifdef ALLOW_EXPERIMENTAL_API
    .hw_miss_packet_recover = netdev_offload_doca_hw_miss_packet_recover,
#endif
};

void
netdev_doca_register(const struct smap *ovs_other_config)
{
    parse_mempool_config(ovs_other_config);
    parse_user_mempools_list(ovs_other_config);

    netdev_register_provider(&doca_class);
    netdev_register_provider(&doca_vdpa_class);
}
