/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2009-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#include <config.h>
#include "dpif-doca.h"
#include "dpif-doca-metrics.h"
#include "dpif-doca-private.h"
#include "dpif-doca-private-dfc.h"

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <inttypes.h>
#include <net/if.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <unistd.h>

#include "bitmap.h"
#include "ccmap.h"
#include "cmap.h"
#include "conntrack.h"
#include "conntrack-offload.h"
#include "conntrack-tp.h"
#include "coverage.h"
#include "ct-dpif.h"
#include "csum.h"
#include "dp-packet.h"
#include "dpif.h"
#include "dpif-doca-lookup.h"
#include "dpif-doca-private-extract.h"
#include "dpif-netdev-perf.h"
#include "dpif-provider.h"
#include "dummy.h"
#include "fat-rwlock.h"
#include "flow.h"
#include "histogram.h"
#include "hmapx.h"
#include "id-fpool.h"
#include "id-pool.h"
#include "ipf.h"
#include "metrics.h"
#include "mov-avg.h"
#include "mpsc-queue.h"
#include "netdev.h"
#include "netdev-offload.h"
#include "netdev-offload-doca.h"
#include "netdev-provider.h"
#include "netdev-vport.h"
#include "netdev-doca.h"
#include "netlink.h"
#include "odp-execute.h"
#include "odp-util.h"
#include "openvswitch/dynamic-string.h"
#include "openvswitch/list.h"
#include "openvswitch/match.h"
#include "openvswitch/ofp-parse.h"
#include "openvswitch/ofp-print.h"
#include "openvswitch/ofpbuf.h"
#include "openvswitch/shash.h"
#include "openvswitch/vlog.h"
#include "ovs-doca.h"
#include "ovs-numa.h"
#include "ovs-rcu.h"
#include "packets.h"
#include "openvswitch/poll-loop.h"
#include "pvector.h"
#include "random.h"
#include "rtnetlink.h"
#include "seq.h"
#include "smap.h"
#include "sset.h"
#include "timeval.h"
#include "tnl-neigh-cache.h"
#include "tnl-ports.h"
#include "unixctl.h"
#include "util.h"
#include "uuid.h"

VLOG_DEFINE_THIS_MODULE(dpif_doca_metrics);

/* Equivalent to 'dpif_is_doca' but usable in the
 * metrics context. */
static bool
metrics_dpif_is_doca(void *it)
{
    return dpif_is_doca(it);
}

METRICS_IF(foreach_dpif, foreach_dpif_doca, metrics_dpif_is_doca);
METRICS_IF(foreach_dpif_doca, foreach_dpif_doca_ext, metrics_ext_enabled);

static bool
dpif_doca_offload_enabled(void *it OVS_UNUSED)
{
    return netdev_is_flow_api_enabled();
}

METRICS_IF(foreach_dpif_doca, dpif_doca_offload, dpif_doca_offload_enabled);

struct hw_offload_it {
    struct dp_doca *dp;
    struct dp_offload_thread *thread;
    unsigned int tid;
};

static void
do_foreach_hw_offload_threads(metrics_visitor_cb callback,
                              struct metrics_visitor *visitor,
                              struct metrics_node *node,
                              struct metrics_label *labels,
                              size_t n OVS_UNUSED)
{
    struct dp_offload_thread *thread;
    struct hw_offload_it it;
    unsigned int tid;
    char id[50];

    it.dp = get_dp_doca(visitor->it);
    labels[0].value = id;
    DP_NETDEV_OFFLOAD_FOREACH_THREAD (thread, tid) {
        /* Show offload statistics & processing for the hw-offload
         * thread on the default page. If the debug page is requested,
         * show them for all threads. */
        if (tid != FIRST_QUEUE && !metrics_dbg_enabled(NULL)) {
            continue;
        }
        snprintf(id, sizeof id, "%u", tid);
        it.thread = thread;
        it.tid = tid;
        visitor->it = &it;
        callback(visitor, node);
    }
}

METRICS_FOREACH(dpif_doca_offload, foreach_hw_offload_threads,
                do_foreach_hw_offload_threads, "thread_num");
METRICS_IF(foreach_hw_offload_threads, foreach_hw_offload_threads_dbg, metrics_dbg_enabled);

enum {
    HWOL_METRICS_ENQUEUED,
    HWOL_METRICS_INSERTED,
    HWOL_METRICS_CT_UNIDIR,
    HWOL_METRICS_CT_BIDIR,
    HWOL_METRICS_N_ENTRIES,
};

#define HWOL_METRICS_ENTRIES \
    [HWOL_METRICS_ENQUEUED] = METRICS_GAUGE(n_enqueued,                  \
        "Number of hardware offload requests waiting to be processed."), \
    [HWOL_METRICS_INSERTED] = METRICS_GAUGE(n_inserted,                  \
        "Number of hardware offload rules currently inserted."),         \
    [HWOL_METRICS_CT_UNIDIR] = METRICS_GAUGE(n_ct_unidir,                \
        "Number of uni-directional connections offloaded in hardware."), \
    [HWOL_METRICS_CT_BIDIR] = METRICS_GAUGE(n_ct_bidir,                  \
        "Number of bi-directional connections offloaded in hardware."),

static void
hw_offload_read_value(double *values, void *_it)
{
    struct netdev_offload_stats per_port_nos[MAX_OFFLOAD_THREAD_NB];
    struct netdev_offload_stats total_nos[MAX_OFFLOAD_THREAD_NB];
    struct hw_offload_it *it = _it;
    unsigned int tid = it->tid;
    struct dp_doca *dp = it->dp;
    struct dp_offload_thread *t = it->thread;
    struct dp_doca_port *port;
    uint64_t count;

    atomic_read_relaxed(&t->enqueued_offload, &count);
    values[HWOL_METRICS_ENQUEUED] = count;

    memset(total_nos, 0, sizeof total_nos);
    dp_doca_port_rdlock(dp);
    HMAP_FOR_EACH (port, node, &dp->ports) {
        memset(per_port_nos, 0, sizeof per_port_nos);
        if (!netdev_offload_get_stats(port->netdev, per_port_nos)) {
            netdev_offload_stats_add(&total_nos[tid], per_port_nos[tid]);
        }
    }
    ovs_rwlock_unlock(&dp->port_rwlock);

    values[HWOL_METRICS_INSERTED] = total_nos[tid].n_inserted;
    values[HWOL_METRICS_CT_UNIDIR] = total_nos[tid].n_unidir_conns;
    values[HWOL_METRICS_CT_BIDIR] = total_nos[tid].n_conns / 2;
}

METRICS_ENTRIES(foreach_hw_offload_threads_dbg, hw_offload_threads_dbg_entries,
                "hw_offload", hw_offload_read_value, HWOL_METRICS_ENTRIES);

static struct histogram *
hw_offload_latency_get(void *_it)
{
    struct hw_offload_it *it = _it;

    return &it->thread->latency;
}

METRICS_HISTOGRAM(foreach_hw_offload_threads_dbg, hw_offload_latency,
                  "Latency in milliseconds between an offload request and its "
                  "completion.", hw_offload_latency_get);

static void
do_foreach_hw_offload_types(metrics_visitor_cb callback,
                            struct metrics_visitor *visitor,
                            struct metrics_node *node,
                            struct metrics_label *labels,
                            size_t n OVS_UNUSED)
{
    const char *hw_offload_type_names[] = {
        [DP_OFFLOAD_FLOW] = "flow",
        [DP_OFFLOAD_FLUSH] = "flush",
        [DP_OFFLOAD_CONN] = "conn",
        [DP_OFFLOAD_STATS_CLEAR] = "stats_clear",
    };
    struct hw_offload_it *ctx_it = visitor->it;

    for (int i = 0; i < DP_OFFLOAD_TYPE_NUM; i++) {
        /* Show offload statistics & processing for flow messages only on the default page.
         * If the debug page is requested, show them for all types. */
        if (i != DP_OFFLOAD_FLOW && !metrics_dbg_enabled(NULL)) {
            continue;
        }
        labels[0].value = hw_offload_type_names[i];
        visitor->it = &ctx_it->thread->queue_metrics[i];
        callback(visitor, node);
    }
}

/* Iterates on offload (thread x type). */
METRICS_FOREACH(foreach_hw_offload_threads, foreach_hw_offload_types,
                do_foreach_hw_offload_types, "type");

static struct histogram *
hw_offload_queue_sojourn_time_get(void *_it)
{
    struct dp_offload_queue_metrics *m = _it;

    return &m->sojourn_time;
}

METRICS_HISTOGRAM(foreach_hw_offload_types, hw_offload_queue_sojourn_time,
                  "Distribution of sojourn time for an offload request "
                  "in milliseconds", hw_offload_queue_sojourn_time_get);

static struct histogram *
hw_offload_queue_wait_time_get(void *_it)
{
    struct dp_offload_queue_metrics *m = _it;

    return &m->wait_time;
}

METRICS_HISTOGRAM(foreach_hw_offload_types, hw_offload_queue_wait_time,
                  "Distribution of wait time for an offload request "
                  "in milliseconds", hw_offload_queue_wait_time_get);

static struct histogram *
hw_offload_queue_service_time_get(void *_it)
{
    struct dp_offload_queue_metrics *m = _it;

    return &m->service_time;
}

METRICS_HISTOGRAM(foreach_hw_offload_types, hw_offload_queue_service_time,
                  "Distribution of service time for an offload request "
                  "in microseconds", hw_offload_queue_service_time_get);

static void
datapath_hw_offload_read_value(double *values, void *_dp)
{
    double t_values[HWOL_METRICS_N_ENTRIES];
    struct dp_offload_thread *thread;
    struct hw_offload_it it;
    size_t i;

    for (i = 0; i < HWOL_METRICS_N_ENTRIES; i++) {
        values[i] = 0.0;
    }

    it.dp = get_dp_doca(_dp);
    DP_NETDEV_OFFLOAD_FOREACH_THREAD (thread, it.tid) {
        it.thread = thread;
        hw_offload_read_value(t_values, &it);
        for (i = 0; i < HWOL_METRICS_N_ENTRIES; i++) {
            values[i] += t_values[i];
        }
    }
}

METRICS_ENTRIES(dpif_doca_offload, datapath_hw_offload_entries,
                "datapath_hw_offload", datapath_hw_offload_read_value,
                HWOL_METRICS_ENTRIES);

static void
poll_threads_n_read_value(double *values, void *it)
{
    struct dp_doca *dp = get_dp_doca(it);

    values[0] = cmap_count(&dp->poll_threads);
}

METRICS_ENTRIES(foreach_dpif_doca, poll_threads_n,
    "poll_threads", poll_threads_n_read_value,
    METRICS_GAUGE(n, "Number of polling threads."),
);

static void
do_foreach_poll_threads(metrics_visitor_cb callback,
                        struct metrics_visitor *visitor,
                        struct metrics_node *node,
                        struct metrics_label *labels,
                        size_t n OVS_UNUSED)
{
    struct dp_doca *dp = get_dp_doca(visitor->it);
    struct dp_doca_pmd_thread *pmd;
    char core[50];
    char numa[50];

    labels[0].value = core;
    labels[1].value = numa;
    CMAP_FOR_EACH (pmd, node, &dp->poll_threads) {
        if (pmd->core_id == NON_PMD_CORE_ID &&
            !metrics_dbg_enabled(NULL)) {
            /* By definition, if the core ID is not one of a PMD,
             * then it is not a poll thread (i.e. 'main').
             * Do not iterate on it as if it was one. */
            continue;
        }
        snprintf(core, sizeof core, "%u", pmd->core_id);
        snprintf(numa, sizeof numa, "%d", pmd->numa_id);
        if (pmd->core_id == NON_PMD_CORE_ID) {
            snprintf(core, sizeof core, "main");
            snprintf(numa, sizeof numa, "0");
        }
        visitor->it = pmd;
        callback(visitor, node);
    }
}

METRICS_FOREACH(foreach_dpif_doca, foreach_poll_threads,
                do_foreach_poll_threads, "core", "numa");

METRICS_IF(foreach_poll_threads, foreach_poll_threads_ext, metrics_ext_enabled);
METRICS_IF(foreach_poll_threads, foreach_poll_threads_dbg, metrics_dbg_enabled);

enum {
    PMD_METRICS_PACKETS,
    PMD_METRICS_RECIRC,
    PMD_METRICS_HIT,
    PMD_METRICS_MISSED,
    PMD_METRICS_LOST,
    PMD_METRICS_AVG_LOOKUPS_PER_HIT,
    PMD_METRICS_AVG_PACKETS_PER_BATCH,
    PMD_METRICS_AVG_RECIRC_PER_PACKET,
    PMD_METRICS_AVG_PASSES_PER_PACKET,
    PMD_METRICS_AVG_CYCLES_PER_PACKET,
    PMD_METRICS_AVG_BUSY_CYCLES_PER_PACKET,
    PMD_METRICS_PERCENT_BUSY_CYCLES,
    PMD_METRICS_PERCENT_IDLE_CYCLES,
};

static void
poll_threads_read_value(double *values, void *it)
{
    struct dp_doca_pmd_thread *pmd = it;
    uint64_t total_cycles, total_packets;
    uint64_t stats[PMD_N_STATS];
    double busy_cycles_per_pkt;
    double packets_per_batch;
    double avg_busy_cycles;
    double avg_idle_cycles;
    double lookups_per_hit;
    double recirc_per_pkt;
    double passes_per_pkt;
    double cycles_per_pkt;
    uint64_t n_hit;

    /* Do not use 'pmd_perf_read_counters'. Counters are supposed to
     * always be increasing, while the pmd perf module is made
     * for debugging purpose and offers a 'clear' operation.
     * Read the counters exactly as they are.
     */
    for (int i = 0; i < PMD_N_STATS; i++) {
        atomic_read_relaxed(&pmd->perf_stats.counters.n[i], &stats[i]);
    }

    n_hit = 0;
    n_hit += stats[PMD_STAT_PHWOL_HIT];
    n_hit += stats[PMD_STAT_SIMPLE_HIT];
    n_hit += stats[PMD_STAT_EXACT_HIT];
    n_hit += stats[PMD_STAT_SMC_HIT];
    n_hit += stats[PMD_STAT_MASKED_HIT];

    total_cycles = stats[PMD_CYCLES_ITER_IDLE] +
                   stats[PMD_CYCLES_ITER_BUSY];
    total_packets = stats[PMD_STAT_RECV];

    lookups_per_hit = 0;
    if (stats[PMD_STAT_MASKED_HIT] > 0) {
        lookups_per_hit = (double) stats[PMD_STAT_MASKED_LOOKUP] /
                          (double) stats[PMD_STAT_MASKED_HIT];
    }

    packets_per_batch = 0;
    if (stats[PMD_STAT_SENT_BATCHES] > 0) {
        packets_per_batch = (double) stats[PMD_STAT_SENT_PKTS] /
                            (double) stats[PMD_STAT_SENT_BATCHES];
    }

    avg_idle_cycles = 0;
    avg_busy_cycles = 0;
    if (total_cycles > 0) {
        avg_idle_cycles = (double) stats[PMD_CYCLES_ITER_IDLE] /
                          (double) total_cycles * 100.0;
        avg_busy_cycles = (double) stats[PMD_CYCLES_ITER_BUSY] /
                          (double) total_cycles * 100.0;
    }

    recirc_per_pkt = 0;
    passes_per_pkt = 0;
    cycles_per_pkt = 0;
    busy_cycles_per_pkt = 0;
    if (total_packets > 0) {
        recirc_per_pkt = (double) stats[PMD_STAT_RECIRC] /
                         (double) total_packets;
        passes_per_pkt = (double) (total_packets + stats[PMD_STAT_RECIRC]) /
                         (double) total_packets;
        cycles_per_pkt = (double) total_cycles / (double) total_packets;
        busy_cycles_per_pkt = (double) stats[PMD_CYCLES_ITER_BUSY] /
                              (double) total_packets;
    }

    values[PMD_METRICS_PACKETS] = stats[PMD_STAT_RECV];
    values[PMD_METRICS_RECIRC] = stats[PMD_STAT_RECIRC];
    values[PMD_METRICS_HIT] = n_hit;
    values[PMD_METRICS_MISSED] = stats[PMD_STAT_MISS];
    values[PMD_METRICS_LOST] = stats[PMD_STAT_LOST];

    values[PMD_METRICS_AVG_LOOKUPS_PER_HIT] = lookups_per_hit;
    values[PMD_METRICS_AVG_PACKETS_PER_BATCH] = packets_per_batch;
    values[PMD_METRICS_AVG_RECIRC_PER_PACKET] = recirc_per_pkt;
    values[PMD_METRICS_AVG_PASSES_PER_PACKET] = passes_per_pkt;
    values[PMD_METRICS_AVG_CYCLES_PER_PACKET] = cycles_per_pkt;
    values[PMD_METRICS_AVG_BUSY_CYCLES_PER_PACKET] = busy_cycles_per_pkt;
    values[PMD_METRICS_PERCENT_BUSY_CYCLES] = avg_busy_cycles;
    values[PMD_METRICS_PERCENT_IDLE_CYCLES] = avg_idle_cycles;
}

METRICS_ENTRIES(foreach_poll_threads, poll_threads_entries,
    "poll_threads", poll_threads_read_value,
    [PMD_METRICS_PACKETS] = METRICS_COUNTER(packets,
        "Number of received packets."),
    [PMD_METRICS_RECIRC] = METRICS_COUNTER(recirculations,
        "Number of executed packet recirculations."),
    [PMD_METRICS_HIT] = METRICS_COUNTER(hit,
        "Number of flow table matches."),
    [PMD_METRICS_MISSED] = METRICS_COUNTER(missed,
        "Number of flow table misses and upcall succeeded."),
    [PMD_METRICS_LOST] = METRICS_COUNTER(lost,
        "Number of flow table misses and upcall failed."),
    [PMD_METRICS_AVG_LOOKUPS_PER_HIT] = METRICS_GAUGE(lookups_per_hit,
        "Average number of lookups per flow table hit."),
    [PMD_METRICS_AVG_PACKETS_PER_BATCH] = METRICS_GAUGE(packets_per_batch,
        "Average number of packets per batch."),
    [PMD_METRICS_AVG_RECIRC_PER_PACKET] = METRICS_GAUGE(recirc_per_packet,
        "Average number of recirculations per packet."),
    [PMD_METRICS_AVG_PASSES_PER_PACKET] = METRICS_GAUGE(passes_per_packet,
        "Average number of datapath passes per packet."),
    [PMD_METRICS_AVG_CYCLES_PER_PACKET] = METRICS_GAUGE(cycles_per_packet,
        "Average number of CPU cycles per packet."),
    [PMD_METRICS_AVG_BUSY_CYCLES_PER_PACKET] = METRICS_GAUGE(
            busy_cycles_per_packet,
        "Average number of active CPU cycles per packet."),
    [PMD_METRICS_PERCENT_BUSY_CYCLES] = METRICS_GAUGE(busy_cycles,
        "Percent of useful CPU cycles."),
    [PMD_METRICS_PERCENT_IDLE_CYCLES] = METRICS_GAUGE(idle_cycles,
        "Percent of idle CPU cycles."),
);

enum {
    PMD_METRICS_SIMPLE_N_ENTRIES,
    PMD_METRICS_SIMPLE_HIT,
    PMD_METRICS_SIMPLE_MISS,
    PMD_METRICS_SIMPLE_UPDATE,
    PMD_METRICS_EMC_N_ENTRIES,
    PMD_METRICS_EMC_HIT,
    PMD_METRICS_EMC_MISS,
    PMD_METRICS_EMC_UPDATE,
    PMD_METRICS_SMC_N_ENTRIES,
    PMD_METRICS_SMC_HIT,
    PMD_METRICS_SMC_MISS,
    PMD_METRICS_SMC_UPDATE,
    PMD_METRICS_CLS_N_ENTRIES,
    PMD_METRICS_CLS_HIT,
    PMD_METRICS_CLS_MISS,
    PMD_METRICS_CLS_UPDATE,
    PMD_METRICS_N_CACHE_ENTRIES,
};

static unsigned int
dpcls_count(struct dpcls *cls)
{
    struct dpcls_subtable *subtable;
    unsigned int count = 0;

    CMAP_FOR_EACH (subtable, cmap_node, &cls->subtables_map) {
        count += cmap_count(&subtable->rules);
    }

    return count;
}

static void
poll_threads_cache_read_value(double *values, void *it)
{
    struct dp_doca_pmd_thread *pmd = it;
    uint64_t stats[PMD_N_STATS];
    unsigned int pmd_n_cls_rules;
    struct dpcls *cls;

    for (int i = 0; i < PMD_N_STATS; i++) {
        atomic_read_relaxed(&pmd->perf_stats.counters.n[i], &stats[i]);
    }

    values[PMD_METRICS_SIMPLE_N_ENTRIES] =
        cmap_count(&pmd->simple_match_table);
    values[PMD_METRICS_SIMPLE_HIT] = stats[PMD_STAT_SIMPLE_HIT];
    values[PMD_METRICS_SIMPLE_MISS] = stats[PMD_STAT_SIMPLE_MISS];
    values[PMD_METRICS_SIMPLE_UPDATE] = stats[PMD_STAT_SIMPLE_UPDATE];

    values[PMD_METRICS_EMC_N_ENTRIES] =
        emc_cache_count(&(pmd->flow_cache).emc_cache);
    values[PMD_METRICS_EMC_HIT] = stats[PMD_STAT_EXACT_HIT];
    values[PMD_METRICS_EMC_MISS] = stats[PMD_STAT_EXACT_MISS];
    values[PMD_METRICS_EMC_UPDATE] = stats[PMD_STAT_EXACT_UPDATE];

    values[PMD_METRICS_SMC_N_ENTRIES] =
        smc_cache_count(&(pmd->flow_cache).smc_cache);
    values[PMD_METRICS_SMC_HIT] = stats[PMD_STAT_SMC_HIT];
    values[PMD_METRICS_SMC_MISS] = stats[PMD_STAT_SMC_MISS];
    values[PMD_METRICS_SMC_UPDATE] = stats[PMD_STAT_SMC_UPDATE];

    pmd_n_cls_rules = 0;
    CMAP_FOR_EACH (cls, node, &pmd->classifiers) {
        pmd_n_cls_rules += dpcls_count(cls);
    }

    values[PMD_METRICS_CLS_N_ENTRIES] = pmd_n_cls_rules;
    values[PMD_METRICS_CLS_HIT] = stats[PMD_STAT_MASKED_HIT];
    values[PMD_METRICS_CLS_MISS] = stats[PMD_STAT_MASKED_LOOKUP] -
                                     stats[PMD_STAT_MASKED_HIT];
    values[PMD_METRICS_CLS_UPDATE] = stats[PMD_STAT_MASKED_UPDATE];
}

/* Use a single point of definition for the cache entries to enforce
 * strict alignment between 'datapath_cache' and 'poll_threads_cache'
 * metrics. */
#define PMD_METRICS_CACHE_ENTRIES                                      \
    /* Simple match cache. */                                          \
    [PMD_METRICS_SIMPLE_N_ENTRIES] = METRICS_GAUGE(simple_n_entries,   \
        "Number of entries in the simple match cache."),               \
    [PMD_METRICS_SIMPLE_HIT] = METRICS_COUNTER(simple_hit,             \
        "Number of lookup hit in the simple match cache."),            \
    [PMD_METRICS_SIMPLE_MISS] = METRICS_COUNTER(simple_miss,           \
        "Number of lookup miss in the simple match cache."),           \
    [PMD_METRICS_SIMPLE_UPDATE] = METRICS_COUNTER(simple_update,       \
        "Number of updates of the simple match cache."),               \
    /* Exact match cache. */                                           \
    [PMD_METRICS_EMC_N_ENTRIES] = METRICS_GAUGE(emc_n_entries,         \
        "Number of entries in the exact match cache."),                \
    [PMD_METRICS_EMC_HIT] = METRICS_COUNTER(emc_hit,                   \
        "Number of lookup hit in the exact match cache."),             \
    [PMD_METRICS_EMC_MISS] = METRICS_COUNTER(emc_miss,                 \
        "Number of lookup miss in the exact match cache."),            \
    [PMD_METRICS_EMC_UPDATE] = METRICS_COUNTER(emc_update,             \
        "Number of updates of the exact match cache."),                \
    /* Signature match cache. */                                       \
    [PMD_METRICS_SMC_N_ENTRIES] = METRICS_GAUGE(smc_n_entries,         \
        "Number of entries in the signature match cache."),            \
    [PMD_METRICS_SMC_HIT] = METRICS_COUNTER(smc_hit,                   \
        "Number of lookup hit in the signature match cache."),         \
    [PMD_METRICS_SMC_MISS] = METRICS_COUNTER(smc_miss,                 \
        "Number of lookup miss in the signature match cache."),        \
    [PMD_METRICS_SMC_UPDATE] = METRICS_COUNTER(smc_update,             \
        "Number of updates of the signature match cache."),            \
    /* Datapath classifiers. */                                        \
    [PMD_METRICS_CLS_N_ENTRIES] = METRICS_GAUGE(cls_n_entries,         \
        "Number of entries in the datapath classifiers."),             \
    [PMD_METRICS_CLS_HIT] = METRICS_COUNTER(cls_hit,                   \
        "Number of lookup hit in the datapath classifiers."),          \
    [PMD_METRICS_CLS_MISS] = METRICS_COUNTER(cls_miss,                 \
        "Number of lookup miss in the datapath classifiers."),         \
    [PMD_METRICS_CLS_UPDATE] = METRICS_COUNTER(cls_update,             \
        "Number of updates of the datapath classifiers."),

METRICS_ENTRIES(foreach_poll_threads_dbg, poll_threads_cache_dbg_entries,
    "poll_threads_cache", poll_threads_cache_read_value, PMD_METRICS_CACHE_ENTRIES);

static void
datapath_cache_read_value(double *values, void *it)
{
    double pmd_values[PMD_METRICS_N_CACHE_ENTRIES];
    struct dp_doca *dp = get_dp_doca(it);
    struct dp_doca_pmd_thread *pmd;
    int i;

    for (i = 0; i < ARRAY_SIZE(pmd_values); i++) {
        values[i] = 0.0;
    }

    CMAP_FOR_EACH (pmd, node, &dp->poll_threads) {
        poll_threads_cache_read_value(pmd_values, pmd);
        for (i = 0; i < ARRAY_SIZE(pmd_values); i++) {
            values[i] += pmd_values[i];
        }
    }
}

METRICS_ENTRIES(foreach_dpif_doca_ext, datapath_cache_ext_entries,
    "datapath_cache", datapath_cache_read_value, PMD_METRICS_CACHE_ENTRIES);

METRICS_DECLARE(hw_offload_threads_dbg_entries);
METRICS_DECLARE(hw_offload_latency);
METRICS_DECLARE(hw_offload_queue_sojourn_time);
METRICS_DECLARE(hw_offload_queue_wait_time);
METRICS_DECLARE(hw_offload_queue_service_time);
METRICS_DECLARE(datapath_hw_offload_entries);

void
dpif_doca_metrics_register(void)
{
    METRICS_REGISTER(datapath_cache_ext_entries);
    METRICS_REGISTER(poll_threads_n);
    METRICS_REGISTER(poll_threads_entries);
    METRICS_REGISTER(poll_threads_cache_dbg_entries);
    METRICS_REGISTER(hw_offload_threads_dbg_entries);
    METRICS_REGISTER(hw_offload_latency);
    METRICS_REGISTER(hw_offload_queue_sojourn_time);
    METRICS_REGISTER(hw_offload_queue_wait_time);
    METRICS_REGISTER(hw_offload_queue_service_time);
    METRICS_REGISTER(datapath_hw_offload_entries);
}
