/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2023-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#include <config.h>

#include <doca_flow.h>
#include <sys/types.h>

#include "dpdk-offload-doca.h"
#include "dpdk-offload-provider.h"
#include "doca-mirror.h"
#include "doca-pipe-group.h"
#include "id-fpool.h"
#include "netdev-doca.h"
#include "netdev-offload.h"
#include "netdev-offload-doca.h"
#include "openvswitch/vlog.h"
#include "ovs-doca.h"
#include "util.h"

/*
 * Mirrros in OVS DOCA OFFLOAD:
 *
 * Upper layer (netdev-dpdk-offload) provides ovs doca offload layer (dpdk-offload-doca) an
 * ordered RTE_FLOW_ACTION list. Mirrors are all the output ports before the last forward
 * action (JUMP, OUTPUT, ...).
 * Upper layer put those mirrored outputs in a OVS_RTE_FLOW_ACTION_TYPE(NESTED) which
 * has a nested sample action with ratio 1 that executes a sample action list with an output.
 *
 * +--------------------------+  +--> +--------------+ ---> +----------------------+
 * | Actions: original        |  |    | Flood pipe   |      | target group         |
 * |          + set FLOW_INFO |  |    | 0: set 0     |      | match: FLOW_INFO     |
 * | FWD: jump flood pipe     |--+    | 1: set 1     |      |        + flood entry |
 * +--------------------------+       | ...          |      | actions: target X    |
 *                                    | N-1: set N-1 |      | FWD: target X        |
 *                                    +--------------+      +----------------------+
 * The flood pipe and the target group are created upon ESW initialization.
 * A mirror-ctx is the set of rules in the target group. Entry 0 is the original packet
 * (not cloned) and performs only the original forward. The rest are cloned packets. They perform
 * the the target X actions and its FWD.
 */

VLOG_DEFINE_THIS_MODULE(doca_mirrors);
static struct vlog_rate_limit rl = VLOG_RATE_LIMIT_INIT(600, 600);

struct doca_mirrors {
    struct id_fpool *id_pool;
    struct netdev *netdev;
    struct doca_pipe_group_ctx *target_group;
    struct doca_flow_pipe *flood_pipe;
    struct doca_flow_pipe_entry *flood_entries[DOCA_MIRRORS_MAX_TARGETS];
};

struct doca_mirror_ctx {
    struct doca_mirrors *doca_mirrors;
    uint32_t id;
    struct doca_pipe_group_mask_entry target_entry_mctx[DOCA_MIRRORS_MAX_TARGETS];
};

static void
doca_mirrors_destroy_flood(struct doca_mirrors *doca_mirrors)
{
    struct doca_offload_esw_ctx *esw = doca_offload_esw_ctx_get(doca_mirrors->netdev);

    for (int ei = 0; ei < DOCA_MIRRORS_MAX_TARGETS; ei++) {
        if (doca_mirrors->flood_entries[ei]) {
            doca_offload_remove_entry(esw, AUX_QUEUE, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                      &doca_mirrors->flood_entries[ei]);
            dpdk_offload_counter_dec(doca_mirrors->netdev);
        }
    }

    if (doca_mirrors->flood_pipe) {
        doca_pipe_group_pipe_destroy(esw, doca_mirrors->flood_pipe);
        doca_mirrors->flood_pipe = NULL;
    }
}

static struct doca_flow_pipe *
doca_mirrors_create_flood_pipe(struct netdev *netdev)
{
    struct doca_flow_port *esw_port = netdev_doca_port_get(netdev);
    struct doca_flow_actions *actions_arr[1];
    struct ovs_doca_flow_actions actions;
    struct doca_flow_pipe *pipe = NULL;
    struct doca_flow_pipe_cfg *cfg;
    struct doca_flow_fwd fwd, miss;
    char pipe_name[50];
    int ret;

    memset(&actions, 0, sizeof actions);
    memset(&fwd, 0, sizeof fwd);
    memset(&miss, 0, sizeof miss);

    actions_arr[0] = &actions.d;
    doca_offload_set_reg_template(&actions.d.meta, REG_FIELD_SCRATCH);
    fwd.type = DOCA_FLOW_FWD_PIPE;
    miss.type = DOCA_FLOW_FWD_DROP;

    snprintf(pipe_name, sizeof pipe_name, "%s_OVS_MIRROR_FLOOD_PIPE", netdev_get_name(netdev));

    ret = doca_flow_pipe_cfg_create(&cfg, esw_port);
    if (ret) {
        VLOG_ERR("%s: Could not create doca_flow_pipe_cfg", netdev_get_name(netdev));
        return NULL;
    }

    if (doca_flow_pipe_cfg_set_name(cfg, pipe_name) ||
        doca_flow_pipe_cfg_set_actions(cfg, actions_arr, actions_arr, NULL, 1) ||
        doca_flow_pipe_cfg_set_type(cfg, DOCA_FLOW_PIPE_HASH) ||
        doca_flow_pipe_cfg_set_hash_map_algorithm(cfg,
                                                  DOCA_FLOW_PIPE_HASH_MAP_ALGORITHM_FLOODING) ||
        doca_flow_pipe_cfg_set_nr_entries(cfg, DOCA_MIRRORS_MAX_TARGETS)) {
        VLOG_ERR("%s: Could not set doca_flow_pipe_cfg", netdev_get_name(netdev));
        goto out;
    }

    ret = doca_flow_pipe_create(cfg, &fwd, &miss, &pipe);
    if (ret) {
        VLOG_ERR("%s: Failed to create hash pipe: %d (%s)", netdev_get_name(netdev),
                 ret, doca_error_get_descr(ret));
        goto out;
    }

out:
    doca_flow_pipe_cfg_destroy(cfg);
    return pipe;
}

static int
doca_mirrors_create_flood(struct doca_mirrors *doca_mirrors)
{
    struct doca_flow_fwd fwd = { .type = DOCA_FLOW_FWD_PIPE, };
    struct ovs_doca_flow_actions actions;

    doca_mirrors->flood_pipe = doca_mirrors_create_flood_pipe(doca_mirrors->netdev);
    if (!doca_mirrors->flood_pipe) {
        return -1;
    }

    for (int ei = 0; ei < DOCA_MIRRORS_MAX_TARGETS; ei++) {
        int ret;

        fwd.next_pipe = doca_pipe_group_get_pipe(doca_mirrors->target_group);
        memset(&actions, 0, sizeof actions);
        doca_set_reg_val(&actions.d.meta, REG_FIELD_SCRATCH, ei);
        ret = doca_offload_add_entry_hash(doca_mirrors->netdev, AUX_QUEUE,
                                          doca_mirrors->flood_pipe, ei, &actions, NULL, &fwd,
                                          DOCA_FLOW_ENTRY_FLAGS_NO_WAIT,
                                          &doca_mirrors->flood_entries[ei]);
        if (ret) {
            VLOG_ERR("%s: Failed to create hash pipe entry. Error: %d (%s)",
                     netdev_get_name(doca_mirrors->netdev), ret, doca_error_get_descr(ret));
            return ret;
        }
    }

    return 0;
}

static void
doca_mirrors_destroy_target_group(struct doca_mirrors *doca_mirrors)
{
    doca_pipe_group_ctx_unref(doca_mirrors->target_group);
    doca_mirrors->target_group = NULL;
}

static int
doca_mirrors_create_target_group(struct doca_mirrors *doca_mirrors)
{
    doca_mirrors->target_group =
        doca_pipe_group_ctx_ref(doca_mirrors->netdev, MIRROR_TARGET_GROUP_TABLE_ID, NULL);
    if (!doca_mirrors->target_group) {
        return -1;
    }

    return 0;
}

struct doca_mirrors *
doca_mirrors_create(struct netdev *netdev)
{
    struct doca_mirrors *doca_mirrors = xzalloc(sizeof *doca_mirrors);
    struct reg_field *reg_field;

    reg_field = netdev_offload_doca_get_reg_field(REG_FIELD_FLOW_INFO);
    doca_mirrors->id_pool = id_fpool_create(MAX_OFFLOAD_THREAD_NB, 1, reg_field->mask);
    if (!doca_mirrors->id_pool) {
        goto err_create_id_pool;
    }

    doca_mirrors->netdev = netdev;
    if (doca_mirrors_create_target_group(doca_mirrors)) {
        goto err_target_group;
    }
    if (doca_mirrors_create_flood(doca_mirrors)) {
        goto err_flood;
    }

    return doca_mirrors;

err_flood:
    doca_mirrors_destroy_flood(doca_mirrors);
err_target_group:
    doca_mirrors_destroy_target_group(doca_mirrors);
    id_fpool_destroy(doca_mirrors->id_pool);
err_create_id_pool:
    free(doca_mirrors);
    return NULL;
}

void
doca_mirrors_destroy(struct doca_mirrors *doca_mirrors)
{
    if (!doca_mirrors) {
        return;
    }

    doca_mirrors_destroy_flood(doca_mirrors);
    doca_mirrors_destroy_target_group(doca_mirrors);

    id_fpool_destroy(doca_mirrors->id_pool);
    free(doca_mirrors);
}

struct doca_mirror_ctx *
doca_mirror_ctx_create(struct doca_mirrors *doca_mirrors,
                       struct ovs_mirror_set *mirror_set,
                       struct ovs_doca_flow_actions *actions,
                       struct ovs_doca_flow_actions *actions_mask,
                       struct doca_flow_fwd *fwd)
{
    struct ovs_doca_flow_actions post_flood_actions, post_flood_actions_mask;
    unsigned int queue_id = netdev_offload_thread_id();
    struct doca_pipe_group_mask_ctx **mask_ctx;
    struct doca_flow_pipe_entry **pentry;
    struct doca_mirror_ctx *mirror_ctx;
    struct ovs_doca_flow_match spec;
    struct ovs_doca_flow_match mask;
    uint32_t queue2prio[] = {
        [AUX_QUEUE] = DPDK_OFFLOAD_PRIORITY_HIGH,
        [FIRST_QUEUE] = DPDK_OFFLOAD_PRIORITY_MED,
    };
    uint32_t prio;
    int ret;

    if (mirror_set->nr_targets >= DOCA_MIRRORS_MAX_TARGETS) {
        VLOG_WARN_RL(&rl, "%s: Too many targets", netdev_get_name(doca_mirrors->netdev));
        return NULL;
    }

    mirror_ctx = xzalloc(sizeof *mirror_ctx);
    mirror_ctx->doca_mirrors = doca_mirrors;

    if (!id_fpool_new_id(mirror_ctx->doca_mirrors->id_pool, queue_id, &mirror_ctx->id)) {
        ret = -1;
        goto out;
    }

    /* The queue translation to priorities are meant that there can't be a shared mask_ctx
     * for both queues. It is assumed the only queues are the AUX_QUEUE and the FIRST_QUEUE,
     * for sflow and mirroring respectively.
     */
    ovs_assert(queue_id == AUX_QUEUE || queue_id == FIRST_QUEUE);
    prio = queue2prio[queue_id];
    pentry = &mirror_ctx->target_entry_mctx[0].entry;
    mask_ctx = &mirror_ctx->target_entry_mctx[0].mctx;
    memset(&spec, 0, sizeof spec);
    memset(&mask, 0, sizeof mask);
    memset(&post_flood_actions, 0, sizeof post_flood_actions);
    memset(&post_flood_actions_mask, 0, sizeof post_flood_actions_mask);
    memcpy(&post_flood_actions.d.meta, &actions->d.meta.pkt_meta,
           sizeof post_flood_actions.d.meta);
    memcpy(&post_flood_actions_mask.d.meta, &actions_mask->d.meta.pkt_meta,
           sizeof post_flood_actions_mask.d.meta);
    doca_set_reg_val_mask(&spec.d.meta, &mask.d.meta, REG_FIELD_FLOW_INFO, mirror_ctx->id);
    doca_set_reg_val_mask(&spec.d.meta, &mask.d.meta, REG_FIELD_SCRATCH, 0);
    ret = doca_pipe_group_add_entry(doca_mirrors->netdev, queue_id, prio,
                                    doca_mirrors->target_group, &spec, &mask, &post_flood_actions,
                                    &post_flood_actions_mask, NULL, NULL, fwd, mask_ctx, pentry);
    if (ret) {
        VLOG_WARN_RL(&rl, "%s: Failed to create group entry. Error: %d (%s)",
                     netdev_get_name(doca_mirrors->netdev), ret, doca_error_get_descr(ret));
        goto out;
    }

    for (int ti = 0; ti < mirror_set->nr_targets; ti++) {
        struct ovs_mirror_target *target = &mirror_set->target[ti];

        pentry = &mirror_ctx->target_entry_mctx[ti + 1].entry;
        mask_ctx = &mirror_ctx->target_entry_mctx[ti + 1].mctx;
        memset(&spec, 0, sizeof spec);
        memset(&mask, 0, sizeof mask);
        doca_set_reg_val_mask(&spec.d.meta, &mask.d.meta, REG_FIELD_FLOW_INFO, mirror_ctx->id);
        doca_set_reg_val_mask(&spec.d.meta, &mask.d.meta, REG_FIELD_SCRATCH, ti + 1);
        ret = doca_pipe_group_add_entry(doca_mirrors->netdev, queue_id, prio,
                                        doca_mirrors->target_group, &spec, &mask,
                                        &target->actions, &target->actions_mask, NULL, NULL,
                                        &target->fwd, mask_ctx, pentry);
        if (ret) {
            VLOG_WARN_RL(&rl, "%s: Failed to create group entry. Error: %d (%s)",
                         netdev_get_name(doca_mirrors->netdev), ret, doca_error_get_descr(ret));
            goto out;
        }
    }

out:
    if (ret) {
        doca_mirror_ctx_destroy(mirror_ctx);
        return NULL;
    }

    /* The provided actions are used with addition of setting REG_FIELD_FLOW_INFO.
     * The provided fwd is used in the first entry of target_groups (original packet) while
     * the original one is replaced by jumping to the flood pipe.
     */
    actions->d.meta.pkt_meta = 0;
    doca_set_reg_val_mask(&actions->d.meta, &actions_mask->d.meta, REG_FIELD_FLOW_INFO,
                          mirror_ctx->id);
    memset(fwd, 0, sizeof *fwd);
    fwd->type = DOCA_FLOW_FWD_PIPE;
    fwd->next_pipe = doca_mirrors->flood_pipe;

    return mirror_ctx;
}

void
doca_mirror_ctx_destroy(struct doca_mirror_ctx *mirror_ctx)
{
    struct doca_offload_esw_ctx *esw;
    struct netdev *netdev;
    unsigned int queue_id;

    if (!mirror_ctx) {
        return;
    }
    ovs_assert(mirror_ctx->doca_mirrors && mirror_ctx->doca_mirrors->netdev);

    netdev = mirror_ctx->doca_mirrors->netdev;
    queue_id = netdev_offload_thread_id();
    esw = doca_offload_esw_ctx_get(netdev);
    for (int ti = 0; ti < DOCA_MIRRORS_MAX_TARGETS; ti++) {
        struct doca_flow_pipe_entry **pentry = &mirror_ctx->target_entry_mctx[ti].entry;
        struct doca_pipe_group_mask_ctx *mctx = mirror_ctx->target_entry_mctx[ti].mctx;

        doca_offload_remove_entry(esw, queue_id, DOCA_FLOW_ENTRY_FLAGS_NO_WAIT, pentry);
        doca_pipe_group_mask_ctx_unref(mctx);
        dpdk_offload_counter_dec(netdev);
    }
    id_fpool_free_id(mirror_ctx->doca_mirrors->id_pool, queue_id, mirror_ctx->id);

    free(mirror_ctx);
}
