// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: BSD-3-Clause

#ifndef LIBVFIO_MLX5_UTIL_H
#define LIBVFIO_MLX5_UTIL_H

#include <stdbool.h>
#include <stdint.h>
#include <sys/types.h>
#include "util/event_log.h"

extern event_log_t libmlx5_logger;

#ifdef DEBUG
#define DEFAULT_LOG_LEVEL MLX5_LOG_LVL_DEBUG
#else
#define DEFAULT_LOG_LEVEL MLX5_LOG_LVL_INFO
#endif

#define log_debug(M, ...) \
	event_log(&libmlx5_logger, MLX5_LOG_LVL_DEBUG, M"\n", ##__VA_ARGS__)
#define log_info(M, ...) \
	event_log(&libmlx5_logger, MLX5_LOG_LVL_INFO, M"\n", ##__VA_ARGS__)
#define log_warn(M, ...) \
	event_log(&libmlx5_logger, MLX5_LOG_LVL_WARN, M"\n", ##__VA_ARGS__)
#define log_error(M, ...) \
	event_log(&libmlx5_logger, MLX5_LOG_LVL_ERR, M"\n", ##__VA_ARGS__)

#define WARN_ON(x) \
	({ \
		int __ret_warn_on = !!(x); \
		if (__ret_warn_on) { \
			log_warn("Warning: %s at %s:%d", #x, __FILE__, __LINE__); \
		} \
		__ret_warn_on; \
	})

#define ARRAY_SIZE(arr) (sizeof(arr) / sizeof((arr)[0]))

/* Check if we have the builtin functions */
#if defined(__GNUC__) || defined(__clang__)
/* Check if specific builtins are available */
#ifndef __has_builtin
#define __has_builtin(x) 0  /* Compatibility with non-clang compilers */
#endif

#if __has_builtin(__builtin_clz) || (defined(__GNUC__) && (__GNUC__ >= 4))
static inline int ilog32(uint32_t v)
{
	return v ? 32 - __builtin_clz(v) : 0;
}
#else
#warning "Using fallback implementation for ilog32 as __builtin_clz is not available"
static inline int ilog32(uint32_t _v)
{
	 static const unsigned char DEBRUIJN_IDX32[32] = {
		 0,  1,  28, 2,  29, 14, 24, 3, 30, 22, 20, 15, 25, 17, 4,  8,
		 31, 27, 13, 23, 21, 19, 16, 7, 26, 12, 18, 6,  11, 5,  10, 9
	 };
	 int ret;

	 ret = _v > 0;
	 _v |= _v >> 1;
	 _v |= _v >> 2;
	 _v |= _v >> 4;
	 _v |= _v >> 8;
	 _v |= _v >> 16;
	 _v = (_v >> 1) + 1;
	 ret += DEBRUIJN_IDX32[_v * 0x77CB531U >> 27 & 0x1F];
	 return ret;
}
#endif

#if __has_builtin(__builtin_clzll) || (defined(__GNUC__) && (__GNUC__ >= 4))
static inline int ilog64(uint64_t v)
{
	return v ? 64 - __builtin_clzll(v) : 0;
}
#else
#warning "Using fallback implementation for ilog64 as __builtin_clzll is not available"
static inline int ilog64(uint64_t _v)
{
	static const unsigned char DEBRUIJN_IDX64[64] = {
		0,  1,  2,  7,  3,  13, 8,  19, 4,  25, 14, 28, 9,  34, 20, 40,
		5,  17, 26, 38, 15, 46, 29, 48, 10, 31, 35, 54, 21, 50, 41, 57,
		63, 6,  12, 18, 24, 27, 33, 39, 16, 37, 45, 47, 30, 53, 49, 56,
		62, 11, 23, 32, 36, 44, 52, 55, 61, 22, 43, 51, 60, 42, 59, 58
	};
	int ret;

	ret = _v > 0;
	_v |= _v >> 1;
	_v |= _v >> 2;
	_v |= _v >> 4;
	_v |= _v >> 8;
	_v |= _v >> 16;
	_v |= _v >> 32;
	_v = (_v >> 1) + 1;
	ret += DEBRUIJN_IDX64[_v * 0x218A392CD3D5DBFULL >> 58 & 0x3F];
	return ret;
}
#endif

#else
#error "Compiler is neither GCC nor Clang"
#endif

static inline uint64_t roundup_pow_of_two(uint64_t n)
{
	 return n == 1 ? 1 : 1ULL << ilog64(n - 1);
}

static inline unsigned long DIV_ROUND_UP(unsigned long n, unsigned long d)
{
	return (n + d - 1) / d;
}

#define ALIGN_UP(x, a) (((x) + (a) - 1) & ~((a) - 1))
#define ALIGN_DOWN(x, a) ((x) & ~((a) - 1))

#define MINMAX_ASSERT_COMPATIBLE(a, b) do { } while (0)

#define min(a, b)								\
		({								\
			 typeof(a) _a = (a);					\
			 typeof(b) _b = (b);					\
			 MINMAX_ASSERT_COMPATIBLE(typeof(_a), typeof(_b));	\
			 _a < _b ? _a : _b;					\
		})

#define min_t(t, a, b)								\
		({								\
			t _ta = (a);						\
			t _tb = (b);						\
			min(_ta, _tb);						\
		})

#define max(a, b)								\
	({									\
		typeof(a) _a = (a);						\
		typeof(b) _b = (b);						\
		MINMAX_ASSERT_COMPATIBLE(typeof(_a), typeof(_b));		\
		_a > _b ? _a : _b;						\
	})

#endif
