// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: BSD-3-Clause

#ifndef __UTIL_MMIO_H
#define __UTIL_MMIO_H

#include <linux/types.h>
#include <stdatomic.h>
#include <stdint.h>
#include <stddef.h>
#include <endian.h>

#include "compiler.h"

/* The first step is to define the 'raw' accessors. To make this very safe
 * with sparse we define two versions of each, a le and a be - however the
 * code is always identical.
 */
#define MAKE_WRITE(_NAME_, _SZ_)					\
	static inline void _NAME_##_be(void *addr, __be##_SZ_ value)	\
	{								\
		atomic_store_explicit((_Atomic(uint##_SZ_##_t) *)addr,	\
				      (__force uint##_SZ_##_t)value,	\
				      memory_order_relaxed);		\
	}								\
	static inline void _NAME_##_le(void *addr, __le##_SZ_ value)	\
	{								\
		atomic_store_explicit((_Atomic(uint##_SZ_##_t) *)addr,	\
				      (__force uint##_SZ_##_t)value,	\
				      memory_order_relaxed);		\
	}
#define MAKE_READ(_NAME_, _SZ_)						\
	static inline __be##_SZ_ _NAME_##_be(const void *addr)		\
	{								\
		return (__force __be##_SZ_)atomic_load_explicit(	\
			(_Atomic(uint##_SZ_##_t) *)addr,		\
			memory_order_relaxed);				\
	}								\
	static inline __le##_SZ_ _NAME_##_le(const void *addr)		\
	{								\
		return (__force __le##_SZ_)atomic_load_explicit(	\
			(_Atomic(uint##_SZ_##_t) *)addr,		\
			memory_order_relaxed);				\
	}

static inline void mmio_write8(void *addr, uint8_t value)
{
	atomic_store_explicit((_Atomic(uint8_t) *)addr, value,
			      memory_order_relaxed);
}
static inline uint8_t mmio_read8(const void *addr)
{
	return atomic_load_explicit((_Atomic(uint8_t) *)addr,
				    memory_order_relaxed);
}

MAKE_WRITE(mmio_write16, 16)
MAKE_WRITE(mmio_write32, 32)

MAKE_READ(mmio_read16, 16)
MAKE_READ(mmio_read32, 32)

#if SIZEOF_LONG == 8
MAKE_WRITE(mmio_write64, 64)
MAKE_READ(mmio_read64, 64)
#else
void mmio_write64_be(void *addr, __be64 val);
static inline void mmio_write64_le(void *addr, __le64 val)
{
	mmio_write64_be(addr, (__be64 __force)val);
}

/* There is no way to do read64 atomically, rather than provide some sketchy
 * implementation we leave these functions undefined, users should not call
 * them if SIZEOF_LONG != 8, but instead implement an appropriate version.
 */
__be64 mmio_read64_be(const void *addr);
__le64 mmio_read64_le(const void *addr);
#endif /* SIZEOF_LONG == 8 */

#undef MAKE_WRITE
#undef MAKE_READ

/* Now we can define the host endian versions of the operator, this just includes
 * a call to htole.
 */
#define MAKE_WRITE(_NAME_, _SZ_)					\
	static inline void _NAME_(void *addr, uint##_SZ_##_t value)	\
	{								\
		_NAME_##_le(addr, htole##_SZ_(value));			\
	}
#define MAKE_READ(_NAME_, _SZ_)						\
	static inline uint##_SZ_##_t _NAME_(const void *addr)		\
	{								\
		return le##_SZ_##toh(_NAME_##_le(addr));		\
	}

MAKE_WRITE(mmio_write16, 16)
MAKE_WRITE(mmio_write32, 32)
MAKE_WRITE(mmio_write64, 64)

MAKE_READ(mmio_read16, 16)
MAKE_READ(mmio_read32, 32)
MAKE_READ(mmio_read64, 64)

#undef MAKE_WRITE
#undef MAKE_READ

#endif /* MMIO_H */
