// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: BSD-3-Clause

#ifndef __MLX5_DEV_H__
#define __MLX5_DEV_H__

#include "abi.h"
#include "include/vfio_mlx5.h"
#include "pagealloc.h"
#include "cmd.h"
#include "hcacap.h"
#include "eq.h"
#include "fwpages.h"
#include "func.h"
#include "util/event_log.h"

#define MLX5_ADAPTER_PAGE_SIZE 4096
#define MLX5_ADAPTER_PAGE_SHIFT 12

#define PCI_BDF_LEN 13 /* 0000:08:00.0 */

#define MLX5_MAX_ADJ_PFS 4

/* pointers to dynamically allocated data */
struct mlx5_dev_heap {
	uintptr_t vmh_vaddr; /* main storage area vaddr */

	struct mlx5_fwpage_reqs fwpage_reqs;
	event_log_t devlog;
	event_log_t cmdlog;

	/* BAR0 map */
	size_t bar_map_size;
	struct mlx5_bar *bar_map;
};

enum vfio_mlx5_state {
	VFIO_MLX5_STATE_DISABLED = 0,
	VFIO_MLX5_STATE_ENABLED = 1,
	VFIO_MLX5_STATE_SUSPENDED = 2,
};

struct __packed_aligned64 mlx5_health_state {
	uint64_t prev_time; /* ms */
	uint32_t prev_count;
	uint32_t miss_counter;
	/* add new fields here, struct is padded to 64 bytes */
};
_ABI_sz_assert(1.0, struct mlx5_health_state, 64);

struct __packed_aligned4k vfio_mlx5_dev {
	/* data */
	char pci_bdf[PCI_BDF_LEN];
	int device_fd; /* VFIO device file descriptor */
	uint16_t index;
	uint16_t num_vfs;

	/* state */
	uint8_t state;
	uint32_t resume_token; /* protect against API misuse */

	struct mlx5_health_state health_state __aligned(64);
	struct mlx5_vfio_cmd cmd __ptr_align;
	struct mlx5_eq async_eq __aligned(256);
	struct mlx5_health_record health_rec;
	bool have_eq;
	bool pci_err;

	struct {
		bool inflight; /* true if a page request is in progress */
		struct fwpage_fifo fifo;
	} page_reqs __ptr_align;

	struct mlx5_dev_caps caps __packed_aligned(256);

	struct mlx5_adj_pf adj_pfs[MLX5_MAX_ADJ_PFS];
	uint8_t adj_pfs_count;

	/* stats */
	mlx5_cmd_stats_t cmd_stats __ptr_align;
	mlx5_pg_events_t page_events;
	uint64_t firmware_pages;
	/* base iova of the storage area, used for vaddr2iova and iova2vaddr */
	uint64_t vmh_iova;

	/* pointers to dynamically allocated data */
	/* must be freed on suspend, re-alloc and fill on resume */
	struct mlx5_dev_heap *heap;
	/* Global page allocator */
	struct page_allocator *page_alloc;
};

_ABI_offset_assert(1.0, struct vfio_mlx5_dev, vmh_iova, 1936);
_ABI_sz_assert(1.0, struct vfio_mlx5_dev, 4096);

_ABI_sz_assert(1.0, struct mlx5_dev_info, 128);
_ABI_offset_assert(1.0, struct mlx5_dev_info, num_vfs, 4);

_ABI_sz_assert(1.0, struct mlx5_dev_stats, 880);
_ABI_offset_assert(1.0, struct mlx5_dev_stats, health_rec, 768);

/* ABI check for mlx5_pg_events_t */
_Static_assert(FWP_EVENT_MAX == 32 && FWP_EVENT_COUNT <= FWP_EVENT_MAX,
	       "FWP_EVENT_MAX must be less than or equal to 32");

/* ABI checks for mlx5_cmd_stats_t */
_Static_assert(CMD_EVENT_MAX == 32 && CMD_EVENT_COUNT <= CMD_EVENT_MAX,
	       "CMD_EVENT_MAX must be less than or equal to 32");

#define dev_info(dev, fmt, args...) \
	event_log(&dev->heap->devlog, MLX5_LOG_LVL_INFO, fmt "\n", ##args)
#define dev_err(dev, fmt, args...) \
	event_log(&dev->heap->devlog, MLX5_LOG_LVL_ERR, fmt "\n", ##args)
#define dev_warn(dev, fmt, args...) \
	event_log(&dev->heap->devlog, MLX5_LOG_LVL_WARN, fmt "\n", ##args)
#define dev_dbg(dev, fmt, args...) \
	event_log(&dev->heap->devlog, MLX5_LOG_LVL_DEBUG, fmt "\n", ##args)

static inline void *iova2vaddr(const struct vfio_mlx5_dev *dev, uint64_t iova)
{
	size_t offset = iova - dev->vmh_iova;
	return (void *)(dev->heap->vmh_vaddr + offset);
}

static inline uint64_t vaddr2iova(const struct vfio_mlx5_dev *dev,
				  const void *vaddr)
{
	size_t offset = (uintptr_t)vaddr - dev->heap->vmh_vaddr;
	return dev->vmh_iova + offset;
}

int mlx5_vfio_device_init(struct vfio_mlx5_dev *dev,
			  struct page_allocator *page_alloc,
			  const void *vmh_vaddr);
void mlx5_vfio_device_uninit(struct vfio_mlx5_dev *dev);

int mlx5_vfio_dev_suspend(struct vfio_mlx5_dev *dev);
int mlx5_vfio_dev_resume(struct vfio_mlx5_dev *dev,
			 struct page_allocator *page_alloc,
			 const void *vmh_vaddr);

int mlx5_device_heap_alloc(struct vfio_mlx5_dev *dev,
			   struct page_allocator *page_alloc,
			   const void *vmh_vaddr);
void mlx5_device_heap_free(struct vfio_mlx5_dev *dev);

#endif /* __MLX5_DEV_H__ */
