/*
 * Copyright (c) 2026 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <config.h>

#include "coverage.h"
#include "doca-telemetry.h"
#include "metrics.h"
#include "openvswitch/util.h"
#include "openvswitch/vlog.h"
#include "openvswitch/poll-loop.h"
#include "timeval.h"

VLOG_DEFINE_THIS_MODULE(doca_telemetry);

/* Telemetry interval should be between
 * 100ms and 24h included. */
#define DOCA_TELEMETRY_MIN_INTERVAL_MS 100
#define DOCA_TELEMETRY_MAX_INTERVAL_MS (1000 * 60 * 60 * 24)

struct doca_telemetry_config {
    /* Time in milliseconds between wake-up events on the main
     * thread to send the telemetry. */
    long long int interval_ms;
    /* Whether the module is currently requested by the user. */
    bool enabled;
    bool http_exporter;
    uint16_t http_port;
    bool ipc_exporter;
};

static struct doca_telemetry_config default_config = {
    .enabled = false,
    .interval_ms = 1000,
    .ipc_exporter = false,
    .http_exporter = false,
    .http_port = 6104,
};

static void
doca_telemetry_dynamic_config(const struct smap *ovs_other_config,
                              struct doca_telemetry_config *config,
                              bool log_changes)
{
    static const char *status_str[] = {
        [false] = "disabled",
        [true] = "enabled",
    };
    long long int interval_ms;
    bool http_exporter;
    bool ipc_exporter;
    int http_port;
    bool enabled;

    http_exporter = smap_get_bool(ovs_other_config, "doca-telemetry-http",
                                  default_config.http_exporter);
    if (http_exporter != config->http_exporter) {
        if (log_changes) {
            VLOG_INFO("DOCA Telemetry HTTP exporter %s", status_str[http_exporter]);
        }
        config->http_exporter = http_exporter;
    }

    http_port = smap_get_int(ovs_other_config, "doca-telemetry-http-port",
                             default_config.http_port);
    if (http_port != config->http_port) {
        if (log_changes && http_exporter) {
            VLOG_INFO("DOCA Telemetry HTTP using port %d", http_port);
        }
        config->http_port = http_port;
    }

    ipc_exporter = smap_get_bool(ovs_other_config, "doca-telemetry-ipc",
                                 default_config.ipc_exporter);
    if (ipc_exporter != config->ipc_exporter) {
        if (log_changes) {
            VLOG_INFO("DOCA Telemetry IPC exporter %s", status_str[ipc_exporter]);
        }
        config->ipc_exporter = ipc_exporter;
    }

    interval_ms = smap_get_ullong(ovs_other_config, "doca-telemetry-interval",
                                  default_config.interval_ms);
    if (interval_ms < DOCA_TELEMETRY_MIN_INTERVAL_MS) {
        VLOG_WARN_ONCE("Requested DOCA Telemetry interval lower than allowed: %lld < %d",
                       interval_ms, DOCA_TELEMETRY_MIN_INTERVAL_MS);
        interval_ms = DOCA_TELEMETRY_MIN_INTERVAL_MS;
    }
    if (interval_ms > DOCA_TELEMETRY_MAX_INTERVAL_MS) {
        VLOG_WARN_ONCE("Requested DOCA Telemetry interval higher than allowed: %lld > %d",
                       interval_ms, DOCA_TELEMETRY_MAX_INTERVAL_MS);
        interval_ms = DOCA_TELEMETRY_MAX_INTERVAL_MS;
    }
    if (interval_ms != config->interval_ms) {
        if (config->interval_ms != 0 && log_changes) {
            VLOG_INFO("DOCA Telemetry interval configured to %lld ms", interval_ms);
        }
        config->interval_ms = interval_ms;
    }

    enabled = smap_get_bool(ovs_other_config, "doca-telemetry",
                            config->http_exporter ||
                            config->ipc_exporter ||
                            default_config.enabled);
    if (enabled != config->enabled) {
        if (log_changes) {
            VLOG_INFO("DOCA Telemetry %s", status_str[enabled]);
        }
        config->enabled = enabled;
    }
}

#if DOCA_TELEMETRY

COVERAGE_DEFINE(doca_telemetry_flush);

#include <clx_api.h>
#include <clx_api_metrics.h>

struct doca_telemetry_context {
    /* The whole structure is accessed only by the main thread.
     * No synchronization necessary. */

    /* Internal elements maintained during operation. */
    clx_api_context_t *clx_ctx;
    clx_metrics_api_context_t *clx_metrics_ctx;
    clx_api_provider_t provider;

    long long int last_run_ms;
    uint64_t timestamp;

    bool initialized;
    /* Initialization failed, do not retry until user action. */
    bool init_error;

    /* Current configuration.
     * These fields can be modified while operating by the
     * user modifying the relevant 'other_config' entries.
     */
    struct doca_telemetry_config config;
};

static struct doca_telemetry_context gctx;

static void
metrics_to_clx_write(struct metrics_node *node, void *aux,
                     const char **label_keys, const char **label_values, size_t n_labels,
                     double *values, size_t n_values)
{
    struct doca_telemetry_context *dctx;
    clx_metrics_api_context_t *ctx;
    struct metrics_array *array;
    size_t label_set_id;

    /* Only support counters and gauges for now. */
    if (node->type != METRICS_NODE_TYPE_ARRAY) {
        return;
    }

    dctx = aux;
    ctx = dctx->clx_metrics_ctx;

    label_set_id = 0;
    if (label_keys) {
        label_set_id = clx_api_metrics_add_label_names(ctx, label_keys, n_labels);
        if (label_set_id == CLX_METRICS_API_LABEL_SET_ID_IS_INVALID) {
            struct ds labels_ds = DS_EMPTY_INITIALIZER;

            for (size_t i = 0; i < n_labels; i++) {
                if (labels_ds.length) {
                    ds_put_char(&labels_ds, ',');
                }
                ds_put_format(&labels_ds, "%s=%s",
                              label_keys[i], label_values[i]);
            }
            VLOG_ERR("Failed to create label set '{%s}'", ds_cstr(&labels_ds));
            ds_destroy(&labels_ds);
        }
    }

    array = metrics_node_cast(node);
    ovs_assert(n_values == array->n_entries);
    for (size_t i = 0; i < array->n_entries; i++) {
        struct metrics_entry *entry = &array->entries[i];
        struct ds full_name = DS_EMPTY_INITIALIZER;

        metrics_entry_name(node, entry, &full_name);

        if (entry->type == METRICS_ENTRY_TYPE_COUNTER) {
            clx_api_metrics_add_counter(ctx, dctx->timestamp, ds_cstr(&full_name),
                                        values[i], label_set_id, label_values);
        } else if (entry->type == METRICS_ENTRY_TYPE_GAUGE) {
            clx_api_metrics_add_gauge(ctx, dctx->timestamp, ds_cstr(&full_name),
                                      values[i], label_set_id, label_values);
        }
        ds_destroy(&full_name);
    }
}

static void
doca_telemetry_cleanup(void)
{
    if (gctx.clx_metrics_ctx) {
        clx_api_metrics_destroy_context(gctx.clx_metrics_ctx);
        gctx.clx_metrics_ctx = NULL;
    }
    if (gctx.clx_ctx) {
        clx_api_destroy_context(gctx.clx_ctx);
        gctx.clx_ctx = NULL;
    }
    memset(&gctx.provider, 0, sizeof(gctx.provider));
    gctx.initialized = false;
}

void
doca_telemetry_run(void)
{
    long long int elapsed;
    long long int now;

    if (gctx.initialized == false) {
        return;
    }

    now = time_msec();
    elapsed = now - gctx.last_run_ms;
    if (0 < elapsed && elapsed < gctx.config.interval_ms) {
        return;
    }

    gctx.timestamp = clx_api_get_timestamp();
    metrics_foreach_node(metrics_to_clx_write, &gctx, false);
    clx_api_metrics_flush(gctx.clx_metrics_ctx);
    COVERAGE_INC(doca_telemetry_flush);

    gctx.last_run_ms = now;
}

void
doca_telemetry_wait(void)
{
    long long int next_wake;

    if (gctx.initialized == false) {
        return;
    }

    next_wake = gctx.last_run_ms + gctx.config.interval_ms;
    poll_timer_wait_until(next_wake);
}

static bool
doca_telemetry_provider_initialize(clx_api_context_t *ctx OVS_UNUSED,
                                   clx_api_provider_t *provider OVS_UNUSED)
{
    return true;
}

int
doca_telemetry_initialize(const struct smap *ovs_other_config)
{
    static bool run_once = false;

    struct doca_telemetry_config config;
    clx_api_params_t params;

    if (!run_once) {
        memcpy(&gctx.config, &default_config, sizeof gctx.config);
        run_once = true;
    }

    memcpy(&config, &gctx.config, sizeof config);
    doca_telemetry_dynamic_config(ovs_other_config, &config, true);

    /* A change to the configured 'interval_ms' does not trigger a re-init. */
    gctx.config.interval_ms = config.interval_ms;
    if (memcmp(&gctx.config, &config, sizeof config)) {
        doca_telemetry_cleanup();
        gctx.init_error = false;
        memcpy(&gctx.config, &config, sizeof config);
    }

    if (!config.enabled) {
        return 0;
    }

    if (gctx.initialized || gctx.init_error) {
        return 0;
    }

    memset(&params, 0, sizeof params);
    if (config.ipc_exporter) {
        params.ipc_enabled = true;
        params.ipc_sockets_dir = "/opt/mellanox/doca/services/telemetry/ipc_sockets";
        setenv("IPC_ENABLE", "1", 1);
        setenv("IPC_SOCKETS_FOLDER", params.ipc_sockets_dir, 1);
        VLOG_INFO("Starting DOCA Telemetry IPC export to directory '%s'", params.ipc_sockets_dir);
    } else {
        unsetenv("IPC_ENABLE");
        unsetenv("IPC_SOCKETS_FOLDER");
    }
    if (config.http_exporter) {
        char http_exporter_endpoint[] = "http://0.0.0.0:<xxxx>";

        snprintf(http_exporter_endpoint, sizeof http_exporter_endpoint,
                 "http://0.0.0.0:%hd", config.http_port);
        setenv("PROMETHEUS_ENDPOINT", http_exporter_endpoint, 1);
        VLOG_INFO("Starting DOCA Telemetry HTTP export on port %hd", config.http_port);
    } else {
        unsetenv("PROMETHEUS_ENDPOINT");
    }

    gctx.provider = (clx_api_provider_t) {
        .name = CONST_CAST(char *, metrics_root_get_name()),
        .version = (clx_api_version_t) {{1, 0, 0}},
        .description = "OVS-DOCA telemetry exporter",
        .initialize = doca_telemetry_provider_initialize,
    };

    gctx.clx_ctx = clx_api_create_context(&params, &gctx.provider);
    if (!gctx.clx_ctx) {
        VLOG_ERR("Failed to create Collectx metrics context");
        goto err;
    }

    gctx.clx_metrics_ctx = clx_api_metrics_create_context(gctx.clx_ctx);
    if (!gctx.clx_metrics_ctx) {
        VLOG_ERR("Failed to create Collectx metrics context");
        goto err;
    }

    gctx.initialized = true;

    return 0;

err:
    doca_telemetry_cleanup();
    gctx.init_error = true;
    return -1;
}

void
doca_telemetry_finalize(void)
{
    doca_telemetry_cleanup();
}

#else /* ! DOCA_TELEMETRY */

int
doca_telemetry_initialize(const struct smap *ovs_other_config)
{
    struct doca_telemetry_config config = default_config;

    doca_telemetry_dynamic_config(ovs_other_config, &config, false);
    if (memcmp(&config, &default_config, sizeof config)) {
        VLOG_WARN("DOCA Telemetry is disabled in this build of OVS.");
    }

    return 0;
}

void
doca_telemetry_run(void)
{
}

void
doca_telemetry_wait(void)
{
}

void
doca_telemetry_finalize(void)
{
}

#endif
