/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2024-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#include <config.h>

#include <doca_bitfield.h>
#include <doca_flow.h>
#include <rte_flow.h>
#include <sys/types.h>

#include "doca-action-split.h"
#include "dpdk-offload-doca.h"
#include "dpdk-offload-provider.h"
#include "id-fpool.h"
#include "netdev-offload-doca.h"
#include "netdev-vport.h"
#include "util.h"

#define SPLIT_ID_MIN 1
#define SPLIT_ID_NUM (POST_ACT_SPLIT_TABLE_ID_MAX - POST_ACT_SPLIT_TABLE_ID)

#define MAX_PRE_SPLIT_RTE_FLOW_ACTIONS 128

/*
 *  Rule action split works by recursively splitting the original ordered action
 *  list on the first action that diverges doca offload action order, and then
 *  using a continuation rule on a special per split POST_ACT_SPLIT logical table
 *  (reuse the same group_ctx with match on split-id) to continue and execute the
 *  rest of the actions.
 *
 *  Split example of actions A1,A2,A3:
 *
 *                  +----------- post-action group ------------------+
 *                  |   A1                         A2                |
 *                  | +-->--+                   +-->--+              |
 *  Original pipe   | |     |                   |     |              |
 *  +------------+  | |  +--v-------------+     |  +--v------------+ |
 *  | Original   |  | |  | match split-id |-->--+  | match split-id|--> A3
 *  | Match  ---->----+  +----------------+        +---------------+ |
 *  |            |  |    post split 1             post split 2       |
 *  +------------+  +------------------------------------------------+
 *
 */

static struct id_fpool *split_id_pool;

struct doca_action_split {
    struct dpdk_offload_handle doh;
    uint32_t split_id;
};

/* This is the order of execution of entry's actions dictated by doca-flow. */
enum doca_offload_action_order {
    DOCA_OFFLOAD_ACTION_ORDER_START,
    DOCA_OFFLOAD_ACTION_ORDER_DECAP,
    DOCA_OFFLOAD_ACTION_ORDER_POP,
    DOCA_OFFLOAD_ACTION_ORDER_PKT_MODIFY, /* META/OUTER/TUN */
    DOCA_OFFLOAD_ACTION_ORDER_PUSH,
    DOCA_OFFLOAD_ACTION_ORDER_ENCAP,
    DOCA_OFFLOAD_ACTION_ORDER_MIRROR,
    DOCA_OFFLOAD_ACTION_ORDER_METER,
    DOCA_OFFLOAD_ACTION_ORDER_FWD,
};

static int
doca_action_split_get_split_index(struct rte_flow_action *actions, int *split_index)
{
    enum doca_offload_action_order pos = DOCA_OFFLOAD_ACTION_ORDER_START;
    bool has_decap = false;
    int actions_num = 0;

    *split_index = 0;

    for (; actions->type != RTE_FLOW_ACTION_TYPE_END; actions++) {
        enum doca_offload_action_order p;
        int act_type = actions->type;

        actions_num++;
        if (*split_index) {
            continue;
        }

        if (act_type == RTE_FLOW_ACTION_TYPE_OF_SET_VLAN_VID) {
            if (pos == DOCA_OFFLOAD_ACTION_ORDER_PUSH) {
                /* Set vlan id action is combined into the push action */
                continue;
            }
            p = DOCA_OFFLOAD_ACTION_ORDER_PKT_MODIFY;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_OF_PUSH_VLAN) {
            p = DOCA_OFFLOAD_ACTION_ORDER_PUSH;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_TAG ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(FLOW_INFO) ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(HASH) ||
                   act_type == RTE_FLOW_ACTION_TYPE_COUNT) {
            /* Assuming COUNT happens before METER (if used), and METER drops under rate,
             * this allows count and meta data modification at any time,
             * as the order of operations doesn't matter.
             */
            continue;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_SET_MAC_SRC ||
                   act_type == RTE_FLOW_ACTION_TYPE_SET_MAC_DST ||
                   act_type == RTE_FLOW_ACTION_TYPE_SET_IPV4_SRC ||
                   act_type == RTE_FLOW_ACTION_TYPE_SET_IPV4_DST ||
                   act_type == RTE_FLOW_ACTION_TYPE_SET_IPV4_DSCP ||
                   act_type == RTE_FLOW_ACTION_TYPE_SET_IPV6_SRC ||
                   act_type == RTE_FLOW_ACTION_TYPE_SET_IPV6_DST ||
                   act_type == RTE_FLOW_ACTION_TYPE_SET_IPV6_DSCP ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_IPV4_TTL) ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_IPV6_HOP) ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_UDP_SRC) ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_UDP_DST) ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_TCP_SRC) ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(SET_TCP_DST)) {
            p = DOCA_OFFLOAD_ACTION_ORDER_PKT_MODIFY;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_NVGRE_DECAP ||
                   act_type == RTE_FLOW_ACTION_TYPE_VXLAN_DECAP ||
                   act_type == RTE_FLOW_ACTION_TYPE_RAW_DECAP) {
            p = DOCA_OFFLOAD_ACTION_ORDER_DECAP;
            has_decap = true;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_RAW_ENCAP) {
            p = DOCA_OFFLOAD_ACTION_ORDER_ENCAP;
            if (has_decap) {
                *split_index = actions_num - 1;
                continue;
            }
        } else if (act_type == RTE_FLOW_ACTION_TYPE_OF_POP_VLAN) {
            p = DOCA_OFFLOAD_ACTION_ORDER_POP;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_METER) {
            /* METER order is after push actions for example, that change the packet's size,
             * thus the count stats of the meter.
             * However, forcing this causes split in other cases in which it is not necessary,
             * for example meter/rewrite. This split causes performance degradation.
             * The stats issue is waived in order not to defect performance.
             */
            continue;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_OF_SET_VLAN_PCP ||
                   act_type == RTE_FLOW_ACTION_TYPE_SAMPLE ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(USERSPACE) ||
                   act_type == OVS_RTE_FLOW_ACTION_TYPE(SFLOW) ||
                   act_type == RTE_FLOW_ACTION_TYPE_VOID) {
            /* These actions don't affect the flow actions */
            continue;
        } else if (act_type == RTE_FLOW_ACTION_TYPE_REPRESENTED_PORT ||
                   act_type == RTE_FLOW_ACTION_TYPE_DROP ||
                   act_type == RTE_FLOW_ACTION_TYPE_JUMP) {
            p = DOCA_OFFLOAD_ACTION_ORDER_FWD;
        } else if (act_type == OVS_RTE_FLOW_ACTION_TYPE(NESTED)) {
            p = DOCA_OFFLOAD_ACTION_ORDER_MIRROR;
        } else {
            return -1;
        }

        /* Can have multiple mirror targets, packet modifications, and meters */
        if (p < pos || (p == pos && p != DOCA_OFFLOAD_ACTION_ORDER_PKT_MODIFY &&
                                    p != DOCA_OFFLOAD_ACTION_ORDER_MIRROR &&
                                    p != DOCA_OFFLOAD_ACTION_ORDER_METER)) {
            *split_index = actions_num - 1;
        } else {
            pos = p;
        }
    }

    /* Need room to insert split-id/jump/end actions. */
    if (*split_index > MAX_PRE_SPLIT_RTE_FLOW_ACTIONS - 3) {
        return -1;
    }
    return 0;
}

int
doca_action_split_create(doca_offload_create_cb offload_cb,
                         doca_offload_destroy_cb destroy_cb,
                         struct netdev *netdev,
                         const struct rte_flow_attr *attr,
                         struct rte_flow_item *items,
                         struct rte_flow_action *actions,
                         struct dpdk_offload_handle *doh,
                         struct rte_flow_error *error)
{
    struct rte_flow_action pre_split_acts[MAX_PRE_SPLIT_RTE_FLOW_ACTIONS];
    static struct ovsthread_once init_once = OVSTHREAD_ONCE_INITIALIZER;
    struct rte_flow_action_set_tag pre_act_set_tag;
    unsigned int tid = netdev_offload_thread_id();
    struct rte_flow_item_tag post_act_tag_spec;
    struct rte_flow_item_tag post_act_tag_mask;
    struct doca_action_split *action_split;
    struct rte_flow_item post_items[2];
    struct flow_patterns flow_patterns;
    struct flow_actions flow_actions;
    struct rte_flow_action_jump jump;
    struct rte_flow_attr split_attr;
    struct reg_field *reg_field;
    int prefix_acts, err;

    ovs_assert(doh->dfh.flow_res.action_split == NULL);
    reg_field = netdev_offload_doca_get_reg_field(REG_FIELD_RECIRC);

    if (doca_action_split_get_split_index(actions, &prefix_acts)) {
        error->type = RTE_FLOW_ERROR_TYPE_ACTION;
        error->message = "Failed splitting actions";
        return -1;
    }

    if (!prefix_acts) {
        return offload_cb(netdev, attr, items, actions, doh, error);
    }

    action_split = xzalloc(sizeof *action_split);

    if (ovsthread_once_start(&init_once)) {
        split_id_pool = id_fpool_create(MAX_OFFLOAD_THREAD_NB, SPLIT_ID_MIN, SPLIT_ID_NUM);
        ovsthread_once_done(&init_once);
    }

    if (!split_id_pool || !id_fpool_new_id(split_id_pool, tid,
                                           &action_split->split_id)) {
        err = -1;
        error->type = RTE_FLOW_ERROR_TYPE_UNSPECIFIED;
        error->message = "Failed allocating split id";
        goto err_new_id;
    }

    /* Post split - Match split-id entry in split group. */

    split_attr = *attr;
    split_attr.group = POST_ACT_SPLIT_TABLE_ID;

    post_act_tag_spec.index = reg_field->index;
    post_act_tag_spec.data = (action_split->split_id & reg_field->mask) << reg_field->offset;
    post_act_tag_mask.index = reg_field->index;
    post_act_tag_mask.data = reg_field->mask << reg_field->offset;
    post_items[0].type = RTE_FLOW_ITEM_TYPE_TAG;
    post_items[0].spec = &post_act_tag_spec;
    post_items[0].mask = &post_act_tag_mask;
    post_items[1].type = RTE_FLOW_ITEM_TYPE_END;

    err = doca_action_split_create(offload_cb, destroy_cb, netdev, &split_attr,
                                   &post_items[0], &actions[prefix_acts],
                                   &action_split->doh, error);
    netdev_offload_doca_items2flow_patterns(&post_items[0], &flow_patterns);
    netdev_offload_doca_actions2flow_actions(&actions[prefix_acts], &flow_actions);
    netdev_offload_doca_dump_flow(netdev, &split_attr, &flow_patterns, &flow_actions,
                                  &action_split->doh, false, err ? error : NULL);
    if (err) {
        goto err_create;
    }

    /* Pre split - Orig match with added jump to split group */
    memcpy(pre_split_acts, actions, prefix_acts * sizeof pre_split_acts[0]);

    pre_act_set_tag.index = reg_field->index;
    pre_act_set_tag.data = (action_split->split_id & reg_field->mask) << reg_field->offset;
    pre_act_set_tag.mask = reg_field->mask << reg_field->offset;
    pre_split_acts[prefix_acts].type = RTE_FLOW_ACTION_TYPE_SET_TAG;
    pre_split_acts[prefix_acts].conf = &pre_act_set_tag;
    prefix_acts++;

    jump.group = POST_ACT_SPLIT_TABLE_ID;
    pre_split_acts[prefix_acts].type = RTE_FLOW_ACTION_TYPE_JUMP;
    pre_split_acts[prefix_acts].conf = &jump;
    prefix_acts++;

    pre_split_acts[prefix_acts].type = RTE_FLOW_ACTION_TYPE_END;

    err = offload_cb(netdev, attr, items, pre_split_acts, doh, error);
    netdev_offload_doca_items2flow_patterns(items, &flow_patterns);
    netdev_offload_doca_actions2flow_actions(pre_split_acts, &flow_actions);
    netdev_offload_doca_dump_flow(netdev, attr, &flow_patterns, &flow_actions, doh, false,
                                  err ? error : NULL);
    if (err) {
        goto err_offload_pre;
    }
    doh->dfh.flow_res.action_split = action_split;

    return 0;

err_offload_pre:
    destroy_cb(netdev, &action_split->doh, error);
err_create:
    id_fpool_free_id(split_id_pool, tid, action_split->split_id);
err_new_id:
    free(action_split);
    return err;
}


int
doca_action_split_destroy(doca_offload_destroy_cb destroy_cb,
                          struct netdev *netdev,
                          struct dpdk_offload_handle *doh,
                          struct rte_flow_error *error)
{
    unsigned int tid = netdev_offload_thread_id();
    int err;

    err = destroy_cb(netdev, doh, error);
    if (err || !doh->dfh.flow_res.action_split) {
        return err;
    }

    err = doca_action_split_destroy(destroy_cb, netdev, &doh->dfh.flow_res.action_split->doh,
                                    error);
    id_fpool_free_id(split_id_pool, tid, doh->dfh.flow_res.action_split->split_id);
    free(doh->dfh.flow_res.action_split);
    doh->dfh.flow_res.action_split = NULL;
    return err;
}
