// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: BSD-3-Clause

#ifndef EVENT_LOG_H
#define EVENT_LOG_H

#include <stdarg.h>
#include <stdio.h>
#include <stdint.h>
#include <unistd.h>

#include "include/vfio_mlx5.h" /* for log levels */

static inline const char *log_level_str(unsigned int level)
{
	switch (level) {
	case MLX5_LOG_LVL_TRACE:
		return "TRACE";
	case MLX5_LOG_LVL_DEBUG:
		return "DEBUG";
	case MLX5_LOG_LVL_INFO:
		return "INFO";
	case MLX5_LOG_LVL_WARN:
		return "WARN";
	case MLX5_LOG_LVL_ERR:
		return "ERROR";
	case MLX5_LOG_LVL_CRIT:
		return "CRITICAL";
	default:
		return "UNKNOWN";
	}
}
/* Event Logger Structure */
typedef struct event_logger {
	char prefix[128];
	unsigned int level;
	unsigned int events_count;
	const char **event_names;
	uint64_t *event_stats;
	FILE *outf;
	FILE *errf;
} event_log_t;


static inline void event_log_level_set(event_log_t *logger, unsigned int level)
{
	logger->level = level;
}

static inline void event_log_prefix_set(event_log_t *logger, const char *fmt, ...)
{
	va_list args;

	va_start(args, fmt);
	vsnprintf(logger->prefix, sizeof(logger->prefix), fmt, args);
	va_end(args);
}

static inline void event_log_set_stats(event_log_t *logger,
			 uint64_t *event_stats, const char **event_names,
			 unsigned int events_count)
{
	logger->event_names = event_names;
	logger->event_stats = event_stats;
	logger->events_count = events_count;
}

static inline void event_log_init(event_log_t *logger, const char *name,
				  FILE *outf, FILE *errf)
{
	event_log_prefix_set(logger, "%s", name);
	logger->level = MLX5_LOG_LVL_INFO;

	// 0 is preserved for stdin so treat it as invalid
	logger->outf = outf ? outf : stdout;
	logger->errf = errf ? errf : stderr;

	// events structures, 0 means no events
	logger->events_count = 0;
	logger->event_names = NULL;
	logger->event_stats = NULL;
	// use event_log_set_stats to setup for event counting
}


static inline void event_log(event_log_t *logger, unsigned int level,
			     const char *fmt, ...)
{
	if (level > logger->level)
		return;

	FILE *printf = logger->outf;

	if (level <= MLX5_LOG_LVL_ERR)
		printf = logger->errf;

	va_list args;

	va_start(args, fmt);
	fprintf(printf, "[%-7s] %-20s: ", log_level_str(level),
	logger->prefix);
	vfprintf(printf, fmt, args);
	va_end(args);
}

#define log_event(logger, level, event, fmt, ...) \
	do { \
		const char *event_name = "n/a"; \
		unsigned int count = 0; \
		if (event < (logger)->events_count)  { \
			count = (logger)->event_stats[event]++; \
			event_name = (logger)->event_names[event]; \
		} \
		event_log((logger), level, "%-20s(%d):" fmt, \
				 event_name, count, ##__VA_ARGS__); \
	} while (0)

#endif // EVENT_LOG_H
