// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: BSD-3-Clause

#ifndef PAGEALLOC_H
#define PAGEALLOC_H

#include <stddef.h>
#include <stdint.h>
#include <linux/mman.h>
#include "vfio_mlx5.h"
#include "util/bitmap.h"
#include "abi.h"

struct page_allocator {
	struct {
		size_t base_iova; /* base iova for the page allocator */
		unsigned long num_pages;
		/* run-time stats */
		mlx5_pg_alloc_stats_t counters;
		/* add new fields here, struct is padded to 128 bytes
		 * MLX5_MINOR_VERSION >= 2
		 * type new_field_for_minor_version_2;
		 */
	} __packed_aligned(128); /* 128 bytes aligned for future use */
	/* Don't add fields here, use the data struct above */
	BMP_DECLARE(page_bitmap, 0); /* bitmap starts at the bottom */
};

/* ABI break checks */
_ABI_offset_assert(1.0, struct page_allocator, counters, 16);
_ABI_offset_assert(1.0, struct page_allocator, page_bitmap, 128);

void pagealloc_spinlock_init(void);

int mlx5_vfio_page_alloc(struct page_allocator *pg_allocs, uint64_t *iova);
void mlx5_vfio_page_free(struct page_allocator *pg_allocs, uint64_t iova);

int mlx5_vfio_page_alloc_init(struct page_allocator *pg_alloc,
			      size_t page_alloc_size, uint64_t iova);

void mlx5_vfio_page_alloc_stats(const struct page_allocator *pg_alloc,
				mlx5_pg_alloc_stats_t *stats);

int mlx5_alloc_contig_pages(struct page_allocator *page_alloc, uint64_t *iova,
			    size_t npages);
int mlx5_free_contig_pages(struct page_allocator *page_alloc, uint64_t iova,
			   size_t npages);

#endif /* PAGEALLOC_H */
