// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: BSD-3-Clause

#include <stdbool.h>
#include <stdint.h>
#include <errno.h>
#include "pagealloc.h"
#include "mlx5_memlay.h"
#include "util/util.h"

#include <pthread.h>

static pthread_spinlock_t pagealloc_lock;
static bool pagealloc_lock_initialized;

void pagealloc_spinlock_init(void)
{
	if (!pagealloc_lock_initialized) {
		pthread_spin_init(&pagealloc_lock, PTHREAD_PROCESS_PRIVATE);
		pagealloc_lock_initialized = true;
	}
}

int mlx5_vfio_page_alloc(struct page_allocator *pg_alloc, uint64_t *iova)
{
	unsigned long pg;
	int ret = 0;

	pthread_spin_lock(&pagealloc_lock);

	pg = bitmap_find_first_bit(pg_alloc->page_bitmap, 0,
				   pg_alloc->num_pages);
	if (pg == pg_alloc->num_pages) {
		pg_alloc->counters.allocs_failed++;
		ret = -ENOMEM;
		goto unlock;
	}

	pg_alloc->counters.allocs++;
	bitmap_clear_bit(pg_alloc->page_bitmap, pg);
	*iova = pg_alloc->base_iova + (pg * MLX5_ADAPTER_PAGE_SIZE);

unlock:
	pthread_spin_unlock(&pagealloc_lock);
	return ret;
}

void mlx5_vfio_page_free(struct page_allocator *pg_alloc, uint64_t iova)
{
	unsigned long pg;

	pthread_spin_lock(&pagealloc_lock);

	pg = (iova - pg_alloc->base_iova) / MLX5_ADAPTER_PAGE_SIZE;
	if (bitmap_test_bit(pg_alloc->page_bitmap, pg)) {
		log_warn("Double free page iova: %lx", iova);
		pg_alloc->counters.double_frees++;
		goto unlock;
	}
	pg_alloc->counters.frees++;
	bitmap_set_bit(pg_alloc->page_bitmap, pg);

unlock:
	pthread_spin_unlock(&pagealloc_lock);
}

void mlx5_vfio_page_alloc_stats(const struct page_allocator *pg_alloc,
				mlx5_pg_alloc_stats_t *stats)
{
	pthread_spin_lock(&pagealloc_lock);
	*stats = pg_alloc->counters;
	stats->free_pages =
		bitmap_weight(pg_alloc->page_bitmap, pg_alloc->num_pages);
	pthread_spin_unlock(&pagealloc_lock);
}

int mlx5_alloc_contig_pages(struct page_allocator *page_alloc, uint64_t *iova,
			    size_t npages)
{
	unsigned long bit;
	int ret = 0;

	pthread_spin_lock(&pagealloc_lock);
	bit = bitmap_find_free_region(page_alloc->page_bitmap,
				      page_alloc->num_pages, npages);
	if (bit == page_alloc->num_pages) {
		ret = -ENOMEM;
		page_alloc->counters.allocs_failed++;
		goto unlock;
	}

	bitmap_zero_region(page_alloc->page_bitmap, bit, bit + npages);

	*iova = page_alloc->base_iova + (bit * MLX5_ADAPTER_PAGE_SIZE);
	page_alloc->counters.allocs += npages;

unlock:
	pthread_spin_unlock(&pagealloc_lock);
	return ret;
}

int mlx5_free_contig_pages(struct page_allocator *page_alloc, uint64_t iova,
			   size_t npages)
{
	unsigned long bit;

	pthread_spin_lock(&pagealloc_lock);
	bit = (iova - page_alloc->base_iova) / MLX5_ADAPTER_PAGE_SIZE;
	/* TODO: we don't check double frees here */
	bitmap_fill_region(page_alloc->page_bitmap, bit, bit + npages);
	page_alloc->counters.frees += npages;
	pthread_spin_unlock(&pagealloc_lock);
	return 0;
}

#define MIN_USABLE_PAGES 1

/* Initialize the page allocator
 * @pg_alloc: pointer variable to struct page_allocator
 * @pg_alloc_size: available size starting from pg_alloc ptr
 * @pg_alloc_iova: iova addr of pg_alloc ptr
 */
int mlx5_vfio_page_alloc_init(struct page_allocator *pg_alloc,
			      size_t pg_alloc_size, uint64_t pg_alloc_iova)
{
	/* Align the struct size up to 4K (PAGE_SIZE) */
	size_t min_size = MIN_USABLE_PAGES * MLX5_ADAPTER_PAGE_SIZE;
	size_t total_pages = pg_alloc_size / MLX5_ADAPTER_PAGE_SIZE;

	/* Calculate the bitmap size */
	size_t bmp_size = bitmap_size(total_pages);

	min_size += bmp_size + sizeof(struct page_allocator);
	min_size = ALIGN_UP(min_size, MLX5_ADAPTER_PAGE_SIZE);

	if (pg_alloc_size < min_size) {
		log_error("Page allocator size is too small: %zu bytes, "
			  "minimum required: %zu bytes",
			  pg_alloc_size, min_size);
		return -EINVAL;
	}

	/* Calculate the end offset of the bitmap within the structure */
	uint64_t bmp_end_offset = sizeof(struct page_allocator) + bmp_size;

	/* Calculate the IOVA address after the bitmap */
	uint64_t bmp_end_iova = pg_alloc_iova + bmp_end_offset;

	/* Align the IOVA address to the next 4K boundary */
	uint64_t bmp_end_iova_4k =
		ALIGN_UP(bmp_end_iova, MLX5_ADAPTER_PAGE_SIZE);

	/* Calculate the total metadata size (including alignment overhead) */
	uint64_t meta_data_size = bmp_end_iova_4k - pg_alloc_iova;
	size_t pages_for_meta =
		DIV_ROUND_UP(meta_data_size, MLX5_ADAPTER_PAGE_SIZE);

	/* Calculate the number of usable pages */
	size_t usable_pages = total_pages - pages_for_meta;

	/* Initialize the page allocator structure */
	pg_alloc->base_iova = bmp_end_iova_4k;
	pg_alloc->num_pages = usable_pages;
	pg_alloc->counters.total_pages = pg_alloc->num_pages;
	memset(pg_alloc->page_bitmap, 0, bmp_size);

	log_debug("Page allocator initialized: alloc_iova=0x%lx, "
		  "num_pages=%lu, bitmap_size=%zu",
		  pg_alloc->base_iova, pg_alloc->num_pages, bmp_size);

	pagealloc_spinlock_init();

	/* Fill the bitmap for usable pages */
	bitmap_fill(pg_alloc->page_bitmap, usable_pages);

	return 0;
}
