// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: BSD-3-Clause

#ifndef MLX5_MEM_LAY_H
#define MLX5_MEM_LAY_H

#include <asm-generic/errno-base.h>
#include <stdatomic.h>
#include <stddef.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <limits.h>
#include <stdbool.h>
#include <stdlib.h>

#include "vfio_mlx5.h"
#include "dev.h"
#include "abi.h"

struct __packed_aligned4k vfio_mlx5_hdr {
	uint32_t magic_value;
	struct {
		uint16_t major;
		uint16_t minor;
	} version;
	uint64_t base_iova;
	size_t total_mem_size;
	uint16_t max_devices;
	uint16_t dev_cnt;

	/* ABI 1.1 */
	uint8_t state;
	uint32_t resume_token; /* protect against API misuse */
};

/* VFIO MLX5 handle memory layout
 * +--------------------------+ <--- vmh->hdr.base_iova
 * |    vfio_mlx5_hdr         | (Header, 4KB aligned, size = 4KB)
 * +--------------------------+ <---- Offset 4KB
 * |    vfio_mlx5_dev[0]      |
 * |    vfio_mlx5_dev[1]      |
 * |    ...                   |
 * |    vfio_mlx5_dev[7]      | Fixed size: sizeof(vfio_mlx5_dev[MAX_DEVICES])
 * +--------------------------+ <---- 4k aligned (as vfio_mlx5_dev 4k aligned)
 * |    page_allocator        | shared page allocator
 * +--------------------------+ <---- Fixed offset ~500 KB for meta data
 * |    pg_alloc->bitmap      | (Bitmap directly after page_allocator)
 * |    padding               | (Padding to align page[0] to 4KB)
 * +--------------------------+ <--- pg_alloc->base_iova
 * |    page[0]               | addr translation for pages by index:
 * |    page[1]               | pg_iova = pg_alloc->base_iova + 4K * pg_idx
 * |    ...                   | pg_vaddr = vmh + (pg_iova - vmh->hdr.base_iova)
 * |    page[n]               |
 * +--------------------------+ <-- storage_len (as storage is 4k aligned)
 */

struct __packed_aligned4k vfio_mlx5_handle {
	struct vfio_mlx5_hdr hdr;
	struct vfio_mlx5_dev devices[VFIO_MLX5_MAX_DEVICES];
	struct page_allocator page_alloc;
};

/* ABI break checks */
/* Checks for backward compatibility, if must change, update the
 * major version number and the size/offsets below
 * Note: When major version changes, resume will fail on older versions
 */
_ABI_sz_assert(1.0, struct vfio_mlx5_hdr, 4096);
_ABI_offset_assert(1.0, struct vfio_mlx5_hdr, dev_cnt, 26);
_ABI_offset_assert(1.0, struct vfio_mlx5_handle, page_alloc, 36864);

_ABI_offset_assert(1.1, struct vfio_mlx5_hdr, resume_token, 29);

int init_mem_lay(void *storage, size_t storage_len, uint64_t iova,
		 unsigned int num_devices);

#endif /* MLX5_MEM_LAY_H */
