// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: BSD-3-Clause

#ifndef FWPAGES_H
#define FWPAGES_H

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>

#include "vfio_mlx5.h"
#include "util/event_log.h"
#include "abi.h"

#define fwp_event(dev, event, fmt, ...)                                      \
	log_event(&dev->heap->fwpage_reqs.elogger, MLX5_LOG_LVL_INFO, event, \
		  fmt, ##__VA_ARGS__)

#define fwp_event_err(dev, event, fmt, ...)                                 \
	log_event(&dev->heap->fwpage_reqs.elogger, MLX5_LOG_LVL_ERR, event, \
		  fmt, ##__VA_ARGS__)

struct grow_buf {
	size_t len;
	void *buf;
};

static inline void *grow_buf_get(struct grow_buf *buf, size_t *len)
{
	if (buf->len >= *len)
		goto out;

	void *new_buf = realloc(buf->buf, *len);

	if (!new_buf) {
		*len = buf->len;
		goto out;
	}

	buf->buf = new_buf;
	buf->len = *len;
out:
	memset(buf->buf, 0, *len);
	return buf->buf;
}

struct fwpage_req {
	uint16_t func_id;
	int32_t npages;
};

/* Most we've see is 2 pending EQEs, 8 should be more than enough! */
#define FWPAGE_REQ_FIFO_SIZE 8
#define FWPAGE_REQ_FIFO_MOD (FWPAGE_REQ_FIFO_SIZE + 1)

struct fwpage_fifo {
	struct fwpage_req reqs[FWPAGE_REQ_FIFO_MOD];
	uint32_t head;
	uint32_t tail;
	uint8_t reserved[32];
};
_ABI_sz_assert(1.1, struct fwpage_fifo, 112);
_ABI_offset_assert(1.1, struct fwpage_fifo, tail, 76);

/* Allocated dynamically no packing/alignment needed */
struct mlx5_fwpage_reqs {
	struct grow_buf in;
	struct grow_buf out;
	event_log_t elogger;
};

static inline int fwpage_fifo_empty(const struct fwpage_fifo *fifo)
{
	return fifo->head == fifo->tail;
}

static inline int fwpage_fifo_full(const struct fwpage_fifo *fifo)
{
	return (fifo->tail + 1) % FWPAGE_REQ_FIFO_MOD == fifo->head;
}

static inline int fwpage_fifo_push(struct fwpage_fifo *fifo,
				   const struct fwpage_req *req)
{
	if (fwpage_fifo_full(fifo))
		return -1;

	fifo->reqs[fifo->tail] = *req;
	fifo->tail = (fifo->tail + 1) % FWPAGE_REQ_FIFO_MOD;
	return 0;
}

static inline struct fwpage_req *fwpage_fifo_pop(struct fwpage_fifo *fifo)
{
	if (fwpage_fifo_empty(fifo))
		return NULL;

	struct fwpage_req *req = &fifo->reqs[fifo->head];

	fifo->head = (fifo->head + 1) % FWPAGE_REQ_FIFO_MOD;
	return req;
}

static inline size_t fwpage_fifo_len(const struct fwpage_fifo *fifo)
{
	return (fifo->tail - fifo->head + FWPAGE_REQ_FIFO_MOD) %
	       FWPAGE_REQ_FIFO_MOD;
}

int mlx5_vfio_satisfy_startup_pages(struct vfio_mlx5_dev *dev, int boot,
				    int *pages_given);
void mlx5_vfio_handle_page_req_event(struct vfio_mlx5_dev *dev,
				     uint16_t func_id, int npages);
int mlx5_vfio_page_request_cmd_comp(struct vfio_mlx5_dev *dev,
				    int completion_status);

int mlx5_vfio_fwpage_reqs_init(struct mlx5_fwpage_reqs *fwpage_reqs,
			       const char *bdf, mlx5_pg_events_t page_events);
void mlx5_vfio_fwpage_reqs_free(struct mlx5_fwpage_reqs *fwpage_reqs);

const char **mlx5_vfio_fwpages_event_names(size_t *count);

#endif /* FWPAGES_H */
