// SPDX-FileCopyrightText: Copyright (c) 2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
// SPDX-License-Identifier: BSD-3-Clause

#ifndef CMD_H
#define CMD_H

#include <stddef.h> /* for size_t */
#include <endian.h> /* for htobe32, be32toh, etc */
#include <stdint.h> /* for uint*_t types */
#include <stdbool.h> /* for bool */
#include <linux/types.h>
#include "ifc.h"
#include "abi.h"
#include "util/event_log.h"

#define __devx_nullp(typ) ((struct mlx5_ifc_##typ##_bits *)NULL)
#define __devx_st_sz_bits(typ) sizeof(struct mlx5_ifc_##typ##_bits)
#define __devx_bit_sz(typ, fld) sizeof(__devx_nullp(typ)->fld)
#define __devx_bit_off(typ, fld) offsetof(struct mlx5_ifc_##typ##_bits, fld)
#define __devx_dw_off(bit_off) ((bit_off) / 32)
#define __devx_64_off(bit_off) ((bit_off) / 64)
#define __devx_dw_bit_off(bit_sz, bit_off) (32 - (bit_sz) - ((bit_off) & 0x1f))
#define __devx_mask(bit_sz) ((uint32_t)((1ull << (bit_sz)) - 1))
#define __devx_dw_mask(bit_sz, bit_off) \
	(__devx_mask(bit_sz) << __devx_dw_bit_off(bit_sz, bit_off))

#define DEVX_FLD_SZ_BYTES(typ, fld) (__devx_bit_sz(typ, fld) / 8)
#define DEVX_ST_SZ_BYTES(typ) (sizeof(struct mlx5_ifc_##typ##_bits) / 8)
#define DEVX_ST_SZ_DW(typ) (sizeof(struct mlx5_ifc_##typ##_bits) / 32)
#define DEVX_ST_SZ_QW(typ) (sizeof(struct mlx5_ifc_##typ##_bits) / 64)
#define DEVX_UN_SZ_BYTES(typ) (sizeof(union mlx5_ifc_##typ##_bits) / 8)
#define DEVX_UN_SZ_DW(typ) (sizeof(union mlx5_ifc_##typ##_bits) / 32)
#define DEVX_BYTE_OFF(typ, fld) (__devx_bit_off(typ, fld) / 8)
#define DEVX_ADDR_OF(typ, p, fld) \
	((unsigned char *)(p) + DEVX_BYTE_OFF(typ, fld))

static inline void _devx_set(void *p, uint32_t value, size_t bit_off,
			     size_t bit_sz)
{
	__be32 *fld = (__be32 *)(p) + __devx_dw_off(bit_off);
	uint32_t dw_mask = __devx_dw_mask(bit_sz, bit_off);
	uint32_t mask = __devx_mask(bit_sz);

	*fld = htobe32((be32toh(*fld) & (~dw_mask)) |
		       ((value & mask) << __devx_dw_bit_off(bit_sz, bit_off)));
}

#define DEVX_SET(typ, p, fld, v) \
	_devx_set(p, v, __devx_bit_off(typ, fld), __devx_bit_sz(typ, fld))

static inline uint32_t _devx_get(const void *p, size_t bit_off, size_t bit_sz)
{
	return ((be32toh(*((const __be32 *)(p) + __devx_dw_off(bit_off))) >>
		 __devx_dw_bit_off(bit_sz, bit_off)) &
		__devx_mask(bit_sz));
}

#define DEVX_GET(typ, p, fld) \
	_devx_get(p, __devx_bit_off(typ, fld), __devx_bit_sz(typ, fld))

static inline void _devx_set64(void *p, uint64_t v, size_t bit_off)
{
	*((__be64 *)(p) + __devx_64_off(bit_off)) = htobe64(v);
}

#define DEVX_SET64(typ, p, fld, v) _devx_set64(p, v, __devx_bit_off(typ, fld))

static inline uint64_t _devx_get64(const void *p, size_t bit_off)
{
	return be64toh(*((const __be64 *)(p) + __devx_64_off(bit_off)));
}

#define DEVX_GET64(typ, p, fld) _devx_get64(p, __devx_bit_off(typ, fld))

#define DEVX_ARRAY_SET64(typ, p, fld, idx, v) DEVX_SET64(typ, p, fld[idx], v)

#ifdef __x86_64__
#define udma_to_device_barrier() ({ asm volatile("" ::: "memory"); })
#define udma_from_device_barrier() ({ asm volatile("lfence" ::: "memory"); })
#else
#define udma_to_device_barrier() ({ asm volatile("dmb oshst" ::: "memory"); })
#define udma_from_device_barrier() ({ asm volatile("dmb oshld" ::: "memory"); })
#endif

enum {
	CMD_SLOT_GENERIC = 0,
	CMD_SLOT_PAGE_REQ = 1,
	CMD_SLOT_MAX = 2,
};

enum {
	MLX5_CMD_DATA_BLOCK_SIZE = 512,
	MLX5_PCI_CMD_XPORT = 7,
};

struct __packed health_buffer {
	__be32 assert_var[5];
	__be32 rsvd0[3];
	__be32 assert_exit_ptr;
	__be32 assert_callra;
	__be32 rsvd1[1];
	__be32 time;
	__be32 fw_ver;
	__be32 hw_id;
	uint8_t rfr_severity;
	uint8_t rsvd2[3];
	uint8_t irisc_index;
	uint8_t synd;
	__be16 ext_synd;
};

struct __packed mlx5_bar {
	__be32 fw_rev;
	__be32 cmdif_rev_fw_sub;
	__be32 rsvd0[2];
	__be32 cmdq_addr_h;
	__be32 cmdq_addr_l_sz;
	__be32 cmd_dbell;
	__be32 rsvd1[120];
	__be32 initializing;
	struct health_buffer health;
	__be32 rsvd2[880];
	__be32 internal_timer_h;
	__be32 internal_timer_l;
	__be32 rsvd3[2];
	__be32 health_counter;
	__be32 rsvd4[1019];
	__be64 ieee1588_clk;
	__be32 ieee1588_clk_type;
	__be32 clr_intx;
};

struct __packed mlx5_cmd_layout {
	uint8_t type;
	uint8_t rsvd0[3];
	__be32 ilen;
	__be64 iptr;
	__be32 in[4];
	__be32 out[4];
	__be64 optr;
	__be32 olen;
	uint8_t token;
	uint8_t sig;
	uint8_t rsvd1;
	uint8_t status_own;
};

struct __packed mlx5_cmd_block {
	uint8_t data[MLX5_CMD_DATA_BLOCK_SIZE];
	uint8_t rsvd0[48];
	__be64 next; /* BE iova addr of next block */
	__be32 block_num;
	uint8_t rsvd1;
	uint8_t token;
	uint8_t ctrl_sig;
	uint8_t sig;
};

struct __packed mlx5_cmd_blocks {
	/* iova addr points to the first mlx5_cmd_block in the list */
	uint64_t frst_blk_iova __ptr_align;
	uint32_t num_blocks;
};

/* RESUME: Cmd IF and its slots can be ptrs and allocated dynamically on suspend/resume. */
struct __packed mlx5_vfio_cmd_slot {
	uint8_t slot;
	struct mlx5_cmd_blocks in __ptr_align;
	struct mlx5_cmd_blocks out __ptr_align;
	bool in_use;
	uint8_t reserved[80];
};

struct __packed mlx5_vfio_cmd {
	uint64_t iova;
	uint8_t log_sz;
	uint8_t log_stride;
	uint8_t rsvd0[32]; /* reserved for future use */
	struct mlx5_vfio_cmd_slot cmds[CMD_SLOT_MAX] __ptr_align;
};

/* ABI break checks */
_ABI_sz_assert(1.0, struct mlx5_cmd_blocks, 16);
_ABI_offset_assert(1.0, struct mlx5_cmd_blocks, num_blocks, 8);

_ABI_sz_assert(1.0, struct mlx5_vfio_cmd_slot, 128);
_ABI_offset_assert(1.0, struct mlx5_vfio_cmd_slot, in_use, 40);

_ABI_sz_assert(1.0, struct mlx5_vfio_cmd, 304);
_ABI_offset_assert(1.0, struct mlx5_vfio_cmd, cmds, 48);

static inline void mlx5_cmd_mbox_status(const void *out, uint8_t *status,
					uint32_t *syndrome)
{
	*status = DEVX_GET(mbox_out, out, status);
	*syndrome = DEVX_GET(mbox_out, out, syndrome);
}

struct vfio_mlx5_dev;

int mlx5_vfio_cmd_interface_init(struct vfio_mlx5_dev *dev);
void mlx5_vfio_cmd_interface_uninit(struct vfio_mlx5_dev *dev);
void mlx5_cmd_page_req_try_grow(struct vfio_mlx5_dev *dev, size_t *ilen,
				size_t *olen);

int mlx5_vfio_cmd_exec(struct vfio_mlx5_dev *dev, void *in, int ilen, void *out,
		       int olen);
int mlx5_vfio_cmd_page_req_post(struct vfio_mlx5_dev *dev, void *in,
				size_t ilen, size_t olen);
void mlx5_vfio_cmd_eqe_comp(struct vfio_mlx5_dev *dev, unsigned long vector);

/* For testing only */
int mlx5_vfio_cmd_post(struct vfio_mlx5_dev *dev, void *in, int ilen, int olen);
int mlx5_vfio_cmd_poll(struct vfio_mlx5_dev *dev, void *out, int olen);

size_t mlx5_vfio_cmd_in_size(struct vfio_mlx5_dev *dev, uint16_t slot);
size_t mlx5_vfio_cmd_out_size(struct vfio_mlx5_dev *dev, uint16_t slot);

void mlx5_vfio_cmd_copy_in(struct vfio_mlx5_dev *dev, void *to, size_t size,
			   uint16_t slot);
void mlx5_vfio_cmd_copy_out(struct vfio_mlx5_dev *dev, void *to, size_t size,
			    uint16_t slot);

#define cmd_event(dev, event, fmt, args...)                                 \
	log_event(&dev->heap->cmdlog, MLX5_LOG_LVL_INFO, CMD_EVENT_##event, \
		  fmt "\n", ##args)
#define cmd_event_dbg(dev, event, fmt, args...)                              \
	log_event(&dev->heap->cmdlog, MLX5_LOG_LVL_DEBUG, CMD_EVENT_##event, \
		  fmt "\n", ##args)
#define cmd_event_err(dev, event, fmt, args...)                            \
	log_event(&dev->heap->cmdlog, MLX5_LOG_LVL_ERR, CMD_EVENT_##event, \
		  fmt "\n", ##args)
#define cmd_event_wrn(dev, event, fmt, args...)                             \
	log_event(&dev->heap->cmdlog, MLX5_LOG_LVL_WARN, CMD_EVENT_##event, \
		  fmt "\n", ##args)
#define cmd_event_fatal(dev, event, fmt, args...)                            \
	log_event(&dev->heap->cmdlog, MLX5_LOG_LVL_FATAL, CMD_EVENT_##event, \
		  fmt "\n", ##args)

#endif
