/*
 * Copyright (c) 2022-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <config.h>

#include <inttypes.h>
#include "metrics.h"
#include "netdev.h"
#include "ofproto-private.h"
#include "ofproto-provider.h"
#include "openvswitch/dynamic-string.h"
#include "openvswitch/types.h"
#include "ovs-thread.h"
#include <string.h>
#include "svec.h"
#include "unixctl.h"
#include "util.h"

METRICS_SUBSYSTEM(ofproto);

static void
ovs_vswitchd_type_label(struct metrics_label *labels, size_t n OVS_UNUSED, void *it OVS_UNUSED)
{
    struct svec types = SVEC_EMPTY_INITIALIZER;
    static char type[10] = "";
    struct ofproto *ofproto;

    labels[0].value = type;
    HMAP_FOR_EACH (ofproto, hmap_node, &all_ofprotos) {
        svec_add(&types, ofproto->type);
        svec_sort_unique(&types);
    }

    if (svec_is_empty(&types)) {
        snprintf(type, sizeof type, "undefined");
    } else if (types.n == 1) {
        snprintf(type, sizeof type, "%s", types.names[0]);
    } else {
        snprintf(type, sizeof type, "mixed");
    }

    svec_destroy(&types);
}

METRICS_LABEL(ofproto, ovs_vswitchd_type_label, ovs_vswitchd_type_label, "type");
METRICS_ENTRIES(ovs_vswitchd_type_label, ovs_vswitchd_type, "", metrics_array_read_one,
    METRICS_GAUGE(type,
        "A metric with a constant value '1' labeled by OVS type: Kernel (=system), "
        "DPDK or DOCA."),
);

static void
do_foreach_ofproto(metrics_visitor_cb callback,
                   struct metrics_visitor *visitor,
                   struct metrics_node *node,
                   struct metrics_label *labels,
                   size_t n OVS_UNUSED)
{
    struct ofproto *ofproto;

    HMAP_FOR_EACH (ofproto, hmap_node, &all_ofprotos) {
        visitor->it = ofproto;
        if (labels[0].key) {
            labels[0].value = ofproto->name;
            labels[1].value = ofproto->type;
        }
        callback(visitor, node);
    }
}

METRICS_FOREACH(ofproto, foreach_ofproto,
                do_foreach_ofproto, "name", "type");

METRICS_FOREACH(ofproto, foreach_ofproto_nolabel,
                do_foreach_ofproto, NULL);

static void
bridge_n_read_value(double *values, void *it OVS_UNUSED)
{
    values[0] = hmap_count(&all_ofprotos);
}
METRICS_ENTRIES(ofproto, n_bridges, "bridge", bridge_n_read_value,
    METRICS_GAUGE(n_bridges,
        "Number of bridges present in the instance."),
);

enum {
    OF_BRIDGE_NAME,
    OF_BRIDGE_N_PORTS,
    OF_BRIDGE_N_FLOWS,
};

static void
bridge_read_value(double *values, void *it)
{
    struct ofproto *ofproto = it;
    struct oftable *table;
    unsigned int n_flows;

    n_flows = 0;
    OFPROTO_FOR_EACH_TABLE (table, ofproto) {
        n_flows += table->n_flows;
    }

    values[OF_BRIDGE_NAME] = 1.0;
    values[OF_BRIDGE_N_PORTS] = hmap_count(&ofproto->ports);
    values[OF_BRIDGE_N_FLOWS] = n_flows;
}

METRICS_ENTRIES(foreach_ofproto, bridge_entries, "bridge", bridge_read_value,
    [OF_BRIDGE_NAME] = METRICS_GAUGE(,
        "A metric with a constant value '1' labeled by bridge name and type "
        "present on the instance."),
    [OF_BRIDGE_N_PORTS] = METRICS_GAUGE(n_ports,
        "Number of ports present on the bridge."),
    [OF_BRIDGE_N_FLOWS] = METRICS_GAUGE(n_flows,
        "Number of flows present on the bridge."),
);

static void
do_foreach_ports(metrics_visitor_cb callback,
                 struct metrics_visitor *visitor,
                 struct metrics_node *node,
                 struct metrics_label *labels,
                 size_t n OVS_UNUSED)
{
    struct ofproto *ofproto = visitor->it;
    struct ofport *port;

    HMAP_FOR_EACH (port, hmap_node, &ofproto->ports) {
        labels[0].value = port->ofproto->name;
        labels[1].value = netdev_get_name(port->netdev);
        labels[2].value = netdev_get_type(port->netdev);
        labels[3].value = port->pp.name;
        visitor->it = port;
        callback(visitor, node);
    }
}

METRICS_FOREACH(foreach_ofproto_nolabel, foreach_ports,
                do_foreach_ports, "bridge", "name", "type", "port");

enum {
    OF_NETDEV_ADMIN_STATE,
    OF_NETDEV_MTU,
    OF_NETDEV_OF_PORT,
    OF_NETDEV_IFINDEX,
    OF_NETDEV_POLICY_BIT_RATE,
    OF_NETDEV_POLICY_BIT_BURST,
    OF_NETDEV_POLICY_PKT_RATE,
    OF_NETDEV_POLICY_PKT_BURST,
    OF_NETDEV_DUPLEXITY,
    OF_NETDEV_LINK_RESETS,
    OF_NETDEV_LINK_SPEED,
    OF_NETDEV_LINK_STATE,
    OF_NETDEV_STATS_RX_PACKETS,
    OF_NETDEV_STATS_TX_PACKETS,
    OF_NETDEV_STATS_RX_BYTES,
    OF_NETDEV_STATS_TX_BYTES,
    OF_NETDEV_STATS_RX_ERRORS,
    OF_NETDEV_STATS_TX_ERRORS,
    OF_NETDEV_STATS_RX_DROPPED,
    OF_NETDEV_STATS_TX_DROPPED,
    OF_NETDEV_STATS_RX_LENGTH_ERRORS,
    OF_NETDEV_STATS_RX_OVER_ERRORS,
    OF_NETDEV_STATS_RX_CRC_ERRORS,
    OF_NETDEV_STATS_RX_FRAME_ERRORS,
    OF_NETDEV_STATS_RX_FIFO_ERRORS,
    OF_NETDEV_STATS_RX_MISSED_ERRORS,
    OF_NETDEV_STATS_MULTICAST,
    OF_NETDEV_STATS_COLLISIONS,
};

static void
interface_read_value(double *values, void *it)
{
    const size_t n_stats = sizeof(struct netdev_stats) / sizeof(uint64_t);
    uint32_t kbits_rate, kbits_burst;
    uint32_t kpkts_rate, kpkts_burst;
    enum netdev_features current;
    struct netdev_stats stats;
    uint32_t link_speed_mbps;
    enum netdev_flags flags;
    struct netdev *netdev;
    struct ofport *port;
    uint64_t *u64_stats;
    bool full_duplex;
    int ifindex;
    size_t i;
    int mtu;

    port = it;
    netdev = port->netdev;

    ifindex = netdev_get_ifindex(netdev);
    if (netdev_get_flags(netdev, &flags)) {
        flags = 0;
    }

    netdev_get_stats(netdev, &stats);
    /* Overwrite unused / error stats with 0. */
    u64_stats = (void *) &stats;
    for (i = 0; i < n_stats; i++) {
        u64_stats[i] = MAX_IS_ZERO(u64_stats[i]);
    }

    netdev_get_policing(netdev,
                        &kbits_rate, &kbits_burst,
                        &kpkts_rate, &kpkts_burst);

    netdev_get_features(netdev, &current, NULL, NULL, NULL);
    full_duplex = netdev_features_is_full_duplex(current);

    if (netdev_get_speed(netdev, &link_speed_mbps, NULL)) {
        link_speed_mbps = 0;
    } else {
        link_speed_mbps /= 1000000;
    }

    values[OF_NETDEV_ADMIN_STATE] = (flags & NETDEV_UP) ? 1 : 0;
    values[OF_NETDEV_MTU] = netdev_get_mtu(netdev, &mtu) ? 0 : mtu;
    values[OF_NETDEV_OF_PORT] = (OVS_FORCE uint32_t) port->ofp_port;
    values[OF_NETDEV_IFINDEX] = (ifindex > 0) ? ifindex : 0;
    values[OF_NETDEV_POLICY_BIT_RATE] = kbits_rate;
    values[OF_NETDEV_POLICY_BIT_BURST] = kbits_burst;
    values[OF_NETDEV_POLICY_PKT_RATE] = kpkts_rate;
    values[OF_NETDEV_POLICY_PKT_BURST] = kpkts_burst;
    values[OF_NETDEV_DUPLEXITY] = full_duplex ? 1 : 0;
    values[OF_NETDEV_LINK_RESETS] = netdev_get_carrier_resets(netdev);
    values[OF_NETDEV_LINK_SPEED] = link_speed_mbps;
    values[OF_NETDEV_LINK_STATE] = !!netdev_get_carrier(netdev);

    values[OF_NETDEV_STATS_RX_PACKETS] = stats.rx_packets;
    values[OF_NETDEV_STATS_TX_PACKETS] = stats.tx_packets;
    values[OF_NETDEV_STATS_RX_BYTES] = stats.rx_bytes;
    values[OF_NETDEV_STATS_TX_BYTES] = stats.tx_bytes;
    values[OF_NETDEV_STATS_RX_ERRORS] = stats.rx_errors;
    values[OF_NETDEV_STATS_TX_ERRORS] = stats.tx_errors;
    values[OF_NETDEV_STATS_RX_DROPPED] = stats.rx_dropped;
    values[OF_NETDEV_STATS_TX_DROPPED] = stats.tx_dropped;
    values[OF_NETDEV_STATS_RX_LENGTH_ERRORS] = stats.rx_length_errors;
    values[OF_NETDEV_STATS_RX_OVER_ERRORS] = stats.rx_over_errors;
    values[OF_NETDEV_STATS_RX_CRC_ERRORS] = stats.rx_crc_errors;
    values[OF_NETDEV_STATS_RX_FRAME_ERRORS] = stats.rx_frame_errors;
    values[OF_NETDEV_STATS_RX_FIFO_ERRORS] = stats.rx_fifo_errors;
    values[OF_NETDEV_STATS_RX_MISSED_ERRORS] = stats.rx_missed_errors;
    values[OF_NETDEV_STATS_MULTICAST] = stats.multicast;
    values[OF_NETDEV_STATS_COLLISIONS] = stats.collisions;
}

METRICS_ENTRIES(foreach_ports, port_entries,
    "interface", interface_read_value,
    [OF_NETDEV_ADMIN_STATE] = METRICS_GAUGE(admin_state,
        "The administrative state of the interface: down(0) or up(1)."),
    [OF_NETDEV_MTU] = METRICS_GAUGE(mtu,
        "The MTU of the interface."),
    [OF_NETDEV_OF_PORT] = METRICS_GAUGE(of_port,
        "The OpenFlow port ID associated with the interface."),
    [OF_NETDEV_IFINDEX] = METRICS_GAUGE(ifindex,
        "The ifindex of the interface."),
    [OF_NETDEV_POLICY_BIT_RATE] = METRICS_GAUGE(ingress_policy_bit_rate,
        "Maximum receive rate in kbps on the interface. "
        "Disabled if set to 0."),
    [OF_NETDEV_POLICY_BIT_BURST] = METRICS_GAUGE(ingress_policy_bit_burst,
        "Maximum receive burst size in kb."),
    [OF_NETDEV_POLICY_PKT_RATE] = METRICS_GAUGE(ingress_policy_pkt_rate,
        "Maximum receive rate in pps on the interface. "
        "Disabled if set to 0."),
    [OF_NETDEV_POLICY_PKT_BURST] = METRICS_GAUGE(ingress_policy_pkt_burst,
        "Maximum receive burst size in number of packets."),
    [OF_NETDEV_DUPLEXITY] = METRICS_GAUGE(duplex,
        "The duplex mode of the interface: half(0) or full(1)."),
    [OF_NETDEV_LINK_RESETS] = METRICS_COUNTER(link_resets,
        "The number of time the interface link changed."),
    [OF_NETDEV_LINK_SPEED] = METRICS_GAUGE(link_speed,
        "The current speed of the interface link in Mbps."),
    [OF_NETDEV_LINK_STATE] = METRICS_GAUGE(link_state,
        "The state of the interface link: down(0) or up(1)."),
    [OF_NETDEV_STATS_RX_PACKETS] = METRICS_COUNTER(rx_packets,
        "The number of packets received."),
    [OF_NETDEV_STATS_TX_PACKETS] = METRICS_COUNTER(tx_packets,
        "The number of packets transmitted."),
    [OF_NETDEV_STATS_RX_BYTES] = METRICS_COUNTER(rx_bytes,
        "The number of bytes received."),
    [OF_NETDEV_STATS_TX_BYTES] = METRICS_COUNTER(tx_bytes,
        "The number of bytes transmitted."),
    [OF_NETDEV_STATS_RX_ERRORS] = METRICS_COUNTER(rx_errors,
        "Total number of bad packets received on this interface. "
        "This counter includes all rx_length_errors, rx_crc_errors, "
        "rx_frame_errors and other errors not otherwise counted."),
    [OF_NETDEV_STATS_TX_ERRORS] = METRICS_COUNTER(tx_errors,
        "Total number of transmit issues on this interface."),
    [OF_NETDEV_STATS_RX_DROPPED] = METRICS_COUNTER(rx_dropped,
        "Number of packets received but not processed, "
        "e.g. due to lack of resources or unsupported protocol. "
        "For hardware interface this counter should not include packets "
        "dropped by the device due to buffer exhaustion which are counted "
        "separately in rx_missed_errors."),
    [OF_NETDEV_STATS_TX_DROPPED] = METRICS_COUNTER(tx_dropped,
        "The number of packets dropped on their way to transmission, "
        "e.g. due to lack of resources."),
    [OF_NETDEV_STATS_RX_LENGTH_ERRORS] = METRICS_COUNTER(rx_length_errors,
        "The number of packets dropped due to invalid length."),
    [OF_NETDEV_STATS_RX_OVER_ERRORS] = METRICS_COUNTER(rx_over_errors,
        "Receiver FIFO overflow event counter. This statistics was "
        "used interchangeably with rx_fifo_errors. This statistics "
        "corresponds to hardware events and is not commonly used on "
        "software devices."),
    [OF_NETDEV_STATS_RX_CRC_ERRORS] = METRICS_COUNTER(rx_crc_errors,
        "The number of packets with CRC errors received by the interface."),
    [OF_NETDEV_STATS_RX_FRAME_ERRORS] = METRICS_COUNTER(rx_frame_errors,
        "The number of received packets with frame alignment errors on "
        "the interface."),
    [OF_NETDEV_STATS_RX_FIFO_ERRORS] = METRICS_COUNTER(rx_fifo_errors,
        "Receiver FIFO error counter. This statistics was used "
        "interchangeably with rx_over_errors but is not recommended for use "
        "in drivers for high speed interfaces. This statistics is used on "
        "software devices, e.g. to count software packets queue overflow or "
        "sequencing errors."),
    [OF_NETDEV_STATS_RX_MISSED_ERRORS] = METRICS_COUNTER(rx_missed_errors,
        "The number of packets missed by the host due to lack of buffer "
        "space. This usually indicates that the host interface is slower than "
        "the hardware interface. This statistics corresponds to hardware "
        "events and is not used on software devices."),
    [OF_NETDEV_STATS_MULTICAST] = METRICS_COUNTER(multicast,
        "The number of multicast packets received by the interface."),
    [OF_NETDEV_STATS_COLLISIONS] = METRICS_COUNTER(collisions,
        "The number of collisions during packet transmission."),
);

static void
netdev_info_label(struct metrics_label *labels, size_t n, void *it)
{
    static char values[3][100];
    struct smap netdev_status;
    struct netdev *netdev;
    struct ofport *port;
    size_t i;

    port = it;
    netdev = port->netdev;

    labels[0].value = values[0];
    labels[1].value = values[1];
    labels[2].value = values[2];

    /* By default, reset the labels values to an empty string. */
    for (i = 0; i < n; i++) {
        values[i][0] = '\0';
    }

    smap_init(&netdev_status);
    if (netdev_get_status(netdev, &netdev_status) == 0) {
        for (i = 0; i < n; i++) {
            const char *value = smap_get(&netdev_status, labels[i].key);

            if (value != NULL) {
                snprintf(values[i], sizeof values[i], "%s", value);
            }
        }
    }
    smap_destroy(&netdev_status);
}

METRICS_LABEL(foreach_ports,
    netdev_labeled_info, netdev_info_label,
    "driver_name", "driver_version", "firmware_version");

METRICS_ENTRIES(netdev_labeled_info, netdev_info,
    "interface", metrics_array_read_one,
    METRICS_GAUGE(info,
        "A metric with a constant value '1' labeled with the driver name, "
        "version and firmware version of the interface."),
);

struct flow_metric {
    struct ovs_list node;
    ovs_be64 cookie;
    char *name;
};

static struct flow_metric *
flow_metric_find(struct ofproto *ofproto, ovs_be64 cookie)
{
    struct flow_metric *rec;

    LIST_FOR_EACH (rec, node, &ofproto->flow_metrics) {
        if (rec->cookie == cookie) {
            return rec;
        }
    }
    return NULL;
}

void
ofproto_metrics_unregister_flow_cookie(struct ofproto *ofproto, ovs_be64 cookie)
{
    struct flow_metric *rec, *next;

    LIST_FOR_EACH_SAFE (rec, next, node, &ofproto->flow_metrics) {
        if (rec->cookie == cookie) {
            ovs_list_remove(&rec->node);
            free(rec->name);
            free(rec);
            return;
        }
    }
}

void
ofproto_metrics_clear_flow_metrics(struct ofproto *ofproto)
{
    struct flow_metric *rec, *next;

    LIST_FOR_EACH_SAFE (rec, next, node, &ofproto->flow_metrics) {
        ovs_list_remove(&rec->node);
        free(rec->name);
        free(rec);
    }
}

static void
ofproto_metrics_add_flow_metric(struct unixctl_conn *conn, int argc OVS_UNUSED,
                                const char *argv[], void *aux OVS_UNUSED)
{
    const char *bridge = argv[1];
    const char *name = argv[3];
    unsigned long long cookie;
    struct flow_metric *rec;
    struct ofproto *ofproto;
    char reply[512];

    if (!str_to_ullong(argv[2], 0, &cookie)) {
        unixctl_command_reply_error(conn, "invalid cookie");
        return;
    }

    if (!str_is_id(name)) {
        unixctl_command_reply_error(conn, "invalid metric name");
        return;
    }

    ofproto = ofproto_lookup(bridge);
    if (!ofproto) {
        unixctl_command_reply_error(conn, "unknown bridge");
        return;
    }

    /* If an entry for this (bridge,cookie) exists, update the name. */
    rec = flow_metric_find(ofproto, htonll(cookie));
    if (rec) {
        snprintf(reply, sizeof reply,
                 "Flow-metric %s:%s successfully renamed from '%s' to '%s'",
                 bridge, argv[2], rec->name, name);
        free(rec->name);
    } else {
        rec = xzalloc(sizeof *rec);
        rec->cookie = htonll(cookie);
        ovs_list_push_back(&ofproto->flow_metrics, &rec->node);
        snprintf(reply, sizeof reply,
                 "Flow-metric %s:%s named '%s' successfully added",
                 bridge, argv[2], name);
    }

    rec->name = xstrdup(name);
    unixctl_command_reply(conn, reply);
}

static void
ofproto_metrics_del_flow_metric(struct unixctl_conn *conn, int argc OVS_UNUSED,
                                const char *argv[], void *aux OVS_UNUSED)
{
    const char *bridge = argv[1];
    unsigned long long cookie;
    struct flow_metric *rec;
    struct ofproto *ofproto;
    char reply[256];

    if (!str_to_ullong(argv[2], 0, &cookie)) {
        unixctl_command_reply_error(conn, "invalid cookie");
        return;
    }

    ofproto = ofproto_lookup(bridge);
    if (!ofproto) {
        unixctl_command_reply_error(conn, "unknown bridge");
        return;
    }

    rec = flow_metric_find(ofproto, htonll(cookie));
    if (!rec) {
        unixctl_command_reply_error(conn, "flow counter not found");
        return;
    }

    snprintf(reply, sizeof reply,
             "Flow-metric %s:%s named '%s' successfully removed",
             bridge, argv[2], rec->name);
    ovs_list_remove(&rec->node);
    free(rec->name);
    free(rec);
    unixctl_command_reply(conn, reply);
}

struct flow_metric_ctx {
    struct ofproto *ofproto;
    ovs_be64 cookie;
    const char *name;
};

static void
do_foreach_flow_metric(metrics_visitor_cb callback,
                       struct metrics_visitor *visitor,
                       struct metrics_node *node,
                       struct metrics_label *labels,
                       size_t n OVS_UNUSED)
{
    struct ofproto *ofproto = visitor->it;
    struct flow_metric *rec;

    LIST_FOR_EACH (rec, node, &ofproto->flow_metrics) {
        struct flow_metric_ctx ctx = {
            .ofproto = ofproto,
            .cookie = rec->cookie,
            .name = rec->name,
        };
        labels[0].value = ofproto->name;
        labels[1].value = rec->name;
        visitor->it = &ctx;
        callback(visitor, node);
    }
}

METRICS_FOREACH(foreach_ofproto_nolabel, foreach_flow_metric,
                do_foreach_flow_metric, "bridge", "name");

enum {
    FLOW_METRIC_PACKETS,
    FLOW_METRIC_BYTES,
};

static void
flow_metric_read(double *values, void *it)
{
    const struct flow_metric_ctx *ctx = it;
    struct pkt_stats sum;

    memset(&sum, 0, sizeof(sum));
    ofproto_get_cookie_stats(ctx->ofproto, ctx->cookie, &sum);
    values[FLOW_METRIC_PACKETS] = sum.n_packets;
    values[FLOW_METRIC_BYTES] = sum.n_bytes;
}

METRICS_ENTRIES(foreach_flow_metric, flow_metric_entries, "flow",
                flow_metric_read,
    [FLOW_METRIC_PACKETS] = METRICS_COUNTER(packets,
        "Number of packets matched by this registered flow cookie."),
    [FLOW_METRIC_BYTES] = METRICS_COUNTER(bytes,
        "Number of bytes matched by this registered flow cookie.")
);

static void
ofproto_metrics_list_flow_metrics(struct unixctl_conn *conn, int argc OVS_UNUSED,
                                  const char *argv[] OVS_UNUSED, void *aux OVS_UNUSED)
{
    struct ds out = DS_EMPTY_INITIALIZER;
    size_t bridge_w = strlen("BRIDGE");
    size_t cookie_w = strlen("COOKIE");
    size_t name_w = strlen("NAME");
    struct flow_metric *rec;
    struct ofproto *ofproto;

    /* Compute column widths. */
    HMAP_FOR_EACH (ofproto, hmap_node, &all_ofprotos) {
        LIST_FOR_EACH (rec, node, &ofproto->flow_metrics) {
            /* Compute un-padded hex length via dry-run snprintf on 0x<hex>. */
            int len = snprintf(NULL, 0, "0x%"PRIx64, ntohll(rec->cookie));

            if (strlen(ofproto->name) > bridge_w) {
                bridge_w = strlen(ofproto->name);
            }

            if (len > 0 && (size_t) len > cookie_w) {
                cookie_w = (size_t) len;
            }

            if (strlen(rec->name) > name_w) {
                name_w = strlen(rec->name);
            }
        }
    }

    /* Header */
    ds_put_format(&out, "%-*s  %-*s  %-*s\n",
                  (int) bridge_w, "BRIDGE",
                  (int) cookie_w, "COOKIE",
                  (int) name_w,   "NAME");

    /* Separator */
    for (size_t i = 0; i < bridge_w; i++) {
        ds_put_char(&out, '-');
    }
    ds_put_cstr(&out, "  ");

    for (size_t i = 0; i < cookie_w; i++) {
        ds_put_char(&out, '-');
    }
    ds_put_cstr(&out, "  ");

    for (size_t i = 0; i < name_w; i++) {
        ds_put_char(&out, '-');
    }
    ds_put_char(&out, '\n');

    /* Print rows */
    HMAP_FOR_EACH (ofproto, hmap_node, &all_ofprotos) {
        LIST_FOR_EACH (rec, node, &ofproto->flow_metrics) {
            ds_put_format(&out, "%-*s  %-#*"PRIx64"  %-*s\n",
                          (int) bridge_w, ofproto->name,
                          (int) cookie_w, ntohll(rec->cookie),
                          (int) name_w,   rec->name);
        }
    }

    unixctl_command_reply(conn, ds_cstr(&out));
    ds_destroy(&out);
}


void
ofproto_metrics_register(void)
{
    static bool registered;
    if (registered) {
        return;
    }
    registered = true;

    METRICS_REGISTER(ovs_vswitchd_type);
    METRICS_REGISTER(n_bridges);
    METRICS_REGISTER(bridge_entries);
    METRICS_REGISTER(port_entries);
    METRICS_REGISTER(netdev_info);
    METRICS_REGISTER(flow_metric_entries);

    unixctl_command_register("ofproto/flow-metric-add",
                             "bridge cookie name", 3, 3,
                             ofproto_metrics_add_flow_metric, NULL);
    unixctl_command_register("ofproto/flow-metric-del",
                             "bridge cookie", 2, 2,
                             ofproto_metrics_del_flow_metric, NULL);
    unixctl_command_register("ofproto/flow-metric-list",
                             "", 0, 0,
                             ofproto_metrics_list_flow_metrics, NULL);
}
