/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2023-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#ifndef NETDEV_DOCA_EXT_H
#define NETDEV_DOCA_EXT_H

#include <config.h>
#include <dirent.h>
#include <errno.h>
#include <net/if.h>

#include "openvswitch/netdev.h"
#include "openvswitch/ofp-meter.h"
#include "smap.h"

struct doca_flow_port;
struct dp_packet;
struct netdev;
struct ovs_doca_esw_ctx;

enum netdev_doca_port_dir {
    NETDEV_DOCA_PORT_DIR_RX,
    NETDEV_DOCA_PORT_DIR_TX,
    NUM_NETDEV_DOCA_PORT_DIR,
};

#ifdef DPDK_NETDEV

#include <rte_flow.h>
#include <rte_ethdev.h>

#define NR_HAIRPINQ (ovs_doca_initialized() ? 0 : 1)

typedef uint16_t dpdk_port_t;

bool
netdev_doca_is_ethdev(struct netdev *netdev);

struct doca_flow_port *
netdev_doca_port_get(struct netdev *netdev);

struct ovs_doca_esw_ctx *
netdev_doca_ovs_doca_esw_ctx(struct netdev *netdev);

bool
netdev_doca_is_esw_mgr(const struct netdev *netdev);

struct rte_flow_action_handle *
netdev_doca_indirect_action_create(struct netdev *,
                                   int port_id,
                                   const struct rte_flow_action *,
                                   struct rte_flow_error *);
int
netdev_doca_indirect_action_destroy(int port_id,
                                    struct rte_flow_action_handle *,
                                    struct rte_flow_error *);
int
netdev_doca_indirect_action_query(int port_id,
                                  struct rte_flow_action_handle *,
                                  void *,
                                  struct rte_flow_error *);

int
netdev_doca_get_esw_mgr_port_id(const struct netdev *netdev);

int
get_sys(const char *prefix, const char *devname, const char *suffix,
        char *outp, size_t maxlen);

enum netdev_doca_port_dir
netdev_doca_get_port_dir(struct netdev *netdev);

int
netdev_doca_meter_set(ofproto_meter_id meter_id,
                      struct ofputil_meter_config *config);
int
netdev_doca_meter_get(ofproto_meter_id meter_id,
                      struct ofputil_meter_stats *stats, uint16_t n_bands);
int
netdev_doca_meter_get_cfg(uint32_t of_meter_id, void *meter_cfg_);

int
netdev_doca_meter_del(ofproto_meter_id meter_id,
                      struct ofputil_meter_stats *stats, uint16_t n_bands);
void
netdev_doca_meters_init(void);
bool
netdev_doca_meter_ref(uint32_t meter_id);
void
netdev_doca_meter_unref(uint32_t meter_id);

bool
netdev_doca_sw_meter_get_protected(struct netdev *netdev,
                                   struct ofputil_meter_config *config,
                                   struct ofputil_meter_band *band);

bool
netdev_doca_sw_meter_get(struct netdev *netdev,
                         struct ofputil_meter_config *config,
                         struct ofputil_meter_band *band);

bool
netdev_doca_sw_meter_get_dry_run(struct netdev *netdev);

struct netdev_doca_sw_meter {
    struct ofputil_meter_config config;
    struct ofputil_meter_band band;
    bool dry_run;
};

#define NETDEV_DPDK_METER_PORT_ID 0
#define NETDEV_DPDK_METER_POLICY_ID 25
#define NETDEV_DPDK_MAX_METERS (MAX_METERS + RTE_MAX_ETHPORTS)

int
netdev_doca_sw_meter_parse(const char *s,
                           struct netdev_doca_sw_meter *m,
                           const char **errp);

void
netdev_doca_sw_meter_copy(struct netdev_doca_sw_meter *dst,
                          const struct netdev_doca_sw_meter *src);

bool
netdev_doca_sw_meter_equal(const struct netdev_doca_sw_meter *m1,
                           const struct netdev_doca_sw_meter *m2);

void
netdev_doca_add_meter_policy(dpdk_port_t port_id, uint32_t policy_id);

void
netdev_doca_get_config_sw_meter(struct smap *args,
                                struct netdev_doca_sw_meter *requested_sw_meter,
                                struct netdev_doca_sw_meter *sw_meter);

int
netdev_doca_set_config_sw_meter(struct netdev *netdev, const struct smap *args,
                                struct netdev_doca_sw_meter *requested_sw_meter,
                                bool dev_started,
                                char **errp);

void
netdev_doca_set_config_port_dir(const char *devargs,
                                enum netdev_doca_port_dir *port_dir);

char *
netdev_doca_generate_devargs(const char *name, char *devargs, size_t maxlen,
                             char iface[IFNAMSIZ]);

int
netdev_doca_get_phys_port_name(const char *devname, char *outp, size_t maxlen);

int
netdev_doca_setup_hairpin_queues(struct netdev *netdev, dpdk_port_t port_id,
                                 int rxq_size, int txq_size,
                                 int n_rxq, int n_txq,
                                 struct rte_eth_hairpin_conf *hairpin_conf);

const char *
netdev_doca_adapt_doca_mlx5_devargs(const char *devargs, char *new_devargs, size_t maxlen);

void
netdev_doca_get_custom_stats_sw_meter(const struct netdev *netdev,
                                      struct netdev_custom_stats *custom_stats);

int
netdev_doca_get_rep_name(const char dev[], char *rep, size_t maxlen);

uint32_t
netdev_doca_get_datapath_priority(const char *netdev_name, const char *dpif_type);

int
get_rep_peer(const char name[], char *peer, size_t maxlen);

#else /* DPDK_NETDEV */

static inline enum netdev_doca_port_dir
netdev_doca_get_port_dir(struct netdev *netdev OVS_UNUSED)
{
    /* If no doca, the return value doesn't matter. */
    return NETDEV_DOCA_PORT_DIR_RX;
}

static inline bool
netdev_doca_is_ethdev(struct netdev *netdev OVS_UNUSED)
{
    return false;
}

static inline int
netdev_doca_get_rep_name(const char dev[] OVS_UNUSED,
                         char *rep OVS_UNUSED, size_t maxlen OVS_UNUSED)
{
    return -1;
}

#endif /* DPDK_NETDEV */

#endif /* netdev-doca-ext.h */
