/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2023-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#include <config.h>

#include <doca_flow.h>
#include <rte_flow.h>
#include <sys/types.h>

#include "conntrack-offload.h"
#include "coverage.h"
#include "doca-mirror.h"
#include "doca-pipe-group.h"
#include "dp-packet.h"
#include "dpdk-offload-doca.h"
#include "dpdk-offload-provider.h"
#include "id-fpool.h"
#include "offload-metadata.h"
#include "openvswitch/vlog.h"
#include "openvswitch/list.h"
#include "ovs-doca.h"
#include "ovs-rcu.h"
#include "netdev-doca.h"
#include "netdev-vport.h"
#include "timeval.h"
#include "refmap.h"
#include "unixctl.h"
#include "util.h"

#define PIPE_RESIZE_ELAPSED_MAX_MS 10

VLOG_DEFINE_THIS_MODULE(doca_pipe_group);
static struct vlog_rate_limit rl = VLOG_RATE_LIMIT_INIT(600, 600);

COVERAGE_DEFINE(doca_pipe_resize);
COVERAGE_DEFINE_WARN(doca_pipe_resize_over_10_ms);

static struct refmap *doca_pipe_group_rfm;
static struct refmap *doca_pipe_group_mask_rfm;

/* Design of the doca pipe groups and doca pipe group masks:
 *
 *  +-----------------+
 *  | Rule outside of |
 *  | group X pointing|
 *  | to it           |
 *  +------+----------+
 *         |
 *         |
 *         |                                   Group X
 *  +---------------------------------------------------------------------------------------------+
 *  |      |                                                                                      |
 *  |      |     +--->----+ Prio 0            +------>     +--------+ Prio 3                      |
 *  |      |     |        |(Highest)          |      |     |        | (Lowest)                    |
 *  |      |     ^  +----------------------+  |      |     |  +---------------------+             |
 *  |      |     |  |     |                |  |      |     |  |     |               |             |
 *  |      |     |  |     +---+            |  |      |     |  |     +---+           |             |
 *  |      |     |  |         |            |  |      |     |  |         |           |             |
 *  |      v     |  | Mask A  v            |  ^      |     ^  | Mask C  v           |             |
 *  |   Group    |  | +-------+---------+  |  |      |     |  | +-------+---------+ |             |
 *  |   Start    ^  | |      Rule       |  |  |      v     |  | |      Rule       | |             |
 *  |   Anchor   |  | |      Rule       |  |  |  +---+--+  |  | |      Rule       | |             |
 *  | +-------+  |  | |      Rule       |  |  |  |      |  |  | |      Rule       | |             |
 *  | | Empty |  |  | |      Rule       |  |  |  |      |  |  | |      Rule       | |  Doca       |
 *  | | Basic |  ^  | +-------+---------+  |  |  | .... |  |  | +-------+---------+ |  Offloads   |
 *  | | Pipe  |  |  |         |            |  |  |      |  |  |         |           |  Miss Path  |
 *  | +--+----+  |  |         |Miss FWD    |  ^  |      |  ^  |         |Miss FWD   |  Pipe       |
 *  |    |       |  |         |            |  |  |      |  |  |         |           |       ^     |
 *  |    |       |  | Mask B  v            |  |  +---+--+  |  | Mask D  v           |       |     |
 *  |    +--->---+  | +-------+---------+  |  |      |     |  | +-------+---------+ |       |     |
 *  |    Miss FWD   | |      Rule       |  |  |      |     |  | |      Rule       | |       |     |
 *  |               | |      Rule       |  |  |      |     |  | |      Rule       | |       |     |
 *  |               | |      Rule       |  |  |      |     |  | |      Rule       | |       |     |
 *  |               | |      Rule       |  |  ^      v     ^  | |      Rule       | |       |     |
 *  |               | +-------+---------+  |  |      |     |  | +-------+---------+ |       ^     |
 *  |               |         |            |  |      |     |  |         |           |       |     |
 *  |               +----------------------+  |      |     |  +---------------------+       |     |
 *  |                         |               |      |     |            |                   |     |
 *  |                         +----->---------+      +-----+            +----->-------------+     |
 *  |                            Miss FWD            Miss FWD              Miss FWD               |
 *  +---------------------------------------------------------------------------------------------+
 */

struct doca_pipe_group_ctx {
    struct doca_offload_esw_ctx *esw;
    struct doca_flow_pipe *start_anchor;
    struct doca_flow_pipe *end_anchor;
    struct ovs_list list;
    struct ovs_mutex lock;
    uint32_t group_id;
};

OVS_ASSERT_PACKED(struct doca_pipe_group_key,
    uint32_t group_id;
    uint32_t esw_mgr_port_id;
    struct doca_pipe_group_mask_ctx *prefix_mask_ctx;
);

struct doca_pipe_resize_ctx {
    struct ovs_list resized_list_node;
    void *esw_ctx;
    atomic_bool resizing;
};

struct doca_pipe_group_mask_ctx {
    struct doca_pipe_group_ctx *group_ctx;
    struct doca_pipe_resize_ctx resize_ctx;
    struct doca_flow_pipe *pipe;
    uint32_t priority;
    struct ovs_list list;
    struct doca_pipe_group_ctx *suffix_group_ctx;
    uint16_t pipe_queue;
    char *match_str;
    char *actions_str;
    uint32_t pipe_size;
    char pipe_name[50];
};

OVS_ASSERT_PACKED(struct doca_pipe_group_mask_key,
    uint32_t priority;
    char pad[4];
    struct doca_pipe_group_ctx *group_ctx;
    struct ovs_doca_flow_match match_mask;
    struct ovs_doca_flow_match match_value;
    struct ovs_doca_flow_actions actions_mask;
    struct ovs_doca_flow_actions actions_value;
    struct doca_flow_action_desc desc;
    struct doca_flow_monitor monitor;
    struct doca_flow_fwd fwd;
    uint32_t encap_type;
    uint32_t encap_size;
);

struct doca_pipe_group_mask_arg {
    struct doca_pipe_group_mask_key *key;
    struct netdev *netdev;
    struct ds match_ds;
    struct ds actions_ds;
    bool *too_big;
    uint64_t allowed_queues_bitmap;
};

/* DOCA has three types of fields in the pipe match value, see below: */
enum doca_field_type {
    DOCA_FIELD_TYPE_IGNORED,
    /**< ignored field type - not matched */
    DOCA_FIELD_TYPE_SPECIFIC,
    /**< specific field type - constant match for all pipe entries */
    DOCA_FIELD_TYPE_CHANGEABLE,
    /**< changeable field type - match determined by entry value and pipe match mask */
};

static void
doca_pipe_group_dump(struct unixctl_conn *conn, int argc OVS_UNUSED,
                     const char *argv[] OVS_UNUSED, void *aux OVS_UNUSED);

static inline enum
doca_field_type field_get_type(const void *field_ptr, uint16_t field_len)
{
    uint16_t changeable_bytes, ignored_bytes, i;
    const uint8_t *data_array;

    data_array = (uint8_t *) field_ptr;
    ignored_bytes = 0;
    changeable_bytes = 0;

    for (i = 0; i < field_len; i++) {
        if (data_array[i] == 0) {
            ignored_bytes++;
        } else if (data_array[i] == 0xFF) {
            changeable_bytes++;
        } else {
            return DOCA_FIELD_TYPE_SPECIFIC;
        }

        if (unlikely(ignored_bytes > 0 && changeable_bytes > 0)) {
            return DOCA_FIELD_TYPE_SPECIFIC;
        }
    }
    if (changeable_bytes > 0) {
        return DOCA_FIELD_TYPE_CHANGEABLE;
    }

    return DOCA_FIELD_TYPE_IGNORED;
}

static bool
doca_pipe_group_is_static(uint32_t group_id)
{
    if (group_id == SAMPLE_TABLE_ID ||
        group_id == SAMPLE_POSTMIRROR_TABLE_ID) {
        return true;
    }

    return false;
}

static uint32_t
doca_pipe_group_get_table_size(uint32_t group_id)
{
    if (doca_pipe_group_is_static(group_id)) {
        return 1;
    }

    if (group_id == 0 || group_id == POSTHASH_TABLE_ID ||
        group_id == POSTCT_TABLE_ID || group_id == POSTMETER_TABLE_ID ||
        IS_SPLIT_TABLE_ID(group_id)) {
        return ctl_pipe_infra_size;
    }

    return ctl_pipe_size;
}

doca_error_t
doca_pipe_cfg_allow_queues(struct doca_flow_pipe_cfg *cfg, uint64_t queues_bitmap)
{
    unsigned int qid;
    doca_error_t err;

    if (!cfg) {
        return DOCA_ERROR_INVALID_VALUE;
    }

    for (qid = 0; qid < ovs_doca_n_offload_queues(); qid++) {
        if ((UINT64_C(1) << qid) & queues_bitmap) {
            continue;
        }
        err = doca_flow_pipe_cfg_set_excluded_queue(cfg, qid);
        if (DOCA_IS_ERROR(err)) {
            VLOG_ERR("failed to exclude queue %u in pipe configuration: %d (%s)",
                     qid, err, doca_error_get_descr(err));
            return err;
        }
    }

    return DOCA_SUCCESS;
}

doca_error_t
doca_pipe_cfg_allow_one_queue(struct doca_flow_pipe_cfg *cfg, unsigned int qid)
{
    ovs_assert(qid < 64);
    return doca_pipe_cfg_allow_queues(cfg, UINT64_C(1) << qid);
}

static doca_error_t
doca_pipe_group_create_anchor_pipe(struct netdev *netdev, char *pipe_name,
                                   bool is_root,
                                   struct doca_flow_pipe *next_pipe,
                                   struct doca_flow_pipe **anchor_pipe)
{
    struct doca_flow_port *doca_port;
    struct doca_flow_pipe_cfg *cfg;
    struct doca_flow_fwd miss_fwd;
    doca_error_t ret;

    doca_port = doca_flow_port_switch_get(netdev_doca_port_get(netdev));
    ret = doca_flow_pipe_cfg_create(&cfg, doca_port);
    if (ret) {
        VLOG_ERR("%s: Could not create doca_flow_pipe_cfg", netdev_get_name(netdev));
        return ret;
    }

    memset(&miss_fwd, 0, sizeof miss_fwd);
    if (next_pipe) {
        miss_fwd.type = DOCA_FLOW_FWD_PIPE;
        miss_fwd.next_pipe = next_pipe;

    } else {
        miss_fwd.type = DOCA_FLOW_FWD_DROP;
    }

    if (doca_flow_pipe_cfg_set_nr_entries(cfg, 0) ||
        doca_flow_pipe_cfg_set_name(cfg, pipe_name) ||
        doca_flow_pipe_cfg_set_type(cfg, DOCA_FLOW_PIPE_BASIC) ||
        doca_pipe_cfg_allow_queues(cfg, 0) ||
        doca_flow_pipe_cfg_set_is_root(cfg, is_root)) {
        VLOG_ERR("%s: Could not set doca_flow_pipe_cfg", netdev_get_name(netdev));
        ret = DOCA_ERROR_UNKNOWN;
        goto out;
    }

    ret = doca_flow_pipe_create(cfg, NULL, &miss_fwd, anchor_pipe);

out:
    doca_flow_pipe_cfg_destroy(cfg);
    return ret;
}

struct doca_pipe_group_arg {
    struct netdev *netdev;
    uint32_t group_id;
    struct doca_pipe_group_mask_ctx *prefix_mask_ctx;
};

void
doca_pipe_group_pipe_destroy(struct doca_offload_esw_ctx *esw_ctx, struct doca_flow_pipe *pipe)
{
    struct destroy_pipe_elem *elem;
    struct ovs_list *list;

    elem = xmalloc(sizeof *elem);
    elem->pipe = pipe;

    list = doca_offload_destroy_list_get(esw_ctx);
    ovs_list_push_back(list, &elem->node);
}

static int
doca_pipe_group_ctx_init(void *ctx_, void *arg_)
{
    struct doca_pipe_group_ctx *group_ctx = ctx_;
    struct doca_flow_pipe *end_anchor_next_pipe;
    struct doca_pipe_group_arg *arg = arg_;
    char pipe_name[50];
    int ret;

    VLOG_DBG_RL(&rl, "%s: %p: group_id=0x%08" PRIx32, __FUNCTION__, group_ctx, arg->group_id);
    group_ctx->esw = doca_offload_esw_ctx_get(arg->netdev);
    group_ctx->group_id = arg->group_id;
    ovs_list_init(&group_ctx->list);
    ovs_mutex_init(&group_ctx->lock);

    snprintf(pipe_name, sizeof pipe_name, "OVS_GROUP_0x%08" PRIx32 "_END_ANCHOR_PIPE",
             arg->group_id);
    end_anchor_next_pipe = doca_offload_get_misspath_pipe(group_ctx->esw);

    /* A provided prefix_mask_ctx argument means this mask_ctx is used as a prefix for the
     * current group being created. Its miss should be the next mask_ctx of that prefix,
     * instead go to SW.
     */
    if (arg->prefix_mask_ctx) {
        struct doca_pipe_group_mask_ctx *miss_mask_ctx = NULL;
        struct doca_pipe_group_ctx *prefix_group_ctx;
        struct ovs_list *prefix_last_item;
        struct ovs_list *prefix_item;

        prefix_group_ctx = arg->prefix_mask_ctx->group_ctx;
        ovs_mutex_lock(&prefix_group_ctx->lock);
        prefix_item = &arg->prefix_mask_ctx->list;
        prefix_last_item = ovs_list_back(&prefix_group_ctx->list);
        if (prefix_item == prefix_last_item) {
            end_anchor_next_pipe = arg->prefix_mask_ctx->group_ctx->end_anchor;
        } else {
            miss_mask_ctx = CONTAINER_OF(prefix_item->next, struct doca_pipe_group_mask_ctx, list);
            end_anchor_next_pipe = miss_mask_ctx->pipe;
        }
        ovs_mutex_unlock(&prefix_group_ctx->lock);
    }

    ret = doca_pipe_group_create_anchor_pipe(arg->netdev, pipe_name, false, end_anchor_next_pipe,
                                             &group_ctx->end_anchor);
    if (ret) {
        VLOG_ERR("%s: Failed to create group end anchor pipe: %d (%s) group %" PRIu32,
                 netdev_get_name(arg->netdev), ret, doca_error_get_descr(ret), arg->group_id);
        return ret;
    }
    VLOG_DBG_RL(&rl, "%s: end_anchor=%p ->%p", pipe_name, group_ctx->end_anchor,
                end_anchor_next_pipe);

    snprintf(pipe_name, sizeof pipe_name, "OVS_GROUP_0x%08" PRIx32 "_START_ANCHOR_PIPE",
             arg->group_id);
    ret = doca_pipe_group_create_anchor_pipe(arg->netdev, pipe_name,
                                             group_ctx->group_id == 0,
                                             group_ctx->end_anchor,
                                             &group_ctx->start_anchor);
    if (ret) {
        VLOG_ERR("%s: Failed to create group start anchor pipe: %d (%s) group %" PRIu32,
                 netdev_get_name(arg->netdev), ret, doca_error_get_descr(ret), arg->group_id);
        goto err_start_anchor;
    }
    VLOG_DBG_RL(&rl, "%s: start_anchor=%p ->%p", pipe_name, group_ctx->start_anchor,
                group_ctx->end_anchor);

    return 0;

err_start_anchor:
    doca_pipe_group_pipe_destroy(group_ctx->esw, group_ctx->end_anchor);
    return ret;
}

static void
doca_pipe_group_ctx_uninit(void *ctx_)
{
    struct doca_pipe_group_ctx *group_ctx = ctx_;

    VLOG_DBG_RL(&rl, "%s: %p: group_id=0x%08" PRIx32, __FUNCTION__, group_ctx,
                group_ctx->group_id);

    ovs_mutex_destroy(&group_ctx->lock);
    doca_pipe_group_pipe_destroy(group_ctx->esw, group_ctx->start_anchor);
    doca_pipe_group_pipe_destroy(group_ctx->esw, group_ctx->end_anchor);
}

static struct ds *
doca_pipe_group_ctx_dump(struct ds *s, void *key_, void *ctx_ OVS_UNUSED, void *arg_ OVS_UNUSED)
{
    struct doca_pipe_group_key *key = key_;

    ds_put_format(s, "esw_mgr_port_id=%d, group_id=%s, ", key->esw_mgr_port_id,
                  dpdk_offload_table_id_str(key->group_id));

    return s;
}

static void
doca_pipe_group_md_init(void)
{
    static struct ovsthread_once init_once = OVSTHREAD_ONCE_INITIALIZER;

    if (ovsthread_once_start(&init_once)) {
        doca_pipe_group_rfm = refmap_create("doca-group", sizeof(struct doca_pipe_group_key),
                                            sizeof(struct doca_pipe_group_ctx),
                                            doca_pipe_group_ctx_init,
                                            doca_pipe_group_ctx_uninit,
                                            doca_pipe_group_ctx_dump,
                                            false);
        unixctl_command_register("doca-pipe-group/dump", "", 0, 0, doca_pipe_group_dump, NULL);
        ovsthread_once_done(&init_once);
    }
}

struct doca_pipe_group_ctx *
doca_pipe_group_ctx_ref(struct netdev *netdev, uint32_t group_id,
                        struct doca_pipe_group_mask_ctx *prefix_mask_ctx)
{
    struct doca_pipe_group_key key = {
        .group_id = group_id,
        .esw_mgr_port_id = netdev_doca_get_esw_mgr_port_id(netdev),
        .prefix_mask_ctx = prefix_mask_ctx,
    };
    struct doca_pipe_group_arg arg = {
        .netdev = netdev,
        .group_id = group_id,
        .prefix_mask_ctx = prefix_mask_ctx,
    };

    /* MISS_TABLE_ID group is handled outside of doca-pipe-group.c */
    ovs_assert(group_id != MISS_TABLE_ID);

    doca_pipe_group_md_init();
    return refmap_ref(doca_pipe_group_rfm, &key, &arg);
}

void
doca_pipe_group_ctx_unref(struct doca_pipe_group_ctx *group_ctx)
{
    if (group_ctx == NULL) {
        return;
    }

    doca_pipe_group_md_init();
    refmap_unref(doca_pipe_group_rfm, group_ctx);
}

static void
doca_pipe_group_put_match_field_to_ds(struct ds *ds, char *prefix, char *field,
                                      unsigned char *value, unsigned char *mask,
                                      size_t size)
{
    enum doca_field_type type = field_get_type(value, size);
    int i;

    if (type == DOCA_FIELD_TYPE_IGNORED) {
         return;
    }

    ds_put_format(ds, "%s.%s[%d,%s]=", prefix, field, (int) size,
                 (type == DOCA_FIELD_TYPE_CHANGEABLE ? "changeable" : "specific"));

    ds_put_cstr(ds, "0x");
    for (i = 0; i < size; i++) {
        ds_put_format(ds, "%02x", value[i]);
    }

    ds_put_format(ds, "/");

    ds_put_cstr(ds, "0x");
    for (i = 0; i < size; i++) {
        ds_put_format(ds, "%02x", mask[i]);
    }

    ds_put_format(ds, ", ");
}

#define log_field(field)                                                                         \
    doca_pipe_group_put_match_field_to_ds(ds, prefix, #field,                                    \
                                          (unsigned char *) &value->field,                       \
                                          (unsigned char *) &used->field,                        \
                                          sizeof(value->field))

/* Verify a field is either ignored, or specifically matched, and set it in used match as to mark
 * that it was handled (so we can assert that no new fields were missed)
 */
#define verify_specific_field(field)                                                              \
    do {                                                                                          \
        size_t __size = sizeof value->field;                                                      \
        enum doca_field_type __type = field_get_type(&value->field, __size);                      \
        char *__used = ((char *) used) + (size_t) (((char *) &value->field) - ((char *) value));  \
                                                                                                  \
        ovs_assert(__type != DOCA_FIELD_TYPE_CHANGEABLE); /* Must be ignored or specific */       \
        log_field(field);                                                                         \
        memset(__used, 0, __size);                                                                \
    } while (0)

/* same as above, but class id is special as it's always specific even if its all 0xFF.. */
#define verify_specific_field_class_id(field)                                                     \
    do {                                                                                          \
        size_t __size = sizeof value->field;                                                      \
        char *__used = ((char *) used) + (size_t) (((char *) &value->field) - ((char *) value));  \
                                                                                                  \
        log_field(field);                                                                         \
        memset(__used, 0, __size);                                                                \
    } while (0)

/* Same as above, just with a pointer to field instead of a field name */
#define mark_specific_field_ptr(field_ptr)                                                        \
    do {                                                                                          \
        size_t __size = sizeof value->field;                                                      \
        enum doca_field_type __type = field_get_type(&value->field, __size);                      \
        char *__used = ((char *) used) + (size_t) (((char *) &value->field) - ((char *) value));  \
                                                                                                  \
        ovs_assert(__type != DOCA_FIELD_TYPE_CHANGEABLE); /* Must be ignored or specific */       \
        log_field(field);                                                                         \
        memset(__used, 0, __size);                                                                \
    } while (0)

/* Extend specifically matched fields masks so they will be considered CHANGEABLE by doca,
 * and set it in used match as to mark that it was handled (so we can assert that no
 * new fields were missed)
 */
#define extend_field_to_changeable(field)                                                         \
    do {                                                                                          \
        size_t __size = sizeof value->field;                                                      \
        enum doca_field_type __type = field_get_type(&mask->field, __size);                       \
        char *__used = ((char *) used) + (size_t) (((char *) &value->field) - ((char *) value));  \
                                                                                                  \
        /* If field isn't ignored, set field as changeable. */                                    \
        if (__type != DOCA_FIELD_TYPE_IGNORED) {                                                  \
            memset(&value->field, 0xFF, __size);                                                  \
        }                                                                                         \
        log_field(field);                                                                         \
        memset(__used, 0, __size);                                                                \
    } while (0)

/* The following functions take in a match mask (or action mask), and from
 * it create a pipe match (value) where fields that can be CHANGEABLE will be
 * CHANGEABLE by extending the pipe match value to the full field size.
 * They also verify we didn't miss and new fields that might have been added.
 */
static void
mask_to_changeable_value_header_format(struct ds *ds, char *prefix,
                                       struct doca_flow_header_format *value,
                                       struct doca_flow_header_format *mask,
                                       struct doca_flow_header_format *used,
                                       enum doca_flow_tun_type encap_tun_type)
{
    /* Doca enforces this field to be specific for tunnel encap */
    if (encap_tun_type) {
        mask->eth.type = 0;
        verify_specific_field(eth.type);
    } else {
        extend_field_to_changeable(eth.type);
    }

    extend_field_to_changeable(eth.dst_mac);
    extend_field_to_changeable(eth.src_mac);

    verify_specific_field(l2_valid_headers);
    if (value->l2_valid_headers & DOCA_FLOW_L2_VALID_HEADER_VLAN_0) {
        extend_field_to_changeable(eth_vlan[0]);
    }

    verify_specific_field(l3_type);
    if (value->l3_type == DOCA_FLOW_L3_TYPE_IP4) {
        extend_field_to_changeable(ip4.src_ip);
        extend_field_to_changeable(ip4.dst_ip);
        extend_field_to_changeable(ip4.dscp_ecn);
        extend_field_to_changeable(ip4.next_proto);
        extend_field_to_changeable(ip4.ttl);
    } else if (value->l3_type == DOCA_FLOW_L3_TYPE_IP6) {
        extend_field_to_changeable(ip6.src_ip);
        extend_field_to_changeable(ip6.dst_ip);
        extend_field_to_changeable(ip6.traffic_class);
        extend_field_to_changeable(ip6.next_proto);
        extend_field_to_changeable(ip6.hop_limit);
    }

    verify_specific_field(l4_type_ext);
    switch (value->l4_type_ext) {
    case DOCA_FLOW_L4_TYPE_EXT_TCP:
        extend_field_to_changeable(tcp.l4_port.src_port);
        extend_field_to_changeable(tcp.l4_port.dst_port);
        extend_field_to_changeable(tcp.flags);
        break;
    case DOCA_FLOW_L4_TYPE_EXT_UDP:
        extend_field_to_changeable(udp.l4_port.src_port);
        if (encap_tun_type == DOCA_FLOW_TUN_GENEVE) {
            /* Doca enforces specific geneve dst port */
            mask->udp.l4_port.dst_port = 0;
            verify_specific_field(udp.l4_port.dst_port);
        } else {
            extend_field_to_changeable(udp.l4_port.dst_port);
        }
        break;
    case DOCA_FLOW_L4_TYPE_EXT_ICMP:
    case DOCA_FLOW_L4_TYPE_EXT_ICMP6:
        extend_field_to_changeable(icmp.type);
        extend_field_to_changeable(icmp.code);
        break;

    case DOCA_FLOW_L4_TYPE_EXT_TRANSPORT:
    case DOCA_FLOW_L4_TYPE_EXT_ROCE_V2:
    case DOCA_FLOW_L4_TYPE_EXT_NONE:
        break;
    }
}

static void
mask_to_changeable_value_tun(struct ds *ds, char *prefix,
                             struct doca_flow_tun *value,
                             struct doca_flow_tun *mask,
                             struct doca_flow_tun *used)
{
    union doca_flow_geneve_option *opt;

    verify_specific_field(type);
    switch (value->type) {
    case DOCA_FLOW_TUN_NONE:
    case DOCA_FLOW_TUN_MAX:
    case DOCA_FLOW_TUN_GTPU:
    case DOCA_FLOW_TUN_ESP:
    case DOCA_FLOW_TUN_MPLS_O_UDP:
    case DOCA_FLOW_TUN_PSP:
    case DOCA_FLOW_TUN_IP_IN_IP:
        break;
    case DOCA_FLOW_TUN_VXLAN:
        extend_field_to_changeable(vxlan_tun_id);
        verify_specific_field(vxlan_type);
        if (value->vxlan_type == DOCA_FLOW_TUN_EXT_VXLAN_GBP) {
            extend_field_to_changeable(vxlan_gbp_group_policy_id);
        }
        break;
    case DOCA_FLOW_TUN_GRE:
        verify_specific_field(key_present);
        if (value->key_present) {
            extend_field_to_changeable(gre_key);
        }
        extend_field_to_changeable(protocol);
        break;
    case DOCA_FLOW_TUN_GENEVE:
        extend_field_to_changeable(geneve.vni);
        extend_field_to_changeable(geneve.next_proto);
        extend_field_to_changeable(geneve.o_c);
        verify_specific_field(geneve.ver_opt_len);

        for (int opt_idx = 0; opt_idx < DOCA_FLOW_GENEVE_OPT_LEN_MAX; opt_idx++) {
            opt = &value->geneve_options[opt_idx];
            uint8_t len = opt->length;
            uint8_t i;

            if (!len || !opt->type) {
                break;
            }

            verify_specific_field(geneve_options[opt_idx].length);
            verify_specific_field(geneve_options[opt_idx].type);
            verify_specific_field_class_id(geneve_options[opt_idx].class_id);

            for (i = 0; i < len; i++) {
                opt_idx++;
                if (opt_idx >= DOCA_FLOW_GENEVE_OPT_LEN_MAX) {
                    break;
                }
                extend_field_to_changeable(geneve_options[opt_idx].data);
            }
        }

        break;
    }
}

static void
mask_to_changeable_value_parser_meta(struct ds *ds, char *prefix,
                                     struct doca_flow_parser_meta *value,
                                     struct doca_flow_parser_meta *mask,
                                     struct doca_flow_parser_meta *used)
{
    extend_field_to_changeable(port_id);
    extend_field_to_changeable(meter_color);
    extend_field_to_changeable(random);
    extend_field_to_changeable(outer_ip_fragmented);
    extend_field_to_changeable(inner_ip_fragmented);
    verify_specific_field(outer_l2_type);
    verify_specific_field(inner_l2_type);
    verify_specific_field(outer_l3_type);
    verify_specific_field(inner_l3_type);
    verify_specific_field(outer_l4_type);
    verify_specific_field(inner_l4_type);
}

static void
mask_to_changeable_value_meta(struct ds *ds, char *prefix,
                              struct doca_flow_meta *value,
                              struct doca_flow_meta *mask,
                              struct doca_flow_meta *used)
{
    int i;

    for (i = 0; i < DOCA_FLOW_META_SCRATCH_PAD_MAX; i++) {
        extend_field_to_changeable(u32[i]);
    }
    extend_field_to_changeable(pkt_meta);
}

static void
mask_to_changeable_value_mark(struct ds *ds, char *prefix,
                              struct ovs_doca_flow_actions *value,
                              struct ovs_doca_flow_actions *mask,
                              struct ovs_doca_flow_actions *used)
{
    extend_field_to_changeable(mark);
}

static void
action_mask_to_changeable_value(struct ovs_doca_flow_actions *ovalue,
                                struct ovs_doca_flow_actions *omask,
                                struct ds *ds)
{
    struct doca_flow_actions *value = &ovalue->d, *mask = &omask->d;
    struct ovs_doca_flow_actions u;
    struct doca_flow_actions *used;
    char *prefix = "actions";

    memcpy(&u, omask, sizeof u);
    used = &u.d;

    verify_specific_field(pop_vlan);

    mask_to_changeable_value_meta(ds, "actions.meta", &value->meta, &mask->meta, &used->meta);
    mask_to_changeable_value_mark(ds, "actions.mark", ovalue, omask, &u);
    mask_to_changeable_value_parser_meta(ds, "actions.parser_meta", &value->parser_meta,
                                         &mask->parser_meta, &used->parser_meta);
    mask_to_changeable_value_header_format(ds, "actions.outer", &value->outer, &mask->outer,
                                           &used->outer, DOCA_FLOW_TUN_NONE);
    mask_to_changeable_value_tun(ds, "actions.tun", &value->tun, &mask->tun, &used->tun);

    verify_specific_field(encap_type);
    if (value->encap_type) {
        if (value->encap_type == DOCA_FLOW_RESOURCE_TYPE_SHARED) {
            extend_field_to_changeable(shared_encap_id);
            value->shared_encap_id = mask->shared_encap_id;
        } else {
            verify_specific_field(encap_cfg.is_l2);
            mask_to_changeable_value_header_format(ds, "actions.encap.outer",
                                                   &value->encap_cfg.encap.outer,
                                                   &mask->encap_cfg.encap.outer,
                                                   &used->encap_cfg.encap.outer,
                                                   value->encap_cfg.encap.tun.type);
            mask_to_changeable_value_tun(ds, "actions.encap.tun",
                                         &value->encap_cfg.encap.tun,
                                         &mask->encap_cfg.encap.tun,
                                         &used->encap_cfg.encap.tun);
        }
    }

    verify_specific_field(decap_type);
    verify_specific_field(decap_cfg.is_l2);

    verify_specific_field(has_push);
    if (value->has_push) {
        verify_specific_field(push.type);
        if (value->push.type == DOCA_FLOW_PUSH_ACTION_VLAN) {
            extend_field_to_changeable(push.vlan.vlan_hdr.tci);
            extend_field_to_changeable(push.vlan.eth_type);
        }
    }

    ovs_assert(is_all_zeros(used, sizeof *used));
}

static void
mask_to_changeable_value_nv_mp(struct ds *ds, char *prefix,
                               struct ovs_doca_flow_match *value,
                               struct ovs_doca_flow_match *mask,
                               struct ovs_doca_flow_match *used)
{
    extend_field_to_changeable(nv_mp.pid);
    extend_field_to_changeable(nv_mp.preferred);
    extend_field_to_changeable(nv_mp.strict);
}

static void
match_mask_to_changeable_value(struct ovs_doca_flow_match *ovalue,
                               struct ovs_doca_flow_match *omask,
                               struct ds *ds)
{
    struct doca_flow_match *value = &ovalue->d, *mask = &omask->d;
    struct ovs_doca_flow_match u;
    struct doca_flow_match *used;

    memcpy(&u, omask, sizeof u);
    used = &u.d;

    mask_to_changeable_value_meta(ds, "match.meta", &value->meta, &mask->meta, &used->meta);
    mask_to_changeable_value_parser_meta(ds, "match.parser_meta", &value->parser_meta,
                                         &mask->parser_meta, &used->parser_meta);
    mask_to_changeable_value_tun(ds, "match.tun", &value->tun, &mask->tun, &used->tun);
    mask_to_changeable_value_header_format(ds, "match.inner", &value->inner, &mask->inner,
                                           &used->inner, DOCA_FLOW_TUN_NONE);
    mask_to_changeable_value_header_format(ds, "match.outer", &value->outer, &mask->outer,
                                           &used->outer, DOCA_FLOW_TUN_NONE);
    mask_to_changeable_value_nv_mp(ds, "nv_mp", ovalue, omask, &u);

    ovs_assert(is_all_zeros(&u, sizeof u));
}

static void
fwd_to_changeable_value(struct doca_flow_fwd *value,
                        const struct doca_flow_fwd *fwd)
{
    /* Try and make forward value changeable per entry */
    switch (fwd->type) {
    case DOCA_FLOW_FWD_PIPE:
        value->type = fwd->type;
        value->next_pipe = 0;
        break;
    case DOCA_FLOW_FWD_PORT:
        memset(value, 0xFF, sizeof(*value));
        value->type = fwd->type;
        break;
    case DOCA_FLOW_FWD_NONE:
    case DOCA_FLOW_FWD_DROP:
    case DOCA_FLOW_FWD_RSS:
    case DOCA_FLOW_FWD_TARGET:
    case DOCA_FLOW_FWD_CHANGEABLE:
    case DOCA_FLOW_FWD_ORDERED_LIST_PIPE:
        /* Use specific action */
        *value = *fwd;
        break;
    case DOCA_FLOW_FWD_HASH_PIPE:
        OVS_NOT_REACHED();
        break;
    }
}

void
doca_pipe_group_set_suffix_group(struct doca_pipe_group_mask_ctx *mask_ctx,
                                 struct doca_pipe_group_ctx *suffix_group_ctx)
{
    mask_ctx->suffix_group_ctx = suffix_group_ctx;
}

struct too_big_item {
    struct ovs_doca_flow_match mask;
    struct cmap_node node;
};

void
doca_pipe_group_too_big_map_init(struct too_big_map *tbm)
{
    cmap_init(&tbm->map);
    ovs_mutex_init(&tbm->lock);
}

void
doca_pipe_group_too_big_map_uninit(struct too_big_map *tbm)
{
    struct too_big_item *item;

    ovs_mutex_lock(&tbm->lock);
    CMAP_FOR_EACH (item, node, &tbm->map) {
        ovsrcu_postpone(free, item);
    }
    cmap_destroy(&tbm->map);
    ovs_mutex_unlock(&tbm->lock);
    ovs_mutex_destroy(&tbm->lock);
}

static void
doca_pipe_group_too_big_map_insert(struct doca_offload_esw_ctx *esw_ctx,
                                   struct ovs_doca_flow_match *mask)
{
    struct too_big_map *tbm = doca_offload_too_big_map_get(esw_ctx);
    struct too_big_item *item;
    uint32_t hash;

    item = xmalloc(sizeof *item);
    memcpy(&item->mask, mask, sizeof item->mask);
    hash = hash_bytes(&item->mask, sizeof item->mask, 0);

    ovs_mutex_lock(&tbm->lock);
    cmap_insert(&tbm->map, &item->node, hash);
    ovs_mutex_unlock(&tbm->lock);
}

static bool
doca_pipe_group_too_big_map_find(struct doca_offload_esw_ctx *esw_ctx,
                                 struct ovs_doca_flow_match *mask)
{
    struct too_big_map *tbm = doca_offload_too_big_map_get(esw_ctx);
    uint32_t hash = hash_bytes(mask, sizeof *mask, 0);
    struct too_big_item *found_item;

    CMAP_FOR_EACH_WITH_HASH (found_item, node, hash, &tbm->map) {
        if (!memcmp(&found_item->mask, mask, sizeof found_item->mask)) {
            return true;
        }
    }
    return false;
}

static int
doca_pipe_group_mask_ctx_init(void *ctx_, void *arg_)
{
    struct doca_pipe_group_mask_ctx *mask_ctx = ctx_, *next, *prev;
    struct doca_flow_actions *actions_list, *actions_mask_list;
    struct doca_flow_action_descs descs, *descs_list;
    struct doca_pipe_group_mask_arg *arg = arg_;
    struct netdev *netdev = arg->netdev;
    struct doca_flow_pipe *prev_pipe;
    struct doca_flow_port *doca_port;
    struct doca_flow_pipe_cfg *cfg;
    struct doca_flow_fwd miss_fwd;
    bool is_resizable = false;
    int ret;

    mask_ctx->priority = arg->key->priority;
    mask_ctx->group_ctx = arg->key->group_ctx;

    VLOG_DBG_RL(&rl, "%s: %p: group_ctx=%p, group_id=0x%08" PRIx32, __FUNCTION__, mask_ctx,
                mask_ctx->group_ctx, mask_ctx->group_ctx->group_id);

    if (!doca_pipe_group_is_static(mask_ctx->group_ctx->group_id)) {
        mask_ctx->resize_ctx.esw_ctx = doca_offload_esw_ctx_get(netdev);
        if (!mask_ctx->resize_ctx.esw_ctx) {
            VLOG_WARN("%s: Failed to create pipe group mask: esw_ctx not found",
                      netdev_get_name(netdev));
            ret = DOCA_ERROR_UNKNOWN;
            goto err_esw_get;
        }

        ovs_list_init(&mask_ctx->resize_ctx.resized_list_node);
        is_resizable = true;
    }

    prev = NULL;
    ovs_mutex_lock(&mask_ctx->group_ctx->lock);
    LIST_FOR_EACH (next, list, &mask_ctx->group_ctx->list) {
        if (mask_ctx->priority < next->priority) {
            break;
        }

        prev = next;
        continue;
    }
    ovs_mutex_unlock(&mask_ctx->group_ctx->lock);

    prev_pipe = prev ? prev->pipe : mask_ctx->group_ctx->start_anchor;

    doca_port = netdev_doca_port_get(netdev);
    doca_port = doca_flow_port_switch_get(doca_port);
    ret = doca_flow_pipe_cfg_create(&cfg, doca_flow_port_switch_get(doca_port));
    if (ret) {
        VLOG_ERR("%s: Could not create doca_flow_pipe_cfg",
                 netdev_get_name(netdev));
        goto err_create_config;
    }

    snprintf(mask_ctx->pipe_name, sizeof mask_ctx->pipe_name,
             "OVS_GROUP_MASK_PIPE_0x%08" PRIx32 "_%d", mask_ctx->group_ctx->group_id,
             mask_ctx->priority);
    mask_ctx->pipe_size = doca_pipe_group_get_table_size(mask_ctx->group_ctx->group_id);
    actions_list = &arg->key->actions_value.d;
    actions_mask_list = &arg->key->actions_mask.d;
    descs.desc_array = &arg->key->desc;
    descs.nb_action_desc = 1;
    descs_list = &descs;

    if ((mask_ctx->pipe_size && doca_flow_pipe_cfg_set_nr_entries(cfg, mask_ctx->pipe_size)) ||
        doca_flow_pipe_cfg_set_name(cfg, mask_ctx->pipe_name) ||
        doca_flow_pipe_cfg_set_type(cfg, DOCA_FLOW_PIPE_BASIC) ||
        doca_flow_pipe_cfg_set_match(cfg, &arg->key->match_value.d,
                                     &arg->key->match_mask.d) ||
        doca_flow_pipe_cfg_set_actions(cfg, &actions_list,
                                       &actions_mask_list,
                                       &descs_list, 1) ||
        ((arg->key->monitor.counter_type ||
          arg->key->monitor.meter_type) &&
         doca_flow_pipe_cfg_set_monitor(cfg, &arg->key->monitor)) ||
        doca_flow_pipe_cfg_set_is_resizable(cfg, is_resizable) ||
        doca_pipe_cfg_allow_queues(cfg, arg->allowed_queues_bitmap) ||
        doca_flow_pipe_cfg_set_user_ctx(cfg, mask_ctx) ||
        doca_flow_pipe_cfg_set_congestion_level_threshold(cfg, doca_congestion_threshold)) {
        ret = DOCA_ERROR_UNKNOWN;
        VLOG_ERR("%s: Could not set doca_flow_pipe_cfg",
                 netdev_get_name(netdev));
        goto err_set_cfg;
    }

    memset(&miss_fwd, 0, sizeof miss_fwd);
    miss_fwd.type = DOCA_FLOW_FWD_PIPE;
    miss_fwd.next_pipe = next ? next->pipe : mask_ctx->group_ctx->end_anchor;

    ret = doca_flow_pipe_create(cfg, &arg->key->fwd, &miss_fwd, &mask_ctx->pipe);
    if (ret) {
        if (ret == DOCA_ERROR_TOO_BIG) {
            if (arg->too_big) {
                *arg->too_big = true;
            }
            doca_pipe_group_too_big_map_insert(doca_offload_esw_ctx_get(netdev),
                                               &arg->key->match_mask);
            VLOG_DBG_RL(&rl, "%s: Failed to create group mask prio %d main pipe: Match too big",
                        netdev_get_name(netdev), mask_ctx->priority);
        } else {
            VLOG_ERR("%s: Failed to create group mask prio %d main pipe: %d (%s)",
                     netdev_get_name(netdev), mask_ctx->priority,
                     ret, doca_error_get_descr(ret));
        }
        goto err_pipe_create_main;
    }
    VLOG_DBG_RL(&rl, "%s: mask_ctx=%p, pipe=%p ->%p", mask_ctx->pipe_name, mask_ctx, mask_ctx->pipe,
                miss_fwd.next_pipe);

    memset(&miss_fwd, 0, sizeof miss_fwd);
    miss_fwd.type = DOCA_FLOW_FWD_PIPE;
    miss_fwd.next_pipe = mask_ctx->pipe;

    ret = doca_flow_pipe_update_miss(prev_pipe, &miss_fwd);
    VLOG_DBG_RL(&rl, "miss update: mask_ctx=%p, pipe=%p ->%p", prev, prev_pipe,
                miss_fwd.next_pipe);
    if (ret) {
        VLOG_ERR("%s: Failed to update group mask miss: %d (%s)",
                 netdev_get_name(netdev), ret, doca_error_get_descr(ret));
        goto err_miss_update_prev;
    }

    ovs_list_insert(next ? &next->list : &mask_ctx->group_ctx->list, &mask_ctx->list);

    /* If it's not the first in the list, check if its previous one has a suffix. If it has,
     * change its end_anchor to miss to the new mask_ctx.
     */
    if (ovs_list_front(&mask_ctx->group_ctx->list) != &mask_ctx->list) {
        prev = CONTAINER_OF(ovs_list_back(&mask_ctx->list), struct doca_pipe_group_mask_ctx, list);
        if (prev->suffix_group_ctx) {
            ret = doca_flow_pipe_update_miss(prev->suffix_group_ctx->end_anchor, &miss_fwd);
            VLOG_DBG_RL(&rl, "suffix miss update: group_ctx=%p, end_anchor=%p ->%p",
                        prev->suffix_group_ctx, prev->suffix_group_ctx->end_anchor,
                        miss_fwd.next_pipe);
            ovs_assert(!ret);
        }
    }

    doca_flow_pipe_cfg_destroy(cfg);

    mask_ctx->match_str = ds_steal_cstr(&arg->match_ds);
    mask_ctx->actions_str = ds_steal_cstr(&arg->actions_ds);

    return 0;

err_miss_update_prev:
    doca_pipe_group_pipe_destroy(mask_ctx->group_ctx->esw, mask_ctx->pipe);
err_pipe_create_main:
err_set_cfg:
    doca_flow_pipe_cfg_destroy(cfg);
err_create_config:
err_esw_get:
    return ret;
}

static void
doca_pipe_group_mask_ctx_uninit(void *ctx_)
{
    struct doca_pipe_group_mask_ctx *mask_ctx = ctx_, *prev, *next;
    struct doca_flow_pipe *prev_pipe;
    struct doca_flow_fwd miss_fwd;
    int ret;

    VLOG_DBG_RL(&rl, "%s: %p: group_ctx=%p, group_id=0x%08" PRIx32, __FUNCTION__, mask_ctx,
                mask_ctx->group_ctx, mask_ctx->group_ctx->group_id);

    ovs_mutex_lock(&mask_ctx->group_ctx->lock);
    next = mask_ctx->list.next == &mask_ctx->group_ctx->list ?
           NULL : CONTAINER_OF(mask_ctx->list.next, struct doca_pipe_group_mask_ctx, list);
    prev = mask_ctx->list.prev == &mask_ctx->group_ctx->list ?
           NULL : CONTAINER_OF(mask_ctx->list.prev, struct doca_pipe_group_mask_ctx, list);
    prev_pipe = prev ? prev->pipe : mask_ctx->group_ctx->start_anchor;
    ovs_list_remove(&mask_ctx->list);
    ovs_mutex_unlock(&mask_ctx->group_ctx->lock);

    memset(&miss_fwd, 0, sizeof miss_fwd);
    miss_fwd.type = DOCA_FLOW_FWD_PIPE;
    miss_fwd.next_pipe = next ? next->pipe : mask_ctx->group_ctx->end_anchor;
    /* In case the previous mask_ctx has a suffix, change its end_anchor to miss to the
     * next mask_ctx.
     */
    if (prev && prev->suffix_group_ctx) {
        ret = doca_flow_pipe_update_miss(prev->suffix_group_ctx->end_anchor, &miss_fwd);
        ovs_assert(!ret);
        VLOG_DBG_RL(&rl, "suffix miss update: group_ctx=%p, end_anchor=%p ->%p",
                    prev->suffix_group_ctx, prev->suffix_group_ctx->end_anchor,
                    miss_fwd.next_pipe);
    }

    ret = doca_flow_pipe_update_miss(prev_pipe, &miss_fwd);
    VLOG_DBG_RL(&rl, "miss update: mask_ctx=%p, pipe=%p, ->%p", prev, prev_pipe,
                miss_fwd.next_pipe);
    ovs_assert(!ret);

    doca_pipe_group_pipe_destroy(mask_ctx->group_ctx->esw, mask_ctx->pipe);

    free(mask_ctx->match_str);
    free(mask_ctx->actions_str);
}

static const char *
doca_pipe_fwd_type_str(enum doca_flow_fwd_type type)
{
    switch (type) {
    case DOCA_FLOW_FWD_NONE:
        return "none";
    case DOCA_FLOW_FWD_RSS:
        return "rss";
    case DOCA_FLOW_FWD_PORT:
        return "port";
    case DOCA_FLOW_FWD_PIPE:
        return "pipe";
    case DOCA_FLOW_FWD_DROP:
        return "drop";
    case DOCA_FLOW_FWD_TARGET:
        return "target";
    case DOCA_FLOW_FWD_ORDERED_LIST_PIPE:
        return "ordered_list_pipe";
    case DOCA_FLOW_FWD_CHANGEABLE:
        return "changeable";
    case DOCA_FLOW_FWD_HASH_PIPE:
        return "fwd_hash_pipe";
    }
    OVS_NOT_REACHED();
    return "ERR";
}

static struct ds *
doca_pipe_group_mask_ctx_dump(struct ds *s, void *key_, void *ctx_, void *arg_ OVS_UNUSED)
{
    struct doca_pipe_group_mask_ctx *mask_ctx = ctx_;
    struct doca_pipe_group_mask_key *key = key_;

    if (!mask_ctx) {
        ds_put_cstr(s, "mask_ctx creation failed");
        return s;
    }

    ds_put_format(s, "esw=%p, group_id=%s, priority=%d, fwd.type=%s, nr_entries=%u/%u, ",
                  key->group_ctx->esw, dpdk_offload_table_id_str(key->group_ctx->group_id),
                  key->priority, doca_pipe_fwd_type_str(key->fwd.type),
                  refmap_value_refcount_read(doca_pipe_group_mask_rfm, mask_ctx),
                  mask_ctx->pipe_size);

    ds_put_format(s, "%s, %s, ", mask_ctx->match_str, mask_ctx->actions_str);

    return s;
}

static void
doca_pipe_group_mask_md_init(void)
{
    static struct ovsthread_once init_once = OVSTHREAD_ONCE_INITIALIZER;

    if (ovsthread_once_start(&init_once)) {
        doca_pipe_group_mask_rfm = refmap_create("doca-group-pipe-mask",
                                                 sizeof(struct doca_pipe_group_mask_key),
                                                 sizeof(struct doca_pipe_group_mask_ctx),
                                                 doca_pipe_group_mask_ctx_init,
                                                 doca_pipe_group_mask_ctx_uninit,
                                                 doca_pipe_group_mask_ctx_dump,
                                                 false);
        ovsthread_once_done(&init_once);
    }
}

struct doca_pipe_group_mask_ctx *
doca_pipe_group_mask_ctx_ref(struct netdev *netdev,
                             struct doca_pipe_group_ctx *group_ctx,
                             const struct ovs_doca_flow_match *match,
                             struct ovs_doca_flow_match *match_mask,
                             struct ovs_doca_flow_actions *actions,
                             struct ovs_doca_flow_actions *actions_mask,
                             const struct doca_flow_action_desc *desc,
                             const struct doca_flow_monitor *monitor,
                             const struct doca_flow_fwd *fwd,
                             uint32_t priority, bool *too_big,
                             uint64_t allowed_queues_bitmap)
{
    struct doca_pipe_group_mask_ctx *mask_ctx;
    struct doca_pipe_group_mask_key key = {
        .group_ctx = group_ctx,
        .priority = priority,
        .encap_type = actions ? actions->encap_type : 0,
        .encap_size = actions ? actions->encap_size : 0,
    };
    struct doca_pipe_group_mask_arg arg = {
        .netdev = netdev,
        .too_big = too_big,
        .key = &key,
        .match_ds = DS_EMPTY_INITIALIZER,
        .actions_ds = DS_EMPTY_INITIALIZER,
        .allowed_queues_bitmap = allowed_queues_bitmap,
    };

    /* Make fields which don't support changeable to specific */
    if (actions_mask && actions->d.tun.type == DOCA_FLOW_TUN_GRE) {
        actions_mask->d.tun.key_present = actions->d.tun.key_present;
    }

    if (match && match_mask) {
        memcpy(&key.match_value, match, sizeof *match);
        memcpy(&key.match_mask, match_mask, sizeof *match_mask);
        match_mask_to_changeable_value(&key.match_value, &key.match_mask, &arg.match_ds);

        if (doca_pipe_group_too_big_map_find(doca_offload_esw_ctx_get(netdev), &key.match_mask)) {
            if (too_big) {
                *too_big = true;
            }
            mask_ctx = NULL;
            goto out;
        }
    }

    if (is_all_zeros(&key.match_value, sizeof key.match_value)) {
        ds_put_format(&arg.match_ds, "empty_match");
    } else {
        ds_chomp(&arg.match_ds, ' ');
        ds_chomp(&arg.match_ds, ',');
    }

    if (actions_mask) {
        memcpy(&key.actions_value, actions, sizeof *actions);
        memcpy(&key.actions_mask, actions_mask, sizeof *actions_mask);
        action_mask_to_changeable_value(&key.actions_value, &key.actions_mask, &arg.actions_ds);
    }

    if (is_all_zeros(&key.actions_value, sizeof key.actions_value)) {
        ds_put_format(&arg.actions_ds, "empty_actions_mask");
    } else {
        ds_chomp(&arg.actions_ds, ' ');
        ds_chomp(&arg.actions_ds, ',');
    }

    fwd_to_changeable_value(&key.fwd, fwd);

    if (monitor) {
        memcpy(&key.monitor, monitor, sizeof *monitor);

        /* Change a shared counter/monitor to be specified per entry to reuse such pipe */
        if (monitor->meter_type == DOCA_FLOW_RESOURCE_TYPE_SHARED) {
            key.monitor.shared_meter.shared_meter_id = UINT32_MAX;
        }

        if (monitor->counter_type == DOCA_FLOW_RESOURCE_TYPE_SHARED) {
            key.monitor.shared_counter.shared_counter_id = UINT32_MAX;
        }
    }

    if (desc) {
        memcpy(&key.desc, desc, sizeof *desc);
    }

    doca_pipe_group_mask_md_init();
    mask_ctx = refmap_ref(doca_pipe_group_mask_rfm, &key, &arg);

out:
    ds_destroy(&arg.match_ds);
    ds_destroy(&arg.actions_ds);
    return mask_ctx;
}

static void
doca_pipe_group_mask_ctx_ref_value(struct doca_pipe_group_mask_ctx *mask_ctx)
{
    refmap_ref_value(doca_pipe_group_mask_rfm, mask_ctx, true);
}

void
doca_pipe_group_mask_ctx_unref(struct doca_pipe_group_mask_ctx *mask_ctx)
{
    if (mask_ctx == NULL) {
        return;
    }

    doca_pipe_group_mask_md_init();
    refmap_unref(doca_pipe_group_mask_rfm, mask_ctx);
}

struct doca_flow_pipe *
doca_pipe_group_get_pipe(struct doca_pipe_group_ctx *group_ctx)
{
    return group_ctx->start_anchor;
}

static bool
doca_pipe_group_mask_resizing(struct doca_pipe_group_mask_ctx *mask_ctx)
{
    bool resizing;

    atomic_read(&mask_ctx->resize_ctx.resizing, &resizing);
    return resizing;
}


static doca_error_t
doca_pipe_group_resize_entry_relocate_cb(void *pipe_user_ctx OVS_UNUSED,
                                         uint16_t pipe_queue OVS_UNUSED, void *entry_user_ctx,
                                         void **new_entry_user_ctx)
{
    *new_entry_user_ctx = entry_user_ctx;

    return 0;
}

static doca_error_t
doca_pipe_group_resize_nr_entries_changed_cb(void *pipe_user_ctx, uint32_t nr_entries)
{
    struct doca_pipe_group_mask_ctx *mask_ctx = pipe_user_ctx;

    VLOG_DBG("Pipe: %s, user context %p: entries increased from %u -> %u", mask_ctx->pipe_name,
             mask_ctx, mask_ctx->pipe_size, nr_entries);

    mask_ctx->pipe_size = nr_entries;

    return 0;
}

static void
doca_pipe_group_mask_ctx_resize_begin(struct doca_pipe_group_mask_ctx *mask_ctx)
{
    long long int resize_start_ms, elapsed_ms;
    struct doca_offload_esw_ctx *esw_ctx;

    /* It is illegal to trigger a resize from a thread that was marked
     * as not using the resize feature. */
    ovs_assert(ovs_doca_queue_use_resize(netdev_offload_thread_id()));

    esw_ctx = mask_ctx->resize_ctx.esw_ctx;
    doca_offload_esw_ctx_pipe_resizing_ref(esw_ctx);
    doca_pipe_group_mask_ctx_ref_value(mask_ctx);
    resize_start_ms = time_msec();
    doca_flow_pipe_resize(mask_ctx->pipe, DOCA_RESIZED_PIPE_CONGESTION_LEVEL,
                          doca_pipe_group_resize_nr_entries_changed_cb,
                          doca_pipe_group_resize_entry_relocate_cb);
    elapsed_ms = time_msec() - resize_start_ms;
    COVERAGE_INC(doca_pipe_resize);
    if (elapsed_ms > PIPE_RESIZE_ELAPSED_MAX_MS) {
        COVERAGE_INC(doca_pipe_resize_over_10_ms);
    }
    atomic_store_relaxed(&mask_ctx->resize_ctx.resizing, true);
}

static void
doca_pipe_group_mask_ctx_resize_end_defer(struct doca_pipe_group_mask_ctx *mask_ctx)
{
    struct doca_offload_esw_ctx *esw_ctx;

    esw_ctx = mask_ctx->resize_ctx.esw_ctx;
    doca_offload_esw_ctx_resized_list_add(esw_ctx, &mask_ctx->resize_ctx.resized_list_node);
}

static void
doca_pipe_group_mask_ctx_resize_end(struct doca_pipe_group_mask_ctx *mask_ctx)
{
    struct doca_offload_esw_ctx *esw_ctx;

    atomic_store(&mask_ctx->resize_ctx.resizing, false);
    esw_ctx = mask_ctx->resize_ctx.esw_ctx;
    doca_offload_esw_ctx_pipe_resizing_unref(esw_ctx);
    doca_pipe_group_mask_ctx_unref(mask_ctx);
}

doca_error_t
doca_pipe_group_add_entry(struct netdev *netdev,
                          uint16_t pipe_queue,
                          uint32_t priority,
                          struct doca_pipe_group_ctx *group_ctx,
                          const struct ovs_doca_flow_match *match,
                          struct ovs_doca_flow_match *match_mask,
                          struct ovs_doca_flow_actions *actions,
                          struct ovs_doca_flow_actions *actions_mask,
                          const struct doca_flow_action_descs *action_descs,
                          struct doca_flow_monitor *monitor,
                          const struct doca_flow_fwd *fwd,
                          struct doca_pipe_group_mask_ctx **mctx,
                          struct doca_flow_pipe_entry **entry)
    OVS_EXCLUDED(mgmt_queue_lock)
{
    struct doca_pipe_group_mask_ctx *mask_ctx;
    struct doca_offload_esw_ctx *esw;
    doca_error_t ret = 0;
    bool too_big = false;

    mask_ctx = doca_pipe_group_mask_ctx_ref(netdev, group_ctx, match, match_mask,
                                            actions, actions_mask,
                                            action_descs ?
                                            action_descs->desc_array : NULL,
                                            monitor, fwd, priority, &too_big,
                                            UINT64_C(1) << pipe_queue);
    if (!mask_ctx) {
        return too_big ? DOCA_ERROR_TOO_BIG : DOCA_ERROR_INVALID_VALUE;
    }

    esw = doca_offload_esw_ctx_get(netdev);
    if (doca_pipe_group_mask_resizing(mask_ctx)) {
        doca_offload_complete_queue_esw(esw, pipe_queue, true);
    }

    ret = doca_offload_add_entry(netdev, pipe_queue, mask_ctx->pipe, match, actions,
                                 monitor, fwd, DOCA_FLOW_NO_WAIT, entry);
    if (ret) {
        goto err_add;
    }

    ret = doca_offload_complete_queue_esw(esw, pipe_queue, true);
    if (ret) {
        goto err_add;
    }

    *mctx = mask_ctx;
    return ret;

err_add:
    doca_pipe_group_mask_ctx_unref(mask_ctx);
    return ret;
}

void
ovs_doca_pipe_process_cb(struct doca_flow_pipe *pipe,
                         enum doca_flow_pipe_status status,
                         enum doca_flow_pipe_op op, void *user_ctx)
{
    static const char *status_desc[] = {
        [DOCA_FLOW_PIPE_STATUS_SUCCESS] = "success",
        [DOCA_FLOW_PIPE_STATUS_ERROR] = "failure",
    };
    static const char *op_desc[] = {
        [DOCA_FLOW_PIPE_OP_CONGESTION_REACHED] = "congested",
        [DOCA_FLOW_PIPE_OP_RESIZED] = "resize",
        [DOCA_FLOW_PIPE_OP_DESTROYED] = "destroy",
    };
    bool error = status == DOCA_FLOW_PIPE_STATUS_ERROR;
    struct doca_pipe_group_mask_ctx *mask_ctx = user_ctx;
    unsigned int tid = netdev_offload_thread_id();

    VLOG_RL(&rl, error ? VLL_ERR : VLL_DBG,
            "%s: [tid:%u] mask=%p, pipe %p %s %s",
            __func__, tid, mask_ctx, pipe, op_desc[op], status_desc[status]);

    /* This case can be called for every pipe, even those
     * non-resizable. For them, 'user_ctx' will never have been
     * set. Ignore them. */
    if (user_ctx == NULL) {
        return;
    }

    switch (op) {
    case DOCA_FLOW_PIPE_OP_CONGESTION_REACHED:
        doca_pipe_group_mask_ctx_resize_begin(mask_ctx);
        break;
    case DOCA_FLOW_PIPE_OP_RESIZED:
        /* Register this context to finish its re-sizing outside of this callback. */
        doca_pipe_group_mask_ctx_resize_end_defer(mask_ctx);
        break;
    case DOCA_FLOW_PIPE_OP_DESTROYED:
    default:
        break;
    }
}

void
doca_pipe_group_ctx_resize_end_list(struct ovs_list *list)
{
    struct doca_pipe_group_mask_ctx *mask_ctx;

    LIST_FOR_EACH_POP (mask_ctx, resize_ctx.resized_list_node, list) {
        doca_pipe_group_mask_ctx_resize_end(mask_ctx);
    }
}

bool
doca_pipe_group_is_ct_zone_group_id(uint32_t group)
{
    return ((group >= CT_TABLE_ID &&
             group < CT_TABLE_ID + NUM_CT_ZONES) ||
            (group >= CTNAT_TABLE_ID &&
             group < CTNAT_TABLE_ID + NUM_CT_ZONES));
}

static void
doca_pipe_group_ctx_dump_masks(struct ds *out, struct doca_pipe_group_ctx *group_ctx)
{
    struct doca_pipe_group_mask_ctx *mask_ctx;

    ovs_mutex_lock(&group_ctx->lock);
    LIST_FOR_EACH (mask_ctx, list, &group_ctx->list) {
        doca_pipe_group_mask_ctx_dump(out,
                                      refmap_key_from_value(doca_pipe_group_mask_rfm, mask_ctx),
                                      mask_ctx,
                                      NULL);
        ds_chomp(out, ' ');
        ds_chomp(out, ',');
        ds_put_format(out, "\n");
    }
    ovs_mutex_unlock(&group_ctx->lock);
}

static void
doca_pipe_group_dump_cb(void *ctx_, void *key_, void *arg_)
{
    struct doca_pipe_group_ctx *group_ctx = ctx_;
    struct doca_pipe_group_key *group_key = key_;
    struct ds *out = arg_;

    doca_pipe_group_ctx_dump(out, group_key, group_ctx, NULL);
    ds_chomp(out, ' ');
    ds_chomp(out, ',');
    ds_put_format(out, "\n");
    doca_pipe_group_ctx_dump_masks(out, group_ctx);
}

static void
doca_pipe_group_dump(struct unixctl_conn *conn, int argc OVS_UNUSED,
                     const char *argv[] OVS_UNUSED, void *aux OVS_UNUSED)
{
    struct ds out = DS_EMPTY_INITIALIZER;

    refmap_for_each(doca_pipe_group_rfm, doca_pipe_group_dump_cb, &out);
    unixctl_command_reply(conn, ds_cstr(&out));
    ds_destroy(&out);
}
