/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2022-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#include <config.h>
#include <errno.h>

#include "batch.h"
#include "conntrack.h"
#include "conntrack-offload.h"
#include "conntrack-private.h"
#include "conntrack-tp.h"
#include "coverage.h"
#include "dp-packet.h"
#include "fatal-signal.h"
#include "netdev-offload.h"
#include "netdev-offload-doca.h"
#include "openvswitch/vlog.h"
#include "ovs-doca.h"
#include "ovs-rcu.h"
#include "timeval.h"
#include "uuid.h"

VLOG_DEFINE_THIS_MODULE(conntrack_offload);

COVERAGE_DEFINE_ERR(conntrack_offload_alloc_fail);

#define CT_OFFLOAD_DEFAULT_SIZE 250000
static unsigned int ct_offload_size = CT_OFFLOAD_DEFAULT_SIZE;

static bool ct_offload_ipv6_enabled = false;
static bool ct_offload_unidir_udp_enabled = false;

static struct ovs_doca_mempool *handles;

bool conntrack_offload_doca_ct_enabled;
bool conntrack_offload_doca_ct_ipv6_enabled;

struct ct_offload_handle *
conntrack_offload_get(struct conn *conn)
{
    return conn_master(conn)->offloads.coh;
}

unsigned int
conntrack_offload_get_insertion_tid(struct conn *conn)
{
    return conn_master(conn)->offloads.insertion_tid;
}

static uintptr_t
conntrack_offload_get_ctid_key(struct conn *conn)
{
    return (uintptr_t) (conn->master_conn ? conn->master_conn : conn);
}

static bool
conntrack_offload_fill_item_common(struct ct_flow_offload_item *item,
                                   struct conntrack *ct,
                                   struct conn *conn,
                                   int dir)
{
    item->ufid = conn->offloads.dir_info[dir].ufid;
    item->ct_match.odp_port = conn->offloads.dir_info[dir].port;
    item->ct = ct;
    if (ct != NULL) {
        item->dp = ct->dp;
    }
    item->ctid_key = conntrack_offload_get_ctid_key(conn);

    /* This field is set by the offload provider once offloading is completed. */
    item->offload_data = ct_dir_info_data_get(&conn->offloads.dir_info[dir]);

    return dir == CT_DIR_REP
           ? !!(conn->offloads.flags & CT_OFFLOAD_REP)
           : !!(conn->offloads.flags & CT_OFFLOAD_INIT);
}

void
conntrack_offload_free(struct conn *conn)
{
    struct ct_offload_handle *coh = conntrack_offload_get(conn);

    if (coh) {
        if (coh->lock.where) {
            ovs_spin_destroy(&coh->lock);
        }
        if (handles) {
            ovs_doca_mempool_free(handles, coh);
        } else {
            free(coh);
        }
        conn_master(conn)->offloads.coh = NULL;
    }
}

void
conntrack_offload_del_conn(struct conntrack *ct,
                           struct conn *conn,
                           bool flush)
    OVS_REQUIRES(conn->lock)
{
    struct conntrack_offload_class *offload_class;
    struct conn *conn_dir;
    int dir;

    if (!conntrack_offload_is_enabled()) {
        return;
    }

    if (conn_master(conn)->offloads.flags & CT_OFFLOAD_DELETED ||
        !(conn_master(conn)->offloads.flags & CT_OFFLOAD_REQUESTED)) {
        return;
    }

    offload_class = ovsrcu_get(struct conntrack_offload_class *,
                               &ct->offload_class);

    if (!offload_class) {
        /* The datapath is being destroyed. If the offload handle is set,
         * do not enqueue a conn-del: The offload resources will be freed
         * when flushing each of the port previously part of the datapath. */
        goto mark_deleted;
    }

    conn = conn_master(conn);

    for (dir = 0; dir < CT_DIR_NUM; dir ++) {
        if (conn->nat_conn &&
            conn->nat_conn->offloads.dir_info[dir].dp) {
            conn_dir = conn->nat_conn;
        } else {
            conn_dir = conn;
        }
        /* Set connection's status to terminated to indicate that the offload
         * of the connection is deleted, but should still bypass tcp seq
         * checking.
         */
        if (flush) {
            /* For flush, keep only bypass flag, so further TCP packets
             * will still bypass seq checking.
             */
            conn_dir->offloads.flags = CT_OFFLOAD_BYPASS_TCP_SEQ_CHK;
        } else {
            conn_dir->offloads.flags |= CT_OFFLOAD_SKIP_BOTH | CT_OFFLOAD_DELETED;
        }
        if (dir == CT_DIR_REP) {
            conn_dir->offloads.flags &= ~CT_OFFLOAD_REP;
        } else {
            conn_dir->offloads.flags &= ~CT_OFFLOAD_INIT;
        }
    }

    offload_class->conn_del(ct, conn);
    conntrack_stats_inc(ct, CT_DPIF_CONN_STATE_OFFLOAD_DEL_REQUESTED);
mark_deleted:
    if (flush) {
        conn_master(conn)->offloads.flags |= CT_OFFLOAD_BYPASS_TCP_SEQ_CHK;
    } else {
        conn_master(conn)->offloads.flags |= CT_OFFLOAD_DELETED;
    }
}

static void
conntrack_swap_conn_key(const struct conn_key *key,
                        struct conn_key *swapped)
{
    memcpy(swapped, key, sizeof *swapped);
    swapped->src = key->dst;
    swapped->dst = key->src;
}

static void
conntrack_offload_fill_item_add(struct ct_flow_offload_item *item,
                                struct conntrack *ct,
                                struct conn *conn,
                                int dir,
                                long long int now)
{
    /* nat_conn has opposite directions. */
    bool reply = !!conn->master_conn ^ dir;

    if (reply) {
        item->ct_match.key = conn->rev_key;
        conntrack_swap_conn_key(&conn->key, &item->nat.key);
    } else {
        item->ct_match.key = conn->key;
        conntrack_swap_conn_key(&conn->rev_key, &item->nat.key);
    }

    item->nat.mod_flags = 0;
    if (memcmp(&item->nat.key.src.addr, &item->ct_match.key.src.addr,
               sizeof item->nat.key.src)) {
        item->nat.mod_flags |= NAT_ACTION_SRC;
    }
    if (item->nat.key.src.port != item->ct_match.key.src.port) {
        item->nat.mod_flags |= NAT_ACTION_SRC_PORT;
    }
    if (memcmp(&item->nat.key.dst.addr, &item->ct_match.key.dst.addr,
               sizeof item->nat.key.dst)) {
        item->nat.mod_flags |= NAT_ACTION_DST;
    }
    if (item->nat.key.dst.port != item->ct_match.key.dst.port) {
        item->nat.mod_flags |= NAT_ACTION_DST_PORT;
    }

    conntrack_offload_fill_item_common(item, ct, conn, dir);
    item->ct_state = conn->offloads.dir_info[dir].pkt_ct_state;
    item->mark_key = conn->offloads.dir_info[dir].pkt_ct_mark;
    item->label_key = conn->offloads.dir_info[dir].pkt_ct_label;
    item->timestamp = now;
}

void
conntrack_offload_build_add_items(struct conntrack *ct, struct conn *conn,
                                  struct ct_flow_offload_item items[CT_DIR_NUM],
                                  long long int now_us)
{
    struct ct_offload_handle *coh = conntrack_offload_get(conn);

    ovs_assert(coh);
    for (int dir = 0; dir < CT_DIR_NUM; dir++) {
        if (conn->nat_conn &&
            conn->nat_conn->offloads.dir_info[dir].dp) {
            conn = conn->nat_conn;
        } else if (conn->master_conn &&
                   conn->master_conn->offloads.dir_info[dir].dp) {
            conn = conn->master_conn;
        }
        conntrack_offload_fill_item_add(&items[dir], ct, conn, dir, now_us);
        items[dir].offload_data = &coh->dir[dir];
        items[dir].ufid = conn->offloads.dir_info[dir].ufid;
    }
}

void
conntrack_offload_prepare(struct conntrack *ct, struct conn *conn,
                          struct dp_packet *packet, bool reply)
    OVS_REQUIRES(conn->lock)
{
    enum ct_offload_flag dir_flag = reply ? CT_OFFLOAD_REP : CT_OFFLOAD_INIT;
    int dir = reply ? CT_DIR_REP : CT_DIR_INIT;
    struct conn *actual;

    /* Always write offload info in the actual conn. */
    actual = conn_master(conn);

    if ((actual->key.nw_proto != IPPROTO_UDP && actual->key.nw_proto != IPPROTO_TCP) ||
        (actual->offloads.flags & CT_OFFLOAD_SKIP_BOTH) == CT_OFFLOAD_SKIP_BOTH) {
        return;
    }

    /* Not established UDP traffic offload is only allowed by unidirectional UDP offload */
    if (!(packet->md.ct_state & CS_ESTABLISHED)) {
        if (actual->key.nw_proto == IPPROTO_TCP) {
            return;
        }
        if (actual->key.nw_proto == IPPROTO_UDP &&
            !conntrack_offload_unidir_udp_is_enabled()) {
            return;
        }
    }

    if (actual->offloads.flags & dir_flag) {
        /* Write metadata only once per direction. */
        return;
    }

    actual->offloads.flags |= dir_flag;
    conn->offloads.flags |= dir_flag;
    conn->offloads.dir_info[dir].dp = ct->dp;
    conn->offloads.dir_info[dir].pkt_ct_state = packet->md.ct_state;
    conn->offloads.dir_info[dir].pkt_ct_mark = packet->md.ct_mark;
    conn->offloads.dir_info[dir].pkt_ct_label = packet->md.ct_label;
    conn->offloads.dir_info[dir].port = packet->md.orig_in_port;
}

void
conntrack_offload_add(struct conntrack *ct, struct batch *conns)
{
    struct conn_batch_md *md = (struct conn_batch_md *) conns->md;
    struct conntrack_offload_class *offload_class;
    struct conn *conn;

    offload_class = ovsrcu_get(struct conntrack_offload_class *,
                               &ct->offload_class);
    if (!offload_class || !offload_class->conns_add) {
        return;
    }

    /* Because of batching, some connections might already been deleted,
     * for example by TCP FIN. Filter those out.
     */
    BATCH_FOREACH_POP (conn, conns) {
        if (conn_master(conn)->offloads.flags & CT_OFFLOAD_DELETED) {
            continue;
        }
        batch_add(conns, conn);
    }

    if (batch_is_empty(conns)) {
        return;
    }

    BATCH_FOREACH (conn, conns) {
        md->tid = conn->offloads.insertion_tid;
        break;
    }

    conntrack_stats_add(ct, CT_DPIF_CONN_STATE_OFFLOAD_ADD_REQUESTED, batch_size(conns));
    offload_class->conns_add(ct, conns);
}

static bool
conntrack_offload_active(struct conntrack *ct,
                         struct conntrack_offload_class *offload_class,
                         struct conn *conn, long long int now, int *ret)
{
    struct ct_offload_handle *coh;
    uint8_t flags;

    conn = conn_master(conn);
    coh = conntrack_offload_get(conn);
    flags = conn->offloads.flags;

    if (flags & CT_OFFLOAD_DELETED) {
        *ret = ENODATA;
    } else if (!ct_offload_handle_active(coh, CT_OFFLOAD_BOTH)) {
        *ret = ENODATA;
    } else {
        *ret = offload_class->conn_active(ct, conn, now, conn->prev_query);
    }

    return (*ret == 0);
}

int
conn_hw_update(struct conntrack *ct,
               struct conntrack_offload_class *offload_class,
               struct conn *conn,
               long long now)
{
    int ret = 0;

    if (conntrack_offload_active(ct, offload_class, conn, now, &ret)) {
        conn_update_expiration(ct, conn, now);
    }

    conn->prev_query = now;
    return ret;
}

void
conntrack_set_offload_class(struct conntrack *ct,
                            struct conntrack_offload_class *cls)
{
    ovsrcu_set(&ct->offload_class, cls);
}

unsigned int
conntrack_offload_size(void)
{
    return ct_offload_size;
}

bool
conntrack_offload_is_enabled(void)
{
    return netdev_is_flow_api_enabled()
        && (conntrack_offload_size() > 0);
}

bool
conntrack_offload_ipv6_is_enabled(void)
{
    return conntrack_offload_is_enabled() && ct_offload_ipv6_enabled;
}

bool
conntrack_offload_unidir_udp_is_enabled(void)
{
    return conntrack_offload_is_enabled() && ct_offload_unidir_udp_enabled;
}

static void
destroy_handle_pool(void *aux OVS_UNUSED)
{
    if (handles) {
        ovs_doca_mempool_destroy(handles);
        handles = NULL;
    }
}

void
conntrack_offload_config(const struct smap *other_config)
{
    static struct ovsthread_once once = OVSTHREAD_ONCE_INITIALIZER;
    unsigned int req_ct_size;

    if (!ovsthread_once_start(&once)) {
        return;
    }

    req_ct_size = smap_get_uint(other_config, "hw-offload-ct-size",
                                CT_OFFLOAD_DEFAULT_SIZE);
    if (req_ct_size != ct_offload_size) {
        ct_offload_size = req_ct_size;
        if (conntrack_offload_is_enabled()) {
            VLOG_INFO("Conntrack offload size set to %u",
                      ct_offload_size);
        }
    }

    if (!conntrack_offload_is_enabled()) {
        VLOG_INFO("Conntrack offloading is disabled");
        goto out;
    }

    VLOG_INFO("Conntrack offloading is enabled");

    ct_offload_ipv6_enabled = smap_get_bool(other_config,
                                            "hw-offload-ct-ipv6-enabled",
                                            false);
    VLOG_INFO("Conntrack IPv6 offloading is %s",
              conntrack_offload_ipv6_is_enabled() ?
              "enabled" : "disabled");

    if (smap_get_bool(other_config, "doca-ct", true)) {
        conntrack_offload_doca_ct_enabled = true;
    }

    if (conntrack_offload_doca_ct_enabled &&
        smap_get_bool(other_config, "doca-ct-ipv6", false) &&
        conntrack_offload_ipv6_is_enabled()) {
        conntrack_offload_doca_ct_ipv6_enabled = true;
    }

    ct_offload_unidir_udp_enabled = smap_get_bool(other_config,
                                                 "hw-offload-ct-unidir-udp-enabled",
                                                 false);
    if (ct_offload_unidir_udp_enabled && !conntrack_offload_doca_ct_enabled) {
         VLOG_WARN("Offloading unidirectional UDP is currently supported with DOCA-CT,"
                   " setting as disabled");
         ct_offload_unidir_udp_enabled = false;
    }

    VLOG_INFO("Conntrack unidirectional UDP offloading is %s",
              conntrack_offload_unidir_udp_is_enabled() ?
              "enabled" : "disabled");

    if (ovs_doca_enabled()) {
        handles = ovs_doca_mempool_create(ct_offload_size, sizeof(struct ct_offload_handle));
        ovs_assert(handles);
        fatal_signal_add_hook(destroy_handle_pool, NULL, NULL, true);
    }

out:
    ovsthread_once_done(&once);
}

static bool
conntrack_offload_on_netdev(struct conn *conn, struct netdev *netdev)
{
    struct ct_offload_handle *coh;

    if (!netdev || !conn) {
        return false;
    }

    coh = conn->offloads.coh;
    return coh &&
        (coh->dir[CT_DIR_INIT].netdev == netdev ||
         coh->dir[CT_DIR_REP].netdev == netdev);
}

void
conntrack_offload_netdev_flush(struct conntrack *ct, struct netdev *netdev)
{
    struct conn *conn;

    CMAP_FOR_EACH (conn, cm_node, &ct->conns) {
        if (conntrack_offload_on_netdev(conn, netdev)) {
            conn_lock(conn);
            conntrack_offload_del_conn(ct, conn, true);
            conn_unlock(conn);
        }
    }
}

void
conntrack_offload_dump(struct ds *ds, struct conn *conn, int dir,
                       uint32_t ct_match_zone_id,
                       uint32_t ct_action_label_id,
                       uint32_t ct_miss_ctx_id)
{
    struct ct_flow_offload_item items[CT_DIR_NUM];
    struct ct_flow_offload_item *ct_offload;
    struct conn_key *nat_key;
    struct ct_match *c;

    conntrack_offload_build_add_items(NULL, conn, items, 0);
    ct_offload = &items[dir];
    c = &ct_offload->ct_match;
    nat_key = &ct_offload->nat.key;

    ds_put_format(ds, "zone %u proto %u ",
                  c->key.zone, c->key.nw_proto);
    if (c->key.dl_type == htons(ETH_TYPE_IP)) {
        ds_put_format(ds, IP_FMT":%u->"IP_FMT":%u",
                      IP_ARGS(c->key.src.addr.ipv4), ntohs(c->key.src.port),
                      IP_ARGS(c->key.dst.addr.ipv4), ntohs(c->key.dst.port));
    } else {
        char ip6_s[INET6_ADDRSTRLEN], ip6_d[INET6_ADDRSTRLEN];

        inet_ntop(AF_INET6, &c->key.src.addr.ipv6, ip6_s, sizeof ip6_s);
        inet_ntop(AF_INET6, &c->key.dst.addr.ipv6, ip6_d, sizeof ip6_d);
        ds_put_format(ds, "%s:%u->%s:%u",
                      ip6_s, ntohs(c->key.src.port),
                      ip6_d, ntohs(c->key.dst.port));
    }

    if (ct_offload->nat.mod_flags) {
        if (c->key.dl_type == htons(ETH_TYPE_IP)) {
            ds_put_format(ds, " nat "IP_FMT":%u->"IP_FMT":%u",
                          IP_ARGS(nat_key->src.addr.ipv4), ntohs(nat_key->src.port),
                          IP_ARGS(nat_key->dst.addr.ipv4), ntohs(nat_key->dst.port));
        } else {
            char ip6_s[INET6_ADDRSTRLEN], ip6_d[INET6_ADDRSTRLEN];

            inet_ntop(AF_INET6, &nat_key->src.addr.ipv6, ip6_s, sizeof ip6_s);
            inet_ntop(AF_INET6, &nat_key->dst.addr.ipv6, ip6_d, sizeof ip6_d);
            ds_put_format(ds, " nat %s:%u->%s:%u",
                          ip6_s, ntohs(nat_key->src.port),
                          ip6_d, ntohs(nat_key->dst.port));
        }
    }

    ds_put_format(ds, " zone_map=%d", ct_match_zone_id);

    /* CT MARK */
    ds_put_format(ds, " mark=0x%08x", ct_offload->mark_key);

    /* CT LABEL */
    ds_put_format(ds, " label=0x%08x", ct_action_label_id);

    /* CT STATE */
    ds_put_format(ds, " state=0x%02x", ct_offload->ct_state);

    /* CT CTX */
    ds_put_format(ds, " ctx=0x%02x", ct_miss_ctx_id);
}

void
conntrack_offload_get_ports(struct conn *conn, odp_port_t ports[CT_DIR_NUM])
{
    ovs_assert(!conn->master_conn);
    for (int dir = 0; dir < CT_DIR_NUM; dir++) {
        struct conn *conn_dir = conn;

        if (conn->nat_conn &&
            conn->nat_conn->offloads.dir_info[dir].dp) {
            conn_dir = conn->nat_conn;
        }

        ports[dir] = conn_dir->offloads.dir_info[dir].port;
    }
}

bool
conntrack_offload_valid(struct conntrack *ct, struct conn *conn, struct dp_packet *packet,
                        bool reply, bool *unidir_update)
    OVS_REQUIRES(conn->lock)
{
    bool offload_unidir_udp = conntrack_offload_unidir_udp_is_enabled();
    struct ct_offload_handle *coh;
    bool took_second_ref = false;
    uint8_t prev_flags;

    if (conn->key.dl_type == htons(ETH_TYPE_IPV6)) {
        /* Unidir IPv6 should be supported by DOCA-CT */
        offload_unidir_udp &= conntrack_offload_doca_ct_ipv6_enabled;
    }

    if (!conntrack_offload_is_enabled() ||
        (!(packet->md.ct_state & CS_ESTABLISHED) &&
         (!offload_unidir_udp || conn->key.nw_proto != IPPROTO_UDP)) ||
        !l4_protos[conn->key.nw_proto]->offload_valid) {
        return false;
    }

    if (!l4_protos[conn->key.nw_proto]->offload_valid(conn)) {
        return false;
    }

    if ((conn->offloads.flags & CT_OFFLOAD_SKIP_BOTH) == CT_OFFLOAD_SKIP_BOTH) {
        return false;
    }

    if (conn->key.dl_type == htons(ETH_TYPE_IPV6) &&
        !conntrack_offload_ipv6_is_enabled()) {
        /* If conn type is IPv6 and IPv6 offload is disabled,
         * do not generate requests. */
        return false;
    }

    /* CT doesn't handle alg */
    if (conn->alg || conn->alg_related) {
        return false;
    }

    conntrack_offload_prepare(ct, conn, packet, reply);

    if (conn->key.nw_proto != IPPROTO_UDP || !offload_unidir_udp) {
        if ((conn->offloads.flags & CT_OFFLOAD_BOTH) != CT_OFFLOAD_BOTH) {
            return false;
        }
    } else {
        /* For unidirectional offload, we need either only INIT side
         * prepared, or both at once.
         *
         * Possible cases:
         *    OFFLOAD_INIT    SKIP_INIT      OFFLOAD_REP   SKIP_REP     validity
         *          -             -               -           -         false
         *          x             x               x           x         false
         *          x             x               -           -         false
         *          -             -               x           -         false
         *          -             -               x           x         false
         *          x             -               -           -         true
         *          x             -               x           -         true
         *          x             x               x           -         true
         */
        if (!(conn->offloads.flags & CT_OFFLOAD_INIT)) {
            return false;
        }
        /* We already offloaded the INIT side. */
        if ((conn->offloads.flags & CT_OFFLOAD_BOTH) == CT_OFFLOAD_INIT &&
             (conn->offloads.flags & CT_OFFLOAD_SKIP_INIT)) {
            return false;
        }
    }

    if (!conntrack_conn_ref(conn)) {
        /* No point in offloading a connection already being deleted. */
        return false;
    }

    prev_flags = conn->offloads.flags;
    *unidir_update = false;

    if (offload_unidir_udp) {
        if (conn->offloads.flags & CT_OFFLOAD_INIT) {
            conn->offloads.flags |= CT_OFFLOAD_SKIP_INIT;
            conn->offloads.unidir = true;
        }

        if (conn->offloads.flags & CT_OFFLOAD_REP) {
            conn->offloads.flags |= CT_OFFLOAD_SKIP_REP;
            conn->offloads.unidir = false;
        }
    } else {
        conn->offloads.flags |= CT_OFFLOAD_SKIP_BOTH;
        conn->offloads.unidir = false;
    }

    if (conn->key.nw_proto == IPPROTO_TCP) {
        /* Take a reference per directions. */
        ignore(conntrack_conn_ref(conn));
        took_second_ref = true;
    } else if (conn->key.nw_proto == IPPROTO_UDP) {
        uint8_t flags_delta = (prev_flags ^ conn->offloads.flags) & CT_OFFLOAD_SKIP_BOTH;

        if (!offload_unidir_udp || flags_delta == CT_OFFLOAD_SKIP_BOTH) {
            /* Take a reference per directions. */
            ignore(conntrack_conn_ref(conn));
            took_second_ref = true;
        } else if (flags_delta == CT_OFFLOAD_SKIP_REP) {
            *unidir_update = true;
        }
    } else {
        OVS_NOT_REACHED();
    }

    if (!conn->offloads.coh) {
        if (handles) {
            if (ovs_doca_mempool_alloc(handles, (void **) &coh)) {
                COVERAGE_INC(conntrack_offload_alloc_fail);
                conntrack_conn_unref(conn);
                if (took_second_ref) {
                    conntrack_conn_unref(conn);
                }
                return false;
            }
            coh->valid = false;
#ifdef DPDK_NETDEV
            coh->dir[CT_DIR_INIT].conn_item.valid = false;
#endif
        } else {
            coh = xzalloc(sizeof *coh);
        }
        conn->offloads.coh = coh;
        conn->offloads.insertion_tid = netdev_offload_thread_id();
    }
    conn->offloads.flags |= CT_OFFLOAD_REQUESTED;

    return true;
}

bool
conntrack_offload_is_unidir(struct conn *conn)
{
    return conn->offloads.unidir;
}
