/*
 * Copyright (c) 2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <config.h>
#undef NDEBUG
#include <assert.h>
#include <getopt.h>
#include <string.h>
#include <sys/time.h>

#include "command-line.h"
#include "openvswitch/vlog.h"
#include "openvswitch/util.h"
#include "ovs-thread.h"
#include "ovs-rcu.h"
#include "ovs-numa.h"
#include "ovstest.h"
#include "random.h"
#include "timeval.h"
#include "util.h"

#include "concurrent-array.h"

static uint32_t
get_idx(int i)
{
    return (i * CONCURRENT_ARRAY_SLAB_SIZE / 3);
}

static void
run_tests(struct ovs_cmdl_context *ctx OVS_UNUSED)
{
    enum { N = 100, };
    struct concurrent_array *c;
    int values[N];

    c = concurrent_array_create();

    for (int i = 0; i < N; i++) {
        values[i] = i;
        ovs_assert(concurrent_array_get(c, get_idx(i)) == NULL);
        concurrent_array_set(c, get_idx(i), &values[i]);
    }

    for (int i = 0; i < N; i++) {
        ovs_assert(concurrent_array_get(c, get_idx(i)) == &values[i]);
    }

    concurrent_array_destroy(c);
}

static const struct ovs_cmdl_command commands[] = {
    {"check", NULL, 0, 0, run_tests, OVS_RO},
    {NULL, NULL, 0, 0, NULL, OVS_RO},
};

static void
concurrent_array_test_main(int argc, char *argv[])
{
    struct ovs_cmdl_context ctx = {
        .argc = argc - optind,
        .argv = argv + optind,
    };

    vlog_set_levels(NULL, VLF_ANY_DESTINATION, VLL_OFF);

    set_program_name(argv[0]);
    ovs_cmdl_run_command(&ctx, commands);
}

OVSTEST_REGISTER("test-concurrent-array", concurrent_array_test_main);
