/*
 * Copyright (c) 2023-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#ifndef NETDEV_OFFLOAD_EXT_H
#define NETDEV_OFFLOAD_EXT_H 1

#include "batch.h"
#include "conntrack-offload.h"
#include "dp-packet.h"
#include "openvswitch/netdev.h"
#include "openvswitch/types.h"

#ifdef  __cplusplus
extern "C" {
#endif

/* Netdev-offload thread pool
 * ==========================
 *
 * Netdev-offload as a module offers per-thread segmentation
 * of operations, as long as participating threads operate
 * within predefined parameters.
 *
 * A thread ID dedicated to a single thread is accessible through
 * calling 'netdev_offload_thread_id()'. It is allocated explicitly
 * with 'netdev_offload_thread_init()' or the first time it is read.
 * To release a thread ID, use 'netdev_offload_thread_uninit()'.
 *
 * IDs are allocated from a pre-defined range and a limited number of
 * threads can participate concurrently in netdev-offload operations.
 *
 * The first ID 0 is reserved to be used by either very early threads
 * (main thread) that might need special access to initialize mutable
 * data, or other threads that would do only reads and no modifications
 * to mutable data (e.g. reading offload statistics, states).
 * If more than one such thread needs to write mutable data using this
 * special ID, external synchronization is required.
 * The main thread ID is the first in the defined range so that it
 * remains a compilation-time constant that is not impacted by any potential
 * sequence of execution of dynamic thread allocations. Some offload
 * providers have stability requirement for this control thread.
 *
 * The other IDs are dynamically allocated on-demand from participating
 * threads. An internal pool of IDs is created that serve as an allocator.
 * Modifying the 'n-offload-threads' configuration creates additional
 * threads that will take their IDs from this dynamic pool.
 *
 * Control thread
 *  |
 *  |  Dynamic offload threads
 *  |  |
 *  v  v
 * [0][1,            ...          , N-2]
 * <-----  MAX_OFFLOAD_THREAD_NB  ----->
 *
 * Some functions such as 'netdev_offload_get_stats' require
 * 'MAX_OFFLOAD_THREAD_NB' number of output objects.
 */

#define NETDEV_OFFLOAD_THREAD_MAIN 0
/* Absolute limit number of threads that can participate concurrently
 * in netdev-offload: Max dynamic offload threads, plus one management thread.
 * The management thread uses the special static ID defined above. */
#if DPDK_NETDEV
/* If OVS is built with DPDK support, the polling threads are allowed to
 * participate in netdev-offload. */
#define MAX_OFFLOAD_THREAD_NB (10 + 1 + RTE_MAX_LCORE)
#else
#define MAX_OFFLOAD_THREAD_NB (10 + 1)
#endif
#define DEFAULT_OFFLOAD_THREAD_NB 1
#define MAX_OFFLOAD_METERS 4

/* Attributes for offload UFID, generated by uuid_set_bits_v4(uuid, attr). */
enum offload_uuid_attr {
    UUID_ATTR_0,	/* Reserved for non-offloads. */
    UUID_ATTR_1,
    UUID_ATTR_2,
    UUID_ATTR_3,
    UUID_ATTR_4,
};

/* Generic statistics of the offload provider of a netdev.
 * It is not related to the 'offload_count' or 'pending_count'
 * stored within the 'netdev_hw_info' and managed entirely
 * by the upcall handler. */
struct netdev_offload_stats {
    uint64_t n_inserted;
    uint64_t n_flows;
    uint64_t n_conns;
    uint64_t n_unidir_conns;
};

#define INVALID_FLOW_MARK 0
#define HAIRPIN_FLOW_MARK 1
#define SAMPLE_FLOW_MARK 2
#define NONSAMPLE_FLOW_MARK 3
#define MIN_FLOW_MARK 4
#define MAX_FLOW_MARK (UINT32_MAX - 1)
#define NB_FLOW_MARK (MAX_FLOW_MARK - MIN_FLOW_MARK + 1)

static inline void
netdev_offload_stats_add(struct netdev_offload_stats *dst,
                         struct netdev_offload_stats src)
{
    dst->n_inserted = ovs_u64_safeadd(dst->n_inserted, src.n_inserted);
    dst->n_flows = ovs_u64_safeadd(dst->n_flows, src.n_flows);
    dst->n_conns = ovs_u64_safeadd(dst->n_conns, src.n_conns);
    dst->n_unidir_conns = ovs_u64_safeadd(dst->n_unidir_conns, src.n_unidir_conns);
}

int netdev_offload_get_stats(struct netdev *netdev,
                             struct netdev_offload_stats *stats);
bool netdev_is_ct_labels_mapping_enabled(void);
bool netdev_is_zone_tables_disabled(void);

/* Upkeep a single netdev, if supported.
 * If 'quiescing' is true, the calling thread is signaling
 * that this is the last upkeep call before starting to wait
 * on more work.
 */
void netdev_offload_upkeep(struct netdev *netdev, bool quiescing);
/* Upkeep all netdev-offload ports. */
void netdev_ports_upkeep(bool quiescing);

/* Find a registered netdev without taking a reference on it. */
struct netdev *netdev_ports_get_short(odp_port_t port_no, const char *dpif_type);

int netdev_offload_get_stats(struct netdev *netdev,
                             struct netdev_offload_stats *stats);

/* Make 'netdev' visible for 'netdev_ports_get'.
 * If set to 'false', this netdev will not be returned on lookup.
 */
void netdev_ports_set_visible(struct netdev *netdev, bool visible);

int
netdev_ct_counter_query(struct netdev *netdev,
                        uintptr_t counter,
                        long long now,
                        long long prev_now,
                        struct dpif_flow_stats *stats);

void netdev_set_flow_api_enabled_ext(const struct smap *ovs_other_config);

int netdev_packet_hw_hash(struct netdev *netdev,
                          struct dp_packet *packet,
                          uint32_t seed,
                          uint32_t *hash);

int netdev_packet_hw_entropy(struct netdev *netdev,
                             struct dp_packet *packet,
                             uint16_t *entropy);

extern bool netdev_offload_ct_on_ct_nat;

#ifdef  __cplusplus
}
#endif

#endif /* netdev-offload-ext.h */
