/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2008-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#ifndef DPIF_DOCA_H
#define DPIF_DOCA_H 1

#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include "ccmap.h"
#include "dpif.h"
#include "dp-packet.h"
#include "dpif-doca-ext.h"
#include "fat-rwlock.h"
#include "mov-avg.h"
#include "netdev-offload.h"
#include "openvswitch/types.h"
#include "packets.h"
#include "pvector.h"

#ifdef __cplusplus
extern "C" {
#endif

bool dpif_is_doca(const struct dpif *);

#define NR_QUEUE 1
#define NR_PMD_THREADS 1

enum dp_offload_type {
    DP_OFFLOAD_FLOW,
    DP_OFFLOAD_CONN,
    DP_OFFLOAD_FLUSH,
    DP_OFFLOAD_STATS_CLEAR,

    /* New entries should be added before this. */
    DP_OFFLOAD_TYPE_NUM
};

enum {
    DP_NETDEV_FLOW_OFFLOAD_OP_NONE,
    DP_NETDEV_FLOW_OFFLOAD_OP_ADD,
    DP_NETDEV_FLOW_OFFLOAD_OP_MOD,
    DP_NETDEV_FLOW_OFFLOAD_OP_DEL,
};

/* Data structure to keep packet order till fastpath processing. */
struct dp_packet_flow_map {
    struct dp_packet *packet;
    struct dp_doca_flow *flow;
    uint16_t tcp_flags;
};

struct dpif_doca_dp_offload_flow_item {
    struct dp_doca_flow *flow;
    int op;
    struct match match;
    struct nlattr *actions;
    size_t actions_len;
    odp_port_t orig_in_port; /* Originating in_port for tnl flows. */
    uintptr_t ct_counter_key;
    uintptr_t flows_counter_key;
};

struct dp_offload_conn_item {
    int op;
    struct conntrack *ct;
    struct conn *conn;
    bool unidir_update;
};

struct dpif_doca_dp_offload_flush_item {
    struct netdev *netdev;
    struct ovs_refcount *count;
    struct ovs_mutex *mutex;
    pthread_cond_t *cond;
};

union dp_offload_thread_data {
    struct dpif_doca_dp_offload_flow_item flow;
    struct dp_offload_conn_item conn;
    struct dpif_doca_dp_offload_flush_item flush;
};

struct dp_offload_thread_item {
    struct mpsc_queue_node node;
    struct ovsrcu_gc_node gc_node;
    enum dp_offload_type type;
    long long int timestamp;
    struct dp_doca *dp;
    union dp_offload_thread_data data[0];
};

struct dp_offload_queue_metrics {
    struct histogram wait_time;
    struct histogram service_time;
    struct histogram sojourn_time;
};

struct dp_offload_thread {
    PADDED_MEMBERS(CACHE_LINE_SIZE,
        struct mpsc_queue offload_queue;
        bool high_latency_event;
        atomic_bool active;
        atomic_uint64_t enqueued_ct_add;
        atomic_uint64_t enqueued_offload;
        struct cmap megaflow_to_mark;
        struct cmap mark_to_flow;
        struct ovs_mutex mark_to_flow_lock;
        struct mov_avg_cma cma;
        struct mov_avg_ema ema;
        struct histogram latency;
        struct dp_offload_queue_metrics queue_metrics[DP_OFFLOAD_TYPE_NUM];
        struct ccmap mark_ref;
    );
};

enum txq_req_mode {
    TXQ_REQ_MODE_THREAD,
    TXQ_REQ_MODE_HASH,
};

enum txq_mode {
    TXQ_MODE_STATIC,
    TXQ_MODE_XPS,
    TXQ_MODE_XPS_HASH,
};

/* A port in a netdev-based datapath. */
struct dp_doca_port {
    odp_port_t port_no;
    enum txq_mode txq_mode;     /* static, XPS, XPS_HASH. */
    bool need_reconfigure;      /* True if we should reconfigure netdev. */
    struct netdev *netdev;
    struct hmap_node node;      /* Node in dp_doca's 'ports'. */
    struct netdev_saved_flags *sf;
    struct dp_doca_rxq *rxqs;
    unsigned n_rxq;             /* Number of elements in 'rxqs' */
    unsigned *txq_used;         /* Number of threads that use each tx queue. */
    struct ovs_mutex txq_used_mutex;
    bool emc_enabled;           /* If true EMC will be used. */
    char *type;                 /* Port type as requested by user. */
    char *rxq_affinity_list;    /* Requested affinity of rx queues. */
    enum txq_req_mode txq_requested_mode;
    bool disabled;
    bool offload_disabled;
};

struct dp_meter_band {
    uint32_t rate;
    uint32_t burst_size;
    atomic_uint64_t bucket;          /* In 1/1000 packets for PKTPS,
                                      * or in bits for KBPS. */
    atomic_uint64_t packet_count;
    atomic_uint64_t byte_count;
};

struct dp_meter {
    struct cmap_node node;
    struct ovs_mutex lock;
    uint32_t id;
    uint16_t flags;
    uint16_t n_bands;
    uint32_t max_delta_t;
    atomic_uint64_t used;  /* Time of a last use in milliseconds. */
    atomic_uint64_t packet_count;
    atomic_uint64_t byte_count;
    struct dp_meter_band bands[];
};

struct pmd_auto_lb {
    bool do_dry_run;
    bool recheck_config;
    bool is_enabled;            /* Current status of Auto load balancing. */
    uint64_t rebalance_intvl;
    uint64_t rebalance_poll_timer;
    uint8_t rebalance_improve_thresh;
    atomic_uint8_t rebalance_load_thresh;
};

enum sched_assignment_type {
    SCHED_ROUNDROBIN,
    SCHED_CYCLES, /* Default.*/
    SCHED_GROUP
};

struct dpcls {
    struct cmap_node node;      /* Within dp_doca_pmd_thread.classifiers */
    odp_port_t in_port;
    struct cmap subtables_map;
    struct pvector subtables;
};

/* Datapath based on the network device interface from netdev.h.
 *
 *
 * Thread-safety
 * =============
 *
 * Some members, marked 'const', are immutable.  Accessing other members
 * requires synchronization, as noted in more detail below.
 *
 * Acquisition order is, from outermost to innermost:
 *
 *    dp_doca_mutex (global)
 *    port_rwlock
 *    bond_mutex
 *    non_pmd_mutex
 */
struct dp_doca {
    const struct dpif_class *const class;
    const char *const name;
    struct ovs_refcount ref_cnt;
    atomic_flag destroyed;

    /* Ports.
     *
     * Any lookup into 'ports' or any access to the dp_doca_ports found
     * through 'ports' requires taking 'port_rwlock'. */
    struct ovs_rwlock port_rwlock;
    struct hmap ports;
    struct seq *port_seq;       /* Incremented whenever a port changes. */

    /* The time that a packet can wait in output batch for sending. */
    atomic_uint32_t tx_flush_interval;

    /* Meters. */
    struct ovs_mutex meters_lock;
    struct cmap meters OVS_GUARDED;

    /* Probability of EMC insertions is a factor of 'emc_insert_min'.*/
    atomic_uint32_t emc_insert_min;
    /* Enable collection of PMD performance metrics. */
    atomic_bool pmd_perf_metrics;
    /* Default max load based sleep request. */
    uint64_t pmd_max_sleep_default;
    /* Register the PMD as quiescent when idle. */
    atomic_bool pmd_quiet_idle;
    /* Enable the SMC cache from ovsdb config */
    atomic_bool smc_enable_db;

    /* Protects access to ofproto-dpif-upcall interface during revalidator
     * thread synchronization. */
    struct fat_rwlock upcall_rwlock;
    upcall_callback *upcall_cb;  /* Callback function for executing upcalls. */
    void *upcall_aux;

    /* Callback function for notifying the purging of dp flows (during
     * reseting pmd deletion). */
    dp_purge_callback *dp_purge_cb;
    void *dp_purge_aux;

    /* Stores all 'struct dp_doca_pmd_thread's. */
    struct cmap poll_threads;
    /* id pool for per thread static_tx_qid. */
    struct id_pool *tx_qid_pool;
    struct ovs_mutex tx_qid_pool_mutex;
    /* Rxq to pmd assignment type. */
    enum sched_assignment_type pmd_rxq_assign_type;
    bool pmd_iso;

    /* Protects the access of the 'struct dp_doca_pmd_thread'
     * instance for non-pmd thread. */
    struct ovs_mutex non_pmd_mutex;

    /* Each pmd thread will store its pointer to
     * 'struct dp_doca_pmd_thread' in 'per_pmd_key'. */
    ovsthread_key_t per_pmd_key;

    struct seq *reconfigure_seq;
    uint64_t last_reconfigure_seq;

    /* Cpu mask for pin of pmd threads. */
    char *pmd_cmask;
    char *req_pmd_cmask;
    bool pmd_paused;

    /* PMD max load based sleep request user string. */
    char *max_sleep_list;

    uint64_t last_tnl_conf_seq;

    struct conntrack *conntrack;
    struct pmd_auto_lb pmd_alb;

    /* Bonds. */
    struct ovs_mutex bond_mutex; /* Protects updates of 'tx_bonds'. */
    struct cmap tx_bonds; /* Contains 'struct tx_bond'. */
};

extern struct dp_offload_thread dp_offload_threads[MAX_OFFLOAD_THREAD_NB];

struct dp_doca *get_dp_doca(const struct dpif *dpif);

/* Time in microseconds to try RCU quiescing. */
#define PMD_RCU_QUIESCE_INTERVAL 10000LL

void
dp_doca_get_mega_ufid(const struct match *match, ovs_u128 *mega_ufid);

uint32_t
dpif_doca_megaflow_to_mark_find(const ovs_u128 *mega_ufid);

struct dp_doca_actions *
dp_doca_flow_get_actions(const struct dp_doca_flow *flow);

#define DP_NETDEV_OFFLOAD_BACKOFF_MIN 1
#define DP_NETDEV_OFFLOAD_BACKOFF_MAX 64
#define DP_NETDEV_OFFLOAD_QUIESCE_INTERVAL_US (100 * 1000) /* 100 ms */

void
dpif_doca_dp_offload_flush(struct dp_offload_thread_item *item);

void
dp_doca_free_offload(struct dp_offload_thread_item *offload);

int
dp_doca_flow_offload_put(struct dp_offload_thread_item *item);

int
dp_doca_flow_offload_del(struct dp_offload_thread_item *item);

struct dp_offload_thread_item *
dp_doca_alloc_flow_offload(struct dp_doca *dp,
                           struct dp_doca_flow *flow,
                           int op, long long now);

void
dpif_doca_dp_offload_flow(struct dp_offload_thread_item *item);

void
dp_doca_offload_init(void);

void
dp_doca_port_rdlock_at(struct dp_doca *dp, unsigned long long int limit_ms,
                       const char *where)
    OVS_ACQ_RDLOCK(dp->port_rwlock);

#define dp_doca_port_rdlock(dp) \
    dp_doca_port_rdlock_at(dp, 1000, OVS_SOURCE_LOCATOR)

#define dp_doca_port_rdlock_limit(dp, limit_ms) \
    dp_doca_port_rdlock_at(dp, limit_ms, OVS_SOURCE_LOCATOR)

void
dp_doca_pmd_pause(struct dp_doca *dp);

void
dp_doca_pmd_resume(struct dp_doca *dp);

void
dpif_doca_log_all_pmd_sleeps(struct dp_doca *dp);

bool
dpif_doca_set_all_pmd_max_sleeps(struct dp_doca *dp, const struct smap *config);

#ifdef __cplusplus
}
#endif

#endif /* dpif-doca.h */
