/*
 * Copyright (c) 2023-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <config.h>
#include <ctype.h>
#include <errno.h>
#include <string.h>
#include <sys/types.h>
/* From netinet/in.h:24:2: error:
 * "Must include <sys/types.h> before <netinet/in.h> for FreeBSD support"
 */
#include <netinet/in.h>

#include "openvswitch/ofpbuf.h"
#include "openvswitch/ofp-port.h"
#include "dpctl.h"
#include "dpif.h"
#include "skiplist.h"
#include "util.h"

/* Extract the keys listed in 'sort_list' into a ds.
 * i.e. if 'sort_list' == 'in_port,recirc_id' and 'flow_desc' contains
 * 'recirc_id(0),in_port(3),eth_type(0x0800), used:never, actions:1'
 * 'vlist' will return with '3,0'.
 */
static void
dpctl_sorted_flow_key_extract(const char *sort_list, const char *flow_desc,
                              struct ds *vlist)
{
    char *sort_cpy = xstrdup(sort_list);
    char *key, value[100];
    char *savep;

    ds_init(vlist);
    for (key = strtok_r(sort_cpy, ",", &savep); key != NULL;
         key = strtok_r(NULL, ",", &savep)) {
        int pcount = 0;
        int num, mask;
        size_t i, d;
        char *s;

        /* ovs_scan does not deal well with parenthesis pairs, so we must
         * scan manually.
         *
         * Extract 'key(%s)' into value.
         */

        s = strstr(flow_desc, key);
        if (s == NULL) {
            continue;
        }
        i = strlen(key);
        if (s[i] != '(') {
            continue;
        }
        for (i += 1, d = 0, pcount = 1;
             s[i] && pcount != 0 && d < sizeof(value) - 1; i++) {
            if (s[i] == '(') {
                pcount += 1;
            } else if (s[i] == ')') {
                pcount -= 1;
            }
            if (pcount > 0) {
                value[d++] = s[i];
            }
        }
        value[d] = '\0';

        /* Attempt to parse some possibly masked values. */
        num = 0;
        if (ovs_scan(value, "0x%x/0x%x", &num, &mask) ||
            ovs_scan(value, "0/0x%x", &mask) ||
            ovs_scan(value, "%d/%d", &num, &mask)) {
            num &= mask;
            /* Writing the masked value in decimal with preceding
             * zeroes makes it sortable using ascii comparison.
             */
            snprintf(value, sizeof value, "%010d", num);
        }

        /* Append the value to the value list. */
        if (vlist->length > 0) {
            ds_put_cstr(vlist, ",");
        }
        ds_put_cstr(vlist, value);
    }
    free(sort_cpy);
}

int
dpctl_sorted_flow_cmp(const void *a, const void *b,
                 const void *conf OVS_UNUSED)
{
    const struct sorted_flow_node *node1 = a;
    const struct sorted_flow_node *node2 = b;

    return strcmp(node1->key, node2->key);
}

void
dpctl_sorted_flow_free(void *_node)
{
    struct sorted_flow_node *node = _node;

    if (node != NULL) {
        free(node->key);
        free(node->flow_desc);
        free(node);
    }
}

void
dpctl_sorted_flow_insert(const char *sort_list,
                         struct skiplist *sl,
                         int pmd_id,
                         const char *flow_desc)
{
    struct ds vlist = DS_EMPTY_INITIALIZER;
    struct sorted_flow_node *node;

    node = xzalloc(sizeof *node);
    dpctl_sorted_flow_key_extract(sort_list, flow_desc, &vlist);
    node->key = xstrdup(ds_cstr(&vlist));
    node->flow_desc = xstrdup(flow_desc);
    node->pmd_id = pmd_id;

    skiplist_insert(sl, node);

    ds_destroy(&vlist);
}

int
dpctl_validate_sort_list(const char *s)
{
    size_t i;

    /* Only accept possible key descriptions: [a-z,_]+
     * Later parsing expects keys to be separated by commas. */
    for (i = 0; s[i] != '\0'; i++) {
        if (!(islower(s[i]) || isdigit(s[i]) ||
              s[i] == ',' || s[i] == '_')) {
            return EINVAL;
        }
    }
    return 0;
}

int
dpctl_offload_stats_clear(int argc, const char *argv[],
                          struct dpctl_params *dpctl_p)
{
    struct dpif *dpif;
    int error;

    error = opt_dpif_open(argc, argv, dpctl_p, 2, &dpif);
    if (error) {
        return error;
    }

    error = dpif_offload_stats_clear(dpif);
    if (error) {
        dpctl_error(dpctl_p, error, "clearing offload statistics");
    } else {
        dpctl_print(dpctl_p, "offload statistics cleared");
    }

    dpif_close(dpif);
    return error;
}

