#!/usr/bin/env bash

[ -f /etc/os-release ] && . /etc/os-release

OVS_SRC_DIR="$(readlink -f $(dirname "$0"))"
DPDK_SRC_DIR=""
DOCA_SRC_DIR=""
NV_HWS_SRC_DIR=""
RDMA_CORE_SRC_DIR=""
LOGFILE="$OVS_SRC_DIR/build.log"
ASAN=0
ASAN_DPDK=0
SPARSE=0
DRY_RUN=0
VERBOSE=0
REBUILD=0
RECOMPILE=0
REBUILD_DOCA=0
REBUILD_NV_HWS=0
REBUILD_RDMA_CORE=0
COMMON_CFLAGS="-g3 -fno-omit-frame-pointer"
INFO_CFLAGS="-O2 $COMMON_CFLAGS"
DEBUG_CFLAGS="-O0 $COMMON_CFLAGS"
CFLAGS=""
CC1=""
PKG_CONFIG_PATH=""
LD_LIBRARY_PATH=""
INSTALL_OVS=0
NPROC=$(nproc)
BUILD_RPM=0
BUILD_DEB=0
ID=${ID:-"unknown"}

DPDK_SRC_ARG=""
DPDK_INSTALL_ARG=""
DOCA_SRC_ARG=""
DOCA_INSTALL_ARG=""
NV_HWS_SRC_ARG=""
NV_HWS_INSTALL_ARG=""
RDMA_CORE_SRC_ARG=""
RDMA_CORE_INSTALL_ARG=""

DPDK_PREFIX='$(readlink -f "${DPDK_SRC_DIR}/../dpdk-install")'
DOCA_PREFIX='$(readlink -f "${DOCA_SRC_DIR}/../doca-install")'
NV_HWS_PREFIX='$(readlink -f "${NV_HWS_SRC_DIR}/../nv_hws-install")'
RDMA_CORE_PREFIX='$(readlink -f "${RDMA_CORE_SRC_DIR}/../rdma-core-install")'

OS_DESC="${ID}${VERSION_ID}-$(uname -m)"
DPDK_DESC='$(describe_repository "${DPDK_SRC_DIR}")-${OS_DESC}'
DOCA_DESC='$(describe_repository "${DOCA_SRC_DIR}")-${OS_DESC}'
NV_HWS_DESC='$(describe_repository "${NV_HWS_SRC_DIR}")-${OS_DESC}'
RDMA_CORE_DESC='$(describe_repository "${RDMA_CORE_SRC_DIR}")-${OS_DESC}'

DPDK_BUILDTYPE="release"
DOCA_BUILDTYPE="release"
NV_HWS_BUILDTYPE="release"
RDMA_CORE_BUILDTYPE="release"

DPDK_INSTALL_DIR='${DPDK_PREFIX}/${DPDK_DESC}-${DPDK_BUILDTYPE}'
DOCA_INSTALL_DIR='${DOCA_PREFIX}/${DOCA_DESC}-${DOCA_BUILDTYPE}'
NV_HWS_INSTALL_DIR='${NV_HWS_PREFIX}/${NV_HWS_DESC}-${NV_HWS_BUILDTYPE}'
RDMA_CORE_INSTALL_DIR='${RDMA_CORE_PREFIX}/${RDMA_CORE_DESC}-${RDMA_CORE_BUILDTYPE}'

# Host CPU from QEMU on Macos M1 will not expose correct CPU info.
# In this case, fallback on a generic platform description for DPDK.
DPDK_MESON_EXTRA=""
MIDR_EL1_PATH='/sys/devices/system/cpu/cpu0/regs/identification/midr_el1'
MIDR_EL1=$(cat $MIDR_EL1_PATH 2> /dev/null)
if [ "$MIDR_EL1" = 0x00000000610f0000 ]; then
    DPDK_MESON_EXTRA="-Dplatform=generic"
fi

# Colors
CYAN="\033[0;36m"
RED="\033[0;31m"
NC="\e[0m"

unset PAGER
unset LESS

# Print a short string uniquely identifying the state of
# the source repository provided in '$1'.
# $1: A directory, pointing to either a git repository
#     or an archive of one.
function describe_repository() {
    local dir="$(readlink -f $1)"
    local desc
    [ -d "$dir" ] || return
    if [ -d "$dir/.git" ]; then
        # If a git repository, description is in order of:
        #   1. tag
        #   2. branch
        #   3. sha1
        desc=$(git -C "$dir" describe --exact-match --tags 2> /dev/null)
        [ ! "$desc" ] && desc=$(git -C "$dir" rev-parse --abbrev-ref HEAD | grep -v HEAD)
        [ ! "$desc" ] && desc=$(git -C "$dir" rev-parse --short HEAD)
    else
        [ -f "$dir/VERSION" ] && desc=$(cat "$dir/VERSION")
    fi
    [ ! "$desc" ] && desc="unknown"
    printf "$desc"
}

# Evaluate $1 and set itself to the result.
# e.g. 'FOO=/tmp; BAR='${FOO}/path';
# reflect BAR ==> BAR is set to '/tmp/path'
function reflect() { eval $1=$(eval echo \$$1); }

function quote() {
tr '\n' ' ' <<EOF | grep '^[-[:alnum:]_=,./:]* $' >/dev/null 2>&1 && { echo "$1" ; return 0; }
$1
EOF
    printf %s\\n "$1" |
        sed -e "s/'/'\\\\''/g" -e "1s/^/'/" \
            -e "\$s/\$/'/" \
            -e "s#^'\([-[:alnum:]_,./:]*\)=\(.*\)\$#\1='\2#"
}

printf "" > "$LOGFILE"
function logfile_append() {
    echo "$@" >> "$LOGFILE"
}

cmdline=$(quote "$0")
for i ; do cmdline="$cmdline $(quote "$i")" ; done
logfile_append "#!/usr/bin/env bash"
logfile_append
logfile_append "# This log was generated by:"
logfile_append "# $cmdline"
logfile_append

function err() {
    logfile_append "# ERROR: $*"
    echo -e "${RED}ERROR: $*${NC}" >&2
    exit 1
}

function log() {
    logfile_append "# $*"
    echo -e "# ${CYAN}$*${NC}"
}

function vlog() {
    if [ "$VERBOSE" == 1 ]
    then log "$*"
    else logfile_append "# $*"
    fi
}

function execute() {
    local redirect
    case $1 in
    --into)
        if [ "$VERBOSE" == 1 ]
        then redirect="| tee -a $2"
        else redirect=">> $2"
        fi
        shift 2;;
    esac

    local line
    for i ; do line="$line$(quote "$i") " ; done
    logfile_append "$line"
    if [ "$DRY_RUN" == 1 ]; then
        echo "$line"
    else
        [ "$VERBOSE" == 1 ] && echo "$line"
        # Set pipefail for using tee but still catch error.
        set -o pipefail
        eval "$line $redirect" || err "Failed to execute: $line"
        set +o pipefail
    fi
}

function check_meson_if_rebuild_needed() {
    local src_dir=${1:?missing arg}
    local install_dir=${2:?missing arg}
    local opts="$src_dir/build/meson-info/intro-buildoptions.json"
    local rc=0
    if [ -f $opts ]; then
        grep -q "$install_dir" $opts && rc=1 && RECOMPILE=1
    fi
    return $rc
}

function check_meson_if_recompile_needed() {
    [ "$DRY_RUN" == 1 ] && return
    ninja -n -C build install | grep -q "\[0/1\] Installing files." || RECOMPILE=1
}

function rebuild_dpdk() {
    [ -z "$DPDK_SRC_DIR" ] && return
    local build
    execute pushd "$DPDK_SRC_DIR"
    if [ -e "$DPDK_SRC_DIR/build/build.ninja" ]; then
        (check_meson_if_rebuild_needed $DPDK_SRC_DIR $DPDK_INSTALL_DIR || [ $REBUILD == "1" ]) && build=1
        check_meson_if_recompile_needed
        if [ "$build" == 1 ]; then
            log "Rebuilding DPDK"
            execute rm -fr "$DPDK_SRC_DIR/build"
        elif [ "$RECOMPILE" == 1 ]; then
            log "Recompiling DPDK"
        else
            log "DPDK is already configured"
            execute popd
            return
        fi
    else
        log "Building DPDK"
        build=1
    fi

    [ "$ASAN_DPDK" == 1 ] && DPDK_MESON_EXTRA+=" -Db_sanitize=address,undefined"

    if [ `uname -m` == "aarch64" ]; then
        local cross_file="config/arm/arm64_bluefield_linux_native_gcc"
        DPDK_MESON_EXTRA+=" --cross-file=$cross_file"
    fi

    local DPDK_OUT=/tmp/dpdk_out
    printf "" > "$DPDK_OUT"
    log "Build output: $DPDK_OUT"
    execute --into "$DPDK_OUT" git log --oneline -1
    if [ "$build" == 1 ]; then
        execute --into "$DPDK_OUT" \
                meson setup -Dc_args="$CFLAGS" --buildtype=$DPDK_BUILDTYPE $DPDK_MESON_EXTRA \
                      -Dtests=false -Denable_drivers=bus/auxiliary,*/mlx5,mempool/*,net/vhost \
                      -Dmachine=default -Dmax_ethports=1024 --prefix="$DPDK_INSTALL_DIR" \
                      build
    fi
    execute --into "$DPDK_OUT" ninja -j $NPROC -C build install
    execute popd
}

function rebuild_rdma_core() {
    local install_dir="$RDMA_CORE_INSTALL_DIR"
    local src_dir="$RDMA_CORE_SRC_DIR"
    local build_dir="$src_dir/build"
    local out_file=/tmp/rdma-core_out
    local need_setup

    [ "$src_dir" ] || return
    execute pushd "$src_dir"

    if [ -e "$src_dir/build/build.ninja" ]; then
        if [ $REBUILD == "1" ] || [ "$REBUILD_RDMA_CORE" == 1 ]; then
            need_setup=yes
        fi
        if [ "$need_setup" ]; then
            log "Rebuilding rdma-core"
            execute rm -fr "$src_dir/build"
        else
            log "rdma-core is already installed"
            execute popd
            return
        fi
    else
        log "Building rdma-core"
        need_setup=yes
    fi

    if [ "$need_setup" ]; then
         execute mkdir -p "$build_dir"
         execute pushd "$build_dir"
         execute --into "$out_file" cmake -GNinja \
                                          -DENABLE_STATIC=1 \
                                          -DNO_MAN_PAGES=1 \
                                          -DNO_PYVERBS=1 \
                                          -DCMAKE_INSTALL_PREFIX="$install_dir" \
                                          ..
         execute popd
    fi
    execute --into "$out_file" ninja -j $NPROC -C build install
    execute popd

    REBUILD_NV_HWS=1
}

function rebuild_nv_hws() {
    [ -z "$NV_HWS_SRC_DIR" ] && return
    local build=0
    execute pushd "$NV_HWS_SRC_DIR"
    if [ -e "$NV_HWS_SRC_DIR/build/build.ninja" ]; then
        (check_meson_if_rebuild_needed $NV_HWS_SRC_DIR $NV_HWS_INSTALL_DIR || [ $REBUILD == "1" ] || [ "$REBUILD_NV_HWS" == 1 ]) && build=1
        check_meson_if_recompile_needed
        if [ "$build" == 1 ]; then
            log "Rebuilding NV_HWS"
            execute rm -fr "$NV_HWS_SRC_DIR/build"
        elif [ "$RECOMPILE" == 1 ]; then
            log "Recompiling NV_HWS"
        else
            log "NV_HWS is already configured"
            execute popd
            return
        fi
    else
        log "Building NV_HWS"
        build=1
    fi

    local NV_HWS_OUT=/tmp/nv_hws_out
    printf "" > "$NV_HWS_OUT"
    log "Build output: $NV_HWS_OUT"
    execute --into "$NV_HWS_OUT" git log --oneline -1
    if [ "$build" == 1 ]; then
        execute --into "$NV_HWS_OUT" \
                meson setup -Dc_args="$CFLAGS" \
                      --buildtype=$NV_HWS_BUILDTYPE \
                      -Dpyhws=false -Dflexio=auto \
                      --prefix="$NV_HWS_INSTALL_DIR" build
    fi
    execute --into "$NV_HWS_OUT" ninja -j $NPROC -C build install
    execute popd
}

# $1: Commit ID
# $2: (optional) Git repository, default to $PWD
# Returns 'true' if the HEAD of the git repository contains
# the provided commit ID as an ancestor.
function git_commit_is_ancestor() {
    local sha=$1
    local dir=${2:-$PWD}

    if [ ! -e "$dir/.git" ] ||
       [ "$(git -C ${dir} rev-parse --is-shallow-repository)" = "true" ]; then
        return 0
    fi

    if git -C ${dir} merge-base --is-ancestor $sha HEAD; then
        return 0
    else
        return 1
    fi
}

function rebuild_doca() {
    [ -z "$DOCA_SRC_DIR" ] && return
    local build=0
    execute pushd "$DOCA_SRC_DIR"
    if [ -e "$DOCA_SRC_DIR/build/build.ninja" ]; then
        (check_meson_if_rebuild_needed $DOCA_SRC_DIR $DOCA_INSTALL_DIR || [ $REBUILD == "1" ] || [ "$REBUILD_DOCA" == 1 ]) && build=1
        check_meson_if_recompile_needed
        if [ "$build" == 1 ]; then
            log "Rebuilding DOCA"
            execute rm -fr "$DOCA_SRC_DIR/build"
        elif [ "$RECOMPILE" == 1 ]; then
            log "Recompiling DOCA"
        else
            log "DOCA is already configured"
            execute popd
            return
        fi
    else
        log "Building DOCA"
        build=1
    fi

    local DOCA_MESON_EXTRA
    [ "$ASAN" == 1 ] && DOCA_MESON_EXTRA="-Db_sanitize=address,undefined"
    local DOCA_ENABLED_LIBS="flow,common,dpdk_bridge"
    local DOCA_ENABLED_DRIVERS="dpdk,nvhws"
    local DOCA_CFLAGS="$CFLAGS"
    [ "$DOCA_BUILDTYPE" == "debug" ] && DOCA_CFLAGS+=" -DDOCA_DEBUG"

    local DOCA_OUT=/tmp/doca_out
    printf "" > "$DOCA_OUT"
    log "Build output: $DOCA_OUT"
    execute --into "$DOCA_OUT" git log --oneline -1

    local doca_min_sha=$(cat "$OVS_SRC_DIR/doca-min-sha.txt")
    if ! git_commit_is_ancestor $doca_min_sha $DOCA_SRC_DIR; then
        err "DOCA API mismatch: the git head is not based on $doca_min_sha, which is required"
    fi

    if [ "$build" == 1 ]; then
        execute --into "$DOCA_OUT" \
                meson setup -Dc_args="$DOCA_CFLAGS" \
                      -Dcpp_args="$DOCA_CFLAGS" \
                      --buildtype=$DOCA_BUILDTYPE \
                      -Denable_libs=$DOCA_ENABLED_LIBS \
                      -Denable_drivers=$DOCA_ENABLED_DRIVERS \
                      -Ddisable_all_tools=true \
                      -Ddisable_all_services=true \
                      -Ddisable_all_extensions=true \
                      -Ddisable_symbol_hiding=true \
                      -Ddisable_system_tests=true \
                      -Denable_grpc_support=false \
                      -Ddisable_tool_flow_tune=true \
                      -Dverification_disable_testsuit=true \
                      $DOCA_MESON_EXTRA --prefix="$DOCA_INSTALL_DIR" build
    fi
    execute --into "$DOCA_OUT" ninja -j $NPROC -C build install
    execute popd
}

function rebuild_openvswitch() {
    local OVS_CONFIGURE_EXTRA

    [ "$ASAN" == 1 ] && OVS_CONFIGURE_EXTRA+=" --with-sanitizer"
    [ "$SPARSE" == 1 ] && OVS_CONFIGURE_EXTRA+=" --enable-sparse"

    execute pushd "$OVS_SRC_DIR"

    if [ -f config.log ] && ! grep -q "$PKG_CONFIG_PATH" config.log ; then
        # Needs rebuild because the PKG_CONFIG_PATH is not the same.
        REBUILD=1
    elif [ -e configure ] && [ -e Makefile ] && [ -e config.h ]; then
        # If a library was detected to be recompiled it sets RECOMPILE=1.
        # If not then there is no reason to force recompile of ovs.
        # Let 'make' decide.
        :
    else
        REBUILD=1
    fi

    if [ "$REBUILD" == 1 ]; then
        log "Rebuild openvswitch"
        execute ./boot.sh
        execute ./configure --prefix=/usr --localstatedir=/var --sysconfdir=/etc \
                            --with-dpdk=static --with-doca=static --with-nvhws=static \
                            --enable-Werror \
                            $OVS_CONFIGURE_EXTRA \
                            ${CC1:+"CC=$CC1"} \
                            ${CFLAGS:+"CFLAGS=$CFLAGS"}
    elif [ "$RECOMPILE" == 1 ]; then
        # Force recompiling ovs in case dpdk or doca changed.
        log "Recompile openvswitch"
        touch lib/ovs-doca.c
    else
        log "Building openvswitch"
    fi

    if [ "$BUILD_RPM" == 1 ]; then
        RPMBUILD_OPT="--with static --with dpdk --with doca --with nvhws_static --without check"
        [ "$ASAN" == 1 ] && RPMBUILD_OPT+=" --with sanitizer"
        if [ -n "$CFLAGS" ];  then
            local optflags=$(rpm --eval "%optflags" | sed -e 's/-D_FORTIFY_SOURCE=2 -Wp,//')
            optflags+=" $CFLAGS"
            RPMBUILD_OPT+=" --define 'optflags $optflags'"
        fi
        execute make rpm-fedora RPMBUILD_OPT="$RPMBUILD_OPT"
    elif [ "$BUILD_DEB" == 1 ]; then
        DEB_BUILD_OPTIONS="with-dpdk with-doca static nocheck parallel=$NPROC"
        EXTRA_CONFIGURE_OPTS="--with-nvhws=static"
        [ "$ASAN" == 1 ] && EXTRA_CONFIGURE_OPTS+=" --with-sanitizer"
        [ -n "$CFLAGS" ] && export DEB_CFLAGS_APPEND="$CFLAGS"
        execute make debian-deb DEB_BUILD_OPTIONS="$DEB_BUILD_OPTIONS" EXTRA_CONFIGURE_OPTS="$EXTRA_CONFIGURE_OPTS"
    else
        execute make -j$NPROC -s

        if [ "$INSTALL_OVS" -eq 1 ]; then
            log "Install openvswitch"
            execute sudo make install -j$NPROC -s
        fi
    fi

    execute popd
}

function print_deps() {
    local deps

    if [ "$ID" == "ubuntu" ] || [ "$ID" == "debian" ]; then
        deps="\
python3-pyelftools meson libjson-c-dev protobuf-compiler binutils-dev
librdmacm-dev libibverbs-dev autoconf automake libtool gcc bc lftp graphviz
libunbound-dev libunwind-dev libssl-dev libelf-dev libnvhws-dev
libnuma-dev libpcap-dev dh-exec python3-all-dev python3-sortedcontainers
build-essential devscripts fakeroot python3-netifaces libdbus-1-dev
python3-sphinx python3-prometheus-client python3-enchant
libcap-ng-dev libsystemd-dev dh-python sparse
libarchive-dev libfdt-dev"
        if [ "$ASAN" -eq 1 ]; then
            if [[ "$VERSION_ID" =~ "20" ]]; then
                deps+=" libasan5 libubsan1"
            elif [[ "$VERSION_ID" =~ "22" ]]; then
                deps+=" libasan6 libubsan1"
            elif [[ "$VERSION_ID" =~ "24" ]]; then
                deps+=" libasan8 libubsan1"
            fi
        fi
    else
        deps="\
librdmacm rdma-core-devel meson json-c-devel protobuf-compiler binutils-devel
zlib-devel libcurl-devel autoconf automake openssl-devel libtool nvhws-devel
unbound-devel unbound libpcap-devel libbsd-devel elfutils-libelf-devel
libunwind-devel rpm-build numactl-devel libcap-ng-devel python3-enchant
python3-sphinx python3-prometheus_client sparse"
        if [ "$ASAN" -eq 1 ]; then
            deps+=" libasan libubsan"
        fi
    fi

    echo $deps
}

function install_deps() {
    local deps=`print_deps`

    if [ "$ID" == "ubuntu" ] || [ "$ID" == "debian" ]; then
        execute sudo apt-get install -y $deps
    else
        execute sudo dnf install -y --skip-broken $deps
    fi
}

function update_pkg_config_path() {
    local path=$1
    local name=$2
    local lib
    local pkg
    local pc

    local pc=$(find "$path" -iname "$name" 2>/dev/null)
    [ -z "$pc" ] && [ "$DRY_RUN" == 1 ] && log "Missing $name. skip because of dry-run." && return
    [ -z "$pc" ] && err "Cannot find $name in $path"
    pkg="$(realpath $(dirname "$pc"))"
    lib="$(realpath $(dirname "$pkg"))"
    log "$name pkgconfig: $pkg"
    log "$name lib: $pkg"
    local version=$(grep Version $pc)
    log "$name $version"
    PKG_CONFIG_PATH+=":$pkg"
    LD_LIBRARY_PATH+=":$lib"
    PKG_CONFIG_PATH=${PKG_CONFIG_PATH##:}
    LD_LIBRARY_PATH=${LD_LIBRARY_PATH##:}
    execute export PKG_CONFIG_PATH=${PKG_CONFIG_PATH##:}
    execute export LD_LIBRARY_PATH=${LD_LIBRARY_PATH##:}
    log "`env | grep PKG_CONFIG_PATH`"
    log "`env | grep LD_LIBRARY_PATH`"
}

function update_dpdk_pkgconfig() {
    update_pkg_config_path "$DPDK_INSTALL_DIR" "libdpdk.pc"
}

function update_doca_pkgconfig() {
    update_pkg_config_path "$DOCA_INSTALL_DIR" "doca-flow.pc"
}

function update_nv_hws_pkgconfig() {
    [ -z "$NV_HWS_SRC_DIR" ] && return
    update_pkg_config_path "$NV_HWS_INSTALL_DIR" "libnvhws.pc"
}

function update_rdma_core_pkgconfig() {
    [ "$RDMA_CORE_SRC_DIR" ] || return
    for pc in libibverbs.pc libmlx5.pc librdmacm.pc; do
        update_pkg_config_path "$RDMA_CORE_INSTALL_DIR" "$pc"
    done
}

function update_install_dirs() {
    # The user arguments supersedes all logic:
    # If some were provided, they take priority over
    # autodetection and default fallbacks.

    [ "$DPDK_SRC_ARG" ]     && DPDK_SRC_DIR="$DPDK_SRC_ARG"
    [ "$DPDK_INSTALL_ARG" ] && DPDK_INSTALL_DIR="$DPDK_INSTALL_ARG"
    [ "$DOCA_SRC_ARG" ]     && DOCA_SRC_DIR="$DOCA_SRC_ARG"
    [ "$DOCA_INSTALL_ARG" ] && DOCA_INSTALL_DIR="$DOCA_INSTALL_ARG"
    [ "$NV_HWS_SRC_ARG" ]     && NV_HWS_SRC_DIR="$NV_HWS_SRC_ARG"
    [ "$NV_HWS_INSTALL_ARG" ] && NV_HWS_INSTALL_DIR="$NV_HWS_INSTALL_ARG"
    [ "$RDMA_CORE_SRC_ARG" ]     && RDMA_CORE_SRC_DIR="$RDMA_CORE_SRC_ARG"
    [ "$RDMA_CORE_INSTALL_ARG" ] && RDMA_CORE_INSTALL_DIR="$RDMA_CORE_INSTALL_ARG"

    if [ ! "$DPDK_INSTALL_ARG" ]; then
        if [ ! "$DPDK_SRC_DIR" ] && [ -d "$OVS_SRC_DIR/../dpdk" ]; then
            DPDK_SRC_DIR="$(readlink -f $OVS_SRC_DIR/../dpdk)"
            vlog "Detected DPDK source: $DPDK_SRC_DIR"
        elif [ ! "$DPDK_SRC_DIR" ] && [ -d "$OVS_SRC_DIR/../dpdk.org" ]; then
            DPDK_SRC_DIR="$(readlink -f $OVS_SRC_DIR/../dpdk.org)"
            vlog "Detected DPDK source: $DPDK_SRC_DIR"
        fi
        if [ -d "$DPDK_SRC_DIR" ]; then
            reflect DPDK_PREFIX
            reflect DPDK_DESC
            reflect DPDK_INSTALL_DIR
            log "DPDK source: $DPDK_SRC_DIR"
        else
            DPDK_INSTALL_DIR="/opt/mellanox/dpdk"
        fi
    fi
    log "DPDK installation: $DPDK_INSTALL_DIR"

    if [ ! "$DOCA_INSTALL_ARG" ]; then
        if [ ! "$DOCA_SRC_DIR" ] && [ -d "$OVS_SRC_DIR/../doca" ]; then
            DOCA_SRC_DIR="$(readlink -f $OVS_SRC_DIR/../doca)"
            vlog "Detected DOCA source: $DOCA_SRC_DIR"
        fi
        if [ -d "$DOCA_SRC_DIR" ]; then
            reflect DOCA_PREFIX
            reflect DOCA_DESC
            reflect DOCA_INSTALL_DIR
            log "DOCA source: $DOCA_SRC_DIR"
        else
            DOCA_INSTALL_DIR="/opt/mellanox/doca"
        fi
    fi
    log "DOCA installation: $DOCA_INSTALL_DIR"

    if [ ! "$NV_HWS_SRC_DIR" ] && [ -d "$OVS_SRC_DIR/../nv_hws" ]; then
        NV_HWS_SRC_DIR="$(readlink -f $OVS_SRC_DIR/../nv_hws)"
        vlog "Detected NV_HWS source: $NV_HWS_SRC_DIR"
    fi
    if [ -d "$NV_HWS_SRC_DIR" ]; then
        reflect NV_HWS_PREFIX
        reflect NV_HWS_DESC
        reflect NV_HWS_INSTALL_DIR
        log "NV_HWS source: $NV_HWS_SRC_DIR"
        log "NV_HWS installation: $NV_HWS_INSTALL_DIR"
    fi

    if [ ! "$RDMA_CORE_SRC_DIR" ] && [ -d "$OVS_SRC_DIR/../rdma-core" ]; then
        RDMA_CORE_SRC_DIR="$(readlink -f $OVS_SRC_DIR/../rdma-core)"
        vlog "Detected rdma-core source: $RDMA_CORE_SRC_DIR"
    fi
    if [ -d "$RDMA_CORE_SRC_DIR" ]; then
        reflect RDMA_CORE_PREFIX
        reflect RDMA_CORE_DESC
        reflect RDMA_CORE_INSTALL_DIR
        log "rdma-core source: $RDMA_CORE_SRC_DIR"
        log "rdma-core installation: $RDMA_CORE_INSTALL_DIR"
    fi
}

function print_help() {
    update_install_dirs

    cat<<EOL
Usage: $(basename "$0") [OPTION...]

Optional arguments:
    -h|--help           Print this help.
    --verbose           More prints.
    --dry-run           Print commands without executing.
    --rebuild           Force rebuild dpdk and doca.
    --rebuild-doca      Force rebuild doca.
    --rebuild-nvhws     Force rebuild nv_hws.
    --rebuild-rdma-core Force rebuild rdma-core.
    --recompile         Do recompile.
    --asan              Build with address sanitizer. Also enables --debug.
    --asan-dpdk         Build DPDK with address sanitizer. Also enables --debug.
    --sparse            Build with sparse checker. Also enables --debug.
    --deps              Print needed dependencies for build.
    --install-deps      Install needed dependencies for build.
    --install-ovs       Install ovs. Executed with sudo, requires privileged user.
    --build-rpm         Build ovs rpm.
    --build-deb         Build ovs deb.
    --info              Build with CFLAGS="$INFO_CFLAGS". Does not change build type.
    --debug             Build with CFLAGS="$DEBUG_CFLAGS". Does not change build type.

    --buildtype ARG         Build type for DPDK, DOCA. [debug, release]. ($DPDK_BUILDTYPE)
                            If set to 'debug', also enables --debug.
    --buildtype-doca ARG    Build type for DOCA. ($DOCA_BUILDTYPE)
                            If set to 'debug', also enables --debug.
    --buildtype-nvhws ARG   Build type for libnvhws. ($NV_HWS_BUILDTYPE)
                            If set to 'debug', also enables --debug.
    --buildtype-rdma-core ARG Build type for rdma-core. ($RDMA_CORE_BUILDTYPE)
                            If set to 'debug', also enables --debug.
    --cc ARG                Compiler to use. ($CC1)
                            This option is ignored when building packages.
    --nproc ARG             Number of cores to use. ($NPROC)
    --dpdk-src-dir ARG      Path to existing dpdk source tree. ($DPDK_SRC_DIR)
    --dpdk-install-dir ARG  Destination for dpdk build. ($DPDK_INSTALL_DIR)
    --doca-src-dir ARG      Path to existing doca source tree. ($DOCA_SRC_DIR)
    --doca-install-dir ARG  Destination for doca build. ($DOCA_INSTALL_DIR)
    --nvhws-src-dir ARG     Path to existing nv_hws source tree. ($NV_HWS_SRC_DIR)
    --nvhws-install-dir ARG Destination for nv_hws build. ($NV_HWS_INSTALL_DIR)
    --rdma-core-src-dir ARG Path to existing rdma-core source tree. ($RDMA_CORE_SRC_DIR)
    --rdma-core-install-dir ARG Destination for rdma-core build. ($RDMA_CORE_INSTALL_DIR)
EOL

    exit 0
}

function parse_args() {
    local print_usage=0
    local print_deps=0
    local install_deps=0
    local debug=0

    while [[ $# -gt 0 ]]; do
      case "$1" in
        --help|-h)
            print_usage=1
            shift
            ;;

        --verbose)
            VERBOSE=1
            shift
            ;;

        --dry-run)
            DRY_RUN=1
            shift
            ;;

        --build-rpm)
            BUILD_RPM=1
            shift
            ;;

        --build-deb)
            BUILD_DEB=1
            shift
            ;;

        --rebuild)
            REBUILD=1
            shift
            ;;

        --rebuild-doca)
            REBUILD_DOCA=1
            shift
            ;;

        --rebuild-nvhws)
            REBUILD_NV_HWS=1
            shift
            ;;

        --rebuild-rdma-core)
            REBUILD_RDMA_CORE=1
            shift
            ;;

        --recompile)
            RECOMPILE=1
            shift
            ;;

        --asan)
            ASAN=1
            debug=1
            shift
            ;;

        --asan-dpdk)
            ASAN_DPDK=1
            debug=1
            shift
            ;;

        --sparse)
            SPARSE=1
            debug=1
            shift
            ;;

        --info)
            info=1
            shift
            ;;

        --debug)
            debug=1
            shift
            ;;

        --cc)
            CC1=$2
            shift 2
            ;;

        --nproc)
            NPROC=$2
            shift 2
            ;;

        --buildtype)
            DPDK_BUILDTYPE=$2
            DOCA_BUILDTYPE=$2
            RDMA_CORE_BUILDTYPE=$2
            shift 2
            ;;

        --buildtype-nvhws)
            NV_HWS_BUILDTYPE=$2
            shift 2
            ;;

        --buildtype-rdma-core)
            RDMA_CORE_BUILDTYPE=$2
            shift 2
            ;;

        --buildtype-doca)
            DOCA_BUILDTYPE=$2
            shift 2
            ;;

        --dpdk-src-dir)
            DPDK_SRC_ARG=$2
            shift 2
            ;;

        --dpdk-install-dir)
            DPDK_INSTALL_ARG=$2
            shift 2
            ;;

        --doca-src-dir)
            DOCA_SRC_ARG=$2
            shift 2
            ;;

        --doca-install-dir)
            DOCA_INSTALL_ARG=$2
            shift 2
            ;;

        --nvhws-src-dir)
            NV_HWS_SRC_ARG=$2
            shift 2
            ;;

        --nvhws-install-dir)
            NV_HWS_INSTALL_ARG=$2
            shift 2
            ;;

        --rdma-core-src-dir)
            RDMA_CORE_SRC_DIR=$2
            shift 2
            ;;

        --rdma-core-install-dir)
            RDMA_CORE_INSTALL_DIR=$2
            shift 2
            ;;

        --deps)
            print_deps=1
            shift
            ;;

        --install-deps)
            install_deps=1
            shift
            ;;

        --install-ovs)
            INSTALL_OVS=1
            shift
            ;;

        --)
            shift
            ;;

        *)
            err "Invalid argument: $1"
            exit 1
            break
            ;;
      esac
    done

    if [ "$DPDK_BUILDTYPE" == "debug" ] ||
       [ "$DOCA_BUILDTYPE" == "debug" ] ||
       [ "$NV_HWS_BUILDTYPE" == "debug" ] ||
       [ "$RDMA_CORE_BUILDTYPE" == "debug" ]; then
        debug=1
    fi

    if [ "$info" == 1 ]; then
        CFLAGS="$INFO_CFLAGS"
    fi

    if [ "$debug" == 1 ]; then
        CFLAGS="$DEBUG_CFLAGS"
    fi

    if [ "$print_usage" == 1 ]; then
        print_help
    fi

    if [ "$install_deps" == 1 ]; then
        install_deps
        exit $?
    fi

    if [ "$print_deps" == 1 ]; then
        print_deps
        exit 0
    fi
}

function main() {
    update_install_dirs

    rebuild_dpdk
    update_dpdk_pkgconfig

    rebuild_rdma_core
    update_rdma_core_pkgconfig

    rebuild_nv_hws
    update_nv_hws_pkgconfig

    rebuild_doca
    update_doca_pkgconfig

    rebuild_openvswitch
}

parse_args "$@"
main
