/*
 * Copyright (c) 2021-2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at:
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

#include <limits.h>
#include <string.h>
#include <stdint.h>

#include <config.h>

#include "openvswitch/util.h"
#include "salloc.h"

struct salloc_header {
    size_t size;
    char data[0];
};

struct salloc *
salloc_init(void *mem, size_t sz)
{
    uintptr_t start = ROUND_UP((uintptr_t) mem, SALLOC_ALIGNMENT);
    struct salloc *s = (void *) start;

    /* If this triggers, the scratch buffer is too small. */
    ovs_assert(sz > sizeof *s);
    s->sz = sz - sizeof *s;
    salloc_reset(s);
    return s;
}

void
salloc_reset(struct salloc *s)
{
    uintptr_t start;
    size_t aligned;

    start = ROUND_UP((uintptr_t) s->mem, SALLOC_ALIGNMENT);
    aligned = (start - (uintptr_t) s->mem);
    /* If this triggers, the scratch buffer is too small. */
    ovs_assert(aligned < s->sz);
    s->pos = aligned;
}

void *
srealloc(struct salloc *s, void *prev, size_t n)
{
    struct salloc_header *hdr;
    size_t old_sz, new_sz;
    char *p = prev;
    size_t next;

    if (prev != NULL) {
        hdr = ((struct salloc_header *) prev) - 1;
        old_sz = hdr->size;
    } else {
        hdr = NULL;
        old_sz = 0;
    }

    new_sz = ROUND_UP(n, SALLOC_ALIGNMENT);
    next = s->pos + new_sz + sizeof(*hdr);
    /* Do not overflow. */
    ovs_assert(s->pos <= next);

    if (p != NULL && &p[old_sz] == &s->mem[s->pos]) {
        /* p was the last allocated chunk. It can grow or shrink. */
        s->pos -= old_sz + sizeof(*hdr);
        next = s->pos + new_sz + sizeof(*hdr);
        p = (void *) hdr;
    } else {
        p = &s->mem[s->pos];
    }

    if (n == 0 || next > s->sz) {
        return NULL;
    }

    hdr = (void *) p;
    hdr->size = new_sz;
    p = hdr->data;

    if (prev != NULL && prev != p) {
        memcpy(p, prev, old_sz);
    }

    s->pos = next;
    return p;
}

void *
salloc(struct salloc *s, size_t n)
{
    return srealloc(s, NULL, n);
}

void *
szalloc(struct salloc *s, size_t n)
{
    void *m = salloc(s, n);

    if (m != NULL) {
        memset(m, 0, n);
    }
    return m;
}

void *
scalloc(struct salloc *s, size_t n, size_t sz)
{
    if (sz && n > (size_t) -1 / sz) {
        return NULL;
    }
    return szalloc(s, n * sz);
}
