/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2022-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#ifndef OVS_DOCA_H
#define OVS_DOCA_H

#include <config.h>

#include "dp-packet.h"
#include "netdev-offload.h"
#include "ovs-atomic.h"
#include "smap.h"
#include "vswitch-idl.h"

#ifdef DPDK_NETDEV

#include <rte_pci.h>

#else

#define PCI_PRI_STR_SIZE sizeof("XXXXXXXX:XX:XX.X")
#define RTE_MAX_ETHPORTS 1

#endif /* DPDK_NETDEV */

enum { MAX_METERS = 5000 };

#if DOCA_OFFLOAD
#include <doca_flow.h>

/* DOCA requires everything upfront. As a WA we define max number of ESW. */
#define OVS_DOCA_DEFAULT_MAX_ESW 2
#define OVS_DOCA_MAX_SUPPORTED_ESW 32

#define OVS_DOCA_QUEUE_DEPTH 32
#define OVS_DOCA_CT_QUEUE_DEPTH 512

#define DOCA_RESIZED_PIPE_CONGESTION_LEVEL 50

/*
 * Shared counter IDs allocation scheme.
 *
 *                ovs_doca_max_shared_counters_per_esw
 * ---------------------------------------------------------------------->
 *                                   ESW_0                                            ESW_N
 * ,----------------------------------+----------------------------------.     ,-------+---------.
 * |                CT                |               METER              |     |       |         |
 * |                                  | ,------------------------------. |     |       |         |
 * |                                  | |    GREEN,RED,GREEN,RED,...   | | ... |  CT   |  METER  |
 * |                                  | `------------------------------' |     |       |         |
 * |                                  |                                  |     |       |         |
 * `----------------------------------+----------------------------------'     `-------+---------'
 *   ovs_doca_max_ct_counters_per_esw | OVS_DOCA_MAX_METER_COUNTERS_PER_ESW
 * ---------------------------------->|--------------------------------->
 *                                    |
 *                                    v
 *                                    OVS_DOCA_METER_COUNTERS_PER_ESW_BASE_ID
 */

/* Using shared counters we need 2 per meter */
#define OVS_DOCA_MAX_METER_COLORS 2 /* green and red */
#define OVS_DOCA_MAX_FLOW_METERS_PER_ESW MAX_METERS
#define OVS_DOCA_MAX_SW_METERS_PER_ESW RTE_MAX_ETHPORTS

#define OVS_DOCA_MAX_PORT_METERS_PER_ESW RTE_MAX_ETHPORTS
#define OVS_DOCA_MAX_CORE_METERS_PER_ESW 1
#define OVS_DOCA_MAX_NORMAL_METERS_PER_ESW \
    (OVS_DOCA_MAX_FLOW_METERS_PER_ESW + OVS_DOCA_MAX_SW_METERS_PER_ESW)
/* As sw-meter use the DPDK port-id, increase the range by one so that
 * all ports can configure a sw-meter. */
#define OVS_DOCA_MAX_GPR_METERS_PER_ESW \
    (OVS_DOCA_MAX_PORT_METERS_PER_ESW + OVS_DOCA_MAX_CORE_METERS_PER_ESW + 1)
#define OVS_DOCA_MAX_METERS_PER_ESW \
    (OVS_DOCA_MAX_NORMAL_METERS_PER_ESW + OVS_DOCA_MAX_GPR_METERS_PER_ESW)
#define OVS_DOCA_MAX_METERS (OVS_DOCA_MAX_GPR_METERS_PER_ESW * ovs_doca_max_eswitch_num_get() + \
                             OVS_DOCA_MAX_NORMAL_METERS_PER_ESW * ovs_doca_max_eswitch_num_get())

#define OVS_DOCA_MAX_METER_COUNTERS_PER_ESW \
    (OVS_DOCA_MAX_FLOW_METERS_PER_ESW * OVS_DOCA_MAX_METER_COLORS)
#define OVS_DOCA_MAX_METER_COUNTERS (OVS_DOCA_MAX_METER_COUNTERS_PER_ESW * \
                                     ovs_doca_max_eswitch_num_get())

enum ovs_doca_gpr_mode {
    OVS_DOCA_GPR_MODE_DISABLED,
    OVS_DOCA_GPR_MODE_REP_ONLY,
    OVS_DOCA_GPR_MODE_ALL_PORTS,
};

/* Number of supported tunnel configurations:
 * GRE-IPv4/GRE-IPv6/GRE-no-key-IPv4/GRE-no-key-IPv6
 * VXLAN-IPv4/VXLAN-IPv6/VXLAN-GBP-IPv4/VXLAN-GBP-IPv6
 * GENEVE-IPv4/GENEVE-IPv6/GENVE-OPTIONS-IPv4/GENEVE-OPTIONS-IPv6
 */
#define OVS_DOCA_MAX_TUNNELS 12
/* 16K different tunnel header + number of different tunnel templates */
#define OVS_DOCA_MAX_SHARED_ENCAPS ((1 << 14) + OVS_DOCA_MAX_TUNNELS)

/* Estimated maximum number of megaflows */
#define OVS_DOCA_MAX_MEGAFLOWS_COUNTERS (1 << 19)
#define OVS_DOCA_FOREACH_ACTIVE_ESWITCH(ID) \
    for (ID = ovs_doca_eswitch_next_active(0); \
         ID < ovs_doca_max_eswitch_num_get(); \
         ID = ovs_doca_eswitch_next_active(ID + 1))
#define AUX_QUEUE NETDEV_OFFLOAD_THREAD_MAIN
#define FIRST_QUEUE (AUX_QUEUE + 1)
#define PMD_QUEUES_BITMAP \
    (UINT64_MAX & ~(UINT64_C(1) << AUX_QUEUE) & ~(UINT64_C(1) << FIRST_QUEUE))
#define ENTRY_PROCESS_TIMEOUT_US 1000

extern uint32_t ctl_pipe_size;
extern uint32_t ctl_pipe_infra_size;
extern unsigned int doca_congestion_threshold;
extern unsigned int ovs_doca_max_eswitch_num;
extern atomic_bool ovs_doca_eswitch_active_ids[OVS_DOCA_MAX_SUPPORTED_ESW];

/* sw-meter range starts immediately after the flow meter's. */
#define OVS_DOCA_SW_METER_BASE_ID OVS_DOCA_MAX_FLOW_METERS_PER_ESW

#define DEFAULT_PER_CORE_METER_RATE 500000

/* DOCA defines max actions memory as HWS_MAX_ACTIONS_MEM_SIZE in hws_port.c:
 *
 *   #define HWS_MAX_ACTIONS_MEM_SIZE (16 * 1024 * 1024 * 64) // max number of flows (16M) * each modify cmd size (64B)
 *
 * The 64B modify cmd size can be twice as big for IPv6 actions. Use max expected number of
 * megaflows as a top boundary for the number of flows with modify / encap actions.
 */
#define OVS_DOCA_DEFAULT_ACTIONS_MEM_SIZE (64 * 2 * OVS_DOCA_MAX_MEGAFLOWS_COUNTERS)

unsigned int ovs_doca_max_ct_conns(void);
unsigned int ovs_doca_max_eswitch_num_get(void);

/* Connections are offloaded with one hardware rule per direction.
 * The netdev-offload layer manages offloads rule-wise, so a
 * connection is handled in two parts. This discrepancy can be
 * misleading.
 * This macro expresses the number of hardware rules required
 * to handle the number of CT connections supported by ovs-doca.
 */
static inline unsigned int
ovs_doca_max_ct_rules(void)
{
    return ovs_doca_max_ct_conns() * 2;
}

/* Using shared counters we need 1 per connection */
static inline unsigned int
ovs_doca_max_ct_counters_per_esw(void)
{
    /* IPv6 connections uses basic pipes and shared counters unless enabled with DOCA-CT.
     * IPv4 connections need those shared counters only if DOCA-CT is disabled.
     *
     * Here is the truth table:
     *
     * CTO: CT-offload enabled/disabled
     * DCT: DOCA-CT enabled/disabled
     * 6CT: CT-offload IPv6 enabled/disabled
     * 6DC: DOCA-CT-IPv6 enabled / disabled
     *
     *   CTO DCT 6CT 6DC OUT
     *    0   0   0   0   0
     *    0   0   0   1   0
     *    0   0   1   0   0
     *    0   0   1   1   0
     *    0   1   0   0   0
     *    0   1   0   1   0
     *    0   1   1   0   0
     *    0   1   1   1   0
     *    1   0   0   0   N
     *    1   0   0   1   N
     *    1   0   1   0   N
     *    1   0   1   1   N
     *    1   1   0   0   0   [Default]
     *    1   1   0   1   0
     *    1   1   1   0   N   [hw-offload-ct-ipv6="true"]
     *    1   1   1   1   0
     */
    if (!conntrack_offload_is_enabled()) {
        return 0;
    }

    if (!conntrack_offload_doca_ct_enabled) {
        return ovs_doca_max_ct_conns();
    }

    if (!conntrack_offload_ipv6_is_enabled()) {
        return 0;
    }

    if (!conntrack_offload_doca_ct_ipv6_enabled) {
        return ovs_doca_max_ct_conns();
    }

    return 0;
}

/* Total shared counters */
static inline unsigned int
ovs_doca_max_shared_counters_per_esw(void)
{
    return ovs_doca_max_ct_counters_per_esw() +
           OVS_DOCA_MAX_METER_COUNTERS_PER_ESW;
}

static inline int
ovs_doca_eswitch_next_active(int next)
{
    bool active;

    while (next < ovs_doca_max_eswitch_num_get()) {
        atomic_read_explicit(&ovs_doca_eswitch_active_ids[next], &active,
                             memory_order_consume);
        if (active) {
            return next;
        }
        next++;
    }
    return next;
}

static inline bool
ovs_doca_eswitch_active(int esw_id)
{
    bool active;

    atomic_read_explicit(&ovs_doca_eswitch_active_ids[esw_id], &active,
                         memory_order_consume);
    return active;
}

void
ovs_doca_pipe_process_cb(struct doca_flow_pipe *pipe,
                         enum doca_flow_pipe_status status,
                         enum doca_flow_pipe_op op, void *user_ctx);

unsigned int
ovs_doca_n_offload_queues(void);

unsigned int
ovs_doca_max_counters(void);

unsigned int
ovs_doca_get_max_megaflows_counters(void);

bool
ovs_doca_queue_use_resize(unsigned int);

uint32_t
ovs_doca_get_post_meter_counter_id(uint32_t meter_id,
                                   enum doca_flow_meter_color meter_color);

/* Translate a DPDK port ID into a meter ID in the extended datapath meter
 * ID range. This ID is valid for use with netdev-dpdk meter functions and
 * then to translate it into a DOCA shared meter ID. */
static inline uint32_t
ovs_doca_sw_meter_dp_ext_id(int dpdk_port_id)
{
    return OVS_DOCA_SW_METER_BASE_ID + dpdk_port_id;
}

uint32_t
ovs_doca_per_port_meter_rate(void);

uint32_t
ovs_doca_per_core_meter_rate(void);

uint32_t
ovs_doca_gpr_mode(void);

bool
ovs_doca_port_rate_configured(void);

#else /* DOCA_OFFLOAD */

#define FIRST_QUEUE 0

#endif /* DOCA_OFFLOAD */

enum ovs_doca_rss_type {
    OVS_DOCA_RSS_IPV4_TCP,
    OVS_DOCA_RSS_IPV4_UDP,
    OVS_DOCA_RSS_IPV4_ICMP,
    OVS_DOCA_RSS_IPV4_ESP,
    OVS_DOCA_RSS_IPV4_OTHER,
    OVS_DOCA_RSS_IPV6_TCP,
    OVS_DOCA_RSS_IPV6_UDP,
    OVS_DOCA_RSS_IPV6_ICMP,
    OVS_DOCA_RSS_IPV6_ESP,
    OVS_DOCA_RSS_IPV6_OTHER,
    OVS_DOCA_RSS_OTHER,
};
/* Must be the last enum type. */
#define OVS_DOCA_RSS_NUM_ENTRIES (OVS_DOCA_RSS_OTHER + 1)

/* For slow path, doca egress the packets by its own pipe, hidden from the
 * user. For hairpin offload, it uses the user's one. Then, we need to use the
 * same mapping as doca, which is simply the port_id.
 */
static inline uint32_t
ovs_doca_port_id_pkt_meta(uint16_t port_id)
{
    return port_id;
}

struct doca_dev;
struct doca_flow_pipe;
struct doca_flow_pipe_entry;
struct doca_flow_port;
struct doca_flow_resource_meter_cfg;
struct ofputil_meter_config;
struct rte_eth_stats;
struct rte_mtr_error;
struct rte_mtr_stats;

struct ovs_doca_port_queue {
    PADDED_MEMBERS(CACHE_LINE_SIZE,
        struct dp_packet_batch batch;
        atomic_uint64_t n_packets;
        atomic_uint64_t n_bytes;
    );
};

struct ovs_doca_tx_stats {
    PADDED_MEMBERS(CACHE_LINE_SIZE,
        atomic_uint64_t n_packets;
        atomic_uint64_t n_bytes;
    );
};

struct ovs_doca_offload_queue {
    PADDED_MEMBERS(CACHE_LINE_SIZE,
        unsigned int n_waiting_entries;
    );
};

struct ovs_doca_esw_ctx {
    char pci_addr[PCI_PRI_STR_SIZE];
    struct doca_flow_pipe *egress_pipe;
    struct doca_flow_pipe *rss_pipe;
    struct doca_dev *dev;
    int cmd_fd;
    uint16_t port_id;
    struct ovs_doca_port_queue *port_queues[RTE_MAX_ETHPORTS];
    struct ovs_doca_offload_queue queue;
    uint32_t op_state;
    int n_rxq;
};

struct ovs_doca_netdev_data {
    const char *devargs;
    struct doca_flow_port *port;
    uint16_t port_id;
    uint16_t *esw_mgr_port_id;
    struct ovs_doca_esw_ctx *esw_ctx;
    struct doca_flow_pipe_entry *egress_entry;
    struct doca_flow_pipe_entry *rss_entries[OVS_DOCA_RSS_NUM_ENTRIES];
    struct doca_dev_rep *dev_rep;
    bool *attached;
    int *n_rxq;
};

/* Whether DOCA support is compiled-in and user configuration
 * requested DOCA to be enabled. DOCA might not yet be initialized,
 * but the user expects a DOCA execution at some point. */
bool
ovs_doca_enabled(void);

/* Both 'ovs_doca_enabled() == true' and DOCA has successfully initialized. */
bool
ovs_doca_initialized(void);

int
ovs_doca_init(const struct smap *ovs_other_config);

int
ovs_doca_sw_meter_init(struct netdev *netdev);

void
ovs_doca_sw_meter_uninit(struct netdev *netdev);

int
ovs_doca_port_start(struct netdev *, struct ovs_doca_netdev_data *, int);

int
ovs_doca_port_stop(struct netdev *netdev, struct ovs_doca_netdev_data *dev_data);

void
ovs_doca_set_properties(struct ovs_doca_netdev_data *dev_data,
                        uint16_t port_id,
                        const char *devargs,
                        uint16_t *esw_mgr_port_id,
                        bool *attached,
                        int *n_rxq);

void
ovs_doca_status(const struct ovsrec_open_vswitch *cfg);

void
print_ovs_doca_release(void);

void
print_doca_version(void);

const char *
ovs_doca_get_version(void);

int
ovs_doca_create_meter(uint32_t meter_id,
                      struct ofputil_meter_config *config,
                      struct doca_flow_resource_meter_cfg *meter_cfg,
                      struct rte_mtr_error *error);

int
ovs_doca_delete_meter(uint16_t port_id, uint32_t meter_id,
                      struct rte_mtr_error *error);

int
ovs_doca_mtr_stats_read(uint16_t port_id,
                        uint32_t mtr_id,
                        struct rte_mtr_stats *stats,
                        struct rte_mtr_error *error);

int
ovs_doca_dev_probe(struct ovs_doca_netdev_data *dev_data,
                   const char *devargs);

void
ovs_doca_dev_close(struct ovs_doca_netdev_data *dev_data);

int
ovs_doca_rx_burst(struct ovs_doca_netdev_data *dev_data,
                  uint16_t queue_id,
                  struct dp_packet_batch *batch);

extern struct ovs_mutex mgmt_queue_lock;

void ovs_doca_mgmt_queue_lock(void)
    OVS_ACQUIRES(mgmt_queue_lock);
void ovs_doca_mgmt_queue_unlock(void)
    OVS_RELEASES(mgmt_queue_lock);

int
ovs_doca_eth_stats_get(struct ovs_doca_netdev_data *dev_data,
                       struct rte_eth_stats *rte_stats);

int
ovs_doca_get_custom_stats(const struct netdev *netdev,
                          const struct ovs_doca_netdev_data *dev_data,
                          const struct ovs_doca_tx_stats *tx_stats,
                          struct netdev_custom_stats *custom_stats);

void
ovs_doca_set_op_state(uint32_t op_state);

bool
ovs_doca_is_offload_trace_enabled(void);

void
ovs_doca_register_classes(const struct smap *ovs_other_config);

struct ovs_doca_mempool;
void ovs_doca_mempool_destroy(struct ovs_doca_mempool *odmp);
struct ovs_doca_mempool *ovs_doca_mempool_create(unsigned n, unsigned elt_size);
void ovs_doca_mempool_free(struct ovs_doca_mempool *odmp, void *obj);
int ovs_doca_mempool_alloc(struct ovs_doca_mempool *odmp, void **obj_p);

struct ds * ovs_doca_packet_ds_put_hex(struct ds *s, struct dp_packet *b,
                                       uint32_t max_bytes);

int
ovs_doca_get_esw_n_rxq(struct ovs_doca_netdev_data *dev_data);

#endif
