/*
 * SPDX-FileCopyrightText: NVIDIA CORPORATION & AFFILIATES
 * Copyright (c) 2009-2025 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 * SPDX-License-Identifier: LicenseRef-NvidiaProprietary
 *
 * NVIDIA CORPORATION, its affiliates and licensors retain all intellectual
 * property and proprietary rights in and to this material, related
 * documentation and any modifications thereto. Any use, reproduction,
 * disclosure or distribution of this material and related documentation
 * without an express license agreement from NVIDIA CORPORATION or
 * its affiliates is strictly prohibited.
 */

#include <config.h>
#include "dirs.h"
#include "dpif-doca-tcpdump.h"
#include "openvswitch/dynamic-string.h"
#include "openvswitch/hmap.h"
#include "openvswitch/util.h"
#include "openvswitch/vlog.h"
#include "ovs-atomic.h"
#include "ovs-thread.h"
#include "unixctl.h"
#include "util.h"

#include "dpif-doca-tcpdump-hooks.inc"

#include <sys/un.h>
#include <unistd.h>

VLOG_DEFINE_THIS_MODULE(dpif_doca_tcpdump);

/* ============================================================================
 * CONSTANTS AND DEFINITIONS
 * ============================================================================ */

#define DPIF_DOCA_TCPDUMP_DEFAULT_HOOKS (UINT32_C(1) << DPIF_DOCA_TCPDUMP_HOOK_RX)

/* Maximum interface name length */
#define DPIF_DOCA_TCPDUMP_MAX_IFACE_NAME 50

/* ============================================================================
 * DATA STRUCTURES
 * ============================================================================ */

OVS_PACKED(
struct dpif_doca_tcpdump_pkt_hdr {
    uint32_t pkt_len;
});

OVS_PACKED(
struct dpif_doca_tcpdump_batch_hdr {
    uint32_t batch_size;
    char dev_name[DPIF_DOCA_TCPDUMP_MAX_IFACE_NAME];
});

struct dpif_doca_tcpdump_iface_config {
    struct hmap_node node;
    char name[DPIF_DOCA_TCPDUMP_MAX_IFACE_NAME];
    uint32_t hook_mask;
};

/* ============================================================================
 * GLOBAL VARIABLES
 * ============================================================================ */

static int dpif_doca_tcpdump_fd = -1;
atomic_bool dpif_doca_tcpdump_enable = false;

/* Socket write protection - only one thread can write at a time */
static struct ovs_mutex dpif_doca_tcpdump_socket_mutex = OVS_MUTEX_INITIALIZER;

/* Interfaces map protection - only one thread can write at a time */
static struct ovs_rwlock dpif_doca_tcpdump_iface_map_rwlock = OVS_RWLOCK_INITIALIZER;
/* Global map of interfaces */
static struct hmap dpif_doca_tcpdump_iface_map OVS_GUARDED_BY(dpif_doca_tcpdump_iface_map_rwlock)
    = HMAP_INITIALIZER(&dpif_doca_tcpdump_iface_map);

/* ============================================================================
 * UTILITY FUNCTIONS
 * ============================================================================ */

/* Helper function to convert hook enum to bit mask */
static uint32_t
dpif_doca_tcpdump_hook_to_mask(enum dpif_doca_tcpdump_hook hook)
{
    return UINT32_C(1) << hook;
}

/* Helper function to check if hook is enabled for interface */
static bool
dpif_doca_tcpdump_hook_enabled_for_iface(const struct dpif_doca_tcpdump_iface_config *iface,
                                         enum dpif_doca_tcpdump_hook hook)
{
    return iface->hook_mask & dpif_doca_tcpdump_hook_to_mask(hook);
}

/* Convert hook bitmask to human-readable string representation */
static void
dpif_doca_tcpdump_hooks_to_str(struct ds *ds, uint32_t hooks)
{
    bool first = true;
    int hook;

    for (hook = 0; hook < ARRAY_SIZE(dpif_doca_tcpdump_hook_names); hook++) {
        if (hooks & dpif_doca_tcpdump_hook_to_mask(hook)) {
            if (!first) {
                ds_put_cstr(ds, ", ");
            }
            ds_put_cstr(ds, dpif_doca_tcpdump_hook_names[hook]);
            first = false;
        }
    }
}

/* Parse hook string and return hook mask */
static bool
dpif_doca_tcpdump_parse_hooks(const char *hooks_str, uint32_t *mask)
{
    char *hooks_copy, *token, *saveptr;
    uint32_t hook_mask = 0;
    int hook;

    *mask = DPIF_DOCA_TCPDUMP_DEFAULT_HOOKS;

    if (!hooks_str || !*hooks_str) {
        return true;
    }

    hooks_copy = xstrdup(hooks_str);
    token = strtok_r(hooks_copy, ",", &saveptr);

    while (token) {
        /* Skip leading/trailing white-space */
        while (*token == ' ' || *token == '\t') {
            token++;
        }
        char *end = token + strlen(token) - 1;
        while (end > token && (*end == ' ' || *end == '\t')) {
            end--;
        }
        *(end + 1) = '\0';

        for (hook = 0; hook < ARRAY_SIZE(dpif_doca_tcpdump_hook_names); hook++) {
            if (strcmp(token, dpif_doca_tcpdump_hook_names[hook]) == 0) {
                hook_mask |= dpif_doca_tcpdump_hook_to_mask(hook);
            }
        }

        token = strtok_r(NULL, ",", &saveptr);
    }

    free(hooks_copy);

    /* If no valid hooks found, use default */
    if (hook_mask == 0) {
        VLOG_DBG("no valid hook found for %s", hooks_str);
        return false;
    }

    *mask = hook_mask;
    return true;
}

/* ============================================================================
 * INTERFACE CONFIGURATION MANAGEMENT
 * ============================================================================ */

/* Lookup by name */
static struct dpif_doca_tcpdump_iface_config *
dpif_doca_tcpdump_iface_lookup_protected(const char *name)
    OVS_REQ_RDLOCK(dpif_doca_tcpdump_iface_map_rwlock)
{
    struct dpif_doca_tcpdump_iface_config *iface = NULL;

    HMAP_FOR_EACH_WITH_HASH (iface, node, hash_string(name, 0),
                             &dpif_doca_tcpdump_iface_map) {
        if (!strcmp(iface->name, name)) {
            return iface;
        }
    }

    return NULL;
}

static struct dpif_doca_tcpdump_iface_config *
dpif_doca_tcpdump_iface_lookup(const char *name)
{
    struct dpif_doca_tcpdump_iface_config *iface;

    ovs_rwlock_rdlock(&dpif_doca_tcpdump_iface_map_rwlock);
    iface = dpif_doca_tcpdump_iface_lookup_protected(name);
    ovs_rwlock_unlock(&dpif_doca_tcpdump_iface_map_rwlock);

    return iface;
}

/* Insert a new interface */
static bool
dpif_doca_tcpdump_iface_insert(const char *name, const char *hooks_str)
{
    struct dpif_doca_tcpdump_iface_config *iface;
    uint32_t mask;

    if (strlen(name) >= DPIF_DOCA_TCPDUMP_MAX_IFACE_NAME) {
        VLOG_DBG("interface name is too long: %s", name);
        return false;
    }

    if (!dpif_doca_tcpdump_parse_hooks(hooks_str, &mask)) {
        return false;
    }

    ovs_rwlock_wrlock(&dpif_doca_tcpdump_iface_map_rwlock);

    /* Check if interface already exists */
    iface = dpif_doca_tcpdump_iface_lookup_protected(name);
    if (iface) {
        /* Update existing interface */
        iface->hook_mask |= mask;
        ovs_rwlock_unlock(&dpif_doca_tcpdump_iface_map_rwlock);
        return true;
    }

    /* Add new interface */
    iface = xmalloc(sizeof *iface);
    ovs_strzcpy(iface->name, name, sizeof iface->name);
    iface->hook_mask = mask;

    hmap_insert(&dpif_doca_tcpdump_iface_map, &iface->node, hash_string(iface->name, 0));

    ovs_rwlock_unlock(&dpif_doca_tcpdump_iface_map_rwlock);

    return true;
}

/* Clear all map */
static void
dpif_doca_tcpdump_iface_clear(void)
{
    struct dpif_doca_tcpdump_iface_config *iface;

    ovs_rwlock_wrlock(&dpif_doca_tcpdump_iface_map_rwlock);

    HMAP_FOR_EACH_SAFE (iface, node, &dpif_doca_tcpdump_iface_map) {
        hmap_remove(&dpif_doca_tcpdump_iface_map, &iface->node);
        free(iface);
    }
    hmap_destroy(&dpif_doca_tcpdump_iface_map);
    hmap_init(&dpif_doca_tcpdump_iface_map);

    ovs_rwlock_unlock(&dpif_doca_tcpdump_iface_map_rwlock);
}

/* Convert interfaces map to human-readable string representation */
static void
dpif_doca_tcpdump_iface_to_str(struct ds *ds)
{
    struct dpif_doca_tcpdump_iface_config *iface;

    ovs_rwlock_rdlock(&dpif_doca_tcpdump_iface_map_rwlock);
    HMAP_FOR_EACH (iface, node, &dpif_doca_tcpdump_iface_map) {
        ds_put_format(ds, "  %s: ", iface->name);
        dpif_doca_tcpdump_hooks_to_str(ds, iface->hook_mask);
        ds_put_char(ds, '\n');
    }
    ovs_rwlock_unlock(&dpif_doca_tcpdump_iface_map_rwlock);
}

/* ============================================================================
 * SOCKET MANAGEMENT
 * ============================================================================ */

static void
dpif_doca_tcpdump_doca_tcpdump_init_sock(void)
{
    struct sockaddr_un addr = {0};

    if (dpif_doca_tcpdump_fd != -1) {
        return;
    }

    dpif_doca_tcpdump_fd = socket(AF_UNIX, SOCK_STREAM, 0);
    if (dpif_doca_tcpdump_fd < 0) {
        VLOG_DBG("socket failed");
        return;
    }

    addr.sun_family = AF_UNIX;
    snprintf(addr.sun_path, sizeof addr.sun_path,
             "%s/%s", ovs_rundir(), "doca-tcpdump.sock");

    if (connect(dpif_doca_tcpdump_fd, (struct sockaddr *) &addr, sizeof addr) == -1) {
        VLOG_DBG("connect failed");
        close(dpif_doca_tcpdump_fd);
        dpif_doca_tcpdump_fd = -1;
    }
}

/* ============================================================================
 * PACKET PROCESSING LOGIC
 * ============================================================================ */

static bool
dpif_doca_tcpdump_doca_tcpdump_should_send(enum dpif_doca_tcpdump_hook hook, const char *dev_name)
{
    struct dpif_doca_tcpdump_iface_config *iface;
    uint32_t num_ifaces;

    /* Check if "any" interface is configured */
    iface = dpif_doca_tcpdump_iface_lookup("any");
    if (iface) {
        return dpif_doca_tcpdump_hook_enabled_for_iface(iface, hook);
    }

    /* If no interfaces configured, capture on all interfaces with default hooks */
    ovs_rwlock_rdlock(&dpif_doca_tcpdump_iface_map_rwlock);
    num_ifaces = hmap_count(&dpif_doca_tcpdump_iface_map);
    ovs_rwlock_unlock(&dpif_doca_tcpdump_iface_map_rwlock);

    if (num_ifaces == 0) {
        return (DPIF_DOCA_TCPDUMP_DEFAULT_HOOKS & dpif_doca_tcpdump_hook_to_mask(hook)) != 0;
    }

    /* Check if this interface is configured and hook is enabled */
    iface = dpif_doca_tcpdump_iface_lookup(dev_name);
    if (iface) {
        return dpif_doca_tcpdump_hook_enabled_for_iface(iface, hook);
    }

    /* Interface not found in configuration */
    return false;
}

void
dpif_doca_tcpdump_send_batch__(const struct dp_packet_batch *batch,
                               const char *dev_name,
                               enum dpif_doca_tcpdump_hook hook)
{
    struct dpif_doca_tcpdump_pkt_hdr pkt_hdrs[NETDEV_MAX_BURST];
    struct dpif_doca_tcpdump_batch_hdr batch_hdr;
    struct iovec iov[1 + NETDEV_MAX_BURST * 2];
    struct dp_packet *packet;
    ssize_t to_send, sent;
    uint32_t batch_size;
    struct msghdr msg;
    int i, num_iov;

    if (!dpif_doca_tcpdump_doca_tcpdump_should_send(hook, dev_name)) {
        return;
    }

    memset(&batch_hdr, 0, sizeof(batch_hdr));
    memset(&msg, 0, sizeof(msg));

    /* 1 batch header + batch_size packet headers + batch_size packet data */
    batch_size = dp_packet_batch_size(batch);
    num_iov = 1 + batch_size * 2;

    msg.msg_iov = iov;
    msg.msg_iovlen = num_iov;

    /* First entry: batch header */
    batch_hdr.batch_size = htonl(batch_size);
    snprintf(batch_hdr.dev_name, sizeof(batch_hdr.dev_name), "%s", dev_name);
    iov[0].iov_base = &batch_hdr;
    iov[0].iov_len = sizeof(batch_hdr);
    to_send = sizeof(batch_hdr);

    /* Build remaining entries: headers and packet data */
    i = 1;

    DP_PACKET_BATCH_FOR_EACH (j, packet, batch) {
        uint32_t pkt_len = dp_packet_size(packet);

        pkt_hdrs[j].pkt_len = htonl(pkt_len);

        /* Header */
        iov[i].iov_base = &pkt_hdrs[j];
        iov[i].iov_len = sizeof(pkt_hdrs[j]);
        to_send += sizeof(pkt_hdrs[j]);
        i++;

        /* Packet data */
        iov[i].iov_base = (void *) dp_packet_data(packet);
        iov[i].iov_len = pkt_len;
        to_send += pkt_len;
        i++;
    }

    ovs_mutex_lock(&dpif_doca_tcpdump_socket_mutex);

    if (dpif_doca_tcpdump_fd == -1) {
        dpif_doca_tcpdump_doca_tcpdump_init_sock();
        if (dpif_doca_tcpdump_fd == -1) {
            ovs_mutex_unlock(&dpif_doca_tcpdump_socket_mutex);
            return;
        }
    }

    /* Send the entire batch in one syscall */
    sent = sendmsg(dpif_doca_tcpdump_fd, &msg, MSG_NOSIGNAL);
    if (sent != to_send) {
        if (sent < 0) {
            VLOG_DBG("sendmsg failed: %s (errno=%d)\n", ovs_strerror(errno), errno);
        } else {
            VLOG_DBG("sendmsg failed: only %lu sent out of %lu", sent, to_send);
        }
    }
    ovs_mutex_unlock(&dpif_doca_tcpdump_socket_mutex);
}

/* ============================================================================
 * UNIXCTL COMMAND HANDLERS
 * ============================================================================ */

static void
dpif_doca_tcpdump_doca_tcpdump_set(struct unixctl_conn *conn,
                                   int argc,
                                   const char *argv[],
                                   void *aux OVS_UNUSED)
{
    struct dpif_doca_tcpdump_iface_config *iface;
    struct ds reply = DS_EMPTY_INITIALIZER;
    int i;

    dpif_doca_tcpdump_iface_clear();

    if (argc == 1) {
        /* No arguments - capture on all interfaces with default hooks */
        atomic_store_relaxed(&dpif_doca_tcpdump_enable, true);
        unixctl_command_reply(conn, "dpif-doca/tcpdump set successfully "
                              "(all interfaces, default hooks)");
        return;
    }

    /* Parse interface specifications */
    for (i = 1; i < argc; i += 2) {
        if (i + 1 < argc) {
            /* Format: interface_name hooks_string */
            const char *iface_name = argv[i];
            const char *hooks_str = argv[i + 1];

            if (!dpif_doca_tcpdump_iface_insert(iface_name, hooks_str)) {
                unixctl_command_reply(conn, "Failed to add interface configuration");
                return;
            }
        } else {
            /* Single interface with no hooks specified - use defaults */
            const char *iface_name = argv[i];
            if (!dpif_doca_tcpdump_iface_insert(iface_name, NULL)) {
                unixctl_command_reply(conn, "Failed to add interface configuration");
                return;
            }
        }
    }

    /* All configurations succeeded, now enable dump */
    atomic_store_relaxed(&dpif_doca_tcpdump_enable, true);

    iface = dpif_doca_tcpdump_iface_lookup("any");
    if (iface) {
        ds_put_format(&reply, "dpif-doca/tcpdump set successfully for all interfaces with hooks: ");
        dpif_doca_tcpdump_hooks_to_str(&reply, iface->hook_mask);
    } else {
        uint32_t num_ifaces;

        ovs_rwlock_rdlock(&dpif_doca_tcpdump_iface_map_rwlock);
        num_ifaces = hmap_count(&dpif_doca_tcpdump_iface_map);
        ovs_rwlock_unlock(&dpif_doca_tcpdump_iface_map_rwlock);

        ds_put_format(&reply, "dpif-doca/tcpdump set successfully with %u interfaces\n",
                      num_ifaces);
        dpif_doca_tcpdump_iface_to_str(&reply);
    }

    unixctl_command_reply(conn, ds_cstr(&reply));
    ds_destroy(&reply);
}

static void
dpif_doca_tcpdump_doca_tcpdump_unset(struct unixctl_conn *conn,
                                     int argc OVS_UNUSED,
                                     const char *argv[] OVS_UNUSED,
                                     void *aux OVS_UNUSED)
{
    atomic_store_relaxed(&dpif_doca_tcpdump_enable, false);
    dpif_doca_tcpdump_iface_clear();

    ovs_mutex_lock(&dpif_doca_tcpdump_socket_mutex);
    if (dpif_doca_tcpdump_fd != -1) {
        close(dpif_doca_tcpdump_fd);
        dpif_doca_tcpdump_fd = -1;
    }
    ovs_mutex_unlock(&dpif_doca_tcpdump_socket_mutex);

    unixctl_command_reply(conn, "dpif-doca/tcpdump unset successfully");
}

static void
dpif_doca_tcpdump_doca_tcpdump_show(struct unixctl_conn *conn,
                                    int argc OVS_UNUSED,
                                    const char *argv[] OVS_UNUSED,
                                    void *aux OVS_UNUSED)
{
    struct dpif_doca_tcpdump_iface_config *iface;
    struct ds reply = DS_EMPTY_INITIALIZER;
    bool enabled;

    atomic_read_relaxed(&dpif_doca_tcpdump_enable, &enabled);

    ds_put_format(&reply, "dpif-doca/tcpdump is %sSET\n", enabled ? "" : "UN");

    if (enabled) {
        iface = dpif_doca_tcpdump_iface_lookup("any");
        if (iface) {
            ds_put_cstr(&reply, "Capturing on all interfaces with hooks: ");
            dpif_doca_tcpdump_hooks_to_str(&reply, iface->hook_mask);
            ds_put_char(&reply, '\n');
        } else {
            uint32_t num_ifaces;

            ovs_rwlock_rdlock(&dpif_doca_tcpdump_iface_map_rwlock);
            num_ifaces = hmap_count(&dpif_doca_tcpdump_iface_map);
            ovs_rwlock_unlock(&dpif_doca_tcpdump_iface_map_rwlock);

            if (num_ifaces == 0) {
                ds_put_cstr(&reply, "Capturing on all interfaces with default hooks (");
                dpif_doca_tcpdump_hooks_to_str(&reply, DPIF_DOCA_TCPDUMP_DEFAULT_HOOKS);
                ds_put_cstr(&reply, ")\n");
            } else {
                ds_put_format(&reply, "Configured interfaces (%u):\n", num_ifaces);
                dpif_doca_tcpdump_iface_to_str(&reply);
            }
        }
    }

    unixctl_command_reply(conn, ds_cstr(&reply));
    ds_destroy(&reply);
}

/* ============================================================================
 * INITIALIZATION AND REGISTRATION
 * ============================================================================ */

void
dpif_doca_tcpdump_init(void)
{
    unixctl_command_register("dpif-doca/tcpdump-set", "[interface1 hooks1] [interface2 hooks2] ...",
                             0, -1, dpif_doca_tcpdump_doca_tcpdump_set, NULL);
    unixctl_command_register("dpif-doca/tcpdump-unset", "",
                             0, 0, dpif_doca_tcpdump_doca_tcpdump_unset, NULL);
    unixctl_command_register("dpif-doca/tcpdump-show", "",
                             0, 0, dpif_doca_tcpdump_doca_tcpdump_show, NULL);
}
