# This watcher tracks /configmap/dts_config_map.ini deployed from k8s master node.
# In addition it tracks configmap files for Fluent Bit: .exp .cset, and .fset.
# Note that, Fluent Bit maps are not copied anywhere and clx should run with:
#       fluentbit-config-dir=/configmap.
#
# The configmap is deployed to container as follows:
#
#   root@telemetry-service:/# ls -al /configmap/
#   drwxr-xr-x 2 root root 4096 Jan 13 16:05 ..2022_01_13_16_05_51.3897081158
#   lrwxrwxrwx 1 root root   32 Jan 13 16:05 ..data -> ..2022_01_13_16_05_51.3897081158
#   lrwxrwxrwx 1 root root   25 Jan 13 16:03 dts_config_map.ini -> ..data/dts_config_map.ini
#
# The directory name in dereferenced path contains timestamp of the last applied config map.
# Note that, the dts_config_map.ini modification time does not change on configmap apply.
#
# On change in dereferenced name watcher copies file content
# to /config/clx_config.ini , adding [clx] section line in the beginning of the file.

import re
import time
import os.path
import logging


def copy_configuration(dts_file_name, clx_file_name):
    with open(dts_file_name, 'r') as dts_file:
        dts_content = "[clx]\n" + dts_file.read()
    with open(clx_file_name, 'w') as clx_file:
        clx_file.write(dts_content)
    return True


def resolve_symlink(linkname, verbose=False):
    res = ""
    if not os.path.exists(dts_config_path):
        logging.error("cannot dereference '%s'.  File do not exist!", linkname)
        return res

    if os.path.islink(linkname):
        res = os.path.realpath(linkname)
        if verbose:
            logging.info("Resolved symlink '%s'. Watching '%s'", linkname, res)
    else:
        logging.info("%s is not a symlink!", linkname)
    return res


def symlink_deref_change(linkname, old_name):
    cur_name = resolve_symlink(linkname)
    if cur_name != old_name:
        logging.info("symlink %s has changed to new location '%s'", linkname, cur_name)
        return True, cur_name
    return False, cur_name

def sleep_forever():
    while True:
        time.sleep(1000)


class FluentConfigmapWatcher:
    """Tracks change in any configmap file of specified extension on call is_changed()
    """
    def __init__(self, configmap_dir="/configmap", extensions=[".exp", ".cset", ".fset"]):
        self.extensions = extensions
        self.configmap_dir = configmap_dir
        self.filelist = []
        self.resolved_names_map = {}

        self.get_fluent_files()
        self.resolve_fluent_files()

    def get_fluent_files(self):
        self.filelist = []
        for name in os.listdir(self.configmap_dir):
            for ext in self.extensions:
                if name.endswith(ext):
                    self.filelist.append(os.path.join(self.configmap_dir, name))

    def resolve_fluent_files(self):
        for name in self.filelist:
            self.resolved_names_map[name] = resolve_symlink(name)

    def is_changed(self):
        changed = False
        for name in self.filelist:
            cur_changed, new_name = symlink_deref_change(name, self.resolved_names_map[name])
            if cur_changed:
                changed = True
                self.resolved_names_map[name] = new_name
        return changed


if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG, format='%(asctime)s : %(levelname)s : %(message)s')
    dts_config_path = '/configmap/dts_config_map.ini'
    clx_config_path = '/config/clx_config.ini'

    deref_filepath  = resolve_symlink(dts_config_path, verbose=True)

    # fall asleep forever if run in standalone k8s deployment
    if clx_config_path == deref_filepath:
        logging.info("%s is a symlink to %s,no need track changes", dts_config_path, clx_config_path)
        sleep_forever()

    if not deref_filepath:
        logging.info("symlink %s does not exist, no need to track changes", dts_config_path)
        sleep_forever()

    copy_configuration(deref_filepath, clx_config_path)

    fcw = FluentConfigmapWatcher()

    while True:
        ini_changed, cur_filepath = symlink_deref_change(dts_config_path, deref_filepath)
        if ini_changed or fcw.is_changed():
            deref_filepath = cur_filepath
            if cur_filepath:
                copy_configuration(cur_filepath, clx_config_path)

        time.sleep(5)
