#!/bin/bash

DEFAULT_NUM_INSTANCES=2
DEFAULT_DELAY=30
CLX_ROOT=${CLX_ROOT:-/opt/mellanox/collectx}

NUM_INSTANCES=${NUM_INSTANCES:-$DEFAULT_NUM_INSTANCES}
DELAY=${DELAY:-$DEFAULT_DELAY}

CLXCLI=${CLXCLI:-$CLX_ROOT/bin/clxcli}
BASE_HTTP_PORT=${BASE_HTTP_PORT:-9050}
WITH_CABLE=${WITH_CABLE:-true}
EXTRA_ARGS=${EXTRA_ARGS:-allow_multiple=t,managed_mode=true,write_files=false,cable=$WITH_CABLE}

LOW_FREQ_XCSET=${LOW_FREQ_XCSET:-low_freq_debug.xcset}
HIGH_FREQ_CSET=${HIGH_FREQ_CSET:-converted_enterprise_example.cset}
CSET_FOLDER=${CSET_FOLDER:-$CLX_ROOT/etc/prometheus_configs/cset}
LOW_FREQ_SAMPLE_RATE=${LOW_FREQ_SAMPLE_RATE:-5m}
HIGH_FREQ_SAMPLE_RATE=${HIGH_FREQ_SAMPLE_RATE:-30s}
LOW_FREQ_CONFIG=${LOW_FREQ_CONFIG:-counter_set=$CSET_FOLDER/$LOW_FREQ_XCSET,sample_rate=$LOW_FREQ_SAMPLE_RATE}
HIGH_FREQ_CONFIG=${HIGH_FREQ_CONFIG:-counter_set=$CSET_FOLDER/$HIGH_FREQ_CSET,sample_rate=$HIGH_FREQ_SAMPLE_RATE}

if [ ! -d "$CSET_FOLDER" ]; then
    echo "counter set folder $CSET_FOLDER does not exist. Please rerun the container with '--env CSET_FOLDER=...' set to the correct folder in the container"
    exit 1
elif [ -z "$(ls -A "$CSET_FOLDER")" ]; then
    echo "counter set folder $CSET_FOLDER is empty"
    exit 1
fi

usage() {
    echo "Usage: $0 [options]"
    echo "General Options:"
    echo "  -d|--delay <seconds>             Delay between consecutive instances. Default: $DEFAULT_DELAY"
    echo "  -n|--num-instances <number>      Number of instances to run given single instance configuration. Default: $DEFAULT_NUM_INSTANCES"
    echo "  -a|--hca  <hca>                  HCA to use for the collections. Default: mlx5_0"
    echo "  -h|--help                        Show this help message and exit"
    echo ""
    echo "Collection Configuration Options (choose one):"
    echo "  -l|--low-freq                    Run instances with low frequency configuration. Default: true"
    echo "  -i|--high-freq                   Run instances with high frequency configuration. Default: false"
    echo ""
    echo "Advanced Configuration Options:"
    echo "  -c|--config <configs>            A Comma separated key=value parameters list per instance, semicolon separated for multiple instances"
    echo "                                   Example: '-c \"KEY1=VALUE1,KEY2=VALUE2;KEY1=VALUE3,KEY2=VALUE4\"'"
    echo "                                            or '-c \"KEY1=VALUE1,KEY2=VALUE2\" -c \"KEY1=VALUE3,KEY2=VALUE4\"'"
    echo "                                   Conflicts with -a, -l, -i options"
}

run_instance_and_sleep() {
    local config=$1
    local instance=$2
    local port=$((BASE_HTTP_PORT + instance))
    config="$config http_port=$port"
    echo "Starting instance $instance with configuration: '$config'"
    $CLXCLI << EOF
collection_start $config
EOF
    echo
    sleep $DELAY
}

error_missing_file() {
    local file="${1}"
    local prefix="${2}"

    echo "$prefix $file does not exist"
    exit 1
}

CONFIGS=()

while [[ $# -gt 0 ]]; do
    key="$1"
    case $key in
        -c|--config)
            CONFIGS+=("$2")
            shift
            shift
            ;;
        -n|--num-instances)
            NUM_INSTANCES="$2"
            shift
            shift
            ;;
        -d|--delay)
            DELAY="$2"
            shift
            shift
            ;;
        -l|--low-freq)
            [ ! -f "$CSET_FOLDER/$LOW_FREQ_XCSET" ] && error_missing_file "$CSET_FOLDER/$LOW_FREQ_XCSET" "Low frequency counter set file"
            [ ! -z "$CONFIGS" ] && usage && echo "Cannot use -l|--low-freq with -c|--config" && exit 1
            CONFIGS=("$LOW_FREQ_CONFIG")
            shift
            ;;
        -i|--high-freq)
            [ ! -f "$CSET_FOLDER/$HIGH_FREQ_CSET" ] && error_missing_file "$CSET_FOLDER/$HIGH_FREQ_CSET" "High frequency counter set file"
            [ ! -z "$CONFIGS" ] && usage && echo "Cannot use -i|--high-freq with -c|--config" && exit 1
            CONFIGS=("$HIGH_FREQ_CONFIG")
            shift
            ;;
        -a|--hca)
            EXTRA_ARGS="hca=$2,$EXTRA_ARGS"
            shift
            shift
            ;;
        -h|--help)
            usage
            exit 0
            ;;
        *)
            echo "Unknown option: $1"
            usage
            exit 1
            ;;
    esac
done

if [ ${#CONFIGS[@]} -eq 0 ]; then
    [ ! -f "$CSET_FOLDER/$LOW_FREQ_XCSET" ] && error_missing_file "$CSET_FOLDER/$LOW_FREQ_XCSET" "Low frequency counter set file"
    CONFIGS=("$LOW_FREQ_CONFIG")
fi

# replace commas with spaces
EXTRA_ARGS=${EXTRA_ARGS//,/ }

# Parse CONFIGS to a resolved array
RESOLVED_CONFIGS=()
for config in "${CONFIGS[@]}"; do
    IFS=';' read -ra instances <<< "${config//,/ }"
    for instance in "${instances[@]}"; do
        RESOLVED_CONFIGS+=("$instance $EXTRA_ARGS")
    done
done

CONFIGS_SIZE=${#RESOLVED_CONFIGS[@]}
if [ $CONFIGS_SIZE -eq 1 ]; then
    for i in $(seq 0 $((NUM_INSTANCES - 1))); do
        run_instance_and_sleep "${RESOLVED_CONFIGS[0]}" $i
    done
else
    for i in $(seq 0 $((CONFIGS_SIZE - 1))); do
        index=$((i % CONFIGS_SIZE))
        run_instance_and_sleep "${RESOLVED_CONFIGS[index]}" $i
    done
fi

sleep infinity
