#!/bin/bash
DTS_CONFIG_CONFIG_MAP=/configmap/dts_config_map.ini
CONFIG_ROOT_DIR=/telemetry.config_original/configs
DTS_CONFIG_PARENT_DIR="${CONFIG_ROOT_DIR}/doca_telemetry"
DTS_COMMON_DIR="${DTS_CONFIG_PARENT_DIR}/common"
dts_config_dir=${DTS_CONFIG_PARENT_DIR}/${DTS_CONFIG_DIR:-"default"}

function error {
    echo "Error: $1"
    exit 1
}

function print_usage {
cat <<HELP_USAGE
Usage: run DOCA telemetry service

$0 [Optional Parameters]

  Optional Parameters:
      --ucx-tls ARG        set UCX_TLS to ARG
      --help               print this message and exit

HELP_USAGE
  exit 2
}

function set_ucx_tls {
    # we set UCX_TLS on DPU to reduce exposed ports in the container,
    # since UCX opens a port per transport candidate

    config_dir=${1}
    transport=${2}
    arch=$( uname -i )
    eth_transport="rc,self"
    ib_transport="ib,self"
    env_file="clx.env"

    if [[ -z ${transport} ]]; then
        if [[ "${arch}" != "aarch64" ]]; then
            return
        fi

        # DPU
        ib=0
        eth=0
        hcas=$( ls /sys/class/infiniband )
        for hca in $hcas; do
            link=$( cat /sys/class/infiniband/${hca}/ports/1/link_layer 2> /dev/null )
            if [[ ${link} = "Ethernet" ]]; then
                eth=1
            elif [[ ${link} = "InfiniBand" ]]; then
                ib=1
            fi
        done

        if [[ $ib -eq 0 ]] && [[ $eth -eq 1 ]] ; then
            transport="${eth_transport}"
        elif [[ $ib -eq 1 ]] && [[ $eth -eq 0 ]] ; then
            transport="${ib_transport}"
        else
            echo "Error: could not find link layer - did not set UCX_TLS"
            return
        fi
    fi

    cd "$config_dir" || error "could not cd to $config_dir"
    ucx_tls_found=$( cat ${env_file} | grep UCX_TLS )
    if [[ -z ${ucx_tls_found} ]]; then
        # append
        echo >> ${env_file}   # add newline
        echo "export UCX_TLS=${transport}" >> ${env_file}
    else
        # replace
        sed -i "/^export UCX_TLS/s|=.*$|=${transport}|g" ${env_file}
    fi
    cd - || error "could not cd back"
}

ucx_tls=
while case "$1" in
        --ucx-tls)      ucx_tls="${2}"; shift;;
        --help)         print_usage;;
        -?*)            echo "Unknown option ${1}"; print_usage;;
        *)              false;;
      esac
do
  shift
done


temp_dir="${CONFIG_ROOT_DIR}/telemetry-init-tmp-dir"
current_path=$(pwd)
cd /

config_dir="/config"
if [[ $(ls -1A $config_dir | wc -l) -eq 0 ]] ; then
    mkdir ${temp_dir}
    # \cp to escape default alias "cp='cp -i'" - might override files from DTS_COMMON_DIR
    \cp -rf ${DTS_COMMON_DIR}/* ${temp_dir}
    \cp -rf ${dts_config_dir}/* ${temp_dir}
    ./get_collectx_configs.sh --config=$( basename ${temp_dir} )
    ret=$?
    rm -rf ${temp_dir}
    if [[ $ret != 0 ]] ; then
        return $ret;
    fi

    if [[ -f ${DTS_CONFIG_CONFIG_MAP} && -r ${DTS_CONFIG_CONFIG_MAP} ]]; then
        # override clx_config.ini with configmap content
        echo "[clx]" > ${config_dir}/clx_config.ini
        cat ${DTS_CONFIG_CONFIG_MAP} >> ${config_dir}/clx_config.ini
    fi

    # create relative symbolic link
    cd "$config_dir" || error "could not cd to $config_dir"
    ln -s clx_config.ini dts_config.ini
    ln -s clx_ad_hoc_runner_config.ini dts_ad_hoc_runner_config.ini || true
    cp clx_ad_hoc_runner_config.ini remote_ad_hoc_runner_config.ini || true
fi

set_ucx_tls ${config_dir} ${ucx_tls}

# setup HW monitoring counters for sysfs provider
/opt/mellanox/collectx/telem/bin/setup_hwmon_counters.sh

cd "$current_path" || error "could not cd to $current_path"
