"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import re
import json
from services.modules.base_module import BaseModule
from services.modules.run_command.command_manager import CommandManager


class Storage(BaseModule):
    """
    Storage class
    """
    def get_hard_disk_info(self):
        """
        get hard disk info
        :return: dict of data
        """

        # output example:
        # #  df -lh -t ext4 --total
        # Filesystem      Size  Used Avail Use% Mounted on
        # /dev/mmcblk0p2   59G  7.8G   48G  15% /
        # total            59G  7.8G   48G  15% -

        data = {}
        res = CommandManager.run_command(connection=self.connection, command='df -lBM -t ext4 --total')
        if res.rc == 0:
            match = re.search(r'total\s+(\S+)\s+(\S+)\s+(\S+)\s+(\d+%)', res.output)
            if match:
                for k, group in zip(['Size','Used','Available','Usages'],match.groups()):
                    result = re.search('(\S+)\s*(\w+|%)',group)
                    if result:
                        data[k] = {'value':result.group(1),'unit':result.group(2)}
            if not data:
                res.update_logger_fail_to_parse_output()
        return data

    def get_disk_wearout_info(self):
        """
        get disk wearout indicator
        :return: list of dict of data
        """
        data = []
        cmd = 'ls /dev/mmcblk0'
        res = CommandManager.run_command(connection=self.connection, command=cmd)
        if res.rc == 0:
            cmd = 'mmc extcsd read /dev/mmcblk0 |grep -i "life\|eol"'
            res = CommandManager.run_command(connection=self.connection, command=cmd)
            if res.rc == 0:
                # mmc extcsd read /dev/mmcblk0 |grep -i "life\|eol"
                # eMMC Life Time Estimation A [EXT_CSD_DEVICE_LIFE_TIME_EST_TYP_A]: 0x01
                # eMMC Life Time Estimation B [EXT_CSD_DEVICE_LIFE_TIME_EST_TYP_B]: 0x01
                # eMMC Pre EOL information [EXT_CSD_PRE_EOL_INFO]: 0x01
                for line in res.output.splitlines():
                    match = re.search(r'LIFE_TIME_EST_TYP_A]:\s+(0x\S+)',line,re.IGNORECASE)
                    if match:
                        value = match.group(1)
                        res = Storage.emmc_estimate_life_time_meaning(value)
                        if res:
                            data.append({'Estimate for life time of SLC': res})
                            continue
                    match = re.search(r'LIFE_TIME_EST_TYP_B]:\s+(0x\S+)', line, re.IGNORECASE)
                    if match:
                        value = match.group(1)
                        res = Storage.emmc_estimate_life_time_meaning(value)
                        if res:
                            data.append({'Estimate for life time of MLC': res})
                            continue
                    match = re.search(r'PRE_EOL_INFO]:\s+(0x\S+)', line, re.IGNORECASE)
                    if match:
                        value = match.group(1)
                        res = Storage.emmc_pre_eol_sevirity_and_meaning(value)
                        if res:
                            data.append({'Pre EOL information is an overall status for reserved blocks on the disks': res})
                            continue
            if not data:
                res.update_logger_fail_to_parse_output()
        return data


    @classmethod
    def emmc_estimate_life_time_meaning(cls,value) -> str:

        info_data = {'0x01':'The disk has used 0%-10% of its estimated life time',
                     '0x02':'The disk has used 10%-20% of its estimated life time',
                     '0x03':'The disk has used 20%-30% of its estimated life time',
                     '0x04':'The disk has used 30%-40% of its estimated life time',
                     '0x05':'The disk has used 40%-50% of its estimated life time',
                     '0x06':'The disk has used 50%-60% of its estimated life time',
                     '0x07':'The disk has used 60%-70% of its estimated life time',
                     '0x08':'The disk has used 70%-80% of its estimated life time',
                     '0x09':'The disk has used 80%-90% of its estimated life time',
                     '0x0a':'The disk has used 90%-100% of its estimated life time',
                     '0x0b':'The disk has used 100%-110% of its estimated life time'}
        res = info_data.get(value)
        if res:
            return res
        return ''

    @classmethod
    def emmc_pre_eol_sevirity_and_meaning(cls,value)-> dict:
        info_data = {'0x01':{'Normal':'The disk has consumed less than 80% of its reserved blocks'},
                     '0x02':{'Warning':'The disk has consumed more than 80% of its reserved blocks'},
                     '0x03':{'Urgent':'The disk has consumed more than 90% of its reserved blocks'}}
        res = info_data.get(value)
        if res:
            return res
        return {}

    def get_nvme_info(self):
        """
        get nvme info
        :return: dict of data
        """

        # output example:
        # nvme list -o json
        # {
        #   "Devices" : [
        #     {
        #       "NameSpace" : 1,
        #       "DevicePath" : "/dev/nvme0n1",
        #       "Firmware" : "AEGA0103",
        #       "Index" : 0,
        #       "ModelNumber" : "KBG40ZPZ128G TOSHIBA MEMORY",
        #       "ProductName" : "Non-Volatile memory controller: KIOXIA Corporation Device 0x0001",
        #       "SerialNumber" : "32U205T2NSJ4",
        #       "UsedBytes" : 128035676160,
        #       "MaximumLBA" : 250069680,
        #       "PhysicalSize" : 128035676160,
        #       "SectorSize" : 512
        #     }
        #   ]
        # }

        data = {}
        res = CommandManager.run_command(connection=self.connection, command='nvme list -o json')
        if res.rc == 0 and res.output:
            try:
                data = json.loads(res.output)
            except Exception as exp:
                self.logger.error(f'failed to get nvme list with error: {exp}')
            if not data:
                res.update_logger_fail_to_parse_output()
        return data

    def get_nvme_smart_log(self, device=None):
        """
        get nvme smart log on specific device, in case device not define get smart log on all devices
        :return: dict of data
        """
        # output example :
        # # nvme smart-log /dev/nvme0n1  -H
        # Smart Log for NVME device:nvme0n1 namespace-id:ffffffff
        # critical_warning                        : 0
        #       Available Spare[0]             : 0
        #       Temp. Threshold[1]             : 0
        #       NVM subsystem Reliability[2]   : 0
        #       Read-only[3]                   : 0
        #       Volatile mem. backup failed[4] : 0
        #       Persistent Mem. RO[5]          : 0
        # temperature                             : 49 C (322 Kelvin)
        # available_spare                         : 100%
        # available_spare_threshold               : 10%
        # percentage_used                         : 0%
        # endurance group critical warning summary: 0
        # data_units_read                         : 2361
        # data_units_written                      : 0
        # host_read_commands                      : 274261
        # host_write_commands                     : 0
        # controller_busy_time                    : 0
        # power_cycles                            : 17
        # power_on_hours                          : 2810
        # unsafe_shutdowns                        : 11
        # media_errors                            : 0
        # num_err_log_entries                     : 0
        # Warning Temperature Time                : 0
        # Critical Composite Temperature Time     : 0
        # Temperature Sensor 1           : 49 C (322 Kelvin)
        # Thermal Management T1 Trans Count       : 0
        # Thermal Management T2 Trans Count       : 0
        # Thermal Management T1 Total Time        : 0
        # Thermal Management T2 Total Time        : 0

        data = {}
        devices_path = []
        if not device:
            res = self.get_nvme_info()
            if res:
                devices_info = res.get("Devices",[])
                for device_info in devices_info:
                    device_path = device_info.get("DevicePath", "")
                    if device_path:
                        devices_path.append(device_path)
        else:
            devices_path.append(device)
        for d_path in devices_path:
            cmd = f'nvme smart-log {d_path} -H'
            res = CommandManager.run_command(connection=self.connection, command=cmd)
            if res.rc == 0 and res.output:
                t_device = ""
                last_1_level_field = ""
                for line in res.output.splitlines():
                    match = re.search(r'device:(\S+)',line)
                    if match:
                        t_device = match.group(1).strip()
                        data.update({t_device: {}})
                        continue
                    match = re.search(r'(^\S+.+):\s*(.+)',line)
                    if match:
                        field = match.group(1).strip()
                        field_value = match.group(2).strip()
                        last_1_level_field = field
                        data[t_device].update({field: field_value})
                    else:
                        match = re.search(r'^\s+(\S+.+):\s*(.+)', line)
                        if match:
                            field = match.group(1).strip()
                            field_value = match.group(2).strip()
                            if last_1_level_field and last_1_level_field in data[t_device]:
                                data[t_device].update({f'{last_1_level_field} {field}': field_value})
                if not data:
                    res.update_logger_fail_to_parse_output()
        return data

    def get_emmc_identification(self) -> dict:
        """
        get disk emmc identification info
        :return: dict of data
        """
        data = {}
        cmd = 'mmc cid read /sys/block/mmcblk0/device/'
        res = CommandManager.run_command(connection=self.connection, command=cmd)
        if res.rc == 0:
            # output example:
            # type: 'MMC'
            # manufacturer: 'Kingston' ''
            # product: 'Y29128' 11.369600370
            # serial: 0x00000005
            # manfacturing date: 2001 oct
            for line in res.output.splitlines():
                if ':' in line:
                    key, value = line.split(':', 1)
                    data[key.strip()] = value.replace("\x00", "").replace("'", "").strip()
            if not data:
                res.update_logger_fail_to_parse_output()
        return data

    def get_emmc_config_info(self) -> str:
        """
        get emmc config info
        :return: config info as string
        """
        data = ""
        cmd = 'mmc extcsd read /dev/mmcblk0'
        res = CommandManager.run_command(connection=self.connection, command=cmd)
        if res.rc == 0:
            # output example:
            # =============================================
            #   Extended CSD rev 1.8 (MMC 5.1)
            # =============================================
            #
            # Card Supported Command sets [S_CMD_SET: 0x01]
            # HPI Features [HPI_FEATURE: 0x01]: implementation based on CMD13
            # Background operations support [BKOPS_SUPPORT: 0x01]
            # Max Packet Read Cmd [MAX_PACKED_READS: 0x3c]
            # Max Packet Write Cmd [MAX_PACKED_WRITES: 0x20]
            # Data TAG support [DATA_TAG_SUPPORT: 0x01]
            data = res.output
        return data