"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import re

from services.modules.base_module import BaseModule
from services.modules.run_command.command_manager import CommandManager


class CPU(BaseModule):
    """
    CPU class
    """

    def get_cpu_info(self):
        """
        get cpu information
        :return: dict of data
        """
        data = {}
        res = CommandManager.run_command(connection=self.connection, command='lscpu')
        if res.rc == 0:
            for raw in res.output.splitlines():
                k, v = raw.split(':', 1)
                data[k.strip()] = v.strip()
            if not data:
                res.update_logger_fail_to_parse_output()

        return data

    def get_cpu_usages_percentage(self):
        """
        get cpu usages percentage
        :return: cpu usages percentage and unit
        """
        data: dict = {}
        res = CommandManager.run_command(connection=self.connection, command='top -b -n 1 | grep Cpu | awk \'{print $8}\'|cut -f 1 -d "."')
        if res.rc == 0:
            match = re.search(r'^\s*(\d+)', res.output)
            if match:
                data['value'] = str(100 - int(match.group(1)))
                data['unit'] = "%"
            if not data:
                res.update_logger_fail_to_parse_output()
        return data

    def get_top_ten_cpu_process(self):

        data: list = []
        # output example :
        # top -b -n 1 |grep -i "PID USER" -A 10
        #     PID USER      PR  NI    VIRT    RES    SHR S  %CPU  %MEM     TIME+ COMMAND
        #      25 root      20   0       0      0      0 R  94.4   0.0  13:44.90 ksoftirqd/3
        #      40 root      20   0       0      0      0 R  94.4   0.0  14:45.57 ksoftirqd/6
        #      45 root      20   0       0      0      0 R  94.4   0.0  14:29.59 ksoftirqd/7
        #  303586 root      20   0    6024   3372   2172 R  94.4   0.0  22:11.39 iperf3
        #  303589 root      20   0    6152   3420   2124 R  94.4   0.0  22:59.85 iperf3
        #  303588 root      20   0    6024   3368   2164 S  83.3   0.0  21:55.82 iperf3
        #  303587 root      20   0    6024   3524   2320 R  61.1   0.0  22:03.40 iperf3
        #  458294 root      20   0   10696   3264   2688 R  11.1   0.0   0:00.04 top
        #       1 root      20   0  169876  10988   7148 S   0.0   0.1   1:12.45 systemd
        #       2 root      20   0       0      0      0 S   0.0   0.0   0:00.77 kthreadd
        res = CommandManager.run_command(connection=self.connection, command='top -b -n 1 |grep -i "PID USER" -A 11')
        if res.rc == 0:
            match = re.findall(
                '\s*(\d+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)',
                res.output)
            if match:
                for raw in match:
                    data_info_raw = {}
                    data_info_raw['PID'] = raw[0]
                    data_info_raw['USER'] = raw[1]
                    data_info_raw['PR'] = raw[2]
                    data_info_raw['NI'] = raw[3]
                    data_info_raw['VIRT'] = raw[4]
                    data_info_raw['RES'] = raw[5]
                    data_info_raw['SHR'] = raw[6]
                    data_info_raw['S'] = raw[7]
                    data_info_raw['%CPU'] = raw[8]
                    data_info_raw['%MEM'] = raw[9]
                    data_info_raw['TIME+'] = raw[10]
                    data_info_raw['COMMAND'] = raw[11]
                    if 'top' == data_info_raw['COMMAND'].strip():
                        continue
                    data.append(data_info_raw)
            if not data:
                res.update_logger_fail_to_parse_output()
        return data

    def get_process_and_command_for_each_core(self):
        """
        get process and command for each core
        :return: dict
        """
        data: dict = {}
        res = CommandManager.run_command(connection=self.connection, command='ps -eo pid,psr,comm --sort -pcpu')
        # output example:
        # # ps -eo pid,psr,comm --sort -pcpu
        #     PID PSR COMMAND
        # 3111656   3 iperf3
        # 3111657   5 iperf3
        # 3111658   2 iperf3
        # 3111659   0 iperf3
        #    2456   2 ovs-vswitchd
        if res.rc == 0:
            match = re.findall(r'\s*(\d+)\s+(\d+)\s+(.*)', res.output)
            if match:
                for raw in match:
                    if data.get(raw[1]):
                        data.get(raw[1]).append({'PID': raw[0], 'COMMAND': raw[2]})
                    else:
                        data[raw[1]] = [{'PID': raw[0], 'COMMAND': raw[2]}]
            if not data:
                res.update_logger_fail_to_parse_output()
        return dict(sorted(data.items()))

    def get_cores_usages(self):
        """
        get cores usages
        :return: dict
        """
        data: dict = {}
        res = CommandManager.run_command(connection=self.connection, command='mpstat -P ALL 1 1')
        if res.rc == 0:
            # output example :
            # Average:     CPU    %usr   %nice    %sys %iowait    %irq   %soft  %steal  %guest  %gnice   %idle
            # Average:     all    1.50    0.00   48.63    0.00    1.00   33.42    0.00    0.00    0.00   15.46
            # Average:       0    2.00    0.00   97.00    0.00    1.00    0.00    0.00    0.00    0.00    0.00
            # Average:       1    0.00    0.00    0.00    0.00    1.00   99.00    0.00    0.00    0.00    0.00
            # Average:       2    0.00    0.00    0.00    0.00    1.00   99.00    0.00    0.00    0.00    0.00
            # Average:       3    0.98    0.00   96.08    0.00    0.98    0.98    0.00    0.00    0.00    0.98
            # Average:       4    1.98    0.00   97.03    0.00    0.99    0.00    0.00    0.00    0.00    0.00
            # Average:       5    0.00    0.00    0.98    0.00    2.94   67.65    0.00    0.00    0.00   28.43
            # Average:       6    3.09    0.00    0.00    0.00    0.00    0.00    0.00    0.00    0.00   96.91
            # Average:       7    4.00    0.00   96.00    0.00    0.00    0.00    0.00    0.00    0.00    0.00
            match = re.findall(r'Average:\s+(\d+|all)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)', res.output, re.IGNORECASE)
            if match:
                for raw in match:
                    data[raw[0]] = str(float(raw[1])+float(raw[2])+float(raw[3])+float(raw[4])\
                                   +float(raw[5])+float(raw[6])+float(raw[7])+float(raw[8])+float(raw[9]))
            if not data:
                res.update_logger_fail_to_parse_output()
        return data

    def get_top_process_and_cpu_usages(self):
        """
        get top process and cpu usages
        :return:
        """
        data: dict = {}
        # output example :
        # top -b -n 1 |grep -i "PID USER" -A 10
        #     PID USER      PR  NI    VIRT    RES    SHR S  %CPU  %MEM     TIME+ COMMAND
        #      25 root      20   0       0      0      0 R  94.4   0.0  13:44.90 ksoftirqd/3
        #      40 root      20   0       0      0      0 R  94.4   0.0  14:45.57 ksoftirqd/6
        #      45 root      20   0       0      0      0 R  94.4   0.0  14:29.59 ksoftirqd/7
        #  303586 root      20   0    6024   3372   2172 R  94.4   0.0  22:11.39 iperf3
        #  303589 root      20   0    6152   3420   2124 R  94.4   0.0  22:59.85 iperf3
        #  303588 root      20   0    6024   3368   2164 S  83.3   0.0  21:55.82 iperf3
        #  303587 root      20   0    6024   3524   2320 R  61.1   0.0  22:03.40 iperf3
        #  458294 root      20   0   10696   3264   2688 R  11.1   0.0   0:00.04 top
        #       1 root      20   0  169876  10988   7148 S   0.0   0.1   1:12.45 systemd
        #       2 root      20   0       0      0      0 S   0.0   0.0   0:00.77 kthreadd
        res = CommandManager.run_command(connection=self.connection, command='top -b -n 1 |grep -i "PID USER" -A 100000')
        if res.rc == 0:
            match = re.findall(
                '\s*(\d+)\s+\S+\s+\S+\s+\S+\s+\d+\s+\d+\s+\d+\s+\S+\s+(\S+)\s+\S+\s+\S+\s+\S+',
                res.output)
            if match:
                for raw in match:
                    data.update({raw[0]: {'%CPU': raw[1]}})
            if not data:
                res.update_logger_fail_to_parse_output()
        return data

    def get_pid_stat(self):
        """
        get pid stat
        return: dict
        """
        data = {}
        res = CommandManager.run_command(connection=self.connection,
                                         command='pidstat 1 1 -l')
        if res.rc == 0:
            # pidstat -u
            # Linux 5.4.0-1042-bluefield    09/12/22        _aarch64_       (8 CPU)
            # 3:45:06      UID       PID    %usr %system  %guest   %wait    %CPU   CPU  Command
            # 13:45:07        0    960491    0.00    0.96    0.00    0.00    0.96     1  python3
            # 13:45:07        0   1216513    0.96    0.00    0.00    0.00    0.96     3  python3.8
            # 13:45:07        0   1218091   98.08    0.00    0.00    0.96   98.08     2  stress
            # 13:45:07        0   1218092   97.12    0.00    0.00    1.92   97.12     3  stress
            # 13:45:07        0   1218093   99.04    0.00    0.00    0.00   99.04     5  stress
            # 13:45:07        0   1218094   97.12    0.96    0.00    0.96   98.08     0  stress
            # 13:45:07        0   1218095   98.08    0.00    0.00    0.96   98.08     7  stress
            # 13:45:07        0   1218096   99.04    0.00    0.00    0.00   99.04     6  stress
            # 13:45:07        0   1218097   99.04    0.00    0.00    0.00   99.04     1  stress
            # 13:45:07        0   1218098   96.15    0.00    0.00    2.88   96.15     4  stress
            # 13:45:07        0   1218099    0.00    1.92    0.00    0.96    1.92     4  pidstat
            #
            # Average:      UID       PID    %usr %system  %guest   %wait    %CPU   CPU  Command
            # Average:        0    960491    0.00    0.96    0.00    0.00    0.96     -  python3
            # Average:        0   1216513    0.96    0.00    0.00    0.00    0.96     -  python3.8
            # Average:        0   1218091   98.08    0.00    0.00    0.96   98.08     -  stress
            # Average:        0   1218092   97.12    0.00    0.00    1.92   97.12     -  stress
            if re.search(r'(UID\s+PID\s+%usr\s+%system\s+%guest\s+%wait\s+%CPU\s+CPU\s+Command)',res.output):
                for line in res.output.splitlines():
                    match = re.search(r'\((\d+)\sCPU\)', line)
                    if match:
                        cores = int(match.group(1))
                        for c in range(cores):
                            if not data.get(str(c)):
                                data.update({str(c): {'Processes': []}})
                        continue
                    match = re.search(r'(\d+)\s+(\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+.\d+)\s+(\d+)\s+(\S+)',line)
                    if match:
                        pid = match.group(2)
                        core_pp = match.group(7)
                        core = match.group(8)
                        command = match.group(9)
                        if not data.get(core):
                            data.update({core: {'Processes':[]}})
                        data.get(core).get('Processes').append({'%CPU': core_pp, 'PID': pid, 'COMMAND': command})
            if not data:
                res.update_logger_fail_to_parse_output()
        return data

    def get_cpu_frequency(self) -> (int, str):
        """
        get cpu frequency
        :return: tuple of value and unit
        """
        data = ()
        res = CommandManager.run_command(connection=self.connection, command='/usr/bin/bfcpu-freq')
        if res.rc == 0:
            # output examples :# /usr/bin/bfcpu-freq
            # core freq = 1999MHz
            # core freq = 2302.0935MHz
            match = re.search(r'core\s*freq\s*=\s*(\d*\.*\d+)\s*([a-zA-Z]*)',res.output)
            if match:
                data = (int(float(match.group(1))), match.group(2))
            if not data:
                res.update_logger_fail_to_parse_output()
        return data