"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import re
import datetime
import logging
from services.blueman.models.model import Model
from services.blueman.models.health.system import SystemHealth
from services.modules.system.general.general import General
logger = logging.getLogger('DTS_models')


class DTS:
    """
    DOCA Telemetry service
    """

    @classmethod
    def get_doca_telemetry_service_counters(cls):
        """
        get DTS service counters, exclude counters with 0 value , exclude none up port_state
        """
        from services.modules.doca_telemetry_service.doca_telemetry_service import DocaTelemetryService
        result = DocaTelemetryService(Model.connection).get_doca_telemetry_service_counters_data()
        result_data = {}
        result_counters_info = []
        for provider_name, provider_data_list in result.items():
            result_data.update({provider_name:[]})
            devices_to_exclude = []
            device_and_sorted_counters = {}
            for provider_device_data in provider_data_list:#list of device data of specific provider
                counters = provider_device_data.get('counters')
                device_name = provider_device_data.get('device_name')
                sorted_device_provider_counter = []
                for counter_data in counters:#counters of device
                    counter_name = counter_data.get('counter_name')
                    counter_value = counter_data.get('counter_value')
                    is_num, counter_value = DTS.is_int_or_float(counter_value)
                    if is_num is False:
                        continue
                    if (counter_name == 'ib_port_state' or counter_name == 'hw_port_state') and counter_value != 4:
                        sorted_device_provider_counter.clear()
                        devices_to_exclude.append(device_name)
                        break
                    elif counter_value == 0:
                        continue
                    else:
                        sorted_device_provider_counter.append(counter_name)
                if sorted_device_provider_counter:
                    sorted_device_provider_counter = sorted(sorted_device_provider_counter)
                    if device_name not in devices_to_exclude:
                        device_and_sorted_counters.update({device_name:sorted_device_provider_counter})
            for device in sorted(device_and_sorted_counters):
                for provider_device_data in provider_data_list:  # list of device data of specific provider
                    device_name = provider_device_data.get('device_name')
                    if device_name == device:
                        sorted_counter_data = []
                        counters = provider_device_data.get('counters')
                        for s_counter in device_and_sorted_counters.get(device):
                            for counter_data in counters:  # counters of device
                                counter_name = counter_data.get('counter_name')
                                if counter_name == s_counter:
                                    counter_value = counter_data.get('counter_value')
                                    if re.search(r'drop|error', counter_name, re.IGNORECASE) and counter_value > 0:
                                        counter_data['failed_counter'] = True
                                        sorted_counter_data.insert(0,counter_data)
                                    else:
                                        counter_data['failed_counter'] = False
                                        sorted_counter_data.append(counter_data)
                                    break
                        provider_device_data.update({'counters':sorted_counter_data})
                        result_data.get(provider_name).append(provider_device_data)
                        break
        for provider in sorted(result_data):
            result_counters_info.append({provider : result_data.get(provider)})
        # resources = DTS.get_resources()
        # result_counters_info.append(resources)
        return result_counters_info

    @classmethod
    def get_resources(cls):
        resources_data = {"resources": []}
        try:
            general_obj = General(Model.connection)
            source_id = general_obj.get_host_name()
            timestamp = int(datetime.datetime.now().timestamp() * 1000000)
            cpu_usages = SystemHealth.verify_cpu_usages()
            memory_usages = SystemHealth.verify_memory_usages()
            disk_usages = SystemHealth.verify_disk_usages()
            if cpu_usages:
                core_info = []
                for core, core_data in cpu_usages.items():
                    usages = core_data.get("usages", "")
                    if usages:
                        counter_name = f"core_{core}_usages"
                        counter_description = "core usages %"
                        core_info.append({"counter_name": counter_name,
                                          "counter_value": float(usages),
                                          "counter_description": counter_description})
                resources_data["resources"].append({"device_name": "cores_usages",
                                                    "source_id": source_id,
                                                    "timestamp": timestamp,
                                                    "data_type": "resources",
                                                    "counters": core_info})
            if memory_usages:
                memory_info = []
                for field, field_value in memory_usages.items():
                    if isinstance(field_value, dict):
                        value = field_value.get("value")
                        unit = field_value.get("unit")
                        if value and unit:
                            memory_info.append({"counter_name": field.lower(),
                                                "counter_value": float(value),
                                                "counter_description": f"{field} memory in {unit}"})
                resources_data["resources"].append({"device_name": "memory_usages",
                                                    "source_id": source_id,
                                                    "timestamp": timestamp,
                                                    "data_type": "resources",
                                                    "counters": memory_info})
            if disk_usages:
                disk_info = []
                for field, field_value in disk_usages.items():
                    if isinstance(field_value, dict):
                        value = field_value.get("value")
                        unit = field_value.get("unit")
                        if value and unit:
                            disk_info.append({"counter_name": field.lower(),
                                                "counter_value": float(value),
                                                "counter_description": f"{field} disk in {unit}"})
                resources_data["resources"].append({"device_name": "disk_usages",
                                                    "source_id": source_id,
                                                    "timestamp": timestamp,
                                                    "data_type": "resources",
                                                    "counters": disk_info})
        except Exception as exp:
            logger.error(f"Failed to get resources {exp}")
        return resources_data

    @classmethod
    def is_int_or_float(cls, s):
        try:
            int_v = int(s)
            return True, int_v
        except ValueError:
            try:
                float_v = float(s)
                return True, float_v
            except ValueError:
                return False, None