"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.

This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import time
import re
import os
import sys
import uuid
import logging
from concurrent import futures
import grpc


_LOGGER = logging.getLogger('blueman_service')
import blueman_pb2, blueman_pb2_grpc

# add collectx directory to PYTHONPATH, to import services
clx_root = os.getenv('CLX_ROOT', "/opt/mellanox/collectx")
sys.path.append(clx_root)
from services.blueman.models.login.models import LoginCredential
from services.dpe.client.dpe_client import DPEClientException
from services.modules.run_command.command_manager import CommandManager

DTS_INTERNAL_ID : str = ''
BLUEMAN_SERVER_VERSION = '2.1.0'

class CheckToken:
    """
    Check Token class
    """

    def __init__(self, empty_response_cls, grpc_status_code=grpc.StatusCode.UNAUTHENTICATED, grpc_details=""):
        self._empty_rsp_cls = empty_response_cls
        self._status = grpc_status_code
        self._details = grpc_details

    def __call__(self, f):
        def wrapped_function(slf, request, context):
            meta = context.invocation_metadata()
            username = ''
            password = ''
            for item in meta:
                if len(item) == 2:
                    if item[0] == 'dts_internal_id' and item[1] == DTS_INTERNAL_ID:
                        return f(slf, request, context)
                    if 'username' == item[0]:
                        username = item[1]
                    elif 'password' == item[0]:
                        password = item[1]
                    if username and password:
                        if LoginCredential.verify_credential(username, password):
                            # use this if you want to update the context and pass some data to the method handler
                            # context.user_by_token = "sample_user"
                            return f(slf, request, context)
            context.set_code(self._status)
            context.set_details(self._details)
            return self._empty_rsp_cls()

        return wrapped_function


class BluemanService(blueman_pb2_grpc.BluemanServiceServicer):
    """
    Blueman Service class
    """

    @CheckToken(blueman_pb2.LoginRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def login(self, request, context):
        """
        verify login request
        :param request:
        :param context:
        :return: LoginResponse
        """
        try:
            response = blueman_pb2.LoginResponse()
            if request.version:
                match_client = re.search(r'(\d+).(\d+).(\d+)',request.version)
                match_server = re.search(r'(\d+).(\d+).(\d+)', BLUEMAN_SERVER_VERSION)
                if match_client and match_server:
                    major_c = match_client.group(1)
                    minor_c = match_client.group(2)
                    major_s = match_server.group(1)
                    minor_s = match_server.group(2)
                    if major_c == major_s and minor_c == minor_s:
                        context.set_code(grpc.StatusCode.OK)
                        return response
            context.set_details(f"Versions of BlueMan service and DOCA Telemetry Service not compatible "
                                f"please update to the latest available versions in NGC https://catalog.ngc.nvidia.com/ .\n"
                                f"Note, if you decide to continue with the current version"
                                f" some part of the data might not be displayed as expected.")
            context.set_code(grpc.StatusCode.INVALID_ARGUMENT)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_details("should update version")
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)


    @CheckToken(blueman_pb2.GeneralInfoRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def general_info(self, request, context):
        """
        General info
        :param request:
        :param context:
        :return: GeneralInfoResponse
        """
        _LOGGER.info('General Information')
        response = blueman_pb2.GeneralInfoResponse()
        try:
            from services.blueman.models.info.general_info import GeneralInfo
            data = GeneralInfo.get_general_info()
            response.data.update(data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get general info")

    @CheckToken(blueman_pb2.InstalledPackagesInfoRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def installed_packages_info(self, request, context):
        """
        General info
        :param request:
        :param context:
        :return: GeneralInfoResponse
        """
        _LOGGER.info('Installed packages Information')
        response = blueman_pb2.InstalledPackagesInfoResponse()
        try:
            from services.blueman.models.info.package_info import PackageInfo
            data = PackageInfo.installed_packages_info()
            for pk_data in data:
                package_name = pk_data.get('package_name')
                version = pk_data.get('version')
                package_data_obj = blueman_pb2.PackageData(package_name=package_name,version=version)
                response.package_data.append(package_data_obj)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get installed packages info")

    @CheckToken(blueman_pb2.CpuInfoRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def cpu_info(self, request, context):
        """
        Cpu information
        :param request:
        :param context:
        :return: CpuInfoResponse
        """
        _LOGGER.info('CPU information')
        response = blueman_pb2.CpuInfoResponse()
        try:
            from services.blueman.models.info.cpu_info import CPUInfo
            data = CPUInfo.cpu_info()
            response.data.update(data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get cpu info")

    @CheckToken(blueman_pb2.FirmwareInfoRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def firmware_info(self, request, context):
        """
        Firmware information
        :param request:
        :param context:
        :return: FirmwareInfoResponse
        """
        _LOGGER.info('Firmware information')
        response = blueman_pb2.FirmwareInfoResponse()
        try:
            from services.blueman.models.info.firmware_info import FirmwareInfo
            data = FirmwareInfo.get_all_ports_fw_info()
            port_counter = 1
            for port_info in data:
                port_info_obj = blueman_pb2.FirmwarePortInfo(port_number=port_counter)
                port_counter += 1
                for field_data in port_info:
                    field_data_obj = blueman_pb2.FwConfigParamData(field_name=field_data.get('field_name'),
                                                                   default=field_data.get('default'),
                                                                   current=field_data.get('current'),
                                                                   next_boot=field_data.get('next_boot'),
                                                                   description=field_data.get('description'),
                                                                   field_config_params=field_data.get('parameters'),
                                                                   changed=field_data.get('changed'))
                    port_info_obj.data.append(field_data_obj)
                response.data.append(port_info_obj)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get firmware info")

    @CheckToken(blueman_pb2.DpuOperationModeInfoRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def dpu_operation_mode_info(self, request, context):
        """
        Dpu operation mode information
        :param request:
        :param context:
        :return: DpuOperationModeInfoResponse
        """
        _LOGGER.info('dpu operation mode information')
        try:
            from services.blueman.models.info.dpu_operation_mode_info import DpuOperationModeInfo
            data = DpuOperationModeInfo.get_dpu_operation_mode_info()
            tmp_dict = {}
            for parameter, dict_value in data.items():
                dict_of_data = blueman_pb2.DictOfData(data=dict_value)
                tmp_dict[parameter] = dict_of_data
            response = blueman_pb2.DpuOperationModeInfoResponse(data=tmp_dict)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get dpu operation mode info")

    @CheckToken(blueman_pb2.NvmeInfoRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def nvme_info(self, request, context):
        """
        nvme information
        :param request:
        :param context:
        :return: NvmeInfoResponse
        """
        _LOGGER.info('nvme information')
        try:
            from services.blueman.models.info.general_info import GeneralInfo
            data = GeneralInfo.show_nvme_info()
            response = blueman_pb2.NvmeInfoResponse()
            if data:
                for list_raw in data:
                    dict_of_data = blueman_pb2.DictOfData(data=list_raw)
                    response.data.append(dict_of_data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get NVME information")

    @CheckToken(blueman_pb2.EmmcInfoRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def emmc_info(self, request, context):
        """
        eMMC identification and config information
        :param request:
        :param context:
        :return: EmmcInfoResponse
        """
        _LOGGER.info('emmc information')
        try:
            from services.blueman.models.info.general_info import GeneralInfo
            data = GeneralInfo.show_emmc_config_info_and_identification()
            response = blueman_pb2.EmmcInfoResponse(identification=data.get('identification', {}),
                                                    config_info=data.get('config_info', ""))

            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get eMMC information")

    @CheckToken(blueman_pb2.LogsHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def logs_health(self, request, context):
        """
        Logs dmesg health
        :param request:
        :param context:
        :return: LogsHealthResponse
        """
        _LOGGER.info('logs health')
        try:
            response = blueman_pb2.LogsHealthResponse()
            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.show_dmesg_log()
            for list_raw in data:
                dic_of_data = blueman_pb2.DictOfData(data=list_raw)
                response.data.insert(0, dic_of_data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get log health")

    @CheckToken(blueman_pb2.RshimLogHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def dpu_rshim_log_health(self, request, context):
        """
        DPU Rshim log
        :param request:
        :param context:
        :return: KernelModulesHealthResponse
        """
        _LOGGER.info('DPU Rshim Log health')
        try:

            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.show_rshim_log()
            data.reverse()
            response = blueman_pb2.RshimLogHealthResponse(data=data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get DPU rshim log health")

    @CheckToken(blueman_pb2.NvmeLogHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def nvme_log_health(self, request, context):
        """
        NVME device log
        :param request:
        :param context:
        :return: NvmeLogHealthResponse
        """
        _LOGGER.info('NVME Log health')
        try:
            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.show_nvme_devices_smart_log()
            if not data:
                response = blueman_pb2.NvmeLogHealthResponse()
            else:
                tmp_dict = {}
                for device, device_log in data.items():
                    dict_of_data = blueman_pb2.DictOfData(data=device_log)
                    tmp_dict.update({device: dict_of_data})
                response = blueman_pb2.NvmeLogHealthResponse(data=tmp_dict)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get NVME log health")

    @CheckToken(blueman_pb2.KernelModulesHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def kernel_modules_health(self, request, context):
        """
        kernel modules health
        :param request:
        :param context:
        :return: KernelModulesHealthResponse
        """
        _LOGGER.info('kernel_modules_health')
        try:
            response = blueman_pb2.KernelModulesHealthResponse()
            from services.blueman.models.health.services import Services
            data = Services.get_kernel_modules_health()
            for list_raw in data:
                dic_of_data = blueman_pb2.DictOfData(data=list_raw)
                response.data.append(dic_of_data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to kernel modules health")

    @CheckToken(blueman_pb2.PortStatusHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def port_status_health(self, request, context):
        """
        Port status health
        :param request:
        :param context:
        :return: PortStatusHealthResponse
        """
        _LOGGER.info('port status health')
        try:
            response = blueman_pb2.PortStatusHealthResponse()
            from services.blueman.models.health.ports import Ports

            data = Ports.get_ports_status()
            for list_raw in data:
                dic_of_data = blueman_pb2.DictOfData(data=list_raw)
                response.data.append(dic_of_data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get port status health")

    @CheckToken(blueman_pb2.SystemServicesHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def system_services_health(self, request, context):
        """
        System services health
        :param request:
        :param context:
        :return: SystemServicesHealthResponse
        """
        _LOGGER.info('system_services_health')
        try:
            response = blueman_pb2.SystemServicesHealthResponse()
            from services.blueman.models.health.services import Services

            data = Services.get_system_service_health()
            for list_raw in data:
                dic_of_data = blueman_pb2.DictOfData(data=list_raw)
                response.data.append(dic_of_data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to system services health")

    @CheckToken(blueman_pb2.CpuHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def cpu_health(self, request, context):
        _LOGGER.info('CPU health')
        try:
            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.verify_cpu_usages()
            cors_datas = {}
            if data:
                for cor_name, core_value in data.items():
                    list_of_process = []
                    for process in core_value.get('Processes'):
                        dict_of_data = blueman_pb2.DictOfData(data=process)
                        list_of_process.append(dict_of_data)
                    res = blueman_pb2.CoresData(usage=core_value.get('usages'), status=core_value.get('status'),
                                                processes=list_of_process)
                    cors_datas[cor_name] = res
            response = blueman_pb2.CpuHealthResponse(cpu_usages=cors_datas)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get cpu health")

    @CheckToken(blueman_pb2.MemoryHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def memory_health(self, request, context):
        _LOGGER.info('Memory health')
        try:
            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.verify_memory_usages()
            ram_temp_data = {}
            if data:
                ram_temp = data
                usages = {str(key): str(value) for key, value in ram_temp.get('usages').items()}
                res = blueman_pb2.RamData(total=ram_temp.get('total'), used=ram_temp.get('used'), free=ram_temp.get('free'),
                                          usage=usages, status=ram_temp.get('status'))

                ram_temp_data = {'RAM': res}
            response = blueman_pb2.MemoryHealthResponse(ram_data=ram_temp_data)
            return response

        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get memory health")

    @CheckToken(blueman_pb2.DiskHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def disk_health(self, request, context):
        _LOGGER.info('Disk health')
        try:
            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.verify_disk_usages()
            hard_disk_data = {}
            if data:
                hard_disk_temp = data
                res = blueman_pb2.HardDiskData(size=hard_disk_temp.get('Size'), used=hard_disk_temp.get('Used'),
                                               available=hard_disk_temp.get('Available'),
                                               usage=hard_disk_temp.get('Usages'), status=hard_disk_temp.get('status'))

                hard_disk_data = {'Hard_disk': res}
            response = blueman_pb2.DiskHealthResponse(hard_disk_data=hard_disk_data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get disk health")

    @CheckToken(blueman_pb2.DiskWearoutHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def disk_wearout_health(self, request, context):
        _LOGGER.info('Disk wearout health')
        try:
            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.verify_disk_wearout()
            response = blueman_pb2.DiskWearoutHealthResponse()
            if data:
                for list_raw in data:
                    dict_of_data = blueman_pb2.DictOfData(data=list_raw)
                    response.data.append(dict_of_data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get disk wearout health")

    @CheckToken(blueman_pb2.DpuTemperatureHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def dpu_temperature_health(self, request, context):
        _LOGGER.info('DPU temperature health')
        try:
            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.verify_dpu_temperature()
            dpu_temp_data = {}
            if data:
                dpu_temp = data
                temperature = {str(key): str(value) for key, value in dpu_temp.get('temperature').items()}
                res = blueman_pb2.DpuTemperatureData(
                    temperature=temperature, status=dpu_temp.get('status'))
                dpu_temp_data = {'DPU_temperature_data': res}
            response = blueman_pb2.DpuTemperatureHealthResponse(dpu_temperature_data=dpu_temp_data)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get dpu temperature health")

    @CheckToken(blueman_pb2.CpuFrequencyHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def cpu_frequency_health(self, request, context):
        _LOGGER.info('CPU frequency health')
        try:
            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.verify_cpu_frequency()

            response = blueman_pb2.CpuFrequencyHealthResponse(frequency_value=data[0],unit=data[1],status=data[2])
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get cpu frequency health")

    @CheckToken(blueman_pb2.PowerConsumptionHealthRequest, grpc.StatusCode.UNAUTHENTICATED, "Invalid token")
    def power_consumption_health(self, request, context):
        _LOGGER.info('Power Consumption health')
        try:
            from services.blueman.models.health.system import SystemHealth
            data = SystemHealth.verify_total_power()
            if not data:
                return blueman_pb2.PowerConsumptionHealthResponse(status="not supported")
            response = blueman_pb2.PowerConsumptionHealthResponse(power_consumption=data[0], unit=data[1], status=data[2])
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get Power Consumption health")

    @CheckToken(blueman_pb2.DocaServicesRequest, grpc.StatusCode.UNAUTHENTICATED,
                "Invalid token")
    def doca_services(self, request, context):
        """
        doca services
        :param request:
        :param context:
        :return: DocaServicesResponse
        """
        _LOGGER.info('DOCA services health')
        try:
            from services.blueman.models.health.doca.doca_services import DocaService
            res = DocaService.get_doca_services()
            response = blueman_pb2.DocaServicesResponse()
            for doca_service_info in res:
                doca_service_obj = blueman_pb2.DocaServiceData(name=doca_service_info.get('NAME'),
                                                               container_id=doca_service_info.get('CONTAINER'),
                                                               state=doca_service_info.get('STATE'),
                                                               logs=doca_service_info.get('LOGS'))

                response.data.append(doca_service_obj)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get doca services health")

    @CheckToken(blueman_pb2.DTSCountersRequest, grpc.StatusCode.UNAUTHENTICATED,
                "Invalid token")
    def doca_telemetry_service_counters(self, request, context):
        """
        DOCA telemetry service counters
        :param request:
        :param context:
        :return: DTSCountersResponse
        """
        _LOGGER.info('DOCA telemetry service counters')
        try:
            from services.blueman.models.dts.dts import DTS
            data = DTS.get_doca_telemetry_service_counters()
            response = blueman_pb2.DTSCountersResponse()
            for provider in data:
                for provider_name , provider_data in provider.items():
                    if not provider_name or not provider_data:
                        continue
                    providers_counters_data_obj = blueman_pb2.ProvidersCountersData(provider_name = provider_name)
                    for device_data  in provider_data:

                        device_name = device_data.get('device_name')
                        source_id = device_data.get('source_id')
                        timestamp = device_data.get('timestamp')
                        counters:list = device_data.get('counters')
                        devices_counters_data_obj = blueman_pb2.DevicesCountersData(device_name=device_name,
                                                                                    source_id=source_id,
                                                                                    timestamp=timestamp)
                        for counter in counters:
                            counter_name = counter.get('counter_name')
                            counter_value = counter.get('counter_value')
                            counter_description = counter.get('counter_description')
                            failed_counter = counter.get('failed_counter')
                            counter_data_obj = blueman_pb2.CountersData(counter_name=counter_name,
                                                                        counter_value= counter_value,
                                                                        counter_description=counter_description,
                                                                        failed_counter = failed_counter)
                            devices_counters_data_obj.counters_data.append(counter_data_obj)
                        providers_counters_data_obj.devices_counters_data.append(devices_counters_data_obj)
                    response.providers_counters_data.append(providers_counters_data_obj)
            return response
        except Exception as exp:
            _LOGGER.error(exp)
            context.set_code(grpc.StatusCode.UNIMPLEMENTED)
            context.set_details("unable to get doca telemetry counters")


class BluemanServiceController:

    def __init__(self,address,port,use_dpe_client=True):
        if use_dpe_client:
            from services.blueman.models.model import Model
            Model.connect_to_dpe_client()
        self._service = None
        max_workers = min(32, os.cpu_count() + 4)
        self._service = grpc.server(futures.ThreadPoolExecutor(max_workers=max_workers))
        blueman_pb2_grpc.add_BluemanServiceServicer_to_server(BluemanService(), self._service)
        self._service.add_insecure_port(f'{address}:{port}')

    def _set_internal_uid(self):
        global DTS_INTERNAL_ID
        if DTS_INTERNAL_ID == '':
            fname = "/var/tmp/dts_internal_id.txt"

            # check if file already exists, in case of server restart
            if os.path.isfile(fname):
                try:
                    with open(fname, 'r') as f:
                        DTS_INTERNAL_ID = f.read()
                except Exception:
                    _LOGGER.info("Regenerating internal ID")
            if DTS_INTERNAL_ID:
                return

            # create a new uid and write to file
            DTS_INTERNAL_ID = str(uuid.uuid4())
            try:
                with open(fname, "w") as f:
                    f.write(DTS_INTERNAL_ID)
            except Exception:
                _LOGGER.error("Could not write internal ID")

    @property
    def start(self):
        if self._service:
            self._set_internal_uid()
            _LOGGER.info('Start BlueMan service')
            self._service.start()
            self._service.wait_for_termination()

    @property
    def stop(self):
        if self._service:
            self._service.stop(0)


if __name__ == '__main__':
    from services.modules.run_command.command_result import CommandResult

    _LOGGER.setLevel(logging.INFO)
    logging.basicConfig(level=logging.INFO,format='%(asctime)s   %(name)-17s   %(levelname)-8s   %(message)s')
    while True:
        try:
            blueman_service_controller = None
            blueman_service_controller = BluemanServiceController(address='localhost',port='50051',use_dpe_client=True)
            blueman_service_controller.start
            CommandResult.is_verbose_mode = True
        except DPEClientException as connect_exp:
            time.sleep(5)
            error_msg = 'Failed to connect to DPE. Please check "systemctl is-active dpe.service" and if not, run "systemctl restart dpe.service"'
            #get last line in log
            result = CommandManager.run_command('tail -1 /var/log/blueman_service.log')
            if error_msg not in result.output:
                _LOGGER.error(error_msg)
        except Exception as exp:
            _LOGGER.error(exp)
        finally:
            if blueman_service_controller:
                blueman_service_controller.stop

