/*
 * Copyright © 2019-2024 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 *
 * This software product is a proprietary product of Nvidia Corporation and its affiliates
 * (the "Company") and all right, title, and interest in and to the software
 * product, including all associated intellectual property rights, are and
 * shall remain exclusively with the Company.
 *
 * This software product is governed by the End User License Agreement
 * provided with the software product.
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <glob.h>
#include <errno.h>
#include <stdbool.h>
#include <unistd.h>

/// Include the following macro to allow proper logger usage between several providers.
/// It is crucial when providers run as part of grpc / plugin runner / clx_api.
#define DEFINE_CLX_REGISTER_LOGGER

#include "common/logger.h"
#include "common/dict.h"
#include "data/source.h"
#include "events/event_provider.h"

#define NUM_EVENTS          2
#define NUM_EVENT_SOURCES   2


/// Customize *example_event_t* according to event that user needs.
/// Keep *timestamp* and *name* fields to allow Collectx query events in the future.
/// *source_index* and *value* are example fields.
/// This event has to be consistent with the schema type in terms of order and offsets
/// (see `example_event_provider_initialize` callback).
typedef struct __attribute__((packed)) example_event_t {
    clx_timestamp_t         timestamp;
    char                    name[16];
    uint16_t                source_index;
    uint64_t                value;
} example_event_t;


struct example_event_provider_t;

// callback for source processor
typedef clx_progress_status_t
        (*example_event_source_processor_t)(struct example_event_provider_t* example_provider,
                                            clx_data_serializer_t* ds);

///
/// Customize provider on the basis of the current example:
///     - fields common to all providers:
///         -# inherit *clx_event_provider_t* to allow clx use event provider callbacks
///         -# save *schema* / *schema_index* to use the correct schema from type_system
///         -# save *type_index* / type_indexes for each data source.
///                 If there are several events types then type_index should be an array
///     - provider specific fields:
///         -# events_cache          is a placeholder for events. It has to be consistent with corresponding schema types.
///         -# pending_events        is the number of events that was not written on the previous `.progress` callback call.
///                                  It usually happens when data buffer is full. If there are pending events, then send it
///                                  to the clx before generating new data.
///         -# val_for_source_[0|1]  values that is incremented on every progress step
///     - optional / implementation specific fields:
///         -# processors      helpers to progress each source
///         -# custom_option_# is the value of option that can be set in the clx_config.ini file (see `construct_event_provider()`)
///

typedef struct example_event_provider_t {
    clx_event_provider_t    base;
    clx_schema_t*           schema;
    uint8_t                 schema_index;
    uint8_t                 type_index;
    example_event_t         events_cache[NUM_EVENTS];
    uint8_t                 pending_events;
    example_event_source_processor_t
                            processors[NUM_EVENT_SOURCES];
    const char*             custom_option_1;
    const char*             custom_option_2;
    int                     val_for_source_0;
    int                     val_for_source_1;
    bool                    example_boolean_param;
} example_event_provider_t;


static void example_event_provider_generate_events(example_event_provider_t* example_provider);


bool example_event_provider_initialize(clx_event_provider_t* provider, clx_type_system_t* ts) {
    ///
    /// Implement `example_event_provider_initialize()` to create the schema and put it into *ts*.
    /// For event provider schema will contain one or several types.
    /// Provider data sources will use one schema, but may use same or different types of the schema.
    ///

    example_event_provider_t* p = (example_event_provider_t*) provider;


    /// 1. create schema as a part of type_system *ts* and save schema_index to provider
    clx_schema_t* schema = clx_type_system_add_schema(ts, provider->name, provider->version,
                                                      &(p->schema_index));
    if (schema == NULL) {
        return false;
    }

    /// 2. Create array of *clx_type_field_info_t* and put it into `clx_schema_add_type`.
    ///    Save type_index for corresponding source.
    ///
    ///    Each clx_type_field_info_t contains 5 fields:
    ///     - field_name    : name without spaces
    ///     - description
    ///     - type_name     : could either built-in type (see the table) or
    ///                       an event name that is already presented in schema.
    ///     - counting_mode : CLX_VALUE_ABSOLUTE, CLX_VALUE_ABSOLUTE_MONOTONIC,
    ///                       CLX_VALUE_RELATIVE, or CLX_VALUE_DERIVATIVE
    ///     - array_length  : set 1 for single value, set > 1 for array
    ///
    ///    see this table of CollectX built-in types
    ///     ______________________ ______________________________________
    ///    |  type name           |   CollectX alias                     |
    ///    |----------------------|--------------------------------------|
    ///    |  bool                |   BOOL                               |
    ///    |  char                |   CHAR                               |
    ///    |  short               |   SHORT                              |
    ///    |  int                 |   INT                                |
    ///    |  long                |   LONG                               |
    ///    |  long long           |   LONGLONG                           |
    ///    |  unsigned char       |   UCHAR                              |
    ///    |  unsigned short      |   USHORT                             |
    ///    |  unsigned int        |   UINT                               |
    ///    |  unsigned long       |   ULONG                              |
    ///    |  unsigned long long  |   ULONGLONG                          |
    ///    |  float               |   FLOAT                              |
    ///    |  double              |   DOUBLE                             |
    ///    |  int8_t              |   INT8                               |
    ///    |  int16_t             |   INT16                              |
    ///    |  int32_t             |   INT32                              |
    ///    |  int64_t             |   INT64                              |
    ///    |  uint8_t             |   UINT8                              |
    ///    |  uint16_t            |   UINT16                             |
    ///    |  uint32_t            |   UINT32                             |
    ///    |  uint64_t            |   UINT64                             |
    ///    |  uint64_t            |   CLX_TYPE_TIMESTAMP or "timestamp"  |
    ///    |  NULL                |   CLX_TYPE_NONE                      |
    ///    |______________________|______________________________________|
    ///


    clx_type_field_info_t example_event_fields[] = {
        {"timestamp",    "Event timestamp",  "timestamp",    CLX_VALUE_ABSOLUTE, 1},
        {"name",         "Event name",       "char",         CLX_VALUE_ABSOLUTE, 16},
        {"source_index", "Source index",     "uint16_t",     CLX_VALUE_ABSOLUTE, 1},
        {"value",        "Some value",       "uint64_t",     CLX_VALUE_ABSOLUTE, 1}
    };

    if (CLX_TS_OK != clx_schema_add_type(schema, "example_event",
                                         example_event_fields, nof_array_elements(example_event_fields),
                                         &(p->type_index))) {
        return false;
    }

    /// 3. create and add as many types as needed

    return true;
}


static
clx_source_array_t* example_event_provider_get_sources(struct clx_event_provider_t* provider) {
    ///
    /// Implement `example_event_provider_get_sources()` to create sources array and to return them clx.
    /// `.progress` callback will deliver data from the provider to clx on source-basis.
    /// Source is a pair of two strings "id" (e.g. node/switch name) and "tag" (used to create bin-file name, e.g. tag_1234556.bin).
    /// Consider using:
    ///     - clx_source_t* source = `clx_create_default_source("id_name", "tag_name");
    ///     - clx_source_array_append(sources, source);             to append source to sources array
    ///

    clx_source_array_t* sources = clx_create_source_array();
    if (sources == NULL) {
        return NULL;
    }

    /// Source #0: use  `clx_create_default_source("tag")` to create source with source_id=host_name
    clx_source_t* source = clx_create_default_source("tag_default");
    if (source == NULL) {
        clx_destroy_source_array(sources);
        return NULL;
    }

    if (!clx_source_array_append(sources, source)) {
        clx_destroy_source_array(sources);
        clx_destroy_source(source);
        return NULL;
    }

    /// Source #1: use `clx_create_source("id", "tag")` to create new source from the scratch
    source = clx_create_source("example_id", "example_tag");
    if (source == NULL) {
        clx_destroy_source_array(sources);
        return NULL;
    }

    if (!clx_source_array_append(sources, source)) {
        clx_destroy_source_array(sources);
        clx_destroy_source(source);
        return NULL;
    }

    return sources;
}


static
bool example_event_provider_start(clx_event_provider_t* provider) {
    ///
    /// Implement `example_event_provider_start()` by picking up one of 2 options:
    ///     -# `return true` if the provider does not use a separate thread
    ///     -# start pthread job for the provider. Consider a separate pthread if
    ///       provider takes a lot of time to prepare events/counters. In that case,
    ///       provider should update and cache data, and write the cached data to clx
    ///       buffer on calls of `.progress` callback.
    ///

    return true;
}


static void example_event_provider_generate_events(example_event_provider_t* example_provider) {
    ///
    /// `example_event_provider_generate_events()` is the helper for retrieving the data
    ///
    memset(example_provider->events_cache, 0, sizeof(example_provider->events_cache));

    example_provider->events_cache[0].timestamp = clx_take_timestamp();
    example_provider->events_cache[0].source_index = 0;
    strncpy(example_provider->events_cache[0].name, "even test event", sizeof(example_provider->events_cache[0].name));
    example_provider->events_cache[0].value = example_provider->val_for_source_0;
    example_provider->val_for_source_0++;

    example_provider->events_cache[1].timestamp = clx_take_timestamp();
    example_provider->events_cache[1].source_index = 0;
    strncpy(example_provider->events_cache[1].name, "odd test event", sizeof(example_provider->events_cache[1].name));
    example_provider->events_cache[1].value = example_provider->val_for_source_0;
    example_provider->val_for_source_0++;
    example_provider->pending_events = 2;
}


static
clx_progress_status_t example_event_provider_progress_source_0(example_event_provider_t* example_provider,
                                                            clx_data_serializer_t* ds) {
    /// Example of how to write events using data serializer with caching.
    /// No data is lost if we run out of buffer.

    if (example_provider->pending_events == 0) {
        /// If there are events pending from previous run do not generate new ones
        example_event_provider_generate_events(example_provider);
    }

    const void* data = &(example_provider->events_cache[NUM_EVENTS - example_provider->pending_events]);
    int num_written = clx_data_serializer_write_events(ds, example_provider->schema_index, example_provider->type_index,
                                                       data, example_provider->pending_events);
    if (num_written < 0) {
        return CLX_PROGRESS_ERROR;
    }
    example_provider->pending_events -= num_written;

    if (example_provider->pending_events != 0) {
        return CLX_PROGRESS_MORE_DATA;
    }

    /// The following block demonstrates how to force clx to write/export events.
    /// Even though the buffer may contain free space for more events.  This is
    /// used for high priority events that need to be delivered immediately
    if (example_provider->events_cache[1].value % 16 == 0) {
        return  CLX_PROGRESS_FORCE_SYNC;
    }

    return CLX_PROGRESS_SUCCESS;
}


static
clx_progress_status_t example_event_provider_progress_source_1(example_event_provider_t* example_provider,
                                                               clx_data_serializer_t* ds) {
    /// Example of how to write events by asking for a buffer and then filling it up with recent data

    size_t buffer_size = 0L;
    int num_events     = 1;
    example_event_t* example_event =
            (example_event_t*) clx_data_serializer_get_events_buffer(ds, example_provider->schema_index,
                                                                    example_provider->type_index, num_events, &buffer_size);

    if (example_event == NULL) {
        return CLX_PROGRESS_MORE_DATA;
    }

    assert(buffer_size == sizeof(example_event_t));

    memset(example_event, 0, buffer_size);

    example_event->timestamp = clx_take_timestamp();
    strncpy(example_event->name, "test event", sizeof(example_event->name));
    example_event->source_index = 1;
    example_event->value = example_provider->val_for_source_1;
    example_provider->val_for_source_1++;
    return CLX_PROGRESS_SUCCESS;
}


static
clx_progress_status_t example_event_provider_progress(clx_event_provider_t* provider,
                                                   uint16_t source_index,
                                                   clx_data_serializer_t* ds) {
    ///
    /// Implement `example_event_provider_progress` function to allow clx to get the cached data from the provider.
    /// clx writes data into its buffer and synchronizes it (writes/exports data) when buffer is full or on force_sync request.
    /// To write the data use one of 2 options:
    ///    - int num_written_events = clx_data_serializer_write_events(ds, schema_index, type_index, cashed_data, pending_events);
    ///    - example_event_t* example_event =
    ///                   (example_event_t*) clx_data_serializer_get_events_buffer(ds, schema_index, type_index, 1, &buffer_size);
    ///
    /// progress callback should return:
    ///    - CLX_PROGRESS_SUCCESS     if all pending events were written to clx buffer successfully.
    ///    - CLX_PROGRESS_MORE_DATA   if clx cannot receive all events. clx will call `.progress`
    ///                               callback again instantly to get the rest of the data.
    ///    - CLX_PROGRESS_FORCE_SYNC  to tell clx to export/write data to disk immediately
    ///    - CLX_PROGRESS_ERROR       if cannot get clx buffer or on the other erroneous scenarios
    ///
    /// see impl functions `example_event_provider_progress_source_[0|1]` for more examples
    assert(source_index < NUM_EVENT_SOURCES);
    example_event_provider_t* example_provider = (example_event_provider_t*) provider;
    return (*example_provider->processors[source_index])(example_provider, ds);
}


static
bool example_event_provider_stop(clx_event_provider_t* provider) {
    ///
    /// Implement `example_event_provider_stop()` by picking up one of 2 options:
    ///     - `return true` if the provider does not use the separate thread
    ///     - finalize the pthread job
    ///

    return true;
}


static
void example_event_provider_finalize(clx_event_provider_t* provider) {
    ///
    /// Implement for finalization step. Destroy all the structures that were allocated in `.initialize` callback.
    ///
}


static
bool example_event_provider_get_source_config(struct clx_event_provider_t* provider,
                                          uint16_t source_index,
                                          clx_event_provider_config_t* const config) {
    ///
    /// Use default values for `example_event_provider_get_source_config()` callback.
    ///

    config->param_mask = CLX_EVENT_PROVIDER_PARAM_BLOCK_SIZE |
                         CLX_EVENT_PROVIDER_PARAM_SHARE_BUFFER |
                         CLX_EVENT_PROVIDER_PARAM_COUNTERS_PROVIDER;

    config->block_size = 0;
    config->share_buffer = false;
    config->counters_provider = false;

    return true;
}


void example_event_provider_command(struct clx_event_provider_t* provider,
                                    const command_request_t* request,
                                    command_response_t* response) {
    /// Optionally implement .command callback to
    /// Process the request from agx server (clxcli) and send the response.
    /// Use to change provider configuration, e.g. to enable/disable components
    /// Note that clxcli should be updated with new commands.

    assert(provider != NULL);
    assert(request  != NULL);
    assert(response != NULL);

    /// Get command from request and prepare response with `cmd_set_string_by_key`

    /// Option 1. Accept command with command_key and boolean command_new_val to set it
    ///           to example_boolean_param if new val differs from the old one.
    bool command_new_val;
    const char *command_key = "example_boolean_param";
    example_event_provider_t* provider_ctx = (example_event_provider_t*)provider;
    if (cmd_get_bool_by_key(request, command_key, &command_new_val)) {
        if (command_new_val != provider_ctx->example_boolean_param) {
            provider_ctx->example_boolean_param = command_new_val;
            // do enable/disable
            cmd_set_string_by_key(response, command_key, "example_boolean_param has been changed");
        } else {
            cmd_set_string_by_key(response, command_key, "Nothing to do");
        }
    }


    /// Option 2. Read command as a status string.
    ///           This example considers only 2 available commands:
    ///               - enable
    ///               - disable
    const char* new_string_val = NULL;

    if (cmd_get_string_by_key(request, command_key, &new_string_val)) {
        if (!strcmp("enable", new_string_val)) {
            // enable
            log_info("[example_counters] [%s] do enable", __FUNCTION__);
            cmd_set_string_by_key(response, command_key, "accepted enable command");
        } else if (!strcmp("disable", new_string_val)) {
            // disable
            log_info("[example_counters] [%s] do disable", __FUNCTION__);
            cmd_set_string_by_key(response, command_key, "accepted disable command");
        } else {
            log_error("[example_counters] Unknown '%s' for %s", new_string_val, command_key);
            cmd_set_string_by_key(response, new_string_val, "unknown command");
        }
    }
}




/// Initialize the provider according to its structure and return it in `construct_event_provider()`
static example_event_provider_t example_event_provider;

clx_event_provider_t* construct_event_provider(const clx_dict_t* opts) {
    /// `construct_event_provider()` function is mandatory for event providers.

    example_event_provider.base.name               = "example_events";
    example_event_provider.base.description        = "Description of the provider";
    example_event_provider.base.version            = (clx_version_t){{1, 0, 0}};

    // implement the following callbacks:
    // .initialize, .start and .get sources will be called at the beginning one after another
    example_event_provider.base.initialize         = &example_event_provider_initialize;
    example_event_provider.base.start              = &example_event_provider_start;
    example_event_provider.base.get_sources        = &example_event_provider_get_sources;

    // progress function is the main routine
    example_event_provider.base.progress           = &example_event_provider_progress;

    // finalization steps
    example_event_provider.base.stop               = &example_event_provider_stop;
    example_event_provider.base.finalize           = &example_event_provider_finalize;

    // keep this callback with the default values
    example_event_provider.base.get_source_config  = &example_event_provider_get_source_config;
    example_event_provider.base.command            = &example_event_provider_command;

    example_event_provider.type_index              = CLX_UNDEFINED_TYPE;
    example_event_provider.pending_events          = 0;
    example_event_provider.processors[0]           = &example_event_provider_progress_source_0;
    example_event_provider.processors[1]           = &example_event_provider_progress_source_1;

    example_event_provider.val_for_source_0       = 0;
    example_event_provider.val_for_source_1       = 0;
    example_event_provider.example_boolean_param  = 0;

    /// 1. Get custom options form app_opts which is parsed clx_config.ini file.
    /// clx_config.ini:
    /// custom_opt_1_key=val_1
    /// custom_opt_2_key=val_2
    ///
    /// if opts == NULL `clx_dict_get()` returns NULL

    example_event_provider.custom_option_1 = clx_dict_get(opts, "custom_opt_1_key");
    example_event_provider.custom_option_2 = clx_dict_get(opts, "custom_opt_2_key");

    /// 2. return pointer to the provider
    return (clx_event_provider_t*) &example_event_provider;
}
