#!/usr/bin/env python3
"""
Copyright © 2019-2023 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.

This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import os
import shutil
import subprocess
import re
import sys
import logging

logging.basicConfig(stream=sys.stdout, level=logging.DEBUG)
logger = logging.getLogger()

file_path = "/etc/kubelet.d/doca_telemetry_standalone.yaml"
dst_path = "/var/tmp/doca_telemetry_standalone.yaml"
string_to_find = '/usr/bin/telemetry-init.sh && '
string_to_update = 'DTS_CONFIG_DIR=ufm /usr/bin/telemetry-init.sh && '
config_path = "/opt/mellanox/doca/services/telemetry/config/"
min_doca_version = '2.0.2'


class CommandResult:
    """
    Command result class
    """
    is_verbose_mode = False

    def __init__(self, std_out, std_err, returncode, command, timeout_expired=False):
        self.__output = std_out
        self.__error = std_err
        self.__rc = returncode
        self.__command = command
        self.__timeout_expired = timeout_expired

    @property
    def output(self):
        return self.__output

    @property
    def error(self):
        return self.__error

    @property
    def rc(self):
        return self.__rc

    @property
    def command(self):
        return self.__command


def run_command(command, timeout=30):
    timeout_expired = False
    p = subprocess.Popen(command, shell=True,
                         stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE,
                         universal_newlines=True,
                         stdin=subprocess.PIPE)
    try:
        std_out, std_err = p.communicate(timeout=timeout)
        rc = p.returncode
    except Exception as exp:
        p.kill()
        std_out = ''
        std_err = f'failed to run command {command} with error {exp}'
        rc = 126
        timeout_expired = True
    finally:
        return CommandResult(std_out, std_err, rc, command, timeout_expired)


def verify_doca_version():
    """
    get doca version
    :return: doca version
    """
    logger.info(f"verify DOCA version equal to {min_doca_version} or newer")
    doca_version = ''
    result = run_command(command="cat /etc/mlnx-release")
    if result.rc == 0:
        # output example:
        # # cat /etc/mlnx-release
        # DOCA_v1.1_BlueField_OS_Ubuntu_20.04-5.4.0-1013-bluefield-5.4-1.0.3.0-3.7.0.11805-1.signed-aarch64
        # the new convention: #DOCA_1.3.0_BSP_3.9.0_Ubuntu_20.04-6
        if 'DOCA_v' in result.output:
            match = re.search(r'DOCA_v(\S+)_B', result.output, re.IGNORECASE)
        else:
            match = re.search(r'DOCA_(\S+)_BSP', result.output, re.IGNORECASE)
        if match:
            doca_version = match.group(1)
    if doca_version and doca_version >= min_doca_version:
        return True
    return False


def load_bf_ptm_module():
    logger.info("Load mlxbf-ptm module")
    result = run_command(command="modprobe  mlxbf-ptm")
    if result.rc == 0:
        return True
    return False


def backup_current_dts_yaml():
    try:
        logger.info(f" backup {file_path} in {dst_path}")
        if os.path.exists(file_path):
            shutil.copy(file_path, dst_path)
    except Exception as ex:
        logger.error(ex)
        return False
    return True


def restore_default_dts_yaml():
    try:
        logger.info(f" restore default DTS yaml from  {dst_path} to {file_path}")
        if os.path.exists(dst_path):
            shutil.copy(dst_path, file_path)
    except Exception as ex:
        logger.error(ex)
        return False
    return True


def remove_conf_dir():
    try:
        logger.info(f"remove config directory: {config_path}")
        if os.path.exists(config_path):
            cmd_res = run_command(f"rm -rf {config_path}")
            if cmd_res.rc == 0:
                return True
    except Exception as ex:
        logger.error(ex)
    return False


def edit_yaml_file():
    try:
        logger.info(f" replace string: {string_to_find} with {string_to_update} in {file_path} ")
        if os.path.exists(file_path):
            f = open(file_path, "r")
            res = f.read()
            f.close()
            if res.find(string_to_find) >= 0:
                res2 = res.replace(string_to_find, string_to_update)
                f = open(file_path, "w")
                f.write(res2)
                f.close()
            else:
                return False
    except Exception as ex:
        logger.error(ex)
        return False
    return True


def start_dpe_service():
    try:
        logger.info("start DPE service")
        cmd = "systemctl restart dpe.service"
        cmd_res = run_command(cmd)
        if cmd_res.rc == 0:
            return True
    except Exception as ex:
        logger.error(ex)
    return False


if __name__ == '__main__':
    need_to_restore_configuration = False
    if verify_doca_version():
        if backup_current_dts_yaml():
            if remove_conf_dir():
                if edit_yaml_file():
                    load_bf_ptm_module()
                    start_dpe_service()
                else:
                    need_to_restore_configuration = True
            else:
                need_to_restore_configuration = True
        else:
            need_to_restore_configuration = True
    else:
        logger.warning(f"This feature need DOCA version equal to {min_doca_version} or newer")
        sys.exit(1)

    if need_to_restore_configuration:
        restore_default_dts_yaml()
        sys.exit(1)

    sys.exit(0)