"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import re
from services.modules.base_module import BaseModule
from services.modules.run_command.command_manager import CommandManager
from services.modules.system.general.general import General


class Interface(BaseModule):
    """
    Interface class
    """
    def is_all_pfs_link_layer_ethernet(self) -> bool:
        """
        verify that all the pfs link layer are ethernet
        """
        interfaces_info = self.get_physical_interfaces_info()
        for interface in interfaces_info:
            if isinstance(interface,dict):
                for name,info in interface.items():
                    if isinstance(info,dict):
                        link_layer = info.get('LINK')
                        if link_layer:
                           if not link_layer.lower().startswith('et'):
                               return False
        return True

    def get_physical_interfaces_info(self):
        """
        get physical interfaces info
        :return: list of interface info
        """
        interfaces_info: list = []
        general_obj = General(self.connection)
        interfaces = general_obj.get_dpu_physical_port_interfaces()
        oob_inf = general_obj.get_oob_interface()
        if oob_inf:
            interfaces = [oob_inf] + interfaces
        for interface in interfaces:
            res = self.get_ip_addr_show(interface)
            if res:
                link_layer = res.get(interface).get('LINK')
                if link_layer:
                    speed = self.get_interface_speed(interface,link_layer)
                    res[interface]['Speed'] = speed
                    interfaces_info.append(res)
        return interfaces_info

    def get_interface_speed(self, interface: str, link_layer:str):
        """
        get interface speed
        :param interface: interface name
        :return: speed
        """
        speed = ''
        if link_layer.lower().startswith('et'):# in case of ethernet link layer
            res = CommandManager.run_command(connection=self.connection, command=f'ethtool {interface}')
            if res.rc == 0:
                match = re.search(r'Speed:\s*(\S+)', res.output)
                if match:
                    speed = match.group(1)
        else:
            # cat /sys/class/net/ib0/device/infiniband/mlx5_0/ports/1/rate
            #10 Gb/sec (4X SDR)

            res = CommandManager.run_command(connection=self.connection,
                                             command=f'cat /sys/class/net/{interface}/device/infiniband/*/ports/*/rate')
            if res.rc == 0:
                match = re.search(r'(.*)', res.output)
                if match:
                    speed = match.group(1)
        if not speed:
            res.update_logger_fail_to_parse_output()
        return speed

    def get_ip_addr_show(self, ifname: str = ''):
        """
        :param ifname:
        :return:
        """
        ip_addr_show: dict = {}
        interface_name = ifname
        net_dev_flags: list = []
        interface_properties: list = []
        ipv6_flag = False
        ipv6_addr = ""
        ipv6_config: list = []
        res = CommandManager.run_command(connection=self.connection, command=f'ip addr show {interface_name}')
        # example:
        # ip addr show
        # 1: lo: < LOOPBACK, UP, LOWER_UP > mtu 65536 qdisc noqueue state UNKNOWN qlen 1
        # link / loopback 00:00:00:00:00:00 brd 00:00:00:00:00:00
        # inet 127.0.0.1/8 scope host lo
        #   valid_lft forever preferred_lft forever
        # inet6::1/128 scope host
        #   valid_lft forever preferred_lft forever
        # 2: eth0: < BROADCAST, MULTICAST, UP, LOWER_UP > mtu 1500 qdisc pfifo_fast state UP qlen 1000
        # link / ether 00:50:56:1a:12:07 brd ff:ff:ff:ff:ff:ff
        # inet 10.236.18.7/16 brd 10.236.255.255 scope global dynamic eth0
        #   valid_lft 31292 sec preferred_lft 31292sec
        # inet6 fdfd:fdfd:10:236:250:56ff:fe1a:1207/64 scope global mngtmpaddr dynamic
        #   valid_lft 2591964sec preferred_lft 604764sec
        # inet6 fe80::250:56ff:fe1a:1207/64 scope link
        #   valid_lft forever preferred_lft forever
        # 4: ens7: <BROADCAST,MULTICAST,UP,LOWER_UP> mtu 1500 qdisc mq state UP qlen 1000
        # link/ether ec:0d:9a:6e:76:18 brd ff:ff:ff:ff:ff:ff
        # inet 11.236.18.7/16 brd 11.236.255.255 scope global ens7
        #   valid_lft forever preferred_lft forever
        # inet6 fe80::ee0d:9aff:fe6e:7618/64 scope link
        #   valid_lft forever preferred_lft forever
        try:
            if res.rc == 0:
                lines = res.output.split('\n')
                for line in lines:
                    line = line.strip()
                    # 1: lo: <LOOPBACK,UP,LOWER_UP> mtu 65536 qdisc noqueue state UNKNOWN qlen 1
                    # 2: eth0: <BROADCAST,MULTICAST,UP,LOWER_UP> mtu 1500 qdisc pfifo_fast state UP qlen 1000
                    # 4: ens7: <BROADCAST,MULTICAST,UP,LOWER_UP> mtu 1500 qdisc mq state UP qlen 1000
                    # 26: ens7.2@ens7: <BROADCAST,MULTICAST,UP,LOWER_UP> mtu 1500 qdisc noqueue state UP
                    regex_res = re.search(r"^(\d+):\s+([\w+\.]*)(@?)(\w*)?:\s+<(\S+)>\s+(.*)", line)
                    if regex_res:
                        ipv6_flag = False
                        interface_name = regex_res.group(2)
                        ip_addr_show[interface_name] = {}
                        ip_addr_show[interface_name]["LINK_NUMBER"] = regex_res.group(1)
                        ip_addr_show[interface_name]["INTERFACE"] = regex_res.group(2)
                        if regex_res.group(4) != "":
                            ip_addr_show[interface_name]["VLAN_BASE_INTERFACE"] = regex_res.group(4)
                        net_dev_flags = regex_res.group(5).split(',')
                        ip_addr_show[interface_name]["NET_DEV_FLAGS"] = net_dev_flags
                        interface_properties = regex_res.group(6).split()
                        i = 0
                        while i < len(interface_properties):
                            ip_addr_show[interface_name][interface_properties[i].upper()] = interface_properties[i + 1]
                            i += 2
                        continue
                    # link/ether ec:0d:9a:6e:76:18 brd ff:ff:ff:ff:ff:ff
                    regex_res = re.search(r"^link\/(\S+)\s+(\S+)(\s+brd\s+(\S+))?", line)
                    if regex_res:
                        ip_addr_show[interface_name]["LINK"] = regex_res.group(1)
                        ip_addr_show[interface_name]["MAC"] = regex_res.group(2)
                        if regex_res.group(4):
                            ip_addr_show[interface_name]["LINK_BROADCAST"] = regex_res.group(4)
                        continue
                    # inet 11.236.18.7/16 brd 11.236.255.255 scope global ens7
                    regex_res = re.search(r"^inet\s+(\S+)\/(\S+)(\s+brd\s+(\S+))?(\s+scope\s+(\S+)\s+(\S+))?", line)
                    if regex_res:
                        if "IPv4" not in ip_addr_show[interface_name].keys():
                            ip_addr_show[interface_name]["IPv4"] = {}  # define "IPv4" key pointing to empty dict
                        ipv4_addr = regex_res.group(1)
                        ipv4_netmask = regex_res.group(2)
                        ip_addr_show[interface_name]["IPv4"].update({ipv4_addr: {"IPv4_NETMASK": ipv4_netmask}})
                        if regex_res.group(4):
                            ip_addr_show[interface_name]["IPv4"][ipv4_addr]["IPv4_BROADCAST"] = regex_res.group(4)
                        if regex_res.group(6):
                            ip_addr_show[interface_name]["IPv4"][ipv4_addr]["IPv4_SCOPE"] = regex_res.group(6)
                        if regex_res.group(7):
                            ip_addr_show[interface_name]["IPv4"][ipv4_addr]["IPv4_SCOPE"] += regex_res.group(7)
                        continue
                    #  valid_lft forever preferred_lft forever
                    regex_res = re.search(r"^valid_lft\s+(\S+)\s+preferred_lft\s+(\S+)", line)
                    if regex_res:
                        if ipv6_flag:
                            ip_addr_show[interface_name]["IPv6"][ipv6_addr]["IPv6_VALID_LFT"] = regex_res.group(1)
                            ip_addr_show[interface_name]["IPv6"][ipv6_addr]["IPv6_PREFERRED_LFT"] = regex_res.group(2)
                        else:
                            ip_addr_show[interface_name]["IPv4_VALID_LFT"] = regex_res.group(1)
                            ip_addr_show[interface_name]["IPv4_PREFERRED_LFT"] = regex_res.group(2)
                        continue
                    # inet6 fdfd:fdfd:10:236:250:56ff:fe1a:1207/64 scope global mngtmpaddr dynamic
                    # inet6 fe80::250:56ff:fe1a:1207/64 scope link
                    regex_res = re.search(r"^inet6\s+(\S+)\/(\S+)(\s+scope\s+(\S+))?(\s*)?(.*)?", line)
                    if regex_res:
                        if "IPv6" not in ip_addr_show[interface_name].keys():
                            ip_addr_show[interface_name]["IPv6"] = {}  # define "IPv6" key pointing to empty dict
                        ipv6_flag = True
                        ipv6_addr = regex_res.group(1)
                        ipv6_netmask = regex_res.group(2)
                        ip_addr_show[interface_name]["IPv6"].update({ipv6_addr: {"IPv6_NETMASK": ipv6_netmask}})
                        if regex_res.group(4):
                            ip_addr_show[interface_name]["IPv6"][ipv6_addr]["IPv6_SCOPE"] = regex_res.group(4)
                        if regex_res.group(6):  # all additional configurations after scope are saved as True
                            ipv6_config = regex_res.group(6).split()
                            i = 0
                            while i < len(ipv6_config):
                                ip_addr_show[interface_name]["IPv6"][ipv6_addr][ipv6_config[i].upper()] = True
                                i += 1
                        continue
            return ip_addr_show
        except Exception as exp:
            self.logger.error(exp)
            return ip_addr_show
