#
# Copyright (c) 2021, NVIDIA CORPORATION & AFFILIATES. All rights reserved.
# Copyright (C) 2015-2016 Mellanox Technologies Ltd. All rights reserved.
#
# See file LICENSE for terms.
#
import os
import sys
#sys.path.append("/opt/mellanox/grpc/python3/lib/") (for debug)
import grpc
from typing import Tuple

# add script directory to PYTHONPATH, to import auto-generated grpc modules
script_dir = os.path.dirname(os.path.abspath(__file__))
sys.path.append(script_dir)
import dpe_pb2
import dpe_pb2_grpc

# add collectx directory to PYTHONPATH, to import services
clx_root = os.getenv('CLX_ַROOT', "/opt/mellanox/collectx")
sys.path.append(clx_root)

from services.modules.run_command.command_result import CommandResult
#from services.modules.event_provider.create_result import CreateResult

REQUEST_TIMEOUT = 30
FAILED = 1


class DPEClientException(Exception):
    pass


class DpeClient:
    """
    Dpe Client class
    """
    _SERVER_ADDRESS = "/tmp/ipc_sockets/grpc.sock"

    def __init__(self):
        self._channel = grpc.insecure_channel(f"unix://{self._SERVER_ADDRESS}")
        self._stub = dpe_pb2_grpc.DPEServiceStub(self._channel)
        self._test_run_command()

    def _test_run_command(self):
        try:
            ret = self.run_command("pwd")
            if ret.rc != 0:
                raise Exception(f"error in run_command RPC: {ret.error}")
        except Exception as err:
            raise DPEClientException(f"Incomplete instance: {err}")

    def run_command(self, cmd: str, sudo_required: bool = False, timeout: int = REQUEST_TIMEOUT) -> CommandResult:
        output = None
        try:
            request = dpe_pb2.RunCommandRequest(command=cmd, sudo_required=sudo_required, timeout=timeout)
            response = self._stub.run_command(request)
            if response:
                std_out = response.std_out
                std_err = response.std_err
                rc = response.rc
                timeout_expired = response.timeout_expired
        except Exception as exp:
            std_out = ''
            std_err = f'failed to run command {cmd} with error {exp}'
            rc = 126
            timeout_expired = True
        finally:
            return CommandResult(std_out, std_err, rc, cmd, timeout_expired)

    def verify_login_credentials(self, username: str, password: str) -> bool:
        try:
            request = dpe_pb2.VerifyLoginCredentialsRequest(username=username, password=password)
            response = self._stub.verify_login_credentials(request)
            return response.authorized
        except Exception as exp:
            return False

    def __del__(self):
        self._channel.close()

"""
#  debug:
    def create_counters_meta_provider(self, options: dict) ->  Tuple[dpe_pb2.EventProviderCreateResponse, list]:
        try:
            request = dpe_pb2.CreateCountersMetaProviderRequest(options=options)
            r = self._stub.create_counters_meta_provider(request)
            if r:
                return r.event_provider_create_response, r.loaded_plugins
        except Exception as exp:
            print(f"failed to create event provider with error {exp}")

        return None, None

    def event_provider_create(self, name: str, options: dict) -> CreateResult:
        try:
            request = dpe_pb2.EventProviderCreateRequest(name=name, options=options)
            r = self._stub.event_provider_create(request)
            if r:
                return CreateResult(r.status, r.sources, r.source_configs, r.type_system, r.version, r.name, r.description, r.id)
        except Exception as exp:
            print(f"failed to create event provider with error {exp}")
       
        return FAILED, None, None, None, None, None, None, None

    def event_provider_progress(self, id_arg: dpe_pb2.ProviderId, source_index: int )-> Tuple[int, list]:
        try:
            request = dpe_pb2.EventProviderProgressRequest(id=id_arg, source_index=source_index)
            response = self._stub.event_provider_progress(request)
            if response:
                return response.status, response.collected_data
        
        except Exception as exp:
            print(f"failed to progress event provider with error {exp}")
       
        return FAILED, None
                        
    def event_provider_terminate(self, id_arg: dpe_pb2.ProviderId)-> int:
        try:
            request = dpe_pb2.EventProviderTerminateRequest(id=id_arg)
            response = self._stub.event_provider_terminate(request)
            if response:
                return response.status

        except Exception as exp:
            print(f"failed to terminate event provider with error {exp}")
       
        return FAILED


if __name__ == "__main__":
    client = DpeClient()
    custom_opts = {"enable-provider" : "grpc.hcaperf,grpc.sysfs",
                    "hcaperf.mlx5_0" : "sample",
                    "hcaperf.mlx5_2" :"sample"}

    res = client.create_counters_meta_provider(custom_opts)
    print(res[0]) # event_provider_create_response
    print(res[1]) # loaded_plugins
    
    status, data = client.event_provider_progress(res[0].id, 0)
    print(data)
    client.event_provider_terminate(res[0].id)
"""