"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import logging
import threading
import time
from services.blueman.models.verifier_config import VerifierConfig
import os
logger = logging.getLogger('blueman_models')


def get_modified_time(file_path):
    res = None
    try:
        if os.path.exists(file_path):
            res = os.path.getmtime(file_path)
    except Exception as exp:
        logger.error(f'{exp}')
    if not res:
        logger.warning(f'config file not exist: {file_path}')
    return res


class Model:
    connection = None  # DpeClient or local connection
    verifier: VerifierConfig = None
    base_time: float = time.time()
    file_path = '/config/blueman_config.ini'
    modifiedOn = get_modified_time(file_path)

    @classmethod
    def connect_to_dpe_client(cls):
        from services.dpe.client.dpe_client import DpeClient
        cls.connection = DpeClient()

    @classmethod
    def verifier_config(cls) -> VerifierConfig:
        lock = threading.Lock()
        with lock:
            if os.path.exists(cls.file_path): # verify if the file exist
                cur_time = time.time()
                if (cur_time - cls.base_time) > 60:
                    cls.base_time = cur_time
                    if not cls.modifiedOn:
                        cls.modifiedOn = os.path.getmtime(cls.file_path)
                    modified = os.path.getmtime(cls.file_path)
                    if modified != cls.modifiedOn:
                        cls.modifiedOn = modified
                        logger.info(f'file: {cls.file_path} modified, updating the verifier')
                        cls.verifier = VerifierConfig(verifier_config_file_path=cls.file_path)
                if not cls.verifier:
                    cls.verifier = VerifierConfig(verifier_config_file_path=cls.file_path)
                return cls.verifier
            if not cls.verifier:
                logger.warning(f'config file not exist: {cls.file_path}, build config according to default')
                cls.verifier = VerifierConfig()
            return cls.verifier
