#!/usr/bin/env python3
"""
@copyright:
    Copyright (C) Mellanox Technologies Ltd. 2001-2021. ALL RIGHTS RESERVED.

    This software product is a proprietary product of Mellanox Technologies
    Ltd. (the "Company") and all right, title, and interest in and to the
    software product, including all associated intellectual property rights,
    are and shall remain exclusively with the Company.

    This software product is governed by the End User License Agreement
    provided with the software product.

@author: Avi Berger
@date:   Nov 11, 2020
"""

import sys
import os
import argparse
import shutil
import pprint
import tempfile
import ipaddress
import re
from distutils.command.config import config
from configparser import ConfigParser, NoSectionError, NoOptionError

CONFIG_PATH = os.path.dirname(os.path.abspath(__file__))
DOCKER_CONFIG_PATH = '/config'
GLOBAL_IBDIAGNET_CONFIG_PATH = os.path.join(CONFIG_PATH, 'launch_ibdiagnet_config.ini')
GLOBAL_IBDIAGNET_CONFIG_SECTION = 'ibdiagnet'
GLOBAL_IBDIAGNET_CONFIG_SAMPLE_RATE = 'sample_rate'
GLOBAL_IBDIAGNET_FLUENTBIT_EXPORT_SECTION = 'fluentbit_export'
PLUGIN_ENV_FLUENT_BIT_EXPORT_ENABLE = 'plugin_env_FLUENT_BIT_EXPORT_ENABLE'
FLUENT_BIT_CONFIG = "fluent_bit_configs"
FLUENT_BIT_CONFIG_PATH = os.path.join(CONFIG_PATH, FLUENT_BIT_CONFIG)
DOCKER_FLUENT_BIT_CONFIG_PATH = os.path.join(DOCKER_CONFIG_PATH, FLUENT_BIT_CONFIG)
TARGET_FORWARD_FILE = 'forward.exp'
TARGET_UFM_ENTERPRICE_CSET_FILE = 'ufm_enterprise.cset'
TARGET_UFM_ENTERPRICE_FSET_FILE = 'ufm_enterprise.fset'
CSET_FILE = 'cset_file.cset'
FSET_FILE = 'fset_file.fset'
TARGET_FILE_LIST = [
    (TARGET_FORWARD_FILE, TARGET_FORWARD_FILE),
    (TARGET_UFM_ENTERPRICE_CSET_FILE, CSET_FILE),
    (TARGET_UFM_ENTERPRICE_FSET_FILE, FSET_FILE)
    ]

TARGET_ATRIB_NAME = 'name'
TARGET_ATRIB_HOST = 'host'
TARGET_ATRIB_ENABLE = 'enable'
TARGET_ATRIB_PORT = 'port'
TARGET_ATRIB_LAYOUT = 'msgpack_data_layout'
TARGET_ENABLE = '1'
TARGET_DISABLE = '0'
TARGET_CHOISE_EXTENDED = 'extended'
TARGET_CHOISE_STANDARD = 'standard'
TARGET_FLB_STD = "flb_std"
TARGET_CHOISE_MAP = {
    TARGET_CHOISE_EXTENDED : "custom",
    TARGET_CHOISE_STANDARD : TARGET_FLB_STD
    }
TARGET_ATRIB_COUNTER_SET = '#counterset'
TARGET_ATRIB_FIELD_SET = '#fieldset'

FLUENT_BIT_EXPORT_ENABLE = 1
FLUENT_BIT_EXPORT_DISABLE = 0
TARGET_DOESNT_EXIST_MSG = "Target %s is missing. Please add it first."
TARGET_CHOISE_COUNTERS = 'counters'
TARGET_CHOISE_FIELDS = 'fields'
TARGET_FILES_MAP = {
    TARGET_CHOISE_COUNTERS : CSET_FILE,
    TARGET_CHOISE_FIELDS : FSET_FILE
    }

TARGET_COUNTERSET = 'counterset'
TARGET_FIELDSSET = 'fieldset'
TARGET_FORWARD_FILE_MAP = {
    TARGET_CHOISE_COUNTERS : TARGET_COUNTERSET,
    TARGET_CHOISE_FIELDS : TARGET_FIELDSSET
    }
TARGET_NO = "No"
TARGET_YES= "Yes"


class CfgFileParserError(Exception):
    pass

class CfgFileParser(ConfigParser):
    def safe_get(self, section, option, default=None):
        try:
            return self.get(section, option)
        except (NoSectionError, NoOptionError):
            if default is None:
                raise
            else:
                return default

    def safe_get_bool(self, section, option, default=None):
        try:
            return self.getboolean(section, option)
        except (NoSectionError, NoOptionError):
            if default is None:
                raise
            else:
                return default
        except ValueError:
            if not default:
                raise
            else:
                return default

    def safe_get_int(self, section, option, default=None):
        try:
            return int(self.safe_get(section, option, default))
        except ValueError:
            if default is None:
                raise
            else:
                return default

    def safe_get_float(self, section, option, default=None):
        try:
            return float(self.safe_get(section, option, default))
        except ValueError:
            if default is None:
                raise
            else:
                return default

    def setAndSave(self, conf_file, section, option,value):
        """ Sets the given value to conf object, and saves changes to conf file,
            so any change will be consistent with conf file.
            If the section & option already exist in conf file, it replaces the value,
            otherwise it creates a new entry on the end of the section
            Will fail if section doesn't exist """
        try:
            # Set to conf object

            if not self.has_section(section):
                raise CfgFileParserError('Section %s not exist' % section)
            if self.safe_get(section, option) == value:
                return
            self.set(section, option, value)
            # Save change to conf file (not using conf.write() to keep all comments in conf file)
            f_read = open(conf_file, 'r')
            lines = f_read.readlines()
            f_read.close()
            new_file_lines = []
            in_section = False # current line inside the relevant section
            wasAdded = False # indicates if option & value were added to
            for line in lines:
                line = line.strip()
                if not line or line[0]=='#':
                    # empty line or comment
                    new_file_lines.append(line)
                    continue
                key = line.split('=')[0].strip()
                if in_section and key==option:
                    # wasAdded section and key inside
                    new_file_lines.append(option + ' = ' + str(value))
                    wasAdded = True
                    continue
                elif in_section and line[0]=='[' and line[-1]==']':
                    # relevant section just ended, the given option did not appear in it. adding it.
                    in_section = False
                    if not wasAdded:
                        new_file_lines.append(option + ' = ' + str(value))
                        new_file_lines.append('')
                elif not in_section and '['+section+']' in line:
                    # just entered the relevant section
                    in_section = True
                new_file_lines.append(line)
            f_write = open(conf_file, 'w')
            f_write.write('\n'.join(new_file_lines))
            f_write.close()
            return True
        except Exception as e:
            raise CfgFileParserError('Failed to set option %s in section %s to value %s. Error: %s' % (option,section,value,e))


class NameValidatorValueError(Exception):
    pass


class NameValidator(object):
    MAX = 32
    """
    Validate the target name
    """

    def __call__(self, name):
        if len(name) > NameValidator.MAX:
            raise NameValidatorValueError(
                "Invalid target name '%s': should be less then %d characters" % (name, NameValidator.MAX))

        if not re.match("^[A-Za-z0-9_-]*$", name):
            raise NameValidatorValueError(
                "Invalid target name '%s': should include only [A-Za-z0-9_-]" % name)
        return name


class HostValidatorValueError(Exception):
    pass


class HostValidator(object):

    """
    Validate the Host ipv4 address
    """

    def __call__(self, ip_address):
        try:
            ipaddress.IPv4Address(ip_address)
        except ValueError:
            raise HostValidatorValueError(
                "Invalid target IP address: '%s'" % ip_address)
        return ip_address


class PortValidatorValueError(Exception):
    pass


class PortValidator(object):
    MIN = 1
    MAX = 65536

    """
    Validate the port number
    """

    def __call__(self, port_str):
        try:
            port_num = int(port_str)
        except ValueError:
            raise PortValidatorValueError(
                "Invalid port: '%s'" % port_str)

        if port_num not in range(PortValidator.MIN, PortValidator.MAX):
            raise PortValidatorValueError(
                "Invalid port '%s': should be in range <%d-%d>" % (
                    port_str,PortValidator.MIN, PortValidator.MAX - 1))
        return port_str


class SampleRateValidatorValueError(Exception):
    pass


class SampleRateValidator(object):

    """
    Validate the sample rate number
    """
    MIN = 6
    MAX = 86401  # 24 hours

    def __call__(self, rate_str):
        try:
            rate_num = int(rate_str)
        except ValueError:
            raise SampleRateValidatorValueError(
                "Invalid sample rate: '%s'" % rate_str)
        if rate_num not in range(SampleRateValidator.MIN, SampleRateValidator.MAX):
            raise SampleRateValidatorValueError(
                "Invalid sample rate '%s': should be in range <%d-%d>" % (
                    rate_str,SampleRateValidator.MIN, SampleRateValidator.MAX-1))
        return rate_str


class ConfigureUfmTelemetryTargetError(Exception):
    pass


class ConfigureUfmTelemetryTarget(object):

    @classmethod
    def _read_forward_config(cls, config_filename):
        config = {}
        try:
            with open(config_filename) as configfile:
                for line in configfile:
                    if line.startswith(TARGET_ATRIB_COUNTER_SET):
                        config[TARGET_COUNTERSET] = TARGET_NO
                        continue
                    elif line.startswith(TARGET_ATRIB_FIELD_SET):
                        config[TARGET_FIELDSSET] = TARGET_NO
                        continue
                    elif line.startswith('#'):
                        continue
                    name, var = line.partition("=")[::2]
                    var = var.strip()
                    if (var == ''):
                        continue
                    config[name.strip()] = var
        except Exception as e:
            raise ConfigureUfmTelemetryTargetError(
                "Reading the configuration file %s has failed: %s" % (
                    os.path.basename(config_filename), str(e)))
        return config

    @classmethod
    def _update_configuration_file(cls, filename, keys , un_comment_list, comment_list):
        try:
            with open(filename, 'rU') as f_in, tempfile.NamedTemporaryFile(
                    'w', dir=os.path.dirname(filename), delete=False) as f_out:
                for line in f_in.readlines():
                    # clean line
                    if un_comment_list is not None:
                        for key in un_comment_list:
                            if line.startswith(key):
                                line = line[1:]
                                if line.startswith(" "):
                                   line = line[1:]
                            break
                    # replace key in line
                    if keys is not None:
                        for key, value in keys.items():
                            if line.startswith(key):
                                line = '='.join((line.split('=')[0], '{}'.format(value + "\n")))
                                break
                    # comment line
                    if comment_list is not None:
                        for key in comment_list:
                            if line.startswith(key):
                                line = '#' + line
                            break

                    f_out.write(line)
        except Exception as e:
            raise ConfigureUfmTelemetryTargetError(
                "Updating the configuration file %s has failed: %s" % (
                    os.path.basename(filename), str(e)))

        try:
            # remove old version
            os.unlink(filename)
            # rename new version
            os.rename(f_out.name, filename)
        except Exception as e:
            raise ConfigureUfmTelemetryTargetError(
                "Updating the configuration file %s has failed: %s" % (
                    os.path.basename(filename), str(e)))

    @classmethod
    def _create_link_to_target(cls, target_name):
        target_src = os.path.join(target_name, TARGET_FORWARD_FILE)
        name = "%s_%s" % (target_name, TARGET_FORWARD_FILE)
        target_dst = os.path.join(FLUENT_BIT_CONFIG_PATH, name)
        try:
            os.symlink(target_src, target_dst)
        except Exception as e:
            raise ConfigureUfmTelemetryTargetError(
                "Creating link for target %s configuration has failed: %s" % (
                    target_name, str(e)))

    @classmethod
    def _show_target(cls, target_name):
        target_path = os.path.join(FLUENT_BIT_CONFIG_PATH, target_name)
        if os.path.exists(target_path):
            config_filename = os.path.join(target_path, TARGET_FORWARD_FILE)
            config = cls._read_forward_config(config_filename)
            print("     Name:                  ", config[TARGET_ATRIB_NAME])
            print("     Enabled:                " , end="")
            if config[TARGET_ATRIB_ENABLE] == TARGET_ENABLE:
                print(TARGET_YES)
            else:
                print(TARGET_NO)
            print("     Host:                  ", config[TARGET_ATRIB_HOST])
            print("     Port:                  ", config[TARGET_ATRIB_PORT])
            print("     Message Type:          ", end="")
            if config[TARGET_ATRIB_LAYOUT] == TARGET_FLB_STD:
                print("", TARGET_CHOISE_STANDARD.title())
            else:
                print("", TARGET_CHOISE_EXTENDED.title())
            print("     %s filter file:   " % (TARGET_CHOISE_COUNTERS.capitalize()), end="")
            if config[TARGET_COUNTERSET] == TARGET_NO:
                print(TARGET_NO)
            else:
                print(TARGET_YES)
            print("     %s filter file:     " % (TARGET_CHOISE_FIELDS.capitalize()), end="")
            if config[TARGET_FIELDSSET] == TARGET_NO:
                print(TARGET_NO)
            else:
                print(TARGET_YES)

        else:
            raise ConfigureUfmTelemetryTargetError(TARGET_DOESNT_EXIST_MSG % target_name)

    @classmethod
    def _show_streaming(cls):
        conf = CfgFileParser()
        conf.read([GLOBAL_IBDIAGNET_CONFIG_PATH])
        print("Enabled:", end="     ")
        if (conf.safe_get_int(GLOBAL_IBDIAGNET_FLUENTBIT_EXPORT_SECTION, PLUGIN_ENV_FLUENT_BIT_EXPORT_ENABLE) == FLUENT_BIT_EXPORT_ENABLE):
            print(TARGET_YES)
        elif (conf.safe_get_int(GLOBAL_IBDIAGNET_FLUENTBIT_EXPORT_SECTION, PLUGIN_ENV_FLUENT_BIT_EXPORT_ENABLE) == FLUENT_BIT_EXPORT_DISABLE):
            print(TARGET_NO)
        else:
            print("NA")

    @classmethod
    def _show_targets(cls, args):
        cls._show_streaming()
        if args.target_name is not None:
            cls._show_target(args.target_name)
        else:
            onlydirs = [f for f in os.listdir(FLUENT_BIT_CONFIG_PATH) if os.path.isdir(os.path.join(FLUENT_BIT_CONFIG_PATH, f))]
            print('\nTelemetry Targets:', end=" ")
            if not onlydirs:
                print('\n     No telemetry target is configured')
            else:
                for target_name in onlydirs:
                    print("")
                    cls._show_target(target_name)


    @classmethod
    def  _remove_target(cls, args):
        target_path = os.path.join(FLUENT_BIT_CONFIG_PATH, args.target_name)
         # check target path
        if os.path.isdir(target_path):
            try:
                shutil.rmtree(target_path)
            except Exception as e:
                raise ConfigureUfmTelemetryTargetError(
                    "Removing target %s configuration has failed: %s" % (
                        args.target_name, str(e)))

        # check for target link
        target_name_link = "%s_%s" % (args.target_name, TARGET_FORWARD_FILE)
        target_link_path = os.path.join(FLUENT_BIT_CONFIG_PATH, target_name_link)
        if os.path.islink(target_link_path):
            try:
                os.unlink(target_link_path)
            except Exception as e:
                raise ConfigureUfmTelemetryTargetError(
                    "Removing target %s configuration link has failed: %s" % (
                        args.target_name, str(e)))

    @classmethod
    def _add_target(cls, args):
        # create target directory
        target_path = os.path.join(FLUENT_BIT_CONFIG_PATH, args.target_name)
        cls._remove_target(args)
        try:
            os.mkdir(target_path)
        except OSError as e:
            raise ConfigureUfmTelemetryTargetError("Creating target %s configuration folder has failed: %s" % (
                args.target_name, str(e)))

        # copy templates to target folder
        for template_file, target_file in TARGET_FILE_LIST:
            filename = os.path.join(target_path, target_file)
            shutil.copyfile(os.path.join(FLUENT_BIT_CONFIG_PATH, template_file), filename)

        config_filename = os.path.join(target_path, TARGET_FORWARD_FILE)
        attributs = {}
        un_comment_attrib_list = ['# msgpack_data_layout']
        attributs[TARGET_ATRIB_NAME] = args.target_name
        attributs[TARGET_ATRIB_HOST] = args.target_host
        attributs[TARGET_ATRIB_ENABLE] = TARGET_ENABLE
        attributs[TARGET_ATRIB_PORT] = args.target_port
        attributs[TARGET_ATRIB_LAYOUT] = TARGET_CHOISE_MAP[args.target_message_type]
        attributs[TARGET_ATRIB_COUNTER_SET] = os.path.join(DOCKER_FLUENT_BIT_CONFIG_PATH,
                                                             args.target_name,
                                                             CSET_FILE)
        attributs[TARGET_ATRIB_FIELD_SET] = os.path.join(DOCKER_FLUENT_BIT_CONFIG_PATH,
                                                             args.target_name,
                                                             FSET_FILE)
        cls._update_configuration_file(config_filename, attributs, un_comment_attrib_list, None)

        cls._create_link_to_target(args.target_name)

    @classmethod
    def  _change_global_target_state(cls, state):
            attributs = {}
            attributs[PLUGIN_ENV_FLUENT_BIT_EXPORT_ENABLE] = state
            cls._update_configuration_file(GLOBAL_IBDIAGNET_CONFIG_PATH, attributs, None, None)

    @classmethod
    def  _change_specific_target_state(cls, state, target_name):
            attributs = {}
            target_path = os.path.join(FLUENT_BIT_CONFIG_PATH, target_name)
            config_filename = os.path.join(target_path, TARGET_FORWARD_FILE)
            attributs[TARGET_ATRIB_ENABLE] = state
            cls._update_configuration_file(config_filename, attributs, None, None)

    @classmethod
    def  _enable_target(cls, args):
        # enable a specific target
        cls._change_specific_target_state(TARGET_ENABLE, args.target_name)

    @classmethod
    def  _enable_streaming(cls, args):
        conf = CfgFileParser()
        conf.read([GLOBAL_IBDIAGNET_CONFIG_PATH])
        conf.setAndSave(GLOBAL_IBDIAGNET_CONFIG_PATH, GLOBAL_IBDIAGNET_FLUENTBIT_EXPORT_SECTION,
                        PLUGIN_ENV_FLUENT_BIT_EXPORT_ENABLE,
                        str(FLUENT_BIT_EXPORT_ENABLE))

    @classmethod
    def  _disable_target(cls, args):
        # disable a specific target
        cls._change_specific_target_state(TARGET_DISABLE, args.target_name)

    @classmethod
    def  _disable_streaming(cls, args):
        conf = CfgFileParser()
        conf.read([GLOBAL_IBDIAGNET_CONFIG_PATH])
        conf.setAndSave(GLOBAL_IBDIAGNET_CONFIG_PATH, GLOBAL_IBDIAGNET_FLUENTBIT_EXPORT_SECTION,
                        PLUGIN_ENV_FLUENT_BIT_EXPORT_ENABLE,
                        str(FLUENT_BIT_EXPORT_DISABLE))

    @classmethod
    def  _modify_target(cls, args):
        attributs = {}
        target_path = os.path.join(FLUENT_BIT_CONFIG_PATH, args.target_name)
        config_filename = os.path.join(target_path, TARGET_FORWARD_FILE)
        if os.path.exists(target_path):
            if args.target_host is not None:
                attributs[TARGET_ATRIB_HOST] = args.target_host
            if args.target_port is not None:
                attributs[TARGET_ATRIB_PORT] = args.target_port
            if args.target_message_type is not None:
                if args.target_message_type in TARGET_CHOISE_MAP:
                    attributs[TARGET_ATRIB_LAYOUT] = TARGET_CHOISE_MAP[args.target_message_type]
                else:
                    raise ConfigureUfmTelemetryTargetError("Message type %s is not supported" % args.target_message_type)

            cls._update_configuration_file(config_filename, attributs, None, None)
        else:
            raise ConfigureUfmTelemetryTargetError(TARGET_DOESNT_EXIST_MSG % args.target_name)

    @classmethod
    def  _import_filter_file(cls, args):
        un_comment_attrib_list = []
        target_path = os.path.join(FLUENT_BIT_CONFIG_PATH, args.target_name)
        config_filename = os.path.join(target_path, TARGET_FORWARD_FILE)
        if os.path.exists(target_path):
            if os.path.isfile(args.file_path):
                if (args.target_filter_type is not None) and (args.target_filter_type in TARGET_FILES_MAP) \
                    and (args.target_filter_type in TARGET_FORWARD_FILE_MAP):
                        filename = os.path.join(target_path, TARGET_FILES_MAP[args.target_filter_type])
                        un_comment_attrib_list.append('#'+TARGET_FORWARD_FILE_MAP[args.target_filter_type])
                else:
                    raise ConfigureUfmTelemetryTargetError("Filter type %s is not supported" % args.target_filter_type)
                try:
                    shutil.copy(args.file_path, filename, follow_symlinks=True)
                except Exception as e:
                    raise ConfigureUfmTelemetryTargetError(
                        "Copying the filter file for target %s has failed: %s" % (
                            args.target_name, str(e)))

                cls._update_configuration_file(config_filename, None, un_comment_attrib_list, None)
            else:
                raise ConfigureUfmTelemetryTargetError("Missing filter file %s" % args.file_path)
        else:
            raise ConfigureUfmTelemetryTargetError(TARGET_DOESNT_EXIST_MSG % args.target_name)


    @classmethod
    def  _disable_filter_file(cls, args):
        comment_attrib_list = []
        target_path = os.path.join(FLUENT_BIT_CONFIG_PATH, args.target_name)
        if os.path.exists(target_path):
            config_filename = os.path.join(target_path, TARGET_FORWARD_FILE)
            if (args.target_filter_type is not None) and (args.target_filter_type in TARGET_FORWARD_FILE_MAP):
                    comment_attrib_list.append(TARGET_FORWARD_FILE_MAP[args.target_filter_type])
            else:
                raise ConfigureUfmTelemetryTargetError("Filter type %s is not supported" % args.target_filter_type)
            cls._update_configuration_file(config_filename, None, None, comment_attrib_list)
        else:
            raise ConfigureUfmTelemetryTargetError(TARGET_DOESNT_EXIST_MSG % args.target_name)

    @classmethod
    def  _set_sample_rate(cls, args):
        conf = CfgFileParser()
        conf.read([GLOBAL_IBDIAGNET_CONFIG_PATH])
        conf.setAndSave(GLOBAL_IBDIAGNET_CONFIG_PATH, GLOBAL_IBDIAGNET_CONFIG_SECTION,
                        GLOBAL_IBDIAGNET_CONFIG_SAMPLE_RATE,
                        args.rate)

    @classmethod
    def  _show_sample_rate(cls, args):
        conf = CfgFileParser()
        conf.read([GLOBAL_IBDIAGNET_CONFIG_PATH])
        print("Ibdiagnet sample rate is: ", conf.safe_get(GLOBAL_IBDIAGNET_CONFIG_SECTION,
                                                         GLOBAL_IBDIAGNET_CONFIG_SAMPLE_RATE))


    @classmethod
    def run(cls):
        args = cls._parse_args()
        if args.subparser_name is None:
            print("% No arguments")
        else:
            args.func(args)

    @classmethod
    def _parse_args(cls):
        parser = argparse.ArgumentParser(usage='%(prog)s <command> [<args>]')
        subparsers = parser.add_subparsers(help='Commands', dest='subparser_name')

        parser_add_target = subparsers.add_parser(
            'add-target',
            help='Add a telemetry target')
        parser_add_target.add_argument(
            '-n', '--target-name',
            type=NameValidator(),
            metavar='<[A-Za-z0-9_-] Name size: 32>',
            help='Target name',
            required=True)
        parser_add_target.add_argument(
            '-H', '--target-host',
            type=HostValidator(),
            metavar='<IPv4>',
            help='IPv4 address',

            required=True)
        parser_add_target.add_argument(
            '-p', '--target-port',
            type=PortValidator(),
            metavar='<1-65535>',
            help='Port number',
            required=True)
        parser_add_target.add_argument(
            '-m', '--target-message-type',
            required=True,
            choices=[TARGET_CHOISE_EXTENDED, TARGET_CHOISE_STANDARD])
        parser_add_target.set_defaults(func=cls._add_target)


        parser_show_target = subparsers.add_parser(
            'show-target',
            help='Show telemetry target(s)')
        parser_show_target.add_argument(
            '-n', '--target-name')
        parser_show_target.set_defaults(func=cls._show_targets)


        parser_remove_target = subparsers.add_parser(
            'remove-target',
            help='Remove a telemetry target')
        parser_remove_target.add_argument(
            '-n', '--target-name',
            required=True)
        parser_remove_target.set_defaults(func=cls._remove_target)


        parser_enable_target = subparsers.add_parser(
            'enable-target',
            help='Enable a telemetry target')
        parser_enable_target.add_argument(
            '-n', '--target-name',
            required=True)
        parser_enable_target.set_defaults(func=cls._enable_target)


        parser_enable_streaming = subparsers.add_parser(
            'enable-streaming',
            help='Enable telemetry streaming')
        parser_enable_streaming.set_defaults(func=cls._enable_streaming)


        parser_disable_target = subparsers.add_parser(
            'disable-target',
            help='Disable a telemetry target')
        parser_disable_target.add_argument(
            '-n', '--target-name',
             required=True)
        parser_disable_target.set_defaults(func=cls._disable_target)


        parser_disable_streaming = subparsers.add_parser(
            'disable-streaming',
            help='Disable telemetry streaming')
        parser_disable_streaming.set_defaults(func=cls._disable_streaming)


        parser_modify_target = subparsers.add_parser(
            'modify-target',
            help='Modify a telemetry target')
        parser_modify_target.add_argument(
            '-n', '--target-name',
            required=True)
        parser_modify_target.add_argument(
            '-H', '--target-host',
            type=HostValidator(),
            metavar='<IPv4>',
            help='IPv4 address')
        parser_modify_target.add_argument(
            '-p', '--target-port',
            type=PortValidator(),
            metavar='<1-65535>',
            help='Port number',)
        parser_modify_target.add_argument(
            '-m', '--target-message-type',
            choices=[TARGET_CHOISE_EXTENDED, TARGET_CHOISE_STANDARD])
        parser_modify_target.set_defaults(func=cls._modify_target)


        parser_import_filter_file = subparsers.add_parser(
            'import-filter-file',
            help='Import a telemetry target filter file')
        parser_import_filter_file.add_argument(
            '-n', '--target-name',
            required=True)
        parser_import_filter_file.add_argument(
            '-t', '--target-filter-type',
            required=True,
            choices=[TARGET_CHOISE_COUNTERS, TARGET_CHOISE_FIELDS])
        parser_import_filter_file.add_argument(
            '-f', '--file-path',
            required=True)
        parser_import_filter_file.set_defaults(func=cls._import_filter_file)


        parser_disable_target_files = subparsers.add_parser(
            'disable-filter-file',
            help='Disable telemetry target filter file')
        parser_disable_target_files.add_argument(
            '-n', '--target-name',
            required=True)
        parser_disable_target_files.add_argument(
            '-t', '--target-filter-type',
            required=True,
            choices=[TARGET_CHOISE_COUNTERS, TARGET_CHOISE_FIELDS])
        parser_disable_target_files.set_defaults(func=cls._disable_filter_file)


        parser_set_sample_rate = subparsers.add_parser(
            'set-sample-rate',
            help='Set telemetry sample rate')
        parser_set_sample_rate.add_argument(
            '-r', '--rate',
            type=SampleRateValidator(),
            metavar='<6-86400>',
            required=True,
            help='Sample rate in seconds')
        parser_set_sample_rate.set_defaults(func=cls._set_sample_rate)

        parser_show_sample_rate = subparsers.add_parser(
            'show-sample-rate',
            help='Show telemetry sample rate')
        parser_show_sample_rate.set_defaults(func=cls._show_sample_rate)


        parser.add_argument(
            '-V', '--version',
            action='version',
            help='Print version information',
            version='1.0.0-6')

        args = parser.parse_args()
        return args

def main():
    try:
        ConfigureUfmTelemetryTarget().run()
    except KeyboardInterrupt:
        # handle keyboard interrupt
        return 0
    except Exception as e:
        print((str(e)))
        return 1


if __name__ == "__main__":
    sys.exit(main())
