#!/bin/bash -e
#set -x

script_name=$(basename "$0")
configs_folder_name=configs

function modify_key {
    file_name=$1
    key=$2
    value=$3
    sed -i "/^${key}[ ]*=/s/=.*$/=$(sed 's/\//\\\//g' <<< "${value}")/" "${file_name}"
}

function split_line_by_separator {
    line=$1
    separator=$2

    for item in $(echo "${line}" | tr "${separator}" "\\n")
    do
        echo "${item}"
    done
}

function split_line_on_key_values {
    line_to_split=$1

    split_line_by_separator "$line_to_split" ";"
}

function list_configurations {
    folders=$(find "${original_folder}/${configs_folder_name}" -mindepth 1 -maxdepth 1 -type d -printf '%f\n')
    for folder in ${folders}
    do
        echo "${folder}"
    done
}

function do_generate_service {
    # Download docker binary in order to parse container information and
    # Generate the Telemetry service file with relevant info.
    install_cmd=""
    if [ $EUID -ne 0 ]; then
        install_cmd+="sudo "
    fi
    if [ -n "$(command -v yum)" ]; then
        install_cmd+="yum "
    else
        install_cmd+="apt "
    fi
    update_cmd="${install_cmd} update -y"
    install_curl="${install_cmd} install -y curl"
    eval ${update_cmd}
    eval ${install_curl}
    curl -sfL -o docker.tgz https://download.docker.com/linux/static/stable/x86_64/docker-20.10.7.tgz && \
     tar -xzvf docker.tgz docker/docker --strip=1 --directory /usr/bin && \
     rm -f docker.tgz && cp docker /usr/bin/

    # Get the the container info
    image=$(/usr/bin/docker ps | grep "config-telemetry" | awk '{print $2}')
    telemetry_dir=$(/bin/docker inspect config-telemetry | grep config:/config | awk -F 'config' '{print $1}' | awk -F '"' '{print $2}')

    cat > /etc/systemd/system/ufm-telemetry.service << EOF
[Unit]
Description=UFM Telemetry
After=docker.service openibd.service
Wants=docker.socket
Requires=docker.socket openibd.service

[Service]
ExecStart=/usr/bin/docker run \
--net=host \
--uts=host \
--ipc=host \
--ulimit stack=67108864 \
--ulimit memlock=-1 \
--security-opt seccomp=unconfined \
--cap-add=SYS_ADMIN \
--device=/dev/infiniband/ \
-v "${telemetry_dir}/conf:/config" \
-v "${telemetry_dir}/data:/data" \
-v "${telemetry_dir}/licenses:/opt/ufm/files/licenses" \
-v "/var/log/:/opt/ufm-telemetry/logs" \
--rm \
--name ufm-telemetry \
-i ${image}

ExecStop=/usr/bin/docker stop ufm-telemetry

[Install]
WantedBy=multi-user.target

EOF
}

function print_help {
    echo "${script_name}: Tool for copying ufm-telemetry configuration"
    echo "Usage: ./${script_name} [key1=value1;key2=value2] [src_config_folder] [dst_config_folder] --list --config=predefined_config_name --apply=file_with"
    echo -e "\t key1=value1;key2=value2: semicolon-separated key=value pairs to be replaces in original configuration file, "
    echo -e "\t --apply=file allow to specify file with such pairs to be changed from"
    echo -e ""
    echo -e "\t src_config_folder: folder with original configuration files, by default: CONFIG_ORIGINAL"
    echo -e "\t dst_config_folder: folder where to put configuration files modified, by default: CONFIG"
    echo -e ""
    echo -e "\t --list: prints out all available predefined configuration"
    echo -e ""
    echo -e "\t --config=predefined_config_name: specifies which predefined configuration to use"
}

dst_config_file=launch_ibdiagnet_config.ini
unnamed_variable_idx=0
unnamed_variables=()
generate_service=0
# parse arguments:
for i in "$@"
do
case $i in
    -h|--help)
    do_print_help=1
    ;;
    --list)
    do_list=1
    ;;
    --config=*)
    config="${i#*=}"
    shift # past argument=value
    ;;
    --apply=*)
    apply_file="${i#*=}"
    shift # past argument=value
    ;;
    --dst_config_file=*)
    dst_config_file="${i#*=}"
    shift # past argument=value
    ;;
    --gen_service)
    generate_service=1
    shift
    ;;
    -*)
    echo "Unknown option: '$i', Please use --help for valid options list"
    exit 1
    ;;

    *)
    # unknown option
    unnamed_variables[${unnamed_variable_idx}]=$i
    unnamed_variable_idx=$((unnamed_variable_idx + 1))
    ;;
esac
done

parameter_line=${unnamed_variables[0]:-}
original_folder=${unnamed_variables[1]:-"/telemetry.config_original"}
target_folder=${unnamed_variables[2]:-"/telemetry.config"}

temp_dir=$(mktemp -d -t telemetry-config-XXXXXXXXX)
dst_config_file=${temp_dir}/${dst_config_file}


if [ -v do_print_help ] && [ "${do_print_help}" -eq 1 ]; then
    print_help
    exit 0
fi

if [ -v do_list ] && [ "${do_list}" -eq 1 ]; then
    list_configurations
    exit 0
fi

if [ -v config ] && [ ! -d "${original_folder}/${configs_folder_name}/${config}" ]; then
    echo "Unable to find configuration dir for ${config}"
    exit 1
fi

if [ -v apply_file ]; then
    if [ -f "${apply_file}" ]; then
        change_lines=$(tr '\n' ';' < "${apply_file}" )
        parameter_line="${change_lines}"
    else
        echo "Error: unable apply file ${apply_file} : file doesn't exist"
        exit 1
    fi
fi

if [ -d "${target_folder}" ]; then
    cd "${target_folder}"
    files_to_copy=$(find "${original_folder}" -mindepth 1 -maxdepth 1 ! '(' -name configs ')' -print)
    for file_to_copy in ${files_to_copy}
    do
        cp -r "${file_to_copy}" "${temp_dir}"
    done

    if [ -n "${config}" ]; then
        cp -r "${original_folder}/${configs_folder_name}/${config}"/* "${temp_dir}"
    fi

    for key_value_pair in $(split_line_on_key_values "${parameter_line}"); do
        echo "key_value_pair: ${key_value_pair}"
        set -- $(echo ${key_value_pair} | tr '=' ' ')
        key=$1
        value=$2
        echo "key: ${key}"
        echo "value: ${value}"

        modify_key "${dst_config_file}" "${key}" "${value}"
    done
    cp -r "${temp_dir}"/* "${target_folder}"
    rm -rf "${temp_dir}"

else
    echo "Error: target folder ${target_folder} is not directory"
    exit 1
fi

if [[ ${generate_service} -eq 1 ]]; then
    do_generate_service
fi

exit 0

