"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import re
from services.modules.base_module import BaseModule
from services.modules.run_command.command_manager import CommandManager


class SystemService(BaseModule):
    """
    System service class
    """

    def get_systemctl_list_units_services(self) -> (list, list):
        """
        get systemctl list units services
        :return: tuple of list of dict the first are the failed data and the next are the rest
        """
        res = CommandManager.run_command(connection=self.connection, command='systemctl list-units --type=service')
        # output example :
        #  UNIT                                                     LOAD   ACTIVE SUB     DESCRIPTION
        #   accounts-daemon.service                                  loaded active running Accounts Service
        #   finalrd.service                                          loaded active exited  Create final runtime dir for shutdown pivot root
        # ● fwupd-refresh.service                                    loaded failed failed  Refresh fwupd metadata and update motd
        #   getty@tty1.service                                       loaded active running Getty on tty1
        data: list = []
        failed_data: list = []
        if res.rc == 0:
            if re.search(r'\s*UNIT\s+LOAD\s+ACTIVE\s+SUB\s+DESCRIPTION\s*', res.output):
                for raw in res.output.splitlines():
                    if re.search(r'\s*UNIT\s+LOAD\s+ACTIVE\s+SUB\s+DESCRIPTION\s*', raw):
                        continue
                    match = re.search(r'\s*●*\s*(\S+.service)\s+(\S+)\s+(\S+)\s+(\S+)\s+(.*)', raw)
                    if match:
                        if 'fail' in match.group(3) or 'fail' in match.group(4):
                            failed_data.append(
                                {"UNIT": match.group(1), "LOAD": match.group(2), "ACTIVE": match.group(3),
                                 "SUB": match.group(4), "DESCRIPTION": match.group(5).strip()})
                        else:
                            data.append({"UNIT": match.group(1), "LOAD": match.group(2), "ACTIVE": match.group(3),
                                         "SUB": match.group(4), "DESCRIPTION": match.group(5).strip()})
            if not failed_data and not data:
                res.update_logger_fail_to_parse_output()
        return failed_data, data

    def get_systemctl_service_status(self, service_name: str) -> str:
        """
        get systemctl service status
        :return: return all the status of service
        """
        res = CommandManager.run_command(connection=self.connection, command=f'systemctl status {service_name}')
        if not res.error or res.rc == 0:
            # output example
            ## systemctl status mlx-regex.service
            # ● mlx-regex.service - Regex daemon for BlueField 2
            #      Loaded: loaded (/etc/systemd/system/mlx-regex.service; enabled; vendor preset: enabled)
            #      Active: failed (Result: exit-code) since Mon 2021-11-22 08:01:46 UTC; 1 weeks 6 days ago
            #     Process: 2896 ExecStart=/usr/bin/mlx-regex (code=exited, status=244)
            #    Main PID: 2896 (code=exited, status=244)
            #
            # Nov 22 08:01:45 localhost systemd[1]: Started Regex daemon for BlueField 2.
            # Nov 22 08:01:45 localhost regex[2896]: Failed to register memory!
            # Nov 22 08:01:45 localhost regex[2896]: Registration failed.
            # Nov 22 08:01:45 localhost regex[2896]: Please make sure huge pages in the system
            # Nov 22 08:01:45 localhost regex[2896]: Hint: cat /proc/meminfo
            # Nov 22 08:01:45 localhost regex[2896]:       echo NUM_PAGES > /proc/sys/vm/nr_hugepages
            # Nov 22 08:01:46 localhost systemd[1]: mlx-regex.service: Main process exited, code=exited, status=244/n/a
            # Nov 22 08:01:46 localhost systemd[1]: mlx-regex.service: Failed with result 'exit-code'.
            return res.output
        return ''
