"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import re
from services.modules.base_module import BaseModule
from services.modules.run_command.command_manager import CommandManager


class General(BaseModule):
    """
    general class
    """
    def get_host_name(self) -> str:
        """
        get host name
        :return:
        """
        hostname = ''
        res = CommandManager.run_command(connection=self.connection, command='hostname')
        if res.rc == 0:
            match = re.search(r'(\S+)', res.output)
            if match:
                hostname = match.group(1)
            if not hostname:
                res.update_logger_fail_to_parse_output()
        return hostname

    def get_os_name(self) -> str:
        """
        show os name
        Returns:
            string: os name
        """
        os_name = ''
        res = CommandManager.run_command(connection=self.connection, command='cat /etc/*release')
        if res.rc == 0:
            result = re.search('^NAME=(.*)', res.output, re.MULTILINE)
            if not result:
                result = re.search(r'(CentOs\s+release.*|Red\s+hat.*)', res.output, re.IGNORECASE)
            if result:
                os_name = result.group(1)
            if not os_name:
                res.update_logger_fail_to_parse_output()
        return os_name

    def get_os_version(self) -> str:
        """
        Get os version
        Returns:
            string: os version
        """
        osversion = ''
        res = CommandManager.run_command(connection=self.connection, command='cat /etc/*release')
        if res.rc == 0 and res.output:
            output = res.output
            result = re.search('VERSION_ID=(.*)', output)
            if not result:
                result = re.search(r'[CentOs|Red\s+hat].*\s+(\d+\.?\d*)', output, re.IGNORECASE)
            if result:
                osversion = result.group(1)
            if not osversion:
                res.update_logger_fail_to_parse_output()
        return osversion

    def get_kernel_version(self) -> str:
        """
        Get os version
        Returns:
            string: kernel version
        """
        kernel_version = ''
        res = CommandManager.run_command(connection=self.connection, command='uname -v')
        if res.rc == 0:
            match = re.search(r'(.+)', res.output)
            if match:
                kernel_version = match.group(1)
            if not kernel_version:
                res.update_logger_fail_to_parse_output()
        return kernel_version

    def get_kernel_release(self) -> str:
        """
        Get os version
        Returns:
            string: kernel release
        """
        kernel_release = ''
        res = CommandManager.run_command(connection=self.connection, command='uname -r')
        if res.rc == 0:
            match = re.search(r'(\S+)', res.output)
            if match:
                kernel_release = match.group(1)
            if not kernel_release:
                res.update_logger_fail_to_parse_output()
        return kernel_release

    def get_architecture(self) -> str:
        """
        Get architecture
        Returns:
            string: architecture
        """
        architecture = ''
        res = CommandManager.run_command(connection=self.connection, command='uname -p')
        if res.rc == 0:
            match = re.search(r'(\S+)', res.output)
            if match:
                architecture = match.group(1)
            if not architecture:
                res.update_logger_fail_to_parse_output()
        return architecture

    def get_board_id_part_serial_number(self):
        """
        get board id part and serial number
        :return:
        """
        data = {'Board Id': '',
                'Part Number': '',
                'Serial Number': ''}
        all_matched:int = 3
        buses = self.get_dpu_port_buses()
        if buses:
            res = CommandManager.run_command(connection=self.connection, command=f'lspci -vvv -s {buses[0]}')
            if res.rc == 0:
                #output example
                # #Capabilities: [48] Vital Product Data
                #         Product Name: BlueField-2 E-Series DPU, 200GbE/HDR single-port QSFP56, Secure Boot Enabled, Crypto Enabled, 16GB on-board DDR, 1GbE OOB management, HHHL
                #         Read-only fields:
                #                 [PN] Part number: MBF2M345A-HECOT
                #                 [EC] Engineering changes: A2
                #                 [V2] Vendor specific: MBF2M345A-HECOT
                #                 [SN] Serial number: MT2130X18208
                #                 [V3] Vendor specific: be2a885fadfdeb11800008c0eb14cb50
                #                 [VA] Vendor specific: MLX:MN=MLNX:CSKU=V2:UUID=V3:PCI=V0:MODL=BF2M345A
                #                 [V0] Vendor specific: PCIeGen4 x16
                #                 [RV] Reserved: checksum good, 1 byte(s) reserved
                for raw in res.output.splitlines():
                    if "Product Name:" in raw:
                        match = re.search(r'Product Name:\s+(.*)', raw, re.IGNORECASE)
                        if match:
                            data['Board Id'] = match.group(1)
                            all_matched -= 1
                    elif "Part number:" in raw:
                        match = re.search(r'Part number:\s+(\S+)', raw, re.IGNORECASE)
                        if match:
                            data['Part Number'] = match.group(1)
                            all_matched -= 1
                    elif "Serial number:" in raw:
                        match = re.search(r'Serial number:\s+(\S+)', raw, re.IGNORECASE)
                        if match:
                            data['Serial Number'] = match.group(1)
                            all_matched -= 1
        if all_matched != 0:
            mst_device = self.get_mst_device()
            res = CommandManager.run_command(connection=self.connection, command=f'mlxburn -d {mst_device} -vpd')
            if res.rc == 0:
                # mlxburn -d /dev/mst/mt41686_pciconf0 -vpd

                #   VPD-KEYWORD    DESCRIPTION             VALUE
                #   -----------    -----------             -----
                # Read Only Section:
                #
                #   PN             Part Number             MBF2M516A-CECOT
                #   EC             Revision                B2
                #   V2             N/A                     MBF2M516A-CECOT
                #   SN             Serial Number           MT2128X10544
                #   V3             N/A                     2c7f1c47c3e5eb118000b8cef6e99756
                #   VA             N/A                     MLX:MN=MLNX:CSKU=V2:UUID=V3:PCI=V0:MODL=BF2M516A
                #   V0             Misc Info               PCIeGen4 x16
                #   RV             Checksum Complement     0x61
                #   IDTAG          Board Id                BlueField-2 DPU 100GbE Dual-Port QSFP56, Crypto and Secure Boot Enabled, 16GB on-board DDR, 1GbE OOB management, Tall Bracket
                #                 #
                for raw in res.output.splitlines():
                    if 'Serial Number' in raw:
                        match = re.search(r'Serial Number\s+(\S+)', raw, re.IGNORECASE)
                        if match:
                            data['Serial Number'] = match.group(1)
                    elif 'Part Number' in raw:
                        match = re.search(r'Part Number\s+(\S+)', raw, re.IGNORECASE)
                        if match:
                            data['Part Number'] = match.group(1)
                    elif 'Board Id' in raw:
                        match = re.search(r'Board Id\s+(.*)', raw, re.IGNORECASE)
                        if match:
                            data['Board Id'] = match.group(1)
                for d_v in data.values():
                    if not d_v:
                        res.update_logger_fail_to_parse_output()
                        break
        return data

    def get_drivername_and_version(self):
        """
        get driver name and version
        :return: driver name and version
        """
        driver_info = ''
        res = CommandManager.run_command(connection=self.connection, command='ofed_info -s')
        if res.rc == 0:
            # output example : MLNX_OFED_LINUX-23.04-0.5.0.0:
            math = re.search(r'(\S+)', res.output)
            if math:
                driver_info = math.group(1).rstrip(":")
            if not driver_info:
                res.update_logger_fail_to_parse_output()
        return driver_info

    def get_mst_device(self):
        """
        get mst device
        :return: mst device
        """
        mst_device = ''
        res = CommandManager.run_command(connection=self.connection, command='mst status |grep -i "/dev/mst/"')
        if res.rc == 0:
            match = re.search(r'(\S+)', res.output)
            if match:
                mst_device = match.group(1)
            if not mst_device:
                res.update_logger_fail_to_parse_output()
        return mst_device

    def get_temperature_of_dpu(self):
        """
        get temperature of dpu
        :return: tuple of temperature and unit
        """
        temperature = ''
        res = CommandManager.run_command(connection=self.connection, command=f'mget_temp -d {self.get_mst_device()}')
        if res.rc == 0:
            match = re.search(r'(\d+)', res.output)
            if match:
                temperature = int(match.group(1))
            if not temperature:
                res.update_logger_fail_to_parse_output()
        return (temperature, '°C')

    def get_dpu_port_buses(self):
        """
        get dpu buses
        :return: list of buses
        """
        dpu_port_buses: list = []
        cmd = 'lspci -D |grep -i Mellanox | egrep -i "(Ethernet|Infiniband|Network)" | grep -iv Virtual'
        result = CommandManager.run_command(connection=self.connection,
                                         command=cmd)
        if result.rc == 0:
            match = re.findall(r'([0-9a-f]{4}:[0-9a-f]{2}:[0-9a-f]{2}.[0-9a-f]+)\s+', result.output)
            for bus in match:
                dpu_port_buses.append(bus)
            if not dpu_port_buses:
                result.update_logger_fail_to_parse_output()
        return dpu_port_buses

    def get_dpu_physical_port_interfaces(self):
        """
        get dpu buses
        :return: list of buses
        """
        dpu_pfs: list = []
        buses = self.get_dpu_port_buses()
        for bus in buses:
            interface = self.show_bus_interface(bus)
            if interface:
                dpu_pfs.append(interface)
        return dpu_pfs

    def show_bus_interface(self, bus) -> str:
        """
        Display each bus number related interface
        Args:
            bus (str): bus number

        Returns:
                interface
        """
        bus = bus.replace(r':', r'\:')
        result = CommandManager.run_command(connection=self.connection,command="ls /sys/bus/pci/devices/" + bus + "/net/")
        expected_interface = ""
        if result.rc == 0 and result.output:
            # example of output:
            # ls   /sys/bus/pci/devices/0000\:00\:06.0/net/
            # ens6  ens6d1
            interfaces = re.findall(r'\w+', result.output)
            for interface in interfaces:
                result = CommandManager.run_command(connection=self.connection,
                                                    command=f"ethtool -i {interface}")
                if result.rc == 0:
                    match = re.search(r'driver:\s*(mlx\d+_(core|en|ib))', result.output)
                    if match:
                        expected_interface = interface
            if not expected_interface:
                result.update_logger_fail_to_parse_output()
        return expected_interface

    def show_lifecycle_state(self):
        """
        show lifecycle state
        :return: dict of data
        """
        data = {}
        result = CommandManager.run_command(connection=self.connection,
                                            command="mlxbf-bootctl | grep lifecycle")
        if result.rc == 0 and result.output:
            #output example :
            #lifecycle state: GA Secured
            match = re.search(r'lifecycle\s+state:(.*)',result.output)
            if match:
                data['Lifecycle state'] = match.group(1)
            if not data:
                result.update_logger_fail_to_parse_output()
        return data

    def show_secure_boot_state(self):
        """
        show lifecycle state
        :return: dict of data
        """
        data = {}
        result = CommandManager.run_command(connection=self.connection,
                                            command="mokutil --sb-state")
        if result.rc == 0 and result.output:
            # output example :
            # SecureBoot enabled
            match = re.search(r'SecureBoot\s+(.*)', result.output)
            if match:
                data['SecureBoot'] = match.group(1)
            if not data:
                result.update_logger_fail_to_parse_output()
        return data

    def get_oob_interface(self):
        """
        get oob interface
        :return: interface
        """
        oob_interface: str = ''
        cmd = 'ls /sys/class/net/'
        result = CommandManager.run_command(connection=self.connection,
                                            command=cmd)
        if result.rc == 0:
            infs = result.output.split()
            for inf in infs:
                result = CommandManager.run_command(connection=self.connection,
                                                    command=f'ethtool -i {inf}')
                if result.rc == 0 and 'mlxbf_gige' in result.output:
                    oob_interface = inf
                    break
            if not oob_interface:
                result.update_logger_fail_to_parse_output()
        return oob_interface

    def get_doca_version(self):
        """
        get doca version
        :return: doca version
        """
        doca_version = ''
        result = CommandManager.run_command(connection=self.connection,
                                            command="cat /etc/mlnx-release")
        if result.rc == 0:
            # output example:
            # # cat /etc/mlnx-release
            # DOCA_v1.1_BlueField_OS_Ubuntu_20.04-5.4.0-1013-bluefield-5.4-1.0.3.0-3.7.0.11805-1.signed-aarch64
            # the new convention: #DOCA_1.3.0_BSP_3.9.0_Ubuntu_20.04-6
            if 'DOCA_v' in result.output:
                match = re.search(r'DOCA_v(\S+)_B',result.output,re.IGNORECASE)
            else:
                match = re.search(r'DOCA_(\S+)_BSP', result.output, re.IGNORECASE)
            if match:
                doca_version = match.group(1)
            if not doca_version:
                result.update_logger_fail_to_parse_output()
        return doca_version

    def get_bluefield_os_version(self):
        """
        get doca version
        :return: doca version
        """
        bluefield_os_version = ''
        result = CommandManager.run_command(connection=self.connection,
                                            command="cat /etc/mlnx-release")
        if result.rc == 0:
            # output example:
            # # cat /etc/mlnx-release
            # DOCA_v1.1_BlueField_OS_Ubuntu_20.04-5.4.0-1013-bluefield-5.4-1.0.3.0-3.7.0.11805-1.signed-aarch64
            # the new convention: #DOCA_1.3.0_BSP_3.9.0_Ubuntu_20.04-6
            # DOCA_1.9.5_BSP_4.0.bu1_Ubuntu_22.04-69.20221213
            if 'DOCA_v' in result.output:
                match = re.search(r'bluefield\S+-((\d+|\d+.){3}).\d{3,}-\d+.', result.output, re.IGNORECASE)
            else:
                match = re.search(r'BSP_(\d+.[a-z,A-Z,0-9,.]+)_', result.output, re.IGNORECASE)
            if match:
                bluefield_os_version = match.group(1)
            if not bluefield_os_version:
                result.update_logger_fail_to_parse_output()
        return bluefield_os_version

    def is_ubuntu_family(self) -> bool:
        """
        verify if os is ubuntu family
        """
        res = self.get_os_name()
        ubuntu_family = ['ubuntu','debian']
        if res and res.lower().strip('"') in ubuntu_family:
            return True
        return False

    def is_redhat_family(self)-> bool:
        """
        verify if os is redhat family
        """
        res = self.get_os_name()
        redhat_family = ['rh', 'centos','oracle','fedora']
        if res and res.lower().strip('"') in redhat_family:
            return True
        return False

    def get_total_power_consumption(self):
        """
        get total power consumption
        :return: tuple of power and unit
        """
        power = ''
        res = CommandManager.run_command(connection=self.connection,
                                         command=f'cat /sys/kernel/debug/mlxbf-ptm/monitors/status/total_power')
        if res.rc == 0:
            match = re.search(r'(\d+)', res.output)
            if match:
                power = int(match.group(1))
            if not power:
                res.update_logger_fail_to_parse_output()
        return power, 'W'