"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import subprocess


from services.modules.run_command.command_result import CommandResult

class CommandManager:
    """
    Command Manager class
    """
    @classmethod
    def run_command(cls, command, connection=None, timeout=30, sudo_required: bool = False) -> CommandResult:
        """
        run command in case no connection run on local
        """
        if connection:
            from services.dpe.client.dpe_client import DpeClient
            if isinstance(connection, DpeClient):
                return connection.run_command(cmd=command, sudo_required=sudo_required,timeout=timeout)
            else:
                raise Exception(f'the following connection type not supported {type(connection)}')
        else:
            return cls.__run_subprocess_command(command=command, timeout=timeout, sudo_required=sudo_required)

    @classmethod
    def __run_subprocess_command(cls, command, timeout=30, sudo_required: bool = False):
        """
        run command with default time out 30 sec
        :param command: command
        :param timeout: timeout in sec
        :return: command result class
        """
        timeout_expired = False
        p = subprocess.Popen(command, shell=True,
                             stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE,
                             universal_newlines=True,
                             stdin=subprocess.PIPE)
        try:
            std_out, std_err = p.communicate(timeout=timeout)
            rc = p.returncode
        except Exception as exp:
            p.kill()
            std_out = ''
            std_err = f'failed to run command {command} with error {exp}'
            rc = 126
            timeout_expired = True
        finally:
            return CommandResult(std_out, std_err, rc, command, timeout_expired)
