"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
import re


from services.modules.base_module import BaseModule
from services.modules.run_command.command_manager import CommandManager
from services.modules.system.general.general import General


class Firmware(BaseModule):
    """
    Firmware class
    """
    def get_mlxconfig_default_current_and_nextboot_info(self, device: str = ''):
        """
        get mlxconfig default current and next boot info of first port if not defined specific device
        :param device: optional params device
        :return: dict of info
        """
        data = {}
        was_none_default = False
        if device:
            mst_device = device
        else:# took the first mst device
            mst_device = General(self.connection).get_mst_device()
        if mst_device:
            result = CommandManager.run_command(connection=self.connection, command='mlxconfig -d {} -e q'.format(mst_device),
                                                timeout=60)
            # Configurations:                              Default         Current         Next Boot
            # MEMIC_BAR_SIZE                      0               0               0
            # MEMIC_SIZE_LIMIT                    _256KB(1)       _256KB(1)       _256KB(1)
            # HOST_CHAINING_MODE                  DISABLED(0)     DISABLED(0)     DISABLED(0)
            # HOST_CHAINING_CACHE_DISABLE         False(0)        False(0)        False(0)

            if result.rc == 0:
                catch_configuration = False
                for raw in result.output.splitlines():
                    if 'Configurations:' in raw:
                        catch_configuration = True
                        continue
                    if catch_configuration:
                        raw = re.sub(r'^\s*RO\s', ' ', raw)
                        match = re.match(r'^\**\s+(\S+)\s+(\S+)\s+(\S+)\s+(\S+)', raw)
                        if match:
                            status = ''
                            if match.group(2).strip() != match.group(3).strip() or match.group(
                                    3).strip() != match.group(4).strip():
                                status = 'CHANGED'
                            data[match.group(1).strip()] = [match.group(2).strip(), match.group(3).strip(),
                                                            match.group(4).strip(), status]
                if not data:
                    result.update_logger_fail_to_parse_output()
        return data

    def get_mlxconfig_fields_info(self):
        """
        get mlxconfig fields info
        :return:
        """
        data = {}
        mst_device = General(self.connection).get_mst_device()
        if mst_device:
            result = CommandManager.run_command(connection=self.connection,
                                                command='mlxconfig -d {} i'.format(mst_device))
            if result.rc == 0:
                field = ''
                for raw in result.output.splitlines():

                    match = re.match(r'\s*(\S+)=<(\S+)>\s*(.*)', raw)
                    if match:
                        field = match.group(1)
                        desc = match.group(3)
                        if desc:
                            data[field] = {'parameters': match.group(2), 'description': f'{desc}\n'}
                        else:
                            data[field] = {'parameters': match.group(2), 'description': ''}
                    elif field:
                        match = re.search(r'(\S+):$', raw.strip())
                        if match:
                            continue
                        match = re.match(r'(.*)', raw)
                        if match:
                            if data[field]['description'] and match.group(1).strip():
                                data[field]['description'] += f'{match.group(1).strip()}\n'
                if not data:
                    result.update_logger_fail_to_parse_output()
        return data
