"""
Copyright © 2019-2022 NVIDIA CORPORATION & AFFILIATES. ALL RIGHTS RESERVED.
 
This software product is a proprietary product of Nvidia Corporation and its affiliates
(the "Company") and all right, title, and interest in and to the software
product, including all associated intellectual property rights, are and
shall remain exclusively with the Company.

This software product is governed by the End User License Agreement
provided with the software product.
"""
from services.modules.run_command.command_manager import CommandManager

from services.modules.base_module import BaseModule
import json
import re
import urllib.request

class DocaTelemetryService(BaseModule):
    """
    DOCA Telemetry Service class
    """
    def get_doca_telemetry_service_counters_data(self):
        """
        get doca telemetry service counters data
        :return: dict of info
        """
        data: dict = {}
        try:
            result = self._simple_get_request(url='http://0.0.0.0:9100/json/metrics')
            if not result:
                return data
            counters_datas = json.loads(result)
            schemas = self.get_schemas_file_path()
            data_from_schema, counters_desc = self.parse_schemas(schemas)
            return self.match_between_schema_and_json_counter_data(counter_json=counters_datas,counters_desc=counters_desc,
                                                                   data_from_schema=data_from_schema)
        except Exception as exp:
            self.logger.error(exp)
            return data

    def _simple_get_request(self, url, timeout=30):
        try:
            req = urllib.request.Request(url)
            with urllib.request.urlopen(req, timeout=timeout) as response:
                content = response.read()
                return content
        except Exception as e:
            self.logger.warning(f"failed to get data from url: {url} : {str(e)}")

        return None

    def get_schemas_file_path(self) -> list:
        """
        get schemas file path
        :return: list of paths
        """
        schemas = []
        if self.connection:
            path_to_schema = '/data/schema/'
        else:
            path_to_schema = '/opt/mellanox/doca/services/telemetry/data/schema/'

        result = CommandManager.run_command(command=f'ls {path_to_schema}')
        if result.rc != 0:
            return schemas
        if result.output:
            schemas = result.output.split()
            schemas = [path_to_schema+sub for sub in schemas]
        return schemas

    def parse_schemas(self,schemas_paths:list):
        """
        parse schemas
        returns: tuple of dict data_from_schema ,counters_desc
        """
        data_from_schema = {}
        counters_desc = {}
        try:
            for schema in schemas_paths:
                f = open(schema)
                schema_data = json.load(f)
                f.close()
                if 'counter_groups' in schema_data.keys():
                    for cg in schema_data.get('counter_groups'):  # list of counter groups
                        provider = ''
                        if 'provider' in cg.keys() and 'name' in cg.keys():
                            if cg.get('provider') != cg.get('name'):
                                provider = f'{cg.get("provider")}_{cg.get("name")}'
                            else:
                                provider = cg.get("provider")
                        if provider:
                            data_from_schema.update({provider: {}})
                            for counter in cg.get('counters'):  # list of counters
                                counter_name = counter.get('name')
                                counter_description = counter.get('description')
                                if counter_name and counter_description:
                                    data_from_schema[provider].update({counter_name: counter_description})
                elif 'provider_name' in schema_data.keys():
                    provider = schema_data.get('provider_name')
                    data_from_schema.update({provider: {}})
                    for type in schema_data.get('types'):  # list of types
                        for field in type.get('fields'):  # list of fields
                            counter_name = field.get('field_name')
                            counter_description = field.get('description')
                            if counter_name and counter_description:
                                data_from_schema[provider].update({counter_name: counter_description})

                for data_f_s in data_from_schema:
                    for cn, cv in data_from_schema.get(data_f_s).items():
                        counters_desc.update({cn: cv})
            return data_from_schema ,counters_desc
        except Exception as exp:
            self.logger.error(f'Failed to parse schemas: schemas path {schemas_paths}, {exp}')
            return data_from_schema ,counters_desc

    def match_between_schema_and_json_counter_data(self,counter_json,counters_desc,data_from_schema):
        """
        match between schema and json counter data
        return : dict of result_counter_data
        """
        result_counter_data = {}
        try:
            # merge rate provider data with the expected provider
            rate_data = {}
            rate_data_key_to_delete = []
            for pr_name, pr_value in data_from_schema.items():
                if pr_name.endswith('_rate'):
                    rate_data.update(pr_value)
                    rate_data_key_to_delete.append(pr_name)
            for rate_key in rate_data_key_to_delete:
                data_from_schema.pop(rate_key)
            for rate_k, rate_v, in rate_data.items():
                for pr_name, pr_value in data_from_schema.items():
                    counter_n = rate_k.replace('_rate', '')
                    if pr_value.get(counter_n):
                        pr_value.update({rate_k: rate_v})
                        break

            for counters_data in counter_json:
                if 'data_type' not in counters_data:
                    continue
                if 'device_name' in counters_data.keys():
                    data_list = []
                    data_dict = {}
                    saved_field = ['device_name', 'source_id', 'timestamp', 'data_type']
                    device_name = counters_data.get('device_name')
                    if not device_name or device_name == '':
                        continue
                    source_id = counters_data.get('source_id')
                    timestamp = counters_data.get('timestamp')
                    data_type = counters_data.get('data_type')
                    provider_name = ''
                    for cn, cv in counters_data.items():
                        if cn in saved_field:
                            continue
                        c_desc = counters_desc.get(cn)
                        if c_desc:
                            if not provider_name:
                                for provider, schema_c_data in data_from_schema.items():
                                    if schema_c_data.get(cn):
                                        provider_name = provider
                                        break
                            data_list.append({'counter_name': cn, 'counter_value': cv, 'counter_description': c_desc})
                    data_dict.update({'device_name': device_name, 'source_id': source_id,
                                      'timestamp': timestamp, 'data_type': data_type, 'counters': data_list})
                    if not result_counter_data.get(provider_name):
                        result_counter_data.update({provider_name: []})
                    result_counter_data.get(provider_name).append(data_dict)
                else:  # in case of counters data type with no device
                    saved_field = ['source_id', 'timestamp', 'data_type']
                    source_id = counters_data.get('source_id')
                    timestamp = counters_data.get('timestamp')
                    data_type = counters_data.get('data_type')
                    if data_type.startswith('tc_'):
                        continue
                    for cn, cv in counters_data.items():
                        if cn in saved_field:
                            continue
                        c_desc = counters_desc.get(cn)
                        if not c_desc:
                            c_desc = ''
                        sections = cn.split(':')
                        device_name = ''
                        counter_name = ''
                        if 'mr_cache' in sections:
                            device_name = sections[0]
                            del sections[0]
                            counter_name = '_'.join(sections)
                        elif re.search(r'(mlx\d+_\d+:\d+)', cn):
                            device_name = '_'.join(sections[0:2])
                            del sections[0:2]
                            counter_name = '_'.join(sections)
                        else:
                            counter_name = sections[-1]
                            del sections[-1]
                            device_name = '_'.join(sections)
                        for provider, schema_c_data in data_from_schema.items():
                            if schema_c_data.get(cn):
                                provider_name = provider
                                if not result_counter_data.get(provider_name):
                                    result_counter_data.update({provider_name: []})

                                matched = False
                                for device in result_counter_data.get(provider_name):
                                    if device.get('device_name'):
                                        if device.get('device_name') == device_name:
                                            matched = True
                                            break
                                if not matched:
                                    result_counter_data.get(provider_name).append(
                                        {'device_name': device_name, 'source_id': source_id,
                                         'timestamp': timestamp, 'data_type': data_type, 'counters': []})
                                for device in result_counter_data.get(provider_name):
                                    if device.get('device_name') == device_name:
                                        device.get('counters').append(
                                            {'counter_name': counter_name, 'counter_value': cv,
                                             'counter_description': c_desc})
                                        break
                                break
            return result_counter_data
        except Exception as exp:
            self.logger.error(f'failed to match between schema and json counter data: {exp} ')
            return result_counter_data